/*********************************************************************
 * libpp_vsc: mode.c
 *
 * contains mode and frequency/sync related functions
 *
 ********************************************************************/

#include <sys/ioctl.h>
#include <sys/types.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <pp/base.h>
#include <pp/cfg.h>
#include <pp/grab.h>
#include <liberic_pthread.h>
#include <pp/rfb.h>
#include <pp/vsc.h>
#include <lara.h>
#include <pp/intl.h>

#include "adc.h"
#include "debug.h"
#include "mode.h"
#include "mode_intern.h"
#include "modelist.h"
#include "timer.h"
#include "vsc_funcs.h"
#include "vsc_io.h"
#include "main.h"

typedef struct {
    u_int port;
    u_int mode_cnt;
    vsc_mode_data_t mode_data[0];
} port_mode_data_t;

typedef struct {
    u_int port_cnt;
    port_mode_data_t port_mode_data[0];
} mode_data_pdu_t;

static const unsigned long MIN_MODE_TOLERANCE_H	= 10;  // +- 100Hz
static const unsigned long MIN_MODE_TOLERANCE_V	= 1;   // +- 0.1Hz
static const unsigned long MAX_MODE_TOLERANCE_H	= 200; // +- 2Khz
static const unsigned long MAX_MODE_TOLERANCE_V	= 20;  // +- 2Hz
static const unsigned long MODE_TOLERANCE_STEP_H= 10;
static const unsigned long MODE_TOLERANCE_STEP_V= 1;

static const unsigned char SYNC_DELTA_H		= 5;  // %
static const unsigned char SYNC_DELTA_V		= 5;  // %
static const unsigned int  SYNC_SET_WAIT_US	= 10000;
static int vsc_clock; // internal vsc clock, used for mode detection

#define DONT_IGNORE_POLARITY			0
#define IGNORE_POLARITY				1

#define dontUSE_TABLE_LOOKUP			1

static const u_int32_t mode_db_ver		= 0x00000001;
static const char* mode_file_prefix		= "/etc/vsc/modedata/vsc";

/**
 * (kvm port specific) mode data, shared among all contexts (video links)
 */
static common_data_t  *common_data;
static pthread_mutex_t common_data_mtx = PTHREAD_ERRORCHECK_MUTEX_INITIALIZER_NP;
static unsigned short  common_data_max = 0;

/* common, operating on shared data */
static int mode_init_port(int kvm_port);
static int mode_init_data(int kvm_port);
static void clear_custom_modes(int kvm_port);
static int load_custom_modes(int kvm_port, u_int32_t certain_id);
static int load_mode_data(int kvm_port);
static int save_mode_data(int kvm_port);

/* misc */
static u_int32_t mode_build_id(u_int16_t freqH, u_int16_t freqV,
			       u_int8_t polH, u_int8_t polV,
			       u_int8_t custom);
static void mode_to_data(vsc_mode_t *mode, vsc_mode_data_t *data);
static void data_to_mode(vsc_mode_data_t *data, vsc_mode_t *mode);
static custom_mode_handling_t get_custom_mode_handling(void);

#ifdef USE_TABLE_LOOKUP
static vsc_mode_t* search_mode(vsc_context_t *context, u_long tol_h,
			       u_long tol_v,
			       custom_mode_handling_t custom_handling,
			       u_char ignore_polarity);
#endif


int
mode_init()
{
    int ret = -1;
    
    common_data = (common_data_t*) malloc(sizeof(common_data_t));
    
    memset(common_data, 0x00, sizeof(common_data_t));

    /* get vsc clock */
    /* gets the system or bus clock - same for all VSC; so use the first one*/
    if (ioctl(pp_grab_fd(0), PPIOCVSCGETCLOCK, &vsc_clock) != 0) {
	D(D_ERROR, "Getting VSC clock failed\n");
	goto bail;
    }
    D(D_VERBOSE, "VSC internal clock is %d HZ\n", vsc_clock);
    if (vsc_clock == 0) {
	D(D_ERROR, "VSC internal clock is 0, wrong u-boot?\n");
    }   
   
    invalidate_custom_modes();

    ret = 0;
    
 bail:
    return ret;
}

void
mode_cleanup()
{
    int i;

    MUTEX_LOCK(&common_data_mtx);
    for (i=0; i<= common_data_max; i++) {
	pp_hash_delete(common_data[i].mode_hash);
    }

    free(common_data);
    MUTEX_UNLOCK(&common_data_mtx);
}

/**
 * check the current sync signals on ADC input,
 * look if they change
 */
sync_detect_result_t
sync_detection(vsc_context_t *context)
{
    sync_signals_t signals;
    pp_bool_t h, v;
   
#ifdef PP_FEAT_VSC_PANEL_INPUT
    vsc_syncs_on(context, &h, &v);
#else
    h = adc_hsync_on(context);
    v = adc_vsync_on(context);
#endif
   
    /* hardware workaround
       We have a buffer between the sync input and the ADC which cleans the sync signals.
       Before the buffer there is a pullup, which somehow leads to a DC offset of about
       1.4 V. Combined with some glitches from the hsync line the buffer expands the glitches
       to a fine, clean hsync-like signal on the vsync line, which the ADC eventually detects.
       Can be fixed by removing the pullups on new devices, for now (old devices) a workaround
       is the sun_mode, we just ignore vsync here */

    if (context->sun_mode == PP_TRUE) {
	v = PP_FALSE;
    }
    
    if (h && v) {
	signals = HV;
    } else if (h) {
	signals = H;
    } else if (v) {
	signals = V;
    } else {
	signals = NONE;
    }    
    
    if (signals != context->input_sync_signals) {
	D(D_VERBOSE, "Detected new external sync input %d (was %d)\n", signals, context->input_sync_signals);
	context->input_sync_signals = signals;
	return SYNC_CHANGED;
    } else {
	return SYNC_NO_CHANGE;
    }
}

/**
 * configure sync separator according to the
 * measured signals (we use the separator in ADC
 * when a composite signal is used)
 */
#if !defined(PP_FEAT_VSC_PANEL_INPUT)
void
sync_configure_input(vsc_context_t *context)
{
    if (context->input_sync_signals == HV) {
	adc_set_coast_env(context, PP_FALSE, 0, 0);
	adc_sync_mode(context, SEPARATE_SYNC);
    } else if (context->input_sync_signals == H) {
	adc_set_coast_env(context, PP_TRUE, 10, 10);
	adc_sync_mode(context, COMPOSITE_SYNC);

	context->force_irq = PP_TRUE;
    }
}
#endif /* !PP_FEAT_VSC_PANEL_INPUT */

/**   
 * measure the current horizontal and vertical frequency
 * for mode detection and determine polarity
 * vsync is returned in hsync-periods!
 * -additionally check the real sync input connected to VSC
 */
int
sync_measurement(vsc_context_t *context)
{
    u_int32_t newHL, newHH, newVL, newVH, newVLRaw, newVHRaw, deltaH, deltaV; 
    sync_signals_t signals;
    pp_bool_t h, v;
    int ret = -1;
    
    // get counters
    newHL = vsc_rx_reg(context, VSC_REG_HLT);
    newHH = vsc_rx_reg(context, VSC_REG_HHT);
    newVLRaw = vsc_rx_reg(context, VSC_REG_VLT);
    newVHRaw = vsc_rx_reg(context, VSC_REG_VHT);
    newVL = newVLRaw * (newHL + newHH);
    newVH = newVHRaw * (newHL + newHH);

    DCH(D_VERBOSE, "HL=%d, HH=%d, VL=%d, VH=%d\n", newHL, newHH, newVL, newVH);

    // get sync info
    vsc_syncs_on(context, &h, &v);

    if (h && v) {
	signals = HV;
    } else if (h) {
	signals = H;
    } else if (v) {
	signals = V;
    } else {
	signals = NONE;
    }       
    
    // sanity checks
    if ((newHL+newHH == 0) || (newVL+newVH == 0) || (signals != HV)) {
	DCH(D_VERBOSE, "Sync missing, bailing out\n");
	goto bail;
    }
    
    // determine polarity and frequency
    context->newPolH = (newHL >= newHH) ? 1 : 0;
    context->newPolV = (newVL >= newVH) ? 1 : 0;
    context->newFreqH = (vsc_clock/10)/(newHL+newHH); // count in 10 Hz steps
    context->newFreqV = (vsc_clock*10)/(newVL+newVH); // count in 0.1 HZ steps

    DCH(D_VERBOSE, "H/V-Sync Frequency = %d,%03.01f Hz, PolH/V=%d/%d\n",
      context->newFreqH*10, (float)context->newFreqV/10.0, context->newPolH, context->newPolV);

    // calculate new h/v sync deltas
    deltaH = ((newHH+newHL) * SYNC_DELTA_H) / 100;
    deltaV = ((newVHRaw+newVLRaw) * SYNC_DELTA_V) / 100;

    if (deltaH > 255) deltaH = 255;
    if (deltaV > 255) deltaV = 255;
    
    DCH(D_VERBOSE, "New sync deltas H=%d, V=%d\n", deltaH, deltaV);
    vsc_tx_reg(context, VSC_REG_HSD, deltaH);
    vsc_tx_reg(context, VSC_REG_VSD, deltaV);
    
    ret = 0;
    
 bail:
    if (signals != context->sync_signals) {
	DCH(D_VERBOSE, "Detected new internal sync input %d (was %d)\n", 
	    signals, context->input_sync_signals);
	context->sync_signals = signals;
    }
    return ret;
}

#ifdef USE_TABLE_LOOKUP
static vsc_mode_t*
search_mode(vsc_context_t *context, u_long tol_h, u_long tol_v,
	    custom_mode_handling_t custom_handling, u_char ignore_polarity)
{
    vsc_mode_t *mode = (vsc_mode_t *) pp_hash_get_first_entry_i(context->current_mode_hash);

    DCH(D_VERBOSE, "Looking for mode, tol_h=%lu, tol_v=%lu, ignore_polarity=%u\n",
      tol_h, tol_v, ignore_polarity);
    
    while (mode != NULL) {
	DCH(D_VERBOSE, "Check %4dx%4d, H/V %4d/%4d PolH/PolV %1d/%1d C%d\n",
	  mode->resX, mode->resY, mode->freqH, mode->freqV,
	  mode->polH, mode->polV, mode->custom);
	
	if ((mode->custom  && custom_handling == CUSTOM_MODE_OFF) ||
	    (!mode->custom && custom_handling == CUSTOM_MODE_FORCE)) {

	    mode = (vsc_mode_t *) pp_hash_get_next_entry_i(context->current_mode_hash);
	    continue;
	}
	
	if ((ignore_polarity || ((context->newPolH  == mode->polH)   &&
				 (context->newPolV  == mode->polV))) &&

	    (context->newFreqH <= mode->freqH + tol_h) &&
	    (context->newFreqV <= mode->freqV + tol_v) &&
	    (context->newFreqH >= mode->freqH - tol_h) &&
	    (context->newFreqV >= mode->freqV - tol_v))
	    {
		DCH(D_NOTICE, "Found Mode %dx%d\n", mode->resX, mode->resY);
		goto bail;
	    }
	
	mode = (vsc_mode_t *) pp_hash_get_next_entry_i(context->current_mode_hash);
    }

    mode = NULL;
    
 bail:
    return mode;
}
#endif

/**
 * when USE_TABLE_LOOKUP is defined:
 * 1. find the correct mode with the measured frequencies & polarities
 * 2. try again while not looking at the polarity
 * 3. goto 1. with larger tolerance until a certain maximum
 *
 * in any case:
 * 4. find the nearest mode if not too far away (some KVM switch OSDs
 *    generate a mode by themself without input which doesn't fit
 *    correctly)
 */
  
mode_det_res_t
mode_detection(vsc_context_t *context)
{
    mode_det_res_t ret = MODE_DET_OK;
    pp_bool_t error;
#if !defined(PP_FEAT_VSC_PANEL_INPUT)    
    const int use_display_enable = 0;
#else
    const int use_display_enable = context->feats.display_enable;
#endif
    	     
    MUTEX_LOCK(&common_data_mtx);


    if (use_display_enable) {
	free(context->current_mode);
	context->current_mode = (vsc_mode_t *)malloc(sizeof(vsc_mode_t));
	
	memset(context->current_mode, 0, sizeof(vsc_mode_t));
	
	context->current_mode->resX = vsc_rx_reg_secure(context, VSC_REG_MHL, &error);
	if (error == PP_TRUE) {
	    D(D_ERROR, "Error reading x resolution\n");
	    ret = MODE_DET_FAILED;
	    goto bail;
	}
	context->current_mode->resY = vsc_rx_reg_secure(context, VSC_REG_MVL, &error);
	if (error == PP_TRUE) {
	    D(D_ERROR, "Error reading y resolution\n");
	    ret = MODE_DET_FAILED;
	    goto bail;
	}

	/* do a sanity check of the digital resolution */
	if (context->current_mode->resX == 0 || context->current_mode->resY == 0) {
	    D(D_ERROR, "Sanity check for resolution %x x %x failed\n",
	      context->current_mode->resX, context->current_mode->resY);
	    ret = MODE_DET_TRY_AGAIN;
	    goto bail;
	}

        if (context->current_mode->resX % 8) {
            D(D_NOTICE, "Strange resolution detected, padding to multiple of 8.\n");
            context->current_mode->resX = (context->current_mode->resX / 8 + 1) * 8;
        }
    } else {
	
	custom_mode_handling_t custom_handling = get_custom_mode_handling();
	vsc_mode_t *mode;
   
	// (re)init mode hash
	mode_init_port(context->kvm_port);
	context->current_mode_hash = common_data[context->kvm_port].mode_hash;
    
	// save last mode for comparision
	context->last_mode = context->current_mode;

#ifdef USE_TABLE_LOOKUP
	{
	    u_long tol_h = MIN_MODE_TOLERANCE_H;
	    u_long tol_v = MIN_MODE_TOLERANCE_V;
	    
	    while (tol_h <= MAX_MODE_TOLERANCE_H &&
		   tol_v <= MAX_MODE_TOLERANCE_V) {
		
		// search for exact polarity
		if ((mode = search_mode(context, tol_h, tol_v, custom_handling,
					DONT_IGNORE_POLARITY)) != NULL) {
		    context->current_mode = context->nearest_mode = mode;
		    goto bail;
		}
		
		// search again without checking polarity
		if ((mode = search_mode(context, tol_h, tol_v, custom_handling,
					IGNORE_POLARITY)) != NULL) {
		    context->current_mode = context->nearest_mode = mode;
		    goto bail;
		}
		
		// increase tolerance and search again
		tol_h += MODE_TOLERANCE_STEP_H;
		tol_v += MODE_TOLERANCE_STEP_V;	    
	    }
	}
#endif
	
	{
	    // as a last resort search for the nearest mode
	    vsc_mode_t *nearest_mode = NULL;
	    double distance = 0, shortest = 0;
	    double max_distance = sqrt(MAX_MODE_TOLERANCE_H *
				       MAX_MODE_TOLERANCE_H +
				       MAX_MODE_TOLERANCE_V *
				       MAX_MODE_TOLERANCE_V);
	    u_char found = 0;

	    DCH(D_VERBOSE, "Looking for nearest mode\n");

	    mode = (vsc_mode_t *) pp_hash_get_first_entry_i(context->current_mode_hash);
	    while (mode != NULL) {
		if ((mode->custom  && custom_handling == CUSTOM_MODE_OFF) ||
		    (!mode->custom && custom_handling == CUSTOM_MODE_FORCE)) {
		
		    mode = (vsc_mode_t *) pp_hash_get_next_entry_i(context->current_mode_hash);
		    continue;
		}

		distance = sqrt(pow(abs(mode->freqH - context->newFreqH), 2) + pow(abs(mode->freqV - context->newFreqV), 2));
		
		if (found == 0 || distance < shortest) {
		    DCH(D_VERBOSE, "New shortest distance of %04f for %dx%d@%d\n", 
			distance, mode->resX, mode->resY, mode->freqV/10);
		    shortest = distance;
		    context->nearest_mode = nearest_mode = mode;
		    found = 1;
		}
	    
		mode = (vsc_mode_t *) pp_hash_get_next_entry_i(context->current_mode_hash);	
	    }
	
	    if (nearest_mode != NULL && shortest < max_distance) {
		DCH(D_NOTICE, "Found nearest mode %dx%d@%d\n", 
		    nearest_mode->resX, nearest_mode->resY, nearest_mode->freqV/10);
		context->current_mode = nearest_mode;
		goto bail;
	    }
	}
	
	DCH(D_VERBOSE, "No mode found\n");
	context->current_mode = NULL;
	
	ret = MODE_DET_FAILED;
    } /* !display_enable */
    
 bail:
    MUTEX_UNLOCK(&common_data_mtx);
    return ret;
}

void
mode_setup(vsc_context_t *context, vsc_mode_t *mode)
{
    u_int16_t x_padded;
    pp_vsc_event_t event;

    u_char video_link = context->id;

    if (mode == NULL) {
	DCH(D_ERROR, "mode_setup: no mode to setup, doing nothing\n");
	return;
    }

    pp_grab_fbchange_start(video_link);

    MUTEX_LOCK(&common_data_mtx);
  
    // set user changeable settings
#ifdef PP_FEAT_VSC_PANEL_INPUT
    if (!context->feats.display_enable) {
	vsc_set_offset_x(context, mode->ofsX);
	vsc_set_offset_y(context, mode->ofsY);
    }   
#else
    adc_set_pll(context, mode->freqH, mode->clocks);
    adc_set_phase(context, mode->phase);

    vsc_set_offset_x(context, mode->ofsX);
    vsc_set_offset_y(context, mode->ofsY);
#endif

    // pad horizontal video area to multiple of hextile size
    x_padded = mode->resX;
    if (x_padded % PP_FB_TILE_WIDTH) x_padded = (x_padded / PP_FB_TILE_WIDTH + 1) * PP_FB_TILE_WIDTH;
					 
    vsc_tx_reg(context, VSC_REG_HL, x_padded);
    vsc_tx_reg(context, VSC_REG_VL, mode->resY);
    DCH(D_VERBOSE, "mode_setup %d,%d\n", x_padded, mode->resY);
    
    usleep(250000);
    vsc_correct_offsets(context);

    pp_grab_fbchange_ready(video_link);
    
    MUTEX_UNLOCK(&common_data_mtx);

    DCH(D_VERBOSE, "set to new mode - default: %d\n", mode->is_default);

    if (vsc_encoder_hook && vsc_encoder_hook->vsc_event) {
        event.type = VSC_EVENT_MODE_CHANGE;
        event.event.mode_change.type = VSC_MODE_CHANGE_VALID;
        event.event.mode_change.res_x = mode->resX;
        event.event.mode_change.res_y = mode->resY;
#if !defined(PP_FEAT_VSC_PANEL_INPUT)
        event.event.mode_change.refresh = mode->freqV/10;
        event.event.mode_change.is_defaultmode = mode->is_default;
#endif
        vsc_encoder_hook->vsc_event(video_link, &event);
    }

    context->mode_valid = PP_TRUE;
}

int
mode_save_settings(vsc_context_t *context,
		   unsigned short clocks, unsigned short phase,
		   unsigned short ofsX,	unsigned short ofsY)
{
    int ret = -1;
    
    MUTEX_LOCK(&common_data_mtx);

    if (context->current_mode == NULL) {
	goto bail;
    }
	
    context->current_mode->clocks = clocks;
    context->current_mode->phase  = phase;
    context->current_mode->ofsX   = ofsX;
    context->current_mode->ofsY   = ofsY;

    context->current_mode->is_default = 0;
    
    ret = save_mode_data(context->kvm_port);

 bail:
    MUTEX_UNLOCK(&common_data_mtx);
    return ret;
}

void
mode_reset_current(vsc_context_t *context)
{
    vsc_mode_t *defmode = default_modes;
    u_int32_t mode_id;
    pp_bool_t reset_done = PP_FALSE;
    
    MUTEX_LOCK(&common_data_mtx);

    if (context->current_mode == NULL) goto bail;

    DCH(D_VERBOSE, "mode_reset_current\n");

    mode_id = mode_build_id(context->current_mode->freqH,
			    context->current_mode->freqV,
			    context->current_mode->polH,
			    context->current_mode->polV,
			    context->current_mode->custom);

    if (context->current_mode->custom) {
	//special handling for custom mode
	load_custom_modes(context->kvm_port, mode_id);
	reset_done = PP_TRUE;
	goto bail;
    }

    while (defmode->freqH && defmode->freqV) {
	u_int32_t def_id = mode_build_id(defmode->freqH, defmode->freqV,
					 defmode->polH, defmode->polV, defmode->custom);

	if (def_id != mode_id) {
	    defmode++;
	    continue;
	}
	
	memcpy(context->current_mode, defmode, sizeof(vsc_mode_t));
	context->current_mode->def_link = defmode;
	
	DCH(D_BLABLA, "Reset with default %4dx%4d, H/V %4d/%4d PolH/PolV %1d/%1d\n",
	  defmode->resX, defmode->resY, defmode->freqH, defmode->freqV,
	  defmode->polH, defmode->polV);

	reset_done = PP_TRUE;
	break;
    }
    
 bail:
    MUTEX_UNLOCK(&common_data_mtx);
    if (reset_done) {
	context->force_irq = PP_TRUE;
    }
}

void mode_reset_all(vsc_context_t *context) {
    pp_bool_t reset_done = PP_FALSE;
    int i;
    
    MUTEX_LOCK(&common_data_mtx);

    for (i=0; i<= common_data_max; i++) {
	pp_hash_delete(common_data[i].mode_hash);
	common_data[i].mode_hash = NULL;
	save_mode_data(i);
    }
    context->current_mode = NULL;
    reset_done = PP_TRUE;
    
    goto bail;
    
 bail:
    MUTEX_UNLOCK(&common_data_mtx);
    if (reset_done) {
	context->force_irq = PP_TRUE;
    }
}

int
mode_get_res_info(vsc_context_t *context,
		  unsigned short *resX, unsigned short *resY,
		  unsigned short *freqH, unsigned short *freqV)
{
    int ret = -1;
    
    MUTEX_LOCK(&common_data_mtx);

    if (context->current_mode == NULL) goto bail;

    if (resX)  *resX = context->current_mode->resX;
    if (resY)  *resY = context->current_mode->resY;
    if (freqH) *freqH = context->current_mode->freqH;
    if (freqV) *freqV = context->current_mode->freqV;

    ret = 0;
    
 bail:
    MUTEX_UNLOCK(&common_data_mtx);

    return ret;
}

int
mode_get_format(vsc_context_t *context, fb_format_info_t *fb)
{
    static const char* fn = ___F;
    vsc_mode_t *mode;
    int ret = -1;
    
    assert(fb);

    MUTEX_LOCK(&common_data_mtx);
    
    if (context->current_mode != NULL) {
	mode = context->current_mode;
    } else if (context->last_mode != NULL) {
	mode = context->last_mode;
    } else {
	/* no useful mode available */
	DCH(D_BLABLA, "%s: no mode active\n", fn);
	goto bail;
    }
    
    fb->bpp	= VSC_PIXEL_BITS;
    fb->g_w	= mode->resX;
    fb->g_wb	= fb->g_w * ((fb->bpp + 7) / 8);
    fb->g_h	= mode->resY;
    fb->tiles_w = fb->g_w / PP_FB_TILE_WIDTH;
    fb->tiles_h = fb->g_h / PP_FB_TILE_HEIGHT;

    if (fb->g_w % PP_FB_TILE_WIDTH)  fb->tiles_w++;
    if (fb->g_h % PP_FB_TILE_HEIGHT) fb->tiles_h++;

    fb->g_w_pd	= fb->tiles_w * PP_FB_TILE_WIDTH;
    fb->g_wb_pd	= fb->g_w_pd * ((fb->bpp + 7) / 8);
    fb->g_h_pd  = fb->tiles_h * PP_FB_TILE_HEIGHT;

    fb->is_unsupported = 0; /* trigger direct host connection, unused atm */

    /*
    D(D_BLABLA, "%s: %d,%d (pad %d,%d), %d bpp, unsupported %d\n", fn,
      fb->g_w, fb->g_h, fb->g_w_pd, fb->g_h_pd, fb->bpp, fb->is_unsupported);
    */
      
    ret = 0;
    
 bail:
    MUTEX_UNLOCK(&common_data_mtx);
    return ret;    
}

void
invalidate_custom_modes(void)
{
    int i;
    
    D(D_VERBOSE, "invalidate_custom_modes\n");
    MUTEX_LOCK(&common_data_mtx);

    for (i=0; i<=common_data_max; i++) {
	common_data[i].custom_modes_invalid = 1;
    }

    vsc_contexts[0].force_irq = PP_TRUE;
    
    MUTEX_UNLOCK(&common_data_mtx);
}

/* ---------- internal functions, hold common_data_mtx before calling -------------------- */

static int
mode_init_port(int kvm_port)
{
    D(D_VERBOSE, "mode_init_port, kvm_port=%d, max=%d\n", kvm_port, common_data_max);
    
    if (kvm_port > common_data_max) {
	// create space for mode hash on this kvm port
	common_data = (common_data_t*) realloc(common_data,
					       (kvm_port + 1) * sizeof(common_data_t));
	if (common_data == NULL) {
	    D(D_ERROR, "Out of memory in mode_init_port");
	    return -1;
	}

	memset(&common_data[common_data_max + 1], 0x00,
	       (kvm_port - common_data_max) * sizeof(common_data_t));
    }
    
    if (mode_init_data(kvm_port) == -1) {
	return -1;
    }

    if (kvm_port > common_data_max) {
	common_data_max = kvm_port;
    }
   
    return 0;
}

static int
mode_init_data(int kvm_port)
{
    vsc_mode_t *defmode = default_modes;

    // init mode hash for this KVM port if necessary
    if (common_data[kvm_port].mode_hash != NULL) {
	D(D_VERBOSE, "Mode hash already initialized\n");

	if (common_data[kvm_port].custom_modes_invalid) {
	    D(D_VERBOSE, "Reloading custom modes for port %d\n", kvm_port);
	    clear_custom_modes(kvm_port);
	    load_custom_modes(kvm_port, 0);
	    load_mode_data(kvm_port);	    
	    common_data[kvm_port].custom_modes_invalid = 0;
	}
	
	return 0;
    }

    common_data[kvm_port].mode_hash = pp_hash_create_i(sizeof(default_modes)/sizeof(vsc_mode_t));
 
    if (common_data[kvm_port].mode_hash == NULL) {
	D(D_ERROR, "Could not create mode hash in mode_init_data\n");
	return -1;
    }
    // fill hash with default values from internal mode list
    while (defmode->freqH && defmode->freqV) {
	u_int32_t mode_id = mode_build_id(defmode->freqH, defmode->freqV,
					  defmode->polH, defmode->polV, defmode->custom);
	vsc_mode_t *mode = malloc(sizeof(vsc_mode_t));

	memcpy(mode, defmode, sizeof(vsc_mode_t));
	mode->def_link = defmode;
	
	if (pp_hash_set_entry_i(common_data[kvm_port].mode_hash, mode_id,
				(void*) mode, free) != 0) {
	    D(D_ERROR, "Cannot set hash entry for mode %08x", mode_id);
	    free(mode); mode = NULL;
	}

	D(D_BLABLA, "Got default %4dx%4d, H/V %4d/%4d PolH/PolV %1d/%1d\n",
	  defmode->resX, defmode->resY, defmode->freqH, defmode->freqV,
	  defmode->polH, defmode->polV);

	defmode++;
    }

    load_custom_modes(kvm_port, 0);
    load_mode_data(kvm_port);

    return 0;
}


/* only called within the common_mutex */
static void
clear_custom_modes(int kvm_port)
{
    pp_hash_i_t *hash = common_data[kvm_port].mode_hash;
    vsc_mode_t *mode;
    unsigned int key;

    key = pp_hash_get_first_key_i(hash);

    while (key != 0) {
	mode = (vsc_mode_t*) pp_hash_get_entry_i(hash, key);
	if (mode->custom) {
	    if (vsc_contexts[0].current_mode == mode)
		vsc_contexts[0].current_mode = NULL;
	    pp_hash_delete_entry_i(hash, key);
	}
	key = pp_hash_get_next_key_i(hash);
    }
}

static int
load_custom_modes(int kvm_port, u_int32_t certain_id)
{
    unsigned short freqH, freqV, resX, resY, clocks, polarity, polH, polV;
    u_int i = 0;

    for (i = 0; i < MAX_CUSTOM_VIDEO_MODES; i++) {
	vsc_mode_t *mode;
	u_int32_t mode_id;
	const char* key = "video.tft_custom.settings[%u].%s";

	if (PP_FAILED(pp_cfg_get_ushort_nodflt(&freqH, key, i, "h"))
	    || PP_FAILED(pp_cfg_get_ushort_nodflt(&freqV, key, i, "v"))
	    || PP_FAILED(pp_cfg_get_ushort_nodflt(&resX, key, i, "x"))
	    || PP_FAILED(pp_cfg_get_ushort_nodflt(&resY, key, i, "y"))
	    || PP_FAILED(pp_cfg_get_ushort_nodflt(&clocks, key, i, "clock"))
	    || PP_FAILED(pp_cfg_get_ushort_nodflt(&polarity, key, i, "polarity"))) {
	    continue;
	}
	
	freqH /= 10;
	freqV *= 10;
	polH = (polarity & 0x01) ? 1 : 0;
	polV = (polarity & 0x02) ? 1 : 0;

	mode_id = mode_build_id(freqH, freqV, polH, polV, 1);

	if (certain_id != 0) {
	    // we got a mode, load only this one
	    if (certain_id != mode_id) {
		i++;
		continue;
	    }

	    D(D_BLABLA, "Reset custom %d, %4dx%4d, H/V %4d/%4d PolH/PolV %1d/%1d\n",
	       i, resX, resY, freqH, freqV, polH, polV);

	    mode = pp_hash_get_entry_i(common_data[kvm_port].mode_hash, mode_id);

	} else {
	    mode = malloc(sizeof(vsc_mode_t));
	}	

	mode->custom = 1;
	mode->freqH = freqH;
	mode->freqV = freqV;
	mode->polH = polH;
	mode->polV = polV;
	mode->resX = resX;
	mode->resY = resY;
	mode->ofsX = 0;
	mode->ofsY = 0;
	mode->clocks = clocks;
	mode->phase = 0;

	if (certain_id == 0) {
	    if (pp_hash_set_entry_i(common_data[kvm_port].mode_hash, mode_id,
				    (void*) mode, free) != 0) {
		D(D_ERROR, "Cannot set hash entry for mode %08x", mode_id);
		free(mode); mode = NULL;
	    }
	
	    D(D_BLABLA, "Got custom %d, %4dx%4d, H/V %4d/%4d PolH/PolV %1d/%1d\n",
	       i, resX, resY, freqH, freqV, polH, polV);
	}
	
	i++;
    }
    
    return 0;
}

static int
load_mode_data(int kvm_port)
{
    char filepath[strlen(mode_file_prefix) + 5];
    vsc_mode_data_t data_loaded;
    u_int32_t version;
    int fd = -1;
    int ret = -1;

    snprintf(filepath, sizeof(filepath), "%s%02x",
	     mode_file_prefix, kvm_port);

    if ((fd = open(filepath, O_RDONLY, 0664)) == -1) {
	D(D_ERROR, "mode_load_data: could not open %s\n", filepath);
	goto bail;
    }

    if (read(fd, &version, sizeof(version)) != sizeof(version)) {
	D(D_ERROR, "mode_load_data: could not read %s\n", filepath);
	goto bail;
    }

    D(D_NOTICE, "mode_load_data: found mode data version %08x\n", version);

    while (1) {
	vsc_mode_t *mode = NULL;
	u_int32_t mode_id = 0;
	int read_count;

	read_count = read(fd, &data_loaded, sizeof(data_loaded));
	if (read_count == 0) {
	    ret = 0;
	    goto bail;
	}		   
	if (read_count != sizeof(data_loaded)) {
	    D(D_ERROR, "mode_load_data: could not read mode from %s\n", filepath);
	    goto bail;
	}

	mode_id = mode_build_id(data_loaded.freqH,data_loaded.freqV,
				data_loaded.polH, data_loaded.polV,
				data_loaded.custom);

	mode = (vsc_mode_t*) pp_hash_get_entry_i(common_data[kvm_port].mode_hash,
						 mode_id);

	if (mode == NULL) {
	    continue;
	}

	data_to_mode(&data_loaded, mode);
	mode->is_default = 0;

	D(D_BLABLA, "Load data %4dx%4d, H/V %4d/%4d PolH/PolV %1d/%1d C%d\n",
	   mode->resX, mode->resY, mode->freqH, mode->freqV,
	   mode->polH, mode->polV, mode->custom);
    }
   
    ret = 0;
      
 bail:
      if (fd != -1) {
	  close(fd);
      }
      return ret;
}

static int
save_mode_data_backend(int kvm_port, void * data, size_t data_len)
{
    char filepath[strlen(mode_file_prefix) + 5];
    int fd = -1, ret = -1;

    snprintf(filepath, sizeof(filepath), "%s%02x", mode_file_prefix, kvm_port);

    if ((fd = open(filepath, O_WRONLY | O_CREAT | O_TRUNC, 0664)) == -1) {
	D(D_ERROR, "%s(): could not open %s\n", ___F, filepath);
	goto bail;
    }

    if (write(fd, &mode_db_ver, sizeof(mode_db_ver)) != sizeof(mode_db_ver)) {
	D(D_ERROR, "%s(): could not write version to %s\n", ___F, filepath);
	goto bail;
    }

    if (write(fd, data, data_len) != (ssize_t)data_len) {
	D(D_ERROR, "%s(): could not write mode to %s\n", ___F, filepath);
	goto bail;
    }

 bail:
    if (fd != -1) close(fd);
    return ret;
}


static int
save_mode_data(int kvm_port)
{
    pp_hash_i_t *hash = common_data[kvm_port].mode_hash;
    u_int mode_cnt = 0;
    vsc_mode_t *mode;
    vsc_mode_data_t * mode_data = NULL;
    int ret = -1;
    
    if (hash != NULL) {
	u_int max_mode_cnt = pp_hash_get_entry_count_i(hash);
	if (max_mode_cnt) {
	    size_t mode_data_len = max_mode_cnt * sizeof(vsc_mode_data_t);
	    mode_data = malloc(mode_data_len);
	}
	for (mode = (vsc_mode_t *) pp_hash_get_first_entry_i(hash);
	     mode != NULL;
	     mode = (vsc_mode_t *) pp_hash_get_next_entry_i(hash)) {
	    if (!mode->is_default) {
		mode_to_data(mode, &mode_data[mode_cnt]);
		D(D_BLABLA, "Save data %4dx%4d, H/V %4d/%4d PolH/PolV %1d/%1d C%d\n",
		  mode->resX, mode->resY, mode_data[mode_cnt].freqH, mode_data[mode_cnt].freqV,
		  mode_data[mode_cnt].polH, mode_data[mode_cnt].polV, mode_data[mode_cnt].custom);
		++mode_cnt;
	    }
	}
	if (mode_data) {
	    ret = save_mode_data_backend(kvm_port, mode_data, mode_cnt * sizeof(vsc_mode_data_t));
	    free(mode_data);
	}
    }
    return ret;
}

/**
 * build mode id (for hash) from mode identification data
 * bits 31..16	15..3	   2	   1	0
 *	freqH	freqV	polH	polV	custom mode
 */
static u_int32_t
mode_build_id(u_int16_t freqH, u_int16_t freqV,
	      u_int8_t polH, u_int8_t polV,
	      u_int8_t custom)
{
    return ((freqH << 16) | ((freqV & 0x1FFF) << 3) |
	    ((polH  & 0x01) << 2) | ((polV   & 0x01) <<1) | custom);
}

static custom_mode_handling_t
get_custom_mode_handling(void)
{
    custom_mode_handling_t ret;
    char *opt;

    pp_cfg_get(&opt, "video.tft_custom.mode");
    if (!strcmp(opt, "use")) {
	ret = CUSTOM_MODE_USE;
    } else if (!strcmp(opt, "force")) {
	ret = CUSTOM_MODE_FORCE;
    } else {
	ret = CUSTOM_MODE_OFF;
    }
    free(opt);
    
    return ret;
}

static void
mode_to_data(vsc_mode_t *mode, vsc_mode_data_t *data)
{
    data->freqH = mode->freqH;
    data->freqV = mode->freqV;
    data->polH = mode->polH;
    data->polV = mode->polV;
    data->ofsX = mode->ofsX;
    data->ofsY = mode->ofsY;
    data->clocks = mode->clocks;
    data->phase = mode->phase;
    data->custom = mode->custom;
}

static void
data_to_mode(vsc_mode_data_t *data, vsc_mode_t *mode)
{
    mode->freqH = data->freqH;
    mode->freqV = data->freqV;
    mode->polH = data->polH;
    mode->polV = data->polV;
    mode->ofsX = data->ofsX;
    mode->ofsY = data->ofsY;
    mode->clocks = data->clocks;
    mode->phase = data->phase;
    mode->custom = data->custom;
}
