 /*********************************************************************
 * libpp_vsc: timer.c
 *
 * really trivial timer implementation to support the various
 * state changes and other delays for adc/vsc operationm
 *
 * currently counts on doing the 'tick' during the main loop,
 * so its not that accurate, doesn't need to be at the moment
 *
 ********************************************************************/

#include <unistd.h>

#include "timer.h"

static int timeval_subtract(struct timeval *result,
			    struct timeval x, struct timeval y);

int
timer_init(vsc_context_t *context, unsigned long tick_ms,
	   unsigned long tick_fast_ms)
{
    context->timer_tick_ms = tick_ms;
    context->timer_tick_fast_ms = tick_fast_ms;
    gettimeofday(&context->timer_last, NULL);
    return 0;
}

void
timer_cleanup(vsc_context_t *context UNUSED)
{
    /* nothing to do atm */
}

void
timer_tick(vsc_context_t *context, timer_tick_t type)
{
    struct timeval timer_current, timer_delay;
    int i;

    usleep(1000 * ((type == TICK_STANDARD) ? context->timer_tick_ms : context->timer_tick_fast_ms));
    gettimeofday(&timer_current, NULL);

    // we just assume its always positive, cannot travel back in time
    timeval_subtract(&timer_delay, timer_current, context->timer_last);
	
    for (i=0; i<MAX_TIMER_COUNT; i++) {	
	if (context->timers[i].expired == PP_TRUE) {
	    continue;
	}

	if (timer_delay.tv_sec*1000 + timer_delay.tv_usec/1000 >=
	    context->timers[i].remaining_ms) {

	    context->timers[i].expired = PP_TRUE;
	    
	} else {
	    context->timers[i].remaining_ms -=
		(timer_delay.tv_sec*1000 + timer_delay.tv_usec/1000);
	}
    }

    context->timer_last = timer_current;
}

void
timer_set(vsc_context_t *context, unsigned char timer,
	  unsigned long expire_in_ms)
{
    assert(timer < MAX_TIMER_COUNT);
    
    context->timers[timer].active  = PP_TRUE;
    context->timers[timer].expired = PP_FALSE;
    context->timers[timer].remaining_ms = expire_in_ms;
}

void
timer_deactivate(vsc_context_t *context, unsigned char timer)
{
    assert(timer < MAX_TIMER_COUNT);

    context->timers[timer].active  = PP_FALSE;
}

pp_bool_t
timer_expired(vsc_context_t *context, unsigned char timer)
{
    assert(timer < MAX_TIMER_COUNT);

    return context->timers[timer].expired;
}

pp_bool_t
timer_active(vsc_context_t *context, unsigned char timer)
{
    assert(timer < MAX_TIMER_COUNT);

    return (context->timers[timer].active != 0);
}


void
timer_wait_ms(vsc_context_t *context UNUSED, unsigned long ms)
{
    struct timespec ts = { ms/1000, (ms%1000) * 1000000 };
    struct timespec remaining = {0, 0};

    while (nanosleep(&ts, &remaining) == -1) {
	if (errno == EINTR) {
	    ts.tv_sec = remaining.tv_sec;
	    ts.tv_nsec = remaining.tv_nsec;
	    remaining.tv_sec = 0;
	    remaining.tv_nsec = 0;
	}
    }
}

/**
 * Subtract the `struct timeval' values X and Y (from libc manual),
 * storing the result in RESULT.
 * Return 1 if the difference is negative, otherwise 0.
 */
static int
timeval_subtract(struct timeval *result,
		 struct timeval x, struct timeval y)
{
    /* Perform the carry for the later subtraction by updating Y. */
    if (x.tv_usec < y.tv_usec) {
	int nsec = (y.tv_usec - x.tv_usec) / 1000000 + 1;
	y.tv_usec -= 1000000 * nsec;
	y.tv_sec += nsec;
    }
    if (x.tv_usec - y.tv_usec > 1000000) {
	int nsec = (x.tv_usec - y.tv_usec) / 1000000;
	y.tv_usec += 1000000 * nsec;
	y.tv_sec -= nsec;
    }
    
    /* Compute the time remaining to wait.
       `tv_usec' is certainly positive. */
    result->tv_sec  = x.tv_sec - y.tv_sec;
    result->tv_usec = x.tv_usec - y.tv_usec;
    
    /* Return 1 if result is negative. */
    return x.tv_sec < y.tv_sec;
}
