 /*********************************************************************
 * libpp_vsc: vsc_io.c
 *
 * low level register access functions for VSC
 *
 ********************************************************************/

#include <pp/base.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <unistd.h>
#include <lara.h>
#include "vsc_io.h"
#include "debug.h"

#define JBLASTER_PATH		"/bin/jblaster"
#define JBLASTER_CHAIN		"/etc/vsc/vsc_chain.cdf"
#define JBLASTER_CHAIN_CUSTOM	"/etc/vsc/vsc_chain_custom.cdf"
#define JBLASTER_RETRYCOUNT	3

vsc_context_t  *vsc_contexts = NULL;

pp_bool_t
vsc_initialize(vsc_context_t *context, int do_program_fpga)
{
/* FIXME: hack to prevent that fpga a programmed on KX2 automatically */
#if defined(PRODUCT_KX2)
    do_program_fpga = 0;
#endif
    if (do_program_fpga) {
	char cmdline[128];
	/* NOTE: volatile necessary to circumvent a compiler bug; If status is
	 *       not volatile then then broken code is generated
	 *       (final build only!)
	 *       Lets recheck this with gcc 3.3.4.
	 */
	volatile int status;
	int retrycount = JBLASTER_RETRYCOUNT;

	context->vsc_active = PP_FALSE;
    
	while (context->vsc_active == PP_FALSE &&
	       retrycount-- > 0) {

	    /* first try a custom vsc binary in flashdisk */
	    snprintf(cmdline, sizeof(cmdline),"%s %s", JBLASTER_PATH, JBLASTER_CHAIN_CUSTOM);
	    status = pp_system(cmdline);
	    if (WIFEXITED(status) && WEXITSTATUS(status) == 0) {
		context->vsc_active = PP_TRUE;
		break;
	    }

	    /* fallback to internal vsc binary in initrd */
	    snprintf(cmdline, sizeof(cmdline),"%s %s", JBLASTER_PATH, JBLASTER_CHAIN);
	    status = pp_system(cmdline);

	    if (WIFEXITED(status) && WEXITSTATUS(status) == 0) {
		context->vsc_active = PP_TRUE;
	    } else {
		context->vsc_active = PP_FALSE;

		// no need to try again if aborted
		if (WIFSIGNALED(status) && WTERMSIG(status) == SIGINT) break;
	    }
	}
    } else {
	D(D_VERBOSE, "Resetting VSC\n");
	vsc_tx_masked(context, VSC_REG_MR, MR_VSC_RESET, MR_VSC_RESET);
	usleep(100000);
	vsc_tx_masked(context, VSC_REG_MR, 0, MR_VSC_RESET);

	context->vsc_active = PP_TRUE;	
    }
   
    if (context->vsc_active) {
	if (ioctl(context->fd, PPIOCVSCACTIVATE, NULL) == -1) {
	    if (errno != EAGAIN) {
		D(D_ERROR, "PPIOCVSCACTIVATE failed, no VSC for us. (%s)\n", strerror(errno));
		context->vsc_active = PP_FALSE;
	    }
	}
    } else {
	D(D_ERROR, "VSC initialization failed, we better don't touch it\n");
    }

    return context->vsc_active;
}

u_int32_t
vsc_rx_reg(vsc_context_t *context, u_int8_t reg)
{
    vsc_regop_t regop = { VSC_REG_READ, reg, 0, 0 };

    if (ioctl(context->fd, PPIOCVSCREGOP, &regop) == -1) {
	if (errno != EAGAIN) {
	    D(D_ERROR, "Error during vsc_rx_reg ioctl: %s\n", strerror(errno));
	}
	return 0;
    }

    return regop.data;
}

u_int32_t
vsc_rx_reg_shadowed(vsc_context_t *context, u_int8_t reg)
{
    vsc_regop_t regop = { VSC_REG_READ_SHADOWED, reg, 0, 0 };

    if (ioctl(context->fd, PPIOCVSCREGOP, &regop) == -1) {
	if (errno != EAGAIN) {
	    D(D_ERROR, "Error during vsc_rx_reg_shadowed ioctl: %s\n", strerror(errno));
	}
	return 0;
    }

    return regop.data;    
}

u_int32_t
vsc_rx_reg_secure(vsc_context_t *context, u_int8_t reg, pp_bool_t *error_occured)
{
    vsc_regop_t regop = { VSC_REG_READ_SECURE, reg, 0, 0 };
    if (error_occured) *error_occured = PP_FALSE;

    if (ioctl(context->fd, PPIOCVSCREGOP, &regop) == -1) {
	if (errno != EAGAIN) {
	    D(D_ERROR, "Error during vsc_rx_reg_secure ioctl: %s\n", strerror(errno));
	}
	if (error_occured) *error_occured = PP_TRUE;
	return 0;
    }

    return regop.data;   
}

void
vsc_tx_reg(vsc_context_t *context, u_int8_t reg, u_int32_t data)
{
    vsc_regop_t regop = { VSC_REG_WRITE, reg, data, 0 };

    if (ioctl(context->fd, PPIOCVSCREGOP, &regop) == -1) {
	if (errno != EAGAIN) {
	    D(D_ERROR, "Error during vsc_tx_reg ioctl: %s\n", strerror(errno));
	}
    }    
}

void
vsc_tx_masked(vsc_context_t *context, u_int8_t reg,
	      u_int32_t data, u_int32_t mask)
{
    vsc_regop_t regop = { VSC_REG_WRITE_MASKED, reg, data, mask };

    if (ioctl(context->fd, PPIOCVSCREGOP, &regop) == -1) {
	if (errno != EAGAIN) {
	    D(D_ERROR, "Error during vsc_tx_reg_masked ioctl: %s\n", strerror(errno));
	}
    }
}

int
vsc_get_sync_irq(vsc_context_t *context UNUSED, pp_bool_t dowait)
{
    sync_wait_t sync_wait = { dowait, 0 };
    
 again_sync:
    if (ioctl(context->fd, PPIOCVSCGETSYNCIRQ, &sync_wait) == -1) {
	if (errno == EINTR) goto again_sync;
	D(D_ERROR, "Error during ioctl(PPIOCVSCGETSYNCIRQ): %s\n", strerror(errno));
	return 0;
    }

    if (sync_wait.got_irq) D(D_BLABLA, "IRQ\n");
    
    return sync_wait.got_irq;
}

int
vsc_get_ofs_len_measures(vsc_context_t *context UNUSED, vsc_measures_t *measures)
{
 again_ofs:
    if (ioctl(context->fd, PPIOCVSCMEASUREPICTURE, measures) == -1) {
	if (errno == EINTR) goto again_ofs;
	D(D_ERROR, "Error during ioctl(PPIOCVSCMEASUREOFFSETS): %s\n", strerror(errno));
	return -1;
    }
    
    return 0;
}
