#include <unistd.h>

#include <pp/base.h>
#include <pp/vector.h>
#include <liberic_pthread.h>

#include "uuid.h"
#include "debug.h"
#include "resource.h"
#include "enumerate_session.h"

#define SESSION_TIMEOUT	(1 * 60)

/*
 * global variables
 */
static pthread_mutex_t session_mutex = PTHREAD_RECURSIVE_MUTEX_INITIALIZER_NP;
static vector_t *sessions = NULL;
static pthread_t session_thread;
static int session_thread_running;
static int session_id = 0;

/*
 * function prototypes
 */
static void del_session(void* element);
static void* session_thread_func(void* data);

/*
 * implementation
 */

/* initialization and cleanup */
int enumerate_session_init() {
    int ret = -1;
    
    lock_sessions();
    
    sessions = vector_new(NULL, 5, del_session);
    if (!sessions) {
    	pp_log_err("%s: Could not allocate session vector.\n", ___F);
    	goto bail;
    }
    
    session_thread_running = 1;
    if (eric_pthread_create(&session_thread, 0, 65536, session_thread_func, NULL)) {
	pp_log_err("%s() cannot create MSP thread\n", ___F);
	goto bail;
    }

    ret = 0;
    
 bail:
    unlock_sessions();
    return ret;
}

void enumerate_session_cleanup() {
    session_thread_running = 0;
    pthread_join(session_thread, NULL);

    lock_sessions();
    
    vector_delete(sessions);
    sessions = NULL;
    
    unlock_sessions();
}

static void del_session(void* element) {
    enumerate_session_t *session = (enumerate_session_t *)element;
    wsman_resource_t *res = session->resource;
    wsman_instance_t *instance = session->instance;
    
    if (res && instance && res->cleanup_instance) {
    	res->cleanup_instance(res, instance);
    }
    
    free(session);
}

/* session cleaning thread */
static void* session_thread_func(void* data UNUSED) {
    while (session_thread_running) {
    	unsigned long idx = 0;
    	
    	//D(D_BLABLA, "Checking for dead sessions.\n");
    	lock_sessions();
    
    	while (idx < vector_size(sessions)) {
    	    enumerate_session_t *session = (enumerate_session_t *)vector_get(sessions, idx);
    	    D(D_BLABLA, "Checking session %u\n", session->id);
    	    if (session && session->last_timestamp + SESSION_TIMEOUT < time(NULL)) {
    	    	D(D_VERBOSE, "Removing session %u (timeout).\n", session->id);
    	    	vector_remove(sessions, idx);
    	    } else {
    	    	idx++;
    	    }
    	}
    	
    	unlock_sessions();
    	
    	sleep(1);
    }
    
    return NULL;
}

/* locking */
void lock_sessions(void) {
    MUTEX_LOCK(&session_mutex);
}

void unlock_sessions(void) {
    MUTEX_UNLOCK(&session_mutex);
}

/* session handling */
enumerate_session_t * get_session(char *context) {
    enumerate_session_t *ret = NULL;
    unsigned long i;
    lock_sessions();
    
    for (i = 0; i < vector_size(sessions); i++) {
    	enumerate_session_t *session = (enumerate_session_t *)vector_get(sessions, i);
    	if (session && context && !strcmp(session->last_context, context)) {
    	    ret = session;
    	    session->last_timestamp = time(NULL);
    	    break;
    	}
    }
    
    unlock_sessions();
    return ret;
}

enumerate_session_t * create_session(enum_type_t type, char *context, wsman_resource_t *res, wsman_instance_t *instance) {
    enumerate_session_t *session;

    lock_sessions();
    session = malloc(sizeof(enumerate_session_t));
    session->type = type;
    session->id = session_id++;
    session->resource = res;
    D(D_VERBOSE, "Created new session %u\n", session->id);
    vector_add(sessions, session);
    
    update_session(session, context, instance);
    
    unlock_sessions();
    return session;
}

enumerate_session_t * update_session(enumerate_session_t *session, char *context, wsman_instance_t *instance) {
    D(D_VERBOSE, "Updating session %u\n", session->id);
    snprintf(session->last_context, sizeof(session->last_context), "%s", context);
    session->instance = instance;
    session->last_timestamp = time(NULL);

    return session;
}

int destroy_session(enumerate_session_t *session) {
    int ret = -1;
    unsigned long i;    
    lock_sessions();
    
    for (i = 0; i < vector_size(sessions); i++) {
    	enumerate_session_t *s = (enumerate_session_t *)vector_get(sessions, i);
    	if (s && session == s) {
    	    ret = 0;
    	    D(D_VERBOSE, "Deleting session %u\n", session->id);
    	    vector_remove(sessions, i);
    	    break;
    	}
    }
    
    unlock_sessions();
    return ret;
}
