#include <stdio.h>

#include <pp/base.h>
#include <pp/features.h>
#include <pp/cim.h>

#include "wsman.h"
#include "fault.h"
#include "debug.h"
#include "xml_writer.h"
#include "standard_actions.h"
#include "resource.h"
#include "res_cim.h"

/******************************************************************************
* datatypes                                                                   *
******************************************************************************/

typedef struct {
    /* must be first element (for casts)! */
    wsman_resource_t res;
    
    pp_cim_class_t *cim_class;
} ws_cim_resource_t;

typedef struct {
    wsman_instance_t inst;
    
    pp_cim_instance_t *cim_instance;
} ws_cim_instance_t;

typedef struct {
    pp_cim_data_t cim_data;
    
    int array;
    pp_cim_datatype_t type;
} ws_cim_data_t;

/******************************************************************************
* function prototypes                                                         *
******************************************************************************/

static void delete_cim_resource(void *elem);
static int create_cim_resource(pp_cim_class_t *class);
static char* get_prop_type(pp_cim_datatype_t type);

static int invoke_cim_method(xml_writer_t, wsman_resource_t *, wsman_request_t *, char **, int *, int *, char **, char *);
static wsman_instance_t * get_cim_instance_by_request(wsman_resource_t *, wsman_request_t *, int *, const char *auth_string);
static wsman_instance_t * get_first_cim_instance(wsman_resource_t *, wsman_request_t *, int *, const char *auth_string);
static wsman_instance_t * get_next_cim_instance(wsman_resource_t *, wsman_request_t *, wsman_instance_t *, int *, const char *auth_string);
static int get_cim_instance_data(xml_writer_t, wsman_request_t *, wsman_instance_t *, int *, int *, char **);
static int get_cim_instance_fragment(xml_writer_t, wsman_request_t *, wsman_instance_t *, int *, int *, char **);
static int get_cim_selector(xml_writer_t, wsman_request_t *, wsman_instance_t *, int *, int *, char **);
static int put_cim_instance_data(wsman_instance_t *, wsman_request_t *, int *, int *, char **);
static int put_cim_instance_fragment(wsman_instance_t *, wsman_request_t *, int *, int *, char **);
static int release_cim_instance(wsman_resource_t *, wsman_instance_t *);
static int cleanup_cim_instance(wsman_resource_t *, wsman_instance_t *);

static ws_cim_instance_t *cim_instance_new(pp_cim_instance_t *, ws_cim_resource_t *);

static int match_selectors(pp_cim_instance_t *, wsman_request_t *, int *);
static int match_property(pp_cim_propval_t *, wsman_req_selector_t*);
static int property_is_array(pp_cim_propval_t *);
static int property_array_size(pp_cim_propval_t *);
static char* cim_property_to_ws(pp_cim_propval_t *);
static char* cim_property_to_ws_array(pp_cim_propval_t *, int);
static int cim_arg_to_ws(pp_cim_method_arg_t *, xml_writer_t, pp_cim_data_t *);
static void cim_ret_to_ws(xml_writer_t, pp_cim_datatype_t, pp_cim_data_t *);
static pp_cim_data_t * ws_property_to_cim(pp_cim_property_t *, xml_parser_t);
static pp_cim_data_t * ws_arg_to_cim(pp_cim_method_arg_t *, xml_parser_t);
static int set_cim_properties(ws_cim_resource_t *, ws_cim_instance_t *, xml_parser_t, int *, int *, char **);
static int set_cim_fragment(pp_cim_instance_t *, pp_cim_property_t *, const char *val);
static int set_cim_fragment_array(pp_cim_instance_t *, pp_cim_property_t *, xml_parser_t);
static int set_cim_fragment_array_index(pp_cim_instance_t *inst, pp_cim_property_t *, const char *, unsigned int);
static int set_cim_method_parameter(pp_cim_method_call_t *, pp_cim_method_arg_t *, xml_parser_t);
static void del_vector_elem_setprop(void *);
static int new_set_property(ws_cim_resource_t *, vector_t *, vector_t *, const char *, xml_parser_t);
static int new_method_arg(pp_cim_method_arg_t *, vector_t *, xml_parser_t);
static pp_cim_data_t *create_null_cim_data(void);

/******************************************************************************
* data                                                                        *
******************************************************************************/

/* vector of ws_cim_resource_t* */
static vector_t *cim_resources = NULL;

/******************************************************************************
* functions                                                                   *
******************************************************************************/

/* ---- initialization and cleanup ---- */

int cim_resources_init(void) {
    int ret = -1;
    vector_t *cim_classes = NULL;
    size_t i, size;
    
    cim_resources = vector_new(NULL, 0, delete_cim_resource);
    if (!cim_resources) {
    	goto bail;
    }
    
    cim_classes = pp_cim_list_classes();
    if (!cim_classes) {
    	D(D_NOTICE, "Could not query CIM classes.\n");
    	goto bail;
    }
    
    size = vector_size(cim_classes);
    for (i = 0; i < size; i++) {
    	char *classname;
    	pp_cim_class_t *class;

    	classname = vector_get(cim_classes, i);
    	if (!classname) continue;

    	D(D_BLABLA, "Found CIM class %s\n", classname);

    	class = pp_cim_class_lookup(classname);
    	if (!class) continue;
    	
    	if (create_cim_resource(class)) {
    	    D(D_NOTICE, "Could not create CIM resource %s\n", classname);
    	}
    	
    	pp_cim_class_release(class);
    }
    
    ret = 0;
 bail:
    if (cim_classes)	vector_delete(cim_classes);
    return ret;
}

void cim_resources_cleanup(void) {
    if (cim_resources)	vector_delete(cim_resources);
}

/* ---- CIM resource handling ---- */

/* delete one resource */
static void delete_cim_resource(void *elem) {
    ws_cim_resource_t *res = (ws_cim_resource_t *)elem;
    delete_resource((wsman_resource_t *)res);
}

/* create cim resource */
static int create_cim_resource(pp_cim_class_t *class) {
    char name[256];
    ws_cim_resource_t *cim_res;
    wsman_resource_t *res;
    wsman_keyword_t *kw;
    pp_cim_class_t *current;
    wsman_operation_t *enum_ops, *std_ops, *custom_ops;
    vector_t *properties, *methods;
    int no_cimkeys = 0;
    
    /* create the resource */
    D(D_VERBOSE, "Addind CIM resource %s\n", class->cim_name);
    cim_res = realloc(create_resource(NAMESPACE_CIM, class->cim_name), sizeof(ws_cim_resource_t));
    res = (wsman_resource_t *)cim_res;
    cim_res->cim_class = class;
    
    /* add general description */
    if (class->dispname) {
    	resource_set_dispname(res, class->dispname);
    }
    
    kw = add_keywords(res, "cim", NAMESPACE_CIMHELP);
    for (current = class; current; current = current->superclass) {
    	add_keyword(kw, current->cim_name);
    }
    
    /* add selector set "cimkeys" */
    properties = pp_cim_list_properties(class);
    if (properties) {
    	size_t size = vector_size(properties);
    	D(D_BLABLA, "Got %u CIM properties.\n", size);
    	if (size) {
    	    wsman_selector_set_t *cimkeys = resource_add_selectorset(res, "cimkeys");
    	    size_t i;
    	    for (i = 0; i < size; i++) {
    	    	pp_cim_property_t *prop = (pp_cim_property_t *)vector_get(properties, i);
    	    	D(D_BLABLA, "found property %s (key: %d)\n", prop->name, prop->key);
    	    	if (prop->key) {
    	    	    add_selector(cimkeys, prop->name, get_prop_type(prop->type), NULL);
    	    	    no_cimkeys++;
    	    	}
    	    }
    	}
    	vector_delete(properties);
    } else {
    	D(D_BLABLA, "CIM class has no properties.\n");
    }
    
    /* add default operations (Enumeration, Get/Put) */
    if (no_cimkeys) {
    	std_ops = add_operation(res, NULL, NULL);
    	op_add_action(std_ops, ACTION_URI_GET, resource_get);
    	op_add_action(std_ops, ACTION_URI_PUT, resource_put);
    	op_add_selsetref(std_ops, "cimkeys");
    }
    
    enum_ops = add_operation(res, NULL, NULL);
    op_add_action(enum_ops, ACTION_URI_ENUMERATE, resource_enumerate);
    op_add_action(enum_ops, ACTION_URI_PULL, resource_pull);
    op_add_action(enum_ops, ACTION_URI_RELEASE, resource_release);
    
    /* add custom methods (-> CIM methods) */
    if (no_cimkeys) {
    	methods = pp_cim_list_methods(class);
    	if (methods) {
    	    size_t size = vector_size(methods);
    	    D(D_BLABLA, "Got %u CIM methods.\n", size);
    	    if (size) {
    	    	size_t i;
    	    	custom_ops = add_operation(res, NULL, NULL);
    	    	for (i = 0; i < size; i++) {
    	    	    pp_cim_method_t *method = (pp_cim_method_t *)vector_get(methods, i);
    	    	    D(D_BLABLA, "Found method %s\n", method->name);
    	    	    snprintf(name, sizeof(name), "%s/%s/%s",
    	    	    	NAMESPACE_CIM, class->cim_name, method->name);
    	    	    op_add_action(custom_ops, name, invoke_cim_method);
    	    	}
    	    	op_add_selsetref(custom_ops, "cimkeys");
    	    }
    	    vector_delete(methods);
    	}
    }
    
    /* set the required function pointers */
    resource_set_functions(res,
    	get_cim_instance_by_request, get_first_cim_instance, get_next_cim_instance,
    	get_cim_instance_data, get_cim_instance_fragment, get_cim_selector,
    	put_cim_instance_data, put_cim_instance_fragment,
    	release_cim_instance, cleanup_cim_instance);
    
    /* add the resource to our vector and to the resource table */
    vector_add(cim_resources, cim_res);
    add_resource(res, 1);
    
    return 0;
}

/* ---- finding instances ---- */

/* returns an instance addressed by selectors */
static wsman_instance_t * get_cim_instance_by_request(wsman_resource_t *res, wsman_request_t *req, int *fault, const char *auth_string) {
    ws_cim_resource_t *cim_res = (ws_cim_resource_t *)res;
    ws_cim_instance_t *cim_inst = NULL;
    vector_t *instances = NULL;
    unsigned int i;
    
    assert(cim_res);
    
    /* lock access to the class (is released later) */
    pp_cim_class_lock(cim_res->cim_class);
    
    D(D_BLABLA, "Searching CIM instance.\n");
    if (!req->no_selectors) {
    	D(D_NOTICE, "No selectors given to access resource %s\n", cim_res->cim_class->cim_name);
    	goto bail;
    }
    
    /* find all instances */
    D(D_BLABLA, "Finding all CIM instances for %s...\n", cim_res->cim_class->cim_name);
    instances = pp_cim_get_instances(cim_res->cim_class);
    if (!instances) {
    	D(D_VERBOSE, "CIM resource %s has no instances, cannot find instance.\n", cim_res->cim_class->cim_name);
    	goto bail;
    }
    D(D_BLABLA, "...done.\n");
    
    /* go through all instances */
    for (i = 0; i < vector_size(instances); i++) {
    	pp_cim_instance_t *inst = (pp_cim_instance_t *)vector_get(instances, i);
    	D(D_BLABLA, "Checking instance %u\n", i);

        if (pp_cim_get_authorization(inst, auth_string) == PP_ERR)
            continue;
    	
    	if (match_selectors(inst, req, fault)) {
    	    cim_inst = cim_instance_new(inst, cim_res);
    	    break;
    	}
    }
    
 bail:
    pp_cim_class_release(cim_res->cim_class);
    if (instances)	vector_delete(instances);

    return (wsman_instance_t *)cim_inst;
}

/* returns the first instance of a CIM resource */
static wsman_instance_t * get_first_cim_instance(wsman_resource_t *res, wsman_request_t *req, int *is_last, const char *auth_string) {
    ws_cim_resource_t *cim_res = (ws_cim_resource_t *)res;
    ws_cim_instance_t *cim_inst = NULL;
    vector_t *instances = NULL;
    size_t size;
    
    assert(cim_res);

    /* lock access to the class (is released later) */
    pp_cim_class_lock(cim_res->cim_class);

    /* find all instances */
    instances = pp_cim_get_instances(cim_res->cim_class);
    if (!instances) {
    	D(D_VERBOSE, "CIM resource %s has no instances, cannot find instance.\n", cim_res->cim_class->cim_name);
    	goto bail;
    }
    
    size = vector_size(instances);
    if (size) {
    	size_t i;
    	for (i = 0; i < size; i++) {
    	    pp_cim_instance_t *instance = (pp_cim_instance_t *)vector_get(instances, i);
            /* skip instances we are not authorized for */
            if (pp_cim_get_authorization(instance, auth_string) == PP_ERR)
                continue;
    	    /* if selectors are given, process them */
    	    if (req->no_selectors == 0 || match_selectors(instance, req, NULL)) {
    	    	*is_last = size == 1;
    	    	cim_inst = cim_instance_new(instance, cim_res);
    	    	break;
    	    }
    	}
    }
    
 bail:
    pp_cim_class_release(cim_res->cim_class);
    if (instances)	vector_delete(instances);

    return (wsman_instance_t *)cim_inst;
}

/* returns the next instance of a CIM resource */
static wsman_instance_t * get_next_cim_instance(wsman_resource_t *res, wsman_request_t *req, wsman_instance_t *previous, int *is_last, const char *auth_string) {
    ws_cim_resource_t *cim_res = (ws_cim_resource_t *)res;
    ws_cim_instance_t *cim_inst = NULL;
    ws_cim_instance_t *prev = (ws_cim_instance_t *)previous;
    vector_t *instances = NULL;
    size_t size, i, j;
    
    assert(cim_res);
    
    if (!prev) return NULL;

    /* lock access to the class (is released later) */
    pp_cim_class_lock(cim_res->cim_class);

    /* find all instances */
    instances = pp_cim_get_instances(cim_res->cim_class);
    if (!instances) {
    	D(D_VERBOSE, "CIM resource %s has no instances, cannot find instance.\n", cim_res->cim_class->cim_name);
    	goto bail;
    }
    
    size = vector_size(instances);
    for (i = 0; i < size; i++) {
    	pp_cim_instance_t *inst = vector_get(instances, i);
    	if (inst && inst == prev->cim_instance && (i + 1) < size) {
    	    /* we found the last instance, let's find the next one that suits */
    	    for (j = i + 1; j < size; j++) {
    	    	pp_cim_instance_t *new_inst = vector_get(instances, j);
    	    	/* if selectors are given, process them */
                if (pp_cim_get_authorization(new_inst, auth_string) == PP_ERR)
                    continue;
    	    	if (req->no_selectors == 0 || match_selectors(new_inst, req, NULL)) {
    	    	    *is_last = size <= (j + 1);
    	    	    cim_inst = cim_instance_new(new_inst, cim_res);
    	    	    goto bail;
    	    	}
    	    }
    	}
    }
    
 bail:
    pp_cim_class_release(cim_res->cim_class);
    if (instances)	vector_delete(instances);

    return (wsman_instance_t *)cim_inst;
}

/* ---- releasing/cleanup instances ---- */

/* release the lock on a certain instance */
static int release_cim_instance(wsman_resource_t *res UNUSED, wsman_instance_t *instance) {
    ws_cim_instance_t *cim_inst = (ws_cim_instance_t *)instance;

    pp_cim_instance_untag(cim_inst->cim_instance);

    return 0;
}

/* free memory of a certain instance */
static int cleanup_cim_instance(wsman_resource_t *res UNUSED, wsman_instance_t * instance) {
    if (instance) free(instance);
    return 0;
}

/* ---- accessing instance properties ---- */

#define RETURN_ERROR(code, detail, det_str)	\
    do {					\
    	*fault_code = code;			\
    	*fault_detail = detail;			\
    	*fault_detail_string = det_str;		\
    	ret = -1;				\
    	goto bail;				\
    } while(0)

static void print_property(xml_writer_t xml, const char *ns, const char *name, int text_only, char *val) {
    D(D_BLABLA, "Property %s is: %s\n", name, val ? val : "NULL");

    if (text_only) {
    	print_xml_text(xml, val ? val : "xsi:nil");
    } else if (val) {
    	print_xml_tag(xml, ns, name, val, NULL);
    } else {
    	print_xml_tag(xml, ns, name, NULL, "xsi", "nil", "", "true");
    }
    
    free(val);
}

static int get_cim_instance_data(xml_writer_t xml, wsman_request_t *req UNUSED, wsman_instance_t *instance,
				int *fault_code, int *fault_detail, char **fault_detail_string) {
    int ret = 0;
    ws_cim_instance_t *cim_inst = (ws_cim_instance_t *)instance;
    ws_cim_resource_t *cim_res = (ws_cim_resource_t *)instance->resource;
    vector_t *properties = pp_cim_list_properties(cim_res->cim_class);
    unsigned int i;
    char ns_helper[200];
    xml_writer_t cim_xml;
    
    /* find all properties */
    if (!properties || !vector_size(properties)) {
    	D(D_ERROR, "Could not query properties of CIM instance.\n");
    	RETURN_ERROR(FAULT_INTERNAL_ERROR, 0, "Could not query properties of CIM instance");
    }

    /* print all properties */
    snprintf(ns_helper, sizeof(ns_helper), "%s/%s", NAMESPACE_CIM, cim_res->cim_class->cim_name);
    cim_xml = print_xml_tag(xml, "cim_res", cim_res->cim_class->cim_name, NULL,
    	"xmlns", "cim", "", NAMESPACE_CIMHELP,
    	"xmlns", "cim_res", "", ns_helper,
    	"cim", "Class", "", cim_res->cim_class->cim_name);
    pp_cim_update_properties(cim_inst->cim_instance);
    for (i = 0; i < vector_size(properties); i++) {
    	pp_cim_property_t *prop = vector_get(properties, i);
	pp_cim_propval_t *pv = pp_cim_get_property(cim_inst->cim_instance, prop->name);
    	int j;
    	
    	for (j = 0; j < property_array_size(pv); j++) {
    	    print_property(cim_xml, "cim_res", pv->property->name, 0,
    	    	cim_property_to_ws_array(pv, j));
    	}
    }
    
 bail:
    if (properties) vector_delete(properties);

    return ret;
}

static int get_cim_instance_fragment(xml_writer_t xml, wsman_request_t *req UNUSED, wsman_instance_t *instance,
				     int *fault_code, int *fault_detail, char **fault_detail_string) {
    int ret = 0;
    ws_cim_instance_t *cim_inst = (ws_cim_instance_t *)instance;
    ws_cim_resource_t *cim_res = (ws_cim_resource_t *)instance->resource;
    xml_writer_t frag;
    wsman_fragment_t *fragment;
    pp_cim_propval_t *prop = NULL;
    
    /* check for plausibility */
    if (vector_size(req->fragment.fragments) != 2 ||
    	(fragment = (wsman_fragment_t *)vector_get(req->fragment.fragments, 0))->array_index ||
    	strcasecmp(fragment->fragment, cim_res->cim_class->cim_name) ||
    	fragment->array_index) {
    	
    	D(D_ERROR, "Fragment %s not found.\n", req->fragment.fragment_full);
    	RETURN_ERROR(FAULT_INVALID_REPRESENTATION, FAULT_DETAIL_REPRESENTATION_INVALID_FRAGMENT, 0);
    }
    
    /* find the requested fragment property */
    fragment = (wsman_fragment_t *)vector_get(req->fragment.fragments, 1);
    pp_cim_update_properties(cim_inst->cim_instance);
    prop = pp_cim_get_property(cim_inst->cim_instance, fragment->fragment);
    if (!prop) {
    	D(D_ERROR, "Fragment %s not property.\n", req->fragment.fragment_full);
    	RETURN_ERROR(FAULT_INVALID_REPRESENTATION, FAULT_DETAIL_REPRESENTATION_INVALID_FRAGMENT, 0);
    }
    
    /* calculate and set property XML text */
    frag = print_xml_tag(xml, "wsman", "XmlFragment", NULL, NULL);

    if (!prop->property->array && !fragment->array_index) {
    	/* simple element */
    	print_property(frag, "", fragment->fragment, req->fragment.text_only,
    		cim_property_to_ws(prop));
    } else if (prop->property->array && fragment->array_index) {
    	/* array element */
    	print_property(frag, "", fragment->fragment, req->fragment.text_only,
    		cim_property_to_ws_array(prop, fragment->array_index - 1));
    } else if (prop->property->array && !fragment->array_index && !req->fragment.text_only) {
    	int i;
    	for (i = 0; i < property_array_size(prop); i++) {
    	    print_property(frag, "", fragment->fragment, req->fragment.text_only,
    	    	cim_property_to_ws_array(prop, i));
    	}
    } else {
    	/* simple/array mismatch */
    	D(D_NOTICE, "simple/array mismatch for fragment %s\n", req->fragment.fragment_full);
    	RETURN_ERROR(FAULT_INVALID_REPRESENTATION, FAULT_DETAIL_REPRESENTATION_INVALID_FRAGMENT, 0);
    }

 bail:
    if (prop) pp_cim_propval_delete(prop);
    return ret;
}

static int put_cim_instance_data(wsman_instance_t *instance, wsman_request_t *req,
				int *fault_code, int *fault_detail, char **fault_detail_string) {
    int ret = 0;
    xml_parser_t res_xml = NULL, child;
    ws_cim_instance_t *cim_inst = (ws_cim_instance_t *)instance;
    ws_cim_resource_t *cim_res = (ws_cim_resource_t *)instance->resource;
    char ns_helper[200];
    char *namespace;
    vector_t *class_props = NULL;
    size_t i;
    
    /* find the CIM class tag */
    snprintf(ns_helper, sizeof(ns_helper), "%s/%s", NAMESPACE_CIM, cim_res->cim_class->cim_name);
    if (!(res_xml = xml_get_child(req->body, namespace = ns_helper, cim_res->cim_class->cim_name)) &&
    	!(res_xml = xml_get_child(req->body, namespace = NAMESPACE_CIM, cim_res->cim_class->cim_name))) {
    	D(D_NOTICE, "Tag %s not found in body.\n", cim_res->cim_class->cim_name);
    	RETURN_ERROR(FAULT_INVALID_REPRESENTATION, FAULT_DETAIL_REPRESENTATION_MISSING_VALUES, NULL);
    }
    
    /* we got it, now parse it */
    if (!(child = xml_get_first_child(res_xml))) {
    	D(D_NOTICE, "Tag %s has no child tags!\n", cim_res->cim_class->cim_name);
    	RETURN_ERROR(FAULT_INVALID_REPRESENTATION, FAULT_DETAIL_REPRESENTATION_MISSING_VALUES, NULL);
    }
    
    /* check if all values are updated (necessary according to spec) */
    class_props = pp_cim_list_properties(cim_res->cim_class);
    for (i = 0; i < vector_size(class_props); i++) {
    	pp_cim_property_t *p = vector_get(class_props, i);
    	if (!xml_get_child(res_xml, namespace, p->name)) {
    	    D(D_NOTICE, "Value for property %s not in XML!\n", p->name);
    	    RETURN_ERROR(FAULT_INVALID_REPRESENTATION, FAULT_DETAIL_REPRESENTATION_MISSING_VALUES, NULL);
    	}
    }
    
    /* now set the properties */
    ret = set_cim_properties(cim_res, cim_inst, child, fault_code, fault_detail, fault_detail_string);

 bail:
    if (class_props)	vector_delete(class_props);
    
    return ret;
}

static int put_cim_instance_fragment(wsman_instance_t *instance, wsman_request_t *req,
				     int *fault_code, int *fault_detail, char **fault_detail_string) {
    int ret = 0;
    ws_cim_instance_t *cim_inst = (ws_cim_instance_t *)instance;
    ws_cim_resource_t *cim_res = (ws_cim_resource_t *)instance->resource;
    xml_parser_t xml_fragment;
    wsman_fragment_t *fragment;
    pp_cim_property_t *prop = NULL;
    char *val;
    
    /* check for plausibility */
    if (vector_size(req->fragment.fragments) != 2 ||
    	(fragment = (wsman_fragment_t *)vector_get(req->fragment.fragments, 0))->array_index ||
    	strcasecmp(fragment->fragment, cim_res->cim_class->cim_name) ||
    	fragment->array_index) {
    	
    	D(D_ERROR, "Fragment %s not found.\n", req->fragment.fragment_full);
    	RETURN_ERROR(FAULT_INVALID_REPRESENTATION, FAULT_DETAIL_REPRESENTATION_INVALID_FRAGMENT, 0);
    }
    
    /* find the property which matches the fragment */
    fragment = (wsman_fragment_t *)vector_get(req->fragment.fragments, 1);
    prop = pp_cim_class_get_property(cim_res->cim_class, fragment->fragment);
    if (!prop) {
    	D(D_ERROR, "Fragment %s not property.\n", req->fragment.fragment_full);
    	RETURN_ERROR(FAULT_INVALID_REPRESENTATION, FAULT_DETAIL_REPRESENTATION_INVALID_FRAGMENT, 0);
    }
    if (!prop->writable) {
    	D(D_NOTICE, "Trying to update read-only property %s\n", prop->name);
    	RETURN_ERROR(FAULT_ACTION_NOT_SUPPORTED, FAULT_DETAIL_READ_ONLY, NULL);
    }
    
    /* find the fragment in the body */
    xml_fragment = xml_get_child(req->body, namespace_management, "XmlFragment");
    if (xml_fragment && !req->fragment.text_only) {
    	xml_fragment = xml_get_child(xml_fragment, "", fragment->fragment);
    }
    if (!xml_fragment || !(val = xml_get_text(xml_fragment))) {
    	D(D_NOTICE, "Could not find fragment data.\n");
    	RETURN_ERROR(FAULT_INVALID_REPRESENTATION, FAULT_DETAIL_REPRESENTATION_INVALID_FRAGMENT, 0);
    }
    D(D_BLABLA, "fragment value is %s\n", val);
    
    /* calculate the value */
    if (!prop->array && !fragment->array_index) {
    	/* simple element */
    	ret = set_cim_fragment(cim_inst->cim_instance, prop, val);
    } else if (prop->array && fragment->array_index) {
    	/* array element */
    	ret = set_cim_fragment_array_index(cim_inst->cim_instance, prop, val, fragment->array_index - 1);
    } else if (prop->array && !fragment->array_index && !req->fragment.text_only) {
    	/* array */
    	ret = set_cim_fragment_array(cim_inst->cim_instance, prop, xml_fragment);
    } else {
    	/* simple/array mismatch */
    	D(D_NOTICE, "simple/array mismatch for fragment %s\n", req->fragment.fragment_full);
    	ret = -1;
    }
    if (ret) {
    	RETURN_ERROR(FAULT_INVALID_REPRESENTATION, FAULT_DETAIL_REPRESENTATION_INVALID_FRAGMENT, 0);
    }

 bail:
    return ret;
}

static int get_cim_selector(xml_writer_t xml, wsman_request_t *req UNUSED, wsman_instance_t *instance,
			    int *fault_code, int *fault_detail, char **fault_detail_string) {
    int ret = 0;
    ws_cim_instance_t *cim_inst = (ws_cim_instance_t *)instance;
    vector_t *properties;
    unsigned int i;
    xml_writer_t selset = print_xml_tag(xml, "wsman", "SelectorSet", NULL, NULL);

    pp_cim_update_properties(cim_inst->cim_instance);
    properties = pp_cim_get_properties(cim_inst->cim_instance);

    if (!properties || !vector_size(properties)) {
    	D(D_ERROR, "Could not query properties of CIM instance.\n");
    	RETURN_ERROR(FAULT_INTERNAL_ERROR, 0, "Could not query properties of CIM instance");
    }
    
    for (i = 0; i < vector_size(properties); i++) {
    	pp_cim_propval_t *prop = vector_get(properties, i);
    	int j;
    	
    	if (!prop->property->key) continue;
    	
    	for (j = 0; j < property_array_size(prop); j++) {
    	    char *val = cim_property_to_ws_array(prop, j);
    	
    	    if (val) {
    	    	print_xml_tag(selset, "wsman", "Selector", val, "", "Name", "", prop->property->name);
    	    	free(val);
    	    }
    	}
    }
    
 bail:
    if (properties) vector_delete(properties);

    return ret;
}

#define ACTION_SIZE	512

static int invoke_cim_method(xml_writer_t xml, wsman_resource_t *res, wsman_request_t * req, char ** action,
			     int *fault_code, int *fault_detail, char **fault_detail_string, char *auth_string) {
    int ret = 0;
    int fault = 0;
    ws_cim_instance_t *cim_inst;
    ws_cim_resource_t *cim_res = (ws_cim_resource_t *)res;
    char *namespace = NULL;
    char *method_name;
    char tagname[256], ns_helper[256];
    xml_parser_t req_xml;
    xml_writer_t ret_xml;
    pp_cim_method_call_t *call = NULL;
    pp_cim_methodptr_t *method = NULL;
    pp_cim_data_t *result = NULL;
    
    assert(res && req);
    
    req_xml = req->body;
    snprintf(ns_helper, sizeof(ns_helper), "%s/%s", NAMESPACE_CIM, cim_res->cim_class->cim_name);

    /* try to find the correct instance */
    if (!(cim_inst = (ws_cim_instance_t *)res->get_instance_by_request(res, req, &fault, auth_string))) {
    	D(D_ERROR, "the requested instance could not be found.\n");
    	RETURN_ERROR(FAULT_INVALID_SELECTORS, fault ? fault : FAULT_DETAIL_SELECTOR_INVALID_VALUE, NULL);
    }
        
    /* find method name */
    method_name = strrchr(req->action_uri, '/');
    if (method_name) {
    	method_name++; /* skip '/' */
    } else {
    	D(D_ERROR, "CIM method name not found in %s\n", req->action_uri);
    	RETURN_ERROR(FAULT_ACTION_NOT_SUPPORTED, 0, req->action_uri);
    }
    D(D_VERBOSE, "CIM method %s called.\n", method_name);

    /* find the CIM method */
    method = pp_cim_get_method(cim_inst->cim_instance, method_name);
    if (!method) {
    	D(D_NOTICE, "Method %s not found.\n", method_name);
    	RETURN_ERROR(FAULT_ACTION_NOT_SUPPORTED, 0, req->action_uri);
    }
    call = pp_cim_method_call_new(method, cim_inst->cim_instance);
    
    /* extract the message parameters */
    D(D_BLABLA, "method has %d parameters.\n", method->method->num_args);
    if (method->method->num_args > 0) {
    	int i;
    	xml_parser_t input;
    	
    	/* find the CIM method parameters */
    	snprintf(tagname, sizeof(tagname), "%s_INPUT", method_name);
    	if (!(input = xml_get_child(req->body, namespace = ns_helper, tagname)) &&
    	    !(input = xml_get_child(req->body, namespace = NAMESPACE_CIM, tagname))) {
    	    
    	    D(D_NOTICE, "Tag %s not found in body.\n", cim_res->cim_class->cim_name);
    	    RETURN_ERROR(FAULT_INVALID_REPRESENTATION, FAULT_DETAIL_REPRESENTATION_MISSING_VALUES, NULL);
    	}
    	
    	/* all parameters must be present */
    	for (i = 0; i < method->method->num_args; i++) {
    	    xml_parser_t arg_xml = NULL;
    	    pp_cim_method_arg_t *arg = &method->method->args[i];
    	
    	    D(D_BLABLA, "searching argument %s\n", arg->name);
    	    
    	    if (arg->in && !(arg_xml = xml_get_child(input, namespace, arg->name))) {
    	    	D(D_ERROR, "argument %s not found!\n", arg->name);
    	    	RETURN_ERROR(FAULT_INVALID_REPRESENTATION, FAULT_DETAIL_REPRESENTATION_MISSING_VALUES, NULL);
    	    }
    	    
    	    /* we have the XML parameter, add it to the method call
    	       NOTE: this call also allocates space for OUT parameters! */
    	    if (set_cim_method_parameter(call, arg, arg_xml)) {
    	    	D(D_ERROR, "error setting argument %s!\n", arg->name);
    	    	RETURN_ERROR(FAULT_INVALID_REPRESENTATION, FAULT_DETAIL_REPRESENTATION_INVALID_VALUES, NULL);
    	    }
    	}
    }

    /* invoke CIM method */
    result = create_null_cim_data();
    
    if (pp_cim_method_call_exec(call, result)) {
    	D(D_NOTICE, "Error calling method.\n");
    	RETURN_ERROR(FAULT_INTERNAL_ERROR, 0, "Error executing method");
    }
    
    /* set return values */
    snprintf(tagname, sizeof(tagname), "%s_OUTPUT", method_name);
    ret_xml = print_xml_tag(xml, "cim_res", tagname, NULL, 
    	"xmlns", "cim", "", NAMESPACE_CIMHELP,
    	"xmlns", "cim_res", "", ns_helper);

    cim_ret_to_ws(ret_xml, method->method->result, result);
    
    if (method->method->num_args > 0) {
    	int i;
    	for (i = 0; i < method->method->num_args; i++) {
    	    pp_cim_method_arg_t *arg = &method->method->args[i];
    	    pp_cim_data_t *data = vector_get(call->args, i);
    	    if (!arg->out) continue;
    	    cim_arg_to_ws(arg, ret_xml, data);
    	}
    }    
    
    /* set the correct action URI */
    *action = malloc(ACTION_SIZE);
    snprintf(*action, ACTION_SIZE, "%sResponse", req->action_uri);
    *action = realloc(*action, strlen(*action)); // save some space
    
 bail:
    if (cim_inst) {
    	res->release_instance(res, (wsman_instance_t *)cim_inst);
    	res->cleanup_instance(res, (wsman_instance_t *)cim_inst);
    }
    if (result && method) pp_cim_data_delete(method->method->result, 0, result);
    if (call) pp_cim_method_call_delete(call);
    return ret;
}

/* ---- helper functions ---- */

/* map CIM datatype to XML datatype */
static char* get_prop_type(pp_cim_datatype_t type) {
    switch (type) {
    	case PP_CIM_BOOLEAN:	return "xs:boolean";
    	case PP_CIM_SIGNED:	return "xs:int";
    	case PP_CIM_UNSIGNED:	return "xs:unsignedInt";
    	case PP_CIM_REAL:	return "xs:double";
    	case PP_CIM_STRING:	return "xs:string";
    	default:		return NULL;
    }
    return NULL;
}

static int match_selectors(pp_cim_instance_t *inst, wsman_request_t *req, int *fault) {
    int j;
    int okay = 0, failed = 0;

    pp_cim_update_properties(inst);
    /* go through all selectors and match */
    for (j = 0; j < req->no_selectors; j++) {
    	pp_cim_propval_t *prop;
    	
    	prop = pp_cim_get_property(inst, req->selectors[j].name);
    	
    	/* match the property */
    	if (!prop || !prop->property->key || !match_property(prop, &req->selectors[j])) {
    	    D(D_VERBOSE, "Could not match property %s\n", req->selectors[j].name);
    	    if (fault && (!prop || !prop->property->key)) {
    	    	*fault = FAULT_DETAIL_SELECTOR_UNEXPECTED_SELECTORS;
    	    }
    	    failed++;
    	} else {
    	    okay++;
    	    D(D_BLABLA, "Property %i matches.\n", j);
    	}

    	/* cleanup property */
    	pp_cim_propval_delete(prop);
    }
    
    /* if okay is still set to 1, we got the instance */
    D(D_BLABLA, "%d selectors matched, %d did not\n", okay, failed);
    if (okay && !failed) {
    	return 1;
    }
    return 0;
}

/* property matching */
static int match_property(pp_cim_propval_t *cim_prop, wsman_req_selector_t *prop) {
    char *s;
    int ret = 0;
    
    switch (prop->type) {
    	case SELECTOR_TYPE_SIMPLE:
    	    s = cim_property_to_ws(cim_prop);
    	    if (s) {
    	    	if (!strcasecmp(s, prop->value.simple)) {
    	    	    D(D_BLABLA, "Property matching: %s\n", s);
    	    	    ret = 1;
    	    	} else {
    	    	    D(D_BLABLA, "Property mismatch: %s %s\n", prop->value.simple, s);
    	    	}
    	    	free(s);
    	    }
    	    break;
    	default:
    	    /* only simple selectors are supported currently */
    	    break;
    }
    
    return ret;
}

static int property_is_array(pp_cim_propval_t *cim_prop) {
    return cim_prop->property->array;
}

static int property_array_size(pp_cim_propval_t *cim_prop) {
    if (cim_prop->data.null) {
	return 0;
    }
    if (!cim_prop->property->array) {
    	return 1;
    }    
    return vector_size(cim_prop->data.types.array);
}

/* convert CIM property to string */
static char* cim_property_to_ws(pp_cim_propval_t *cim_prop) {
    if (property_is_array(cim_prop)) {
    	return NULL;
    }
    
    return pp_cim_data_print(cim_prop->property->type, cim_prop->data);
}

static char* cim_property_to_ws_array(pp_cim_propval_t *cim_prop, int idx) {
    pp_cim_data_t *data;
    if (!property_is_array(cim_prop)) {
    	return cim_property_to_ws(cim_prop);
    }
    
    if (idx >= property_array_size(cim_prop)) return NULL;
    
    data = (pp_cim_data_t *)vector_get(cim_prop->data.types.array, idx);
    return pp_cim_data_print(cim_prop->property->type, *data);
}

static void cim_arg_to_ws_internal(xml_writer_t xml, const char *name, pp_cim_datatype_t type, pp_cim_data_t *data) {
    char *val = pp_cim_data_print(type, *data);
    if (val) {
    	print_xml_tag(xml, "", name, val, NULL);
    	free(val);
    } else {
    	print_xml_tag(xml, "", name, NULL, "xsi", "nil", "", "true");
    }
}

static int cim_arg_to_ws(pp_cim_method_arg_t *arg, xml_writer_t xml, pp_cim_data_t *data) {
    if (data->null) {
    	print_xml_tag(xml, "", arg->name, NULL, "xsi", "nil", "", "true");
    } else if (arg->array) {
    	size_t i;
    	for (i = 0; i < vector_size(data->types.array); i++) {
    	    pp_cim_data_t *elem = vector_get(data->types.array, i);
    	    cim_arg_to_ws_internal(xml, arg->name, arg->type, elem);
    	}
    } else {
    	cim_arg_to_ws_internal(xml, arg->name, arg->type, data);
    }
    return 0;
}

static void cim_ret_to_ws(xml_writer_t xml, pp_cim_datatype_t type, pp_cim_data_t *data) {
    if (type != PP_CIM_VOID) {
    	if (data->null) {
    	    print_xml_tag(xml, "", "ReturnCode", NULL, "xsi", "nil", "", "true");
    	} else {
    	    cim_arg_to_ws_internal(xml, "ReturnCode", type, data);
    	}
    }
}

/* convert WS-Management properties to CIM properties */
static pp_cim_data_t *ws_property_to_cim_helper(pp_cim_datatype_t type, int array, const char *val) {
    pp_cim_data_t* ret = NULL;
    
    if (array) {
    	ret = malloc(sizeof(pp_cim_data_t));
    	memset(ret, 0, sizeof(pp_cim_data_t));
    	ret->types.array = vector_new(NULL, 1, NULL);
    	return ret;
    } else if (!val || val[0] == '\0') {
    	return create_null_cim_data();
    }
    
    return pp_cim_data_parse(type, val);
}

static pp_cim_data_t *ws_property_to_cim_internal(pp_cim_datatype_t type, int array, const char *name, xml_parser_t xml) {
    if (array) {
    	pp_cim_data_t *data = ws_property_to_cim_helper(type, 1, NULL);
    	if (!data) return NULL;
    	D(D_BLABLA, "Setting property array %s\n", name);
    	
    	while (xml) {
    	    pp_cim_data_t *newdata = ws_property_to_cim_helper(type, 0, xml_get_text(xml));
    	    if (!newdata) {
    	    	pp_cim_data_delete(type, 1, data);
    	    	return NULL;
    	    }
    	    vector_add(data->types.array, newdata);
    	    xml = xml_get_next_same_name(xml);
    	}
    	
    	return data;
    } else {
    	D(D_BLABLA, "Setting property %s\n", name);
    	return ws_property_to_cim_helper(type, 0, xml_get_text(xml));
    }
}

static pp_cim_data_t *ws_property_to_cim(pp_cim_property_t *property, xml_parser_t xml) {
    return ws_property_to_cim_internal(property->type, property->array, property->name, xml);
}

/* method args are of a type similar to properties */
static pp_cim_data_t * ws_arg_to_cim(pp_cim_method_arg_t *arg, xml_parser_t xml) {
    return ws_property_to_cim_internal(arg->type, arg->array, arg->name, xml);
}

/* setting properties in CIM */
static int set_cim_properties(ws_cim_resource_t *cim_res, ws_cim_instance_t *cim_inst, xml_parser_t xml,
			      int *fault_code, int *fault_detail, char **fault_detail_string) {
    int ret = 0;
    vector_t *names = vector_new(NULL, 0, free);
    vector_t *props = vector_new(NULL, 0, del_vector_elem_setprop);
    
    for (; xml; xml = xml_get_next_different_name(xml)) {
    	const char *propname = xml_get_name(xml);
    	
    	D(D_BLABLA, "Got property %s\n", propname);
    	
    	if (new_set_property(cim_res, names, props, propname, xml)) {
    	    D(D_NOTICE, "Could not set CIM property %s!\n", propname);
    	    RETURN_ERROR(FAULT_INVALID_REPRESENTATION, FAULT_DETAIL_REPRESENTATION_INVALID_VALUES, NULL);
    	}
    }
    
    D(D_BLABLA, "Setting properties in CIM server.\n");
    pp_cim_set_properties(cim_inst->cim_instance, names, props);
    pp_cim_commit_properties(cim_inst->cim_instance);

    if (props)		vector_delete(props);
    if (names)		vector_delete(names);
    
 bail:
    return ret;
}

/* setting of properties given by fragment */
static int set_cim_fragment(pp_cim_instance_t *inst, pp_cim_property_t *property, const char *val) {
    pp_cim_data_t *data = NULL;
    int ret = -1;

    /* create property data structure */
    if (!(data = ws_property_to_cim_helper(property->type, 0, val))) {
    	D(D_ERROR, "Could not get parsed data.\n");
    	goto bail;
    }

    /* set this in CIM server */
    D(D_BLABLA, "Setting properties in CIM server.\n");
    pp_cim_set_property(inst, property->name, *data);
    pp_cim_commit_properties(inst);
    
    ret = 0;
    
 bail:
    if (data) pp_cim_data_delete(property->type, property->type, data);
    return ret;
}

static int set_cim_fragment_array(pp_cim_instance_t *inst, pp_cim_property_t *property, xml_parser_t xml) {
    pp_cim_data_t *data = NULL;
    int ret = -1;

    /* create property data structure */
    if (!(data = ws_property_to_cim_helper(property->type, 1, NULL))) {
    	D(D_ERROR, "Could not get parsed data.\n");
    	goto bail;
    }
    
    while (xml) {
    	pp_cim_data_t *datanew = ws_property_to_cim_helper(property->type, 0, xml_get_text(xml));
    	if (!datanew) {
    	    D(D_ERROR, "Could not get parsed data.\n");
    	    goto bail;
    	}
    	vector_add(data->types.array, datanew);
    	xml = xml_get_next_same_name(xml);
    }

    /* set this in CIM server */
    D(D_BLABLA, "Setting properties in CIM server.\n");
    pp_cim_set_property(inst, property->name, *data);
    pp_cim_commit_properties(inst);
    
    ret = 0;
    
 bail:
    if (data) pp_cim_data_delete(property->type, property->type, data);
    return ret;
}

static int set_cim_fragment_array_index(pp_cim_instance_t *inst, pp_cim_property_t *property,
					const char *val, unsigned int idx) {
    int ret = -1;
    pp_cim_propval_t *old_props = NULL;
    pp_cim_data_t *data = NULL;
    pp_cim_data_t *data_new = NULL;
    
    /* get the current data */
    pp_cim_update_properties(inst);
    old_props = pp_cim_get_property(inst, property->name);
    
    /* copy data */
    if (!(data = pp_cim_data_dup(property->type, 1, &old_props->data))) {
    	D(D_ERROR, "Could not get parsed data.\n");
    	goto bail;
    }
    
    /* create new element */
    if (!(data_new = ws_property_to_cim_helper(property->type, 0, val))) {
    	D(D_ERROR, "Could not get parsed data.\n");
    	goto bail;
    }
    
    /* replace the element */
    if (idx < vector_size(data->types.array)) {
    	D(D_BLABLA, "index is %u, size is %u; replacing element.\n", idx, vector_size(data->types.array));
    	pp_cim_data_delete(property->type, 0, vector_get(data->types.array, idx));
    	vector_set(data->types.array, idx, data_new);
    } else {
    	while (idx > vector_size(data->types.array)) {
    	    pp_cim_data_t *d = create_null_cim_data();
    	    D(D_BLABLA, "index is %u, size is %u; filling up with elements.\n", idx, vector_size(data->types.array));
    	    vector_add(data->types.array, d);
    	}
    	D(D_BLABLA, "index is %u, size is %u; adding element at end.\n", idx, vector_size(data->types.array));
    	vector_add(data->types.array, data_new);
    }
    
    /* set this in CIM server */
    D(D_BLABLA, "Setting properties in CIM server.\n");
    pp_cim_set_property(inst, property->name, *data);
    pp_cim_commit_properties(inst);
    
    ret = 0;
    
 bail:
    if (data) pp_cim_data_delete(property->type, property->array, data);
    if (old_props) pp_cim_propval_delete(old_props);
    return ret;
    
}

static int set_cim_method_parameter(pp_cim_method_call_t *call, pp_cim_method_arg_t *arg, xml_parser_t xml) {
    return new_method_arg(arg, call->args, xml);
}

/* Management of set_properties function calls */
static void del_vector_elem_setprop(void *elem) {
    ws_cim_data_t *data = (ws_cim_data_t *)elem;
    pp_cim_data_delete(data->type, data->array, (pp_cim_data_t *)data);
}

static int new_set_property(ws_cim_resource_t *cim_res, vector_t *names, vector_t *props, const char *name, xml_parser_t xml) {
    ws_cim_data_t *data;
    pp_cim_property_t *property = pp_cim_class_get_property(cim_res->cim_class, name);
    
    if (!property) {
    	return -1;
    }
    if (!property->writable) {
    	return 0;
    }
    
    data = (ws_cim_data_t *)ws_property_to_cim(property, xml);
    if (!data) return -1;
    data = realloc(data, sizeof(ws_cim_data_t));
    data->array = property->array;
    data->type = property->type;
    vector_add(names, strdup(name));
    vector_add(props, data);

    return 0;
}

static int new_method_arg(pp_cim_method_arg_t *arg, vector_t *args, xml_parser_t xml) {
    pp_cim_data_t *data;
    if (arg->in && xml) {
    	data = ws_arg_to_cim(arg, xml);
    } else {
    	data = create_null_cim_data();
    }
    
    if (data) {
    	vector_add(args, data);
    	return 0;
    }
    return -1;
}

/* create a new WS-Management CIM instance */
static ws_cim_instance_t *cim_instance_new(pp_cim_instance_t *inst, ws_cim_resource_t *res) {
   ws_cim_instance_t *cim_inst = malloc(sizeof(ws_cim_instance_t));
   initialize_instance((wsman_instance_t *)cim_inst, (wsman_resource_t *)res);
   cim_inst->cim_instance = inst;
   pp_cim_instance_tag(inst);
   
   return cim_inst;
}

static pp_cim_data_t *create_null_cim_data(void) {
    pp_cim_data_t *data;
    data = malloc(sizeof(pp_cim_data_t));
    memset(data, 0, sizeof(pp_cim_data_t));
    data->null = 1;
    return data;
}
