/**
 * ahb_dma.c
 *
 * KIRA100 AHB DMA kernel module
 *
 * (c) 2006 Peppercon AG, Danny Baumann <daba@raritan.com>
 */

/* 
The usage of this module is as follows:

Drivers can request a DMA channel by calling ahb_dma_request_channel. If the driver
got the channel, it can add an action item to the DMA (up to llp_count+1 times):
            
            ahb_dma_parm_t parm;
            parm.src=SRC_ADDR;                      //given source phy addr
            parm.dest=DEST_ADDR;                    //given dest phy addr
            parm.sw=AHBDMA_WIDTH_32BIT;             //source width 32 bit
            parm.dw=AHBDMA_WIDTH_32BIT;             //dest width 32 bit
            parm.sctl=AHBDMA_CTL_FIX;               //source fix control
            parm.dctl=AHBDMA_CTL_FIX;               //dest fix control
            parm.size=DMA_SIZE;                     //transfer count
            parm.irq=AHBDMA_NO_TRIGGER_IRQ;         //irq trigger?
            ahb_dma_channel_add(priv,&parm);        //call add function

The actual transfer is started by calling ahb_dma_channel_start. 

*/

#include <linux/module.h>
#include <linux/mm.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/wait.h>
#include <linux/interrupt.h>
#include <linux/pci.h>
#include <asm/types.h>
#include <asm/io.h>
#include <asm/arch/ahb_dma.h>

/* local function prototypes */
static int ahb_dma_init(void);
static void ahb_dma_cleanup(void);

static ahb_dma_data_t dma_data;

static irqreturn_t ahb_dma_interrupt_handler(int irq, void *_data, struct pt_regs *dummy)
{
    ahb_dma_data_t *data = _data;
    uint32_t stat, i;

    stat = data->base[AHBDMA_INT_ERR] | data->base[AHBDMA_INT_TC];
    i = 0;
    
    while (stat != 0) {
        if (stat & 0x1) {
            /* set the appropriate status bit - must be reset by the user */
            if (data->base[AHBDMA_INT_ERR] & (1 << i)) data->data[i]->status = INT_ERROR;
            else data->data[i]->status = INT_TRIGGER;
            /* call interrupt handler */
            if (data->irq_handler[i]) data->irq_handler[i](data->isr_data[i]);
            /* reset interrupt source bit */
            if (data->base[AHBDMA_INT_ERR] & (1 << i)) data->base[AHBDMA_INT_ERR_CLR] = (1 << i);
            else data->base[AHBDMA_INT_TC_CLR] = (1 << i);
            /* we have transferred everything, so reset the lld counter to get ready for the next transaction */
            data->data[i]->llp_free_idx = 0;
            data->data[i]->llp_last_idx = 0;
            /* mark channel as inactive */
            data->active_channels &= ~(1 << i);
        }
        i++;
        stat >>= 1;
    }

    return IRQ_HANDLED;
}

int __init ahb_dma_init(void)
{
    int r, ret;

    memset(&dma_data, 0, sizeof(ahb_dma_data_t));

    /* set base address */
    dma_data.base = (u32*)(CPE_AHBDMA_VA_BASE);

    /* request irq */
    set_irq_type(IRQ_DMAC, IRQT_HIGH);
    if ((r = request_irq(IRQ_DMAC, ahb_dma_interrupt_handler, SA_INTERRUPT, "ahb_dma", &dma_data)) == 0) {
        dma_data.irq_requested = 1;
    } else {
        ret = -EIO;
        goto fail;
    }
    
    /* all channels are unused at the beginning */
    dma_data.active_channels = 0;
    
    /* enable DMA engine */
    dma_data.base[AHBDMA_CSR] |= 0x1;
    
    return 0;

fail:
    return ret;
}

static void ahb_dma_cleanup(void)
{
    if (dma_data.irq_requested) free_irq(IRQ_DMAC, &dma_data);
}

int ahb_dma_request_channel(ahb_dma_irq_handler_t irq_handler, void *isr_data, int llp_count, ahb_dma_channel_data_t **channel_data)
{
    int size, channel;
    ahb_dma_channel_data_t *data;

    if (channel_data == NULL)
	return -EINVAL;
   
    for (channel=0;((channel < AHB_DMA_MAX_CHANNELS) && (dma_data.channel_used[channel])); channel++);

    if (channel < AHB_DMA_MAX_CHANNELS) {
        if ((*channel_data = (ahb_dma_channel_data_t *)kmalloc(sizeof(ahb_dma_channel_data_t), GFP_KERNEL)) == NULL) {
            return -ENOMEM;
        }
	data = *channel_data;
        dma_data.data[channel] = data;
        memset(data, 0, sizeof(ahb_dma_channel_data_t));
        data->base = (u32*)(((uint32_t)dma_data.base)+0x100+(channel*0x20));
        data->channel = channel;
        data->llp_count = llp_count;
        if(llp_count != 0) {
            size = sizeof(ahb_lld_t)*llp_count;
            data->ahb_dma_lld = pci_alloc_consistent(NULL, size, (dma_addr_t *)&data->ahb_dma_lld_phys);
            if(data->ahb_dma_lld == NULL) {
                return -ENOMEM;
            }
            memset(data->ahb_dma_lld, 0, size);
        } 
        
        data->base[AHBDMA_CHANNEL_LLP] = 0;     //no LLP as default
        data->base[AHBDMA_CHANNEL_CFG] = 0x0;   //enable all interrupts
        data->base[AHBDMA_CHANNEL_CSR] = 0;
        
        dma_data.channel_used[channel] = 1;
        dma_data.irq_handler[channel] = irq_handler;
        dma_data.isr_data[channel] = isr_data;
    } else {
        /* all DMA channels are used */
        return -ENODEV;
    }
    return 0;
}

void ahb_dma_release_channel(ahb_dma_channel_data_t **channel_data)
{
    int channel;
    ahb_dma_channel_data_t *data; 
    
    if (channel_data && *channel_data) {
        data = *channel_data;
        channel = data->channel;
        if (channel < AHB_DMA_MAX_CHANNELS) {
            dma_data.channel_used[channel] = 0;
            
            /* release memory */
            if (data->ahb_dma_lld) pci_free_consistent(NULL, data->llp_count*sizeof(ahb_lld_t), (void*)data->ahb_dma_lld, (dma_addr_t)data->ahb_dma_lld_phys);
            kfree(data);
            data = NULL;
        }
    }
}

void ahb_dma_channel_add(ahb_dma_channel_data_t *priv, ahb_dma_parm_t *parm)
{
    unsigned int    val;
    ahb_lld_t       *lld;
    ahb_lld_t       *lld_phys;
    
    lld=priv->ahb_dma_lld;
    lld_phys=priv->ahb_dma_lld_phys;
    
    if (priv->llp_free_idx <= priv->llp_count) {
	if(priv->llp_free_idx==0) { //first to call ahb_dma_add
	    priv->base[AHBDMA_CHANNEL_TXSZ] = parm->size;
	    priv->base[AHBDMA_CHANNEL_SRC] = parm->src;
	    priv->base[AHBDMA_CHANNEL_DST] = parm->dest;

	    val = ((parm->irq) ? 0 : (1 << 31)) |
	          ((parm->sw & 7) << 11) |
	          ((parm->dw & 7) << 8) | 
	          ((priv->hw_handshake) ? (1 << 7) : 0) |
	          ((parm->sctl & 3) << 5) |
	          ((parm->dctl & 3) << 3) |
	          ((parm->src_data_master) ? (1 << 2) : 0) |
	          ((parm->dest_data_master) ? (1 << 1) : 0);

	    priv->base[AHBDMA_CHANNEL_CSR] = val;
	    priv->base[AHBDMA_CHANNEL_LLP] = 0;
	} else {
	    val = ((parm->irq) ? 0 : (1 << 28)) |
	          ((parm->sw & 7) << 25) |
	          ((parm->dw & 7) << 22) | 
	          ((parm->sctl & 3) << 20) |
	          ((parm->dctl & 3) << 18) |
	          ((parm->src_data_master) ? (1 << 17) : 0) |
	          ((parm->dest_data_master) ? (1 << 16) : 0) |
	          (parm->size);

	    lld[priv->llp_free_idx-1].source = parm->src;
	    lld[priv->llp_free_idx-1].dest = parm->dest;
	    lld[priv->llp_free_idx-1].control = val;
	    lld[priv->llp_free_idx-1].llp = NULL;

	    /* for the first lld insert its address into Cn_LLP,
	       for subsequent lld insert the address into the previous lld */
	    if (priv->llp_free_idx == 1) priv->base[AHBDMA_CHANNEL_LLP] = (unsigned int) (&lld_phys[0]) | priv->llp_master;
            else lld[priv->llp_free_idx-2].llp = (u32*) ((unsigned int)(&lld_phys[priv->llp_free_idx-1]) | priv->llp_master);
	}
	priv->llp_free_idx++;
    }
}

void ahb_dma_channel_start(ahb_dma_channel_data_t *priv)
{
    /* mark channel as active */
    dma_data.active_channels |= (1 << priv->channel);
    
    priv->base[AHBDMA_CHANNEL_CSR] |= 0x1;
}

void ahb_dma_channel_reset(ahb_dma_channel_data_t *priv)
{
    priv->llp_last_idx=0;
    priv->llp_free_idx=0;

    priv->base[AHBDMA_CHANNEL_CSR] = 0;   //disable DMA channel 0         
    priv->base[AHBDMA_CHANNEL_CFG] = 0;
    priv->base[AHBDMA_CHANNEL_LLP] = 0;
    priv->base[AHBDMA_CHANNEL_TXSZ] = 0;  //no transfer size (use LLP)
    priv->base[AHBDMA_CHANNEL_CFG] = 0;   //disable all interrupt
}

EXPORT_SYMBOL(ahb_dma_request_channel);
EXPORT_SYMBOL(ahb_dma_release_channel);
EXPORT_SYMBOL(ahb_dma_channel_add);
EXPORT_SYMBOL(ahb_dma_channel_start);
EXPORT_SYMBOL(ahb_dma_channel_reset);

module_init(ahb_dma_init);
module_exit(ahb_dma_cleanup);
