/*
 *  linux/arch/armnommu/mach-faraday/ftpci.c
 *
 *  Faraday FTPCI100 PCI Bridge Controller Device Driver Implementation
 *
 *  Copyright (C) 2005 Faraday Corp. (http://www.faraday-tech.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * 
 * ChangeLog
 * 
 *  Peter Liao 09/28/2005  Created.
 */

#include <linux/config.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/pci.h>
#include <linux/ptrace.h>
#include <linux/slab.h>
#include <linux/ioport.h>
#include <linux/interrupt.h>
#include <linux/spinlock.h>
#include <linux/init.h>
#include <asm/hardware.h>
#include <asm/irq.h>
#include <asm/system.h>
#include <asm/mach/pci.h>
#include <asm/io.h>
#include <asm/sizes.h>
#include <asm/arch/ftpci.h>
#include <asm/arch/spec.h> 
#include <asm/mach-types.h> 

#define IPMODULE PCIC
#define IPNAME   FTPCI100

#define DEBUGFPCI
#undef DEBUGFPCI

#ifdef DEBUGFPCI
#define DBGFPCI(x...) printk(x)
#else
#define DBGFPCI(x...)
#endif
#define FPCI_VA_BASE		IP_VA_BASE(0)
#define FTPCI_PCI_BRIDGE_VENID  PCI_BRIDGE_VENID
#define FPCI_IO_VA_BASE		PCIIO_VA_BASE
#define FPCI_IO_PA_BASE         PCIIO_PA_BASE
#define FPCI_IO_VA_END		PCIIO_VA_LIMIT
#define FPCI_MEM_VA_BASE	PCIMEM_VA_BASE
#define FPCI_MEM_PA_BASE        PCIMEM_PA_BASE
#define FPCI_MEM_VA_END		PCIMEM_VA_LIMIT

// --------------------------------------------------------------------
//		AHB Control Register
// --------------------------------------------------------------------
#define FTPCI_IOSIZE_REG			0x0
#define FTPCI_PROT_REG				0x4
#define	FTPCI_CTRL_REG				0x8
#define FTPCI_ERREN_REG				0xc
#define FTPCI_SOFTRST_REG			0x10
#define FTPCI_EN64_REG				0x14
#define FTPCI_ADDRH32_REG			0x18
#define FTPCI_CFG_ADR_REG			0x28
#define FTPCI_CFG_DATA_REG			0x2c


// --------------------------------------------------------------------
//		FTPCI_IOSIZE_REG's constant definitions
// --------------------------------------------------------------------
#define FTPCI_BASE_IO_SIZE_1M		0x0
#define FTPCI_BASE_IO_SIZE_2M		0x1
#define FTPCI_BASE_IO_SIZE_4M		0x2
#define FTPCI_BASE_IO_SIZE_8M		0x3
#define FTPCI_BASE_IO_SIZE_16M		0x4
#define FTPCI_BASE_IO_SIZE_32M		0x5
#define FTPCI_BASE_IO_SIZE_64M		0x6
#define FTPCI_BASE_IO_SIZE_128M		0x7
#define FTPCI_BASE_IO_SIZE_256M		0x8
#define FTPCI_BASE_IO_SIZE_512M		0x9
#define FTPCI_BASE_IO_SIZE_1G		0xa
#define FTPCI_BASE_IO_SIZE_2G		0xb



// --------------------------------------------------------------------
//		PCI Configuration Register
// --------------------------------------------------------------------
#define PCI_INT_MASK				0x4e
#define PCI_MEM_BASE_SIZE1			0x50
#define PCI_MEM_BASE_SIZE2			0x54
#define PCI_MEM_BASE_SIZE3			0x58


// --------------------------------------------------------------------
//		PCI_INT_MASK's bit definitions
// --------------------------------------------------------------------
#define PCI_INTA_ENABLE				(1U<<6)
#define PCI_INTB_ENABLE				(1U<<7)
#define PCI_INTC_ENABLE				(1U<<8)
#define PCI_INTD_ENABLE				(1U<<9)
	

// --------------------------------------------------------------------
//		PCI_MEM_BASE_SIZE1's constant definitions
// --------------------------------------------------------------------
#define FTPCI_BASE_ADR_SIZE_1MB		(0x0<<16)
#define FTPCI_BASE_ADR_SIZE_2MB		(0x1<<16)
#define FTPCI_BASE_ADR_SIZE_4MB		(0x2<<16)
#define FTPCI_BASE_ADR_SIZE_8MB		(0x3<<16)
#define FTPCI_BASE_ADR_SIZE_16MB	(0x4<<16)
#define FTPCI_BASE_ADR_SIZE_32MB	(0x5<<16)
#define FTPCI_BASE_ADR_SIZE_64MB	(0x6<<16)
#define FTPCI_BASE_ADR_SIZE_128MB	(0x7<<16)
#define FTPCI_BASE_ADR_SIZE_256MB	(0x8<<16)
#define FTPCI_BASE_ADR_SIZE_512MB	(0x9<<16)
#define FTPCI_BASE_ADR_SIZE_1GB		(0xa<<16)
#define FTPCI_BASE_ADR_SIZE_2GB		(0xb<<16)

#define CONFIG_CMD(bus, device_fn, where)   (0x80000000 | (bus << 16) | (device_fn << 8) | (where & ~3) )

static spinlock_t ftpci_lock;
struct pci_dev *pci_bridge=NULL;
static unsigned int pci_config_addr;
static unsigned int pci_config_data;
int ftpci_probed=0;

static struct resource pcic_resource = {
        .name  = "Faradat PCIC", 
	.start = IP_VA_BASE(0),
	.end   = IP_VA_LIMIT(0),
};

// Luke Lee 03/21/2005 mod begin
static int ftpci_read_config_byte(struct pci_bus *bus, unsigned int devfn, int where, u8 *val)
{
	unsigned long   flags;
	u32             v;
	unsigned int    shift;

	spin_lock_irqsave(&ftpci_lock, flags);
	outl(CONFIG_CMD(bus->number,devfn,where), pci_config_addr);
	v=inl(pci_config_data);
	spin_unlock_irqrestore(&ftpci_lock, flags);
	shift = (where&0x3)*8;
	*val = (v>>shift)&0xff;
	return PCIBIOS_SUCCESSFUL;
}

static int ftpci_read_config_word(struct pci_bus *bus, unsigned int devfn, int where, u16 *val)
{
	unsigned long   flags;
	u32             v;
	unsigned int    shift;

	spin_lock_irqsave(&ftpci_lock, flags);
	outl(CONFIG_CMD(bus->number,devfn,where), pci_config_addr);
	v=inl(pci_config_data);
	spin_unlock_irqrestore(&ftpci_lock, flags);
	shift = (where&0x3)*8;
	*val = (v>>shift)&0xffff;
	return PCIBIOS_SUCCESSFUL;
}

static int ftpci_read_config_dword(struct pci_bus *bus, unsigned int devfn, int where, u32 *val)
{
	unsigned long   flags;
	u32             v;

	spin_lock_irqsave(&ftpci_lock, flags);
	outl(CONFIG_CMD(bus->number,devfn,where), pci_config_addr);
	v=inl(pci_config_data);
	spin_unlock_irqrestore(&ftpci_lock, flags);
	*val = v;
	return PCIBIOS_SUCCESSFUL;
}

static int ftpci_write_config_byte(struct pci_bus *bus, unsigned int devfn, int where, u8 val)
{
	u32             org_val;
	unsigned long   flags;
	unsigned int    shift;

	shift = (where&0x3)*8;
	spin_lock_irqsave(&ftpci_lock, flags);
	outl(CONFIG_CMD(bus->number,devfn,where), pci_config_addr);
	org_val = inl(pci_config_data);
	org_val=(org_val&~(0xff<<shift))|((u32)val<<shift);
	outl(org_val, pci_config_data);
	spin_unlock_irqrestore(&ftpci_lock, flags);
	return PCIBIOS_SUCCESSFUL;
}

static int ftpci_write_config_word(struct pci_bus *bus, unsigned int devfn, int where, u16 val)
{
	u32             org_val;
	unsigned long   flags;
	unsigned int    shift;

	shift = (where&0x3)*8;
	spin_lock_irqsave(&ftpci_lock, flags);
	outl(CONFIG_CMD(bus->number,devfn,where), pci_config_addr);
	org_val = inl(pci_config_data);
	org_val=(org_val&~(0xffff<<shift))|((u32)val<<shift);
	outl(org_val, pci_config_data);
	spin_unlock_irqrestore(&ftpci_lock, flags);
	return PCIBIOS_SUCCESSFUL;
}

static int ftpci_write_config_dword(struct pci_bus *bus, unsigned int devfn, int where, u32 val)
{
	unsigned long flags;
	spin_lock_irqsave(&ftpci_lock, flags);
	outl(CONFIG_CMD(bus->number,devfn,where), pci_config_addr);
	outl(val, pci_config_data);
	spin_unlock_irqrestore(&ftpci_lock, flags);
	return PCIBIOS_SUCCESSFUL;
}
// Luke Lee 03/21/2005 mod end

// Luke Lee 03/21/2005 ins begin
static int ftpci_read_config(struct pci_bus *bus, unsigned int devfn, int where,
			  int size, u32 *val)
{
	int r; 
	switch (size) {
	case 1:
		r = ftpci_read_config_byte(bus,devfn,where,(u8*)val); // Luke Lee TOFIX 03/22/2005 : convert to (u8*) -- beware of endian !
		break;
	case 2:
		r = ftpci_read_config_word(bus,devfn,where,(u16*)val); // Luke Lee TOFIX 03/22/2005 : convert to (u16*) -- beware of endian !
		break;

	default:
		r = ftpci_read_config_dword(bus,devfn,where,val);
		break;
	}

	return r;
}

static int ftpci_write_config(struct pci_bus *bus, unsigned int devfn, int where,
			      int size, u32 val)
{
	int r; 
	switch (size) {
	case 1:
 	        r = ftpci_write_config_byte(bus, devfn, where, val);
		break;
	case 2:
 	        r = ftpci_write_config_word(bus, devfn, where, val);
		break;

	case 4:
 	        r = ftpci_write_config_dword(bus, devfn, where, val);
		break;
	default:
	        printk( "Invalid size for ftpci_write()\n" );
		r = PCIBIOS_FUNC_NOT_SUPPORTED;  // Luke Lee 03/23/2005 ins 1
	}

	return r;
}

// Luke Lee 03/21/2005 ins end

static struct pci_ops ftpci_ops = {
	// Luke Lee 03/21/2005 mod begin
	.read  = ftpci_read_config,
	.write = ftpci_write_config,
	// Luke Lee 03/21/2005 mod end
};

/* using virtual address for pci_resource_start() function*/
static struct resource pci_io = {
	.name	= "Faraday PCI I/O Space",
	.start	= FPCI_IO_VA_BASE,
	.end	= FPCI_IO_VA_END,
	.flags	= IORESOURCE_IO,
};
/* using physical address for memory resource*/
static struct resource pci_mem = {
	.name	= "Faraday PCI non-prefetchable Memory Space",
	.start	= FPCI_MEM_VA_BASE,
	.end	= FPCI_MEM_VA_END,
	.flags	= IORESOURCE_MEM,
};

// Luke Lee 03/23/2005 unrem 1 rem 1
int __init ftpci_setup_resource(struct resource **resource)
{
        DBGFPCI("PCI I/O space from %08lX to %08lX\n", pci_io.start, pci_io.end);
        DBGFPCI("PCI Memory space from %08lX to %08lX\n", pci_mem.start, pci_mem.end);
	if (request_resource(&ioport_resource, &pci_io)) {
		printk(KERN_ERR "PCI: unable to allocate io region\n");
		return -EBUSY;  // Luke Lee 03/23/2005 unrem 1
	}
	if (request_resource(&iomem_resource, &pci_mem)) {
		printk(KERN_ERR "PCI: unable to allocate non-prefetchable "
		       "memory region\n");
		return -EBUSY;  // Luke Lee 03/23/2005 unrem 1
	}

	/*
	 * bus->resource[0] is the IO resource for this bus
	 * bus->resource[1] is the mem resource for this bus
	 * bus->resource[2] is the prefetch mem resource for this bus
	 */

	resource[0] = &pci_io;
	resource[1] = &pci_mem;
	resource[2] = NULL;

	return 1;  // Luke Lee 03/23/2005 unrem 1
}

int ftpci_get_irq(void)
{
	unsigned int     status;
	ftpci_read_config_dword(pci_bridge->bus, pci_bridge->devfn, 0x4c, &status); // Luke Lee 03/22/2005 mod 1
	DBGFPCI("ftpci_get_irq,status=0x%x\n",status);
	status=(status>>28);
	if(status&0x1)
		return 0;
	if(status&0x2)
		return 1;
	if(status&0x4)
		return 2;
	if(status&0x8)
		return 3;
	return -1;
}

void ftpci_clear_irq(unsigned int irq)
{
	//int             i;
	unsigned int     status;
	ftpci_read_config_dword(pci_bridge->bus, pci_bridge->devfn, 0x4c, &status); // Luke Lee 03/22/2005 mod 1
	if(irq==0)
		status=(status&0xfffffff)|((0x1)<<28);
	else if(irq==1)
		status=(status&0xfffffff)|((0x2)<<28);
	else if(irq==2)
		status=(status&0xfffffff)|((0x4)<<28);
	else if(irq==3)
		status=(status&0xfffffff)|((0x8)<<28);
	ftpci_write_config_dword(pci_bridge->bus, pci_bridge->devfn, 0x4c, status);	// Luke Lee 03/22/2005 mod 1
}

static int ftpci_probe(unsigned int addr_p)
{
    unsigned int *addr=(unsigned int*)addr_p;
    *(volatile unsigned int *)addr=0x80000000;
    if(*(volatile unsigned int *)addr==0x80000000) {
	DBGFPCI("Faraday FPCI bridge probed ok\n");
        ftpci_probed=1;
    } else {
        ftpci_probed=0;
    }
    *(volatile unsigned int *)addr=0x0;
    return ftpci_probed;
}


void __init ftpci_preinit(void /**sysdata*/) // Luke Lee 03/22/2005 mod 1
{
	spin_lock_init(&ftpci_lock);
	DBGFPCI( "ftpci_preinit()\n\r" );
	
	pci_config_addr=FPCI_VA_BASE+FTPCI_CFG_ADR_REG;
	pci_config_data=FPCI_VA_BASE+FTPCI_CFG_DATA_REG;
	DBGFPCI("Config addr is %08X, data port is %08X\n",(int)pci_config_addr, (int) pci_config_data);

	if(!ftpci_probe(pci_config_addr))
		return;
}

void __init ftpci_postinit(void /**sysdata*/) // Luke Lee 03/22/2005 mod 1
{
	u16             val;
	DBGFPCI( "ftpci_postinit()\n\r" );
	pci_bridge=pci_find_device(PCI_BRIDGE_VENID,PCI_BRIDGE_DEVID,NULL);
	if (pci_bridge == NULL)
		return;
	// Enable the Interrupt Mask (INTA/INTB/INTC/INTD)
	ftpci_read_config_word(pci_bridge->bus,pci_bridge->devfn,PCI_INT_MASK,&val); // Luke Lee 03/22/2005 mod 1
	val|=(PCI_INTA_ENABLE|PCI_INTB_ENABLE|PCI_INTC_ENABLE|PCI_INTD_ENABLE);
	ftpci_write_config_word(pci_bridge->bus,pci_bridge->devfn,PCI_INT_MASK,val);// Luke Lee 03/22/2005 mod 1

	// Write DMA Start Address/Size Data to the Bridge configuration space 
	ftpci_write_config_dword(pci_bridge->bus, pci_bridge->devfn, PCI_MEM_BASE_SIZE1, FTPCI_BASE_ADR_SIZE_1GB); // Luke Lee 03/22/2005 mod 1
	DBGFPCI("%s: Post init ok\n",__func__);
}

/*
 * This routine handles multiple bridges.
 */
static u8 __init fpci_swizzle(struct pci_dev *dev, u8 *pinp)
{
	// If there are one more bridges on our platfrom, we need to implement this function.
	DBGFPCI( "a320_swizzle(%X,%X)\n\r",(unsigned)dev,(unsigned)pinp );
	return 0;
}

static int __init fpci_map_irq(struct pci_dev *dev, u8 slot, u8 pin)
{
    DBGFPCI("a320_map_irq,slot=%d pin=%d\n",PCI_SLOT(dev->devfn),pin);
    switch(PCI_SLOT(dev->devfn))
    {
        case 8:
            return IP_IRQ0(0);
        case 9:
            return IP_IRQ1(0);
        case 10:
            return IP_IRQ2(0);
        case 11:
            return IP_IRQ3(0);
        default:
            printk(KERN_ERR "Not Support Slot %d\n",slot);
            break;
    }
    return -1;
}

int __init ftpci_setup(int nr, struct pci_sys_data *sys)
{
	int ret = 0;
	if (nr == 0) {
		ret = ftpci_setup_resource(sys->resource);
		sys->mem_offset = FPCI_MEM_VA_BASE - FPCI_MEM_PA_BASE;
        	sys->io_offset  = FPCI_IO_VA_BASE - FPCI_IO_PA_BASE;
	}
	return ret;
}

static struct pci_bus *ftpci_scan_bus(int nr, struct pci_sys_data *sys)
{
	return pci_scan_bus(sys->busnr, &ftpci_ops, sys);
}

static struct hw_pci a320_pci  __initdata = {
	.swizzle		= fpci_swizzle,
	.map_irq		= fpci_map_irq,
	.setup			= ftpci_setup,
	.nr_controllers		= 1,
	.scan			= ftpci_scan_bus,
	.preinit		= ftpci_preinit,	/* The first called init function */
	.postinit		= ftpci_postinit,	/* It is called after hw init and scanned PCI bus */
};

static int __init fpci_init(void)
{
#ifdef MODULE
	printk(KERN_INFO "Faraday PCI driver Init");
#endif
	if ( machine_is_cpe() || machine_is_faraday() ) {
		printk(KERN_DEBUG "Init A321 PCI bridge controller\n");
        	/* Register I/O address range of this PCI Bridge Controller  */
		DBGFPCI("Name:%s, Base=%lX, End=%lX\n",pcic_resource.name, pcic_resource.start, pcic_resource.end);
	        request_resource(&ioport_resource, &pcic_resource);
		pci_common_init(&a320_pci);
	}
	return 0;
}



subsys_initcall(fpci_init);

EXPORT_SYMBOL(ftpci_probed);
EXPORT_SYMBOL(ftpci_clear_irq);
EXPORT_SYMBOL(ftpci_get_irq);
