/*
 *  linux/arch/arm/mach-faraday/template_intc.h
 *
 *  Faraday FTINTC010 Master Interrupt Controller Device Driver Template
 *
 *  Copyright (C) 2005 Faraday Corp. (http://www.faraday-tech.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * Note
 * 
 *  This program implements only the master INTC of the platform. Slave INTCs must
 *  be initialized by themselves.
 *
 * Example:
 * 
 *  Before includeing this template file, define the following:
 *
 *  #define INTC_PLATFORM_NAME              fia321
 *  #define INTC_TEMPLATE_INTC_NAME         "A321 interrupt controller"
 *  #define INTC_TEMPLATE_INTC_ID           PLATFORM_FIA321_INTC_FTINTC010_ID
 *  #define INTC_TEMPLATE_IRQ_BASE          PLATFORM_FIA321_IRQ_BASE
 *  #define INTC_TEMPLATE_FIQ_BASE          PLATFORM_FIA321_FIQ_BASE
 *  #define INTC_TEMPLATE_IRQ_TRIGGER_MODE  PLATFORM_FIA321_FIQ_TRIGGER_MODE 
 *  #define INTC_TEMPLATE_FIQ_TRIGGER_MODE  PLATFORM_FIA321_IRQ_TRIGGER_MODE 
 *  #define INTC_TEMPLATE_IRQ_TRIGGER_LEVEL PLATFORM_FIA321_FIQ_TRIGGER_LEVEL
 *  #define INTC_TEMPLATE_FIQ_TRIGGER_LEVEL PLATFORM_FIA321_IRQ_TRIGGER_LEVEL
 *  #define INTC_TEMPLATE_IRQ_BASE          PLATFORM_FIA321_IRQ_BASE
 *  #define INTC_TEMPLATE_IRQ_COUNT         PLATFORM_FIA321_IRQ_TOTALCOUNT
 *  #define INTC_TEMPLATE_FIQ_BASE          PLATFORM_FIA321_FIQ_BASE
 *  #define INTC_TEMPLATE_FIQ_COUNT         PLATFORM_FIA321_FIQ_TOTALCOUNT
 *  #define INTC_TEMPLATE_PARENT_IRQ        PLATFORM_FIA321_IRQ
 *  #define INTC_TEMPLATE_PARENT_FIQ        PLATFORM_FIA321_FIQ
 *  
 * ChangeLog
 * 
 *  Luke Lee  09/15/2005  Created.
 *  Luke Lee  09/27/2005  Fixed for parent chip registration and notification.
 */

#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <asm/io.h>

#include <asm/mach/irq.h>
#include <asm/arch/intc.h>
#include <asm/arch/spec.h>

#define IPMODULE INTC
#define IPNAME   FTINTC010

/*
 * Edge trigger IRQ chip methods
 */
static void __glue(INTC_PLATFORM_NAME,_intc_ftintc010_edge_ack_irq)(unsigned int irq)
{
        unsigned parent_chip_irq = (unsigned)get_irq_chipdata(irq);
        // Ack parent chip (recursively)
	irq_desc[parent_chip_irq].chip->ack(parent_chip_irq);
        // Ack
        *(volatile unsigned *)(IP_VA_BASE(INTC_TEMPLATE_INTC_ID)+IRQ_CLEAR_REG)
                = 1<<(irq-INTC_TEMPLATE_IRQ_BASE);
}

static void __glue(INTC_PLATFORM_NAME,_intc_ftintc010_edge_mask_irq)(unsigned irq)
{
        unsigned parent_chip_irq = (unsigned)get_irq_chipdata(irq);
        // Disable
        *(volatile unsigned *)(IP_VA_BASE(INTC_TEMPLATE_INTC_ID)+IRQ_MASK_REG)
                &= ~(1<<(irq-INTC_TEMPLATE_IRQ_BASE));
        // Mask parent chip (recursively)
	irq_desc[parent_chip_irq].chip->mask(parent_chip_irq);
}

static void __glue(INTC_PLATFORM_NAME,_intc_ftintc010_edge_unmask_irq)(unsigned int irq)
{
        unsigned parent_chip_irq = (unsigned)get_irq_chipdata(irq);
        // Unmask parent chip (recursively)
	irq_desc[parent_chip_irq].chip->unmask(parent_chip_irq);
        // Enable 
        *(volatile unsigned *)(IP_VA_BASE(INTC_TEMPLATE_INTC_ID)+IRQ_MASK_REG)
                |= 1<<(irq-INTC_TEMPLATE_IRQ_BASE);
}

/*
 * Level trigger IRQ chip methods
 */
static void __glue(INTC_PLATFORM_NAME,_intc_ftintc010_level_ack_mask_irq)(unsigned int irq)
{
        unsigned parent_chip_irq = (unsigned)get_irq_chipdata(irq);
        // Ack parent chip (recursively)
	irq_desc[parent_chip_irq].chip->ack(parent_chip_irq);
        // Ack and disable
        *(volatile unsigned *)(IP_VA_BASE(INTC_TEMPLATE_INTC_ID)+IRQ_CLEAR_REG)
                = 1<<(irq-INTC_TEMPLATE_IRQ_BASE);
        *(volatile unsigned *)(IP_VA_BASE(INTC_TEMPLATE_INTC_ID)+IRQ_MASK_REG)
                &= ~(1<<(irq-INTC_TEMPLATE_IRQ_BASE));
}

static void __glue(INTC_PLATFORM_NAME,_intc_ftintc010_level_mask_irq)(unsigned int irq)
{
        unsigned parent_chip_irq = (unsigned)get_irq_chipdata(irq);
        // Disable
        *(volatile unsigned *)(IP_VA_BASE(INTC_TEMPLATE_INTC_ID)+IRQ_MASK_REG)
                &= ~(1<<(irq-INTC_TEMPLATE_IRQ_BASE));
        // Mask parent chip (recursively)
	irq_desc[parent_chip_irq].chip->mask(parent_chip_irq);
}

static void __glue(INTC_PLATFORM_NAME,_intc_ftintc010_level_unmask_irq)(unsigned int irq)
{
        unsigned parent_chip_irq = (unsigned)get_irq_chipdata(irq);
        // Unmask parent chip (recursively)
	irq_desc[parent_chip_irq].chip->unmask(parent_chip_irq);
        // enable 
        *(volatile unsigned *)(IP_VA_BASE(INTC_TEMPLATE_INTC_ID)+IRQ_MASK_REG)
                |= 1<<(irq-INTC_TEMPLATE_IRQ_BASE);
}

static struct irqchip __glue(INTC_PLATFORM_NAME,_intc_ftintc010_edge_chip) = {
	.ack	= __glue(INTC_PLATFORM_NAME,_intc_ftintc010_edge_ack_irq),
	.mask	= __glue(INTC_PLATFORM_NAME,_intc_ftintc010_edge_mask_irq),
	.unmask = __glue(INTC_PLATFORM_NAME,_intc_ftintc010_edge_unmask_irq)
};

static struct irqchip __glue(INTC_PLATFORM_NAME,_intc_ftintc010_level_chip) = {
	.ack	= __glue(INTC_PLATFORM_NAME,_intc_ftintc010_level_ack_mask_irq),
	.mask	= __glue(INTC_PLATFORM_NAME,_intc_ftintc010_level_mask_irq),
	.unmask = __glue(INTC_PLATFORM_NAME,_intc_ftintc010_level_unmask_irq)
};

static struct resource __glue(INTC_PLATFORM_NAME,_intc_resource) = {
        .name  = INTC_TEMPLATE_INTC_NAME,
        .start = IP_VA_BASE(INTC_TEMPLATE_INTC_ID),
        .end   = IP_VA_BASE(INTC_TEMPLATE_INTC_ID) + IP_VA_SIZE(INTC_TEMPLATE_INTC_ID),
};

int __init __glue(INTC_PLATFORM_NAME,_intc_ftintc010_init_irq)(void)
{
        int i,edge;

        /* Initialize the INTC */
	outl( 0x00000000, IP_VA_BASE(INTC_TEMPLATE_INTC_ID) + IRQ_MASK_REG );
	outl( 0x00000000, IP_VA_BASE(INTC_TEMPLATE_INTC_ID) + FIQ_MASK_REG );
	outl( 0xffffffff, IP_VA_BASE(INTC_TEMPLATE_INTC_ID) + IRQ_CLEAR_REG );
	outl( 0xffffffff, IP_VA_BASE(INTC_TEMPLATE_INTC_ID) + FIQ_CLEAR_REG );
        
        outl( INTC_TEMPLATE_IRQ_TRIGGER_MODE, IP_VA_BASE(INTC_TEMPLATE_INTC_ID) + IRQ_MODE_REG );
        outl( INTC_TEMPLATE_FIQ_TRIGGER_MODE, IP_VA_BASE(INTC_TEMPLATE_INTC_ID) + FIQ_MODE_REG );
        /* FTINTC010: bit 0=active high or rising edge, 1=active low or falling edge. */
        outl( ~INTC_TEMPLATE_IRQ_TRIGGER_LEVEL, IP_VA_BASE(INTC_TEMPLATE_INTC_ID) + IRQ_LEVEL_REG );
        outl( ~INTC_TEMPLATE_FIQ_TRIGGER_LEVEL, IP_VA_BASE(INTC_TEMPLATE_INTC_ID) + FIQ_LEVEL_REG );
        
        /* Register all IRQ */
        for (i = INTC_TEMPLATE_IRQ_BASE, edge=1;
             i < INTC_TEMPLATE_IRQ_BASE + INTC_TEMPLATE_IRQ_COUNT;
             i++, edge<<=1)
        {
                if (INTC_TEMPLATE_IRQ_TRIGGER_MODE & edge) {
                        // edge trigger
                        set_irq_chip(i, &__glue(INTC_PLATFORM_NAME,_intc_ftintc010_edge_chip));
                        set_irq_handler(i, do_edge_IRQ);
                } else {
                        // level trigger
                        set_irq_chip(i, &__glue(INTC_PLATFORM_NAME,_intc_ftintc010_level_chip));
                        set_irq_handler(i, do_level_IRQ);
                }
                set_irq_flags(i, IRQF_VALID | IRQF_PROBE);
                set_irq_chipdata( i, (void*) INTC_TEMPLATE_PARENT_IRQ );
        }
        /* Register all FIQ :
         * Notice that FIQ is not supported by the system yet
         */
        for (i = INTC_TEMPLATE_FIQ_BASE, edge=1;
             i < INTC_TEMPLATE_FIQ_BASE + INTC_TEMPLATE_FIQ_COUNT;
             i++, edge<<=1)
        {
                if (INTC_TEMPLATE_FIQ_TRIGGER_MODE & edge) {
                        // edge trigger
                        set_irq_chip(i, &__glue(INTC_PLATFORM_NAME,_intc_ftintc010_edge_chip));
                        set_irq_handler(i, do_edge_IRQ);
                } else {
                        // level trigger
                        set_irq_chip(i, &__glue(INTC_PLATFORM_NAME,_intc_ftintc010_level_chip));
                        set_irq_handler(i, do_level_IRQ);
                }
                set_irq_flags(i, IRQF_VALID | IRQF_PROBE);
                set_irq_chipdata( i, (void*) INTC_TEMPLATE_PARENT_FIQ );
        }

        /* Register I/O address range of this INTC */
        request_resource(&ioport_resource, &__glue(INTC_PLATFORM_NAME,_intc_resource));
        
        return 0;
}

/*
 * Undefine all the symbols so that this can be included multiple times
 */
#ifdef INTC_TEMPLATE_MULTIPLE_INSTANCE

#undef INTC_PLATFORM_NAME
#undef INTC_TEMPLATE_INTC_ID           
#undef INTC_TEMPLATE_IRQ_BASE          
#undef INTC_TEMPLATE_FIQ_BASE          
#undef INTC_TEMPLATE_INTC_NAME         
#undef INTC_TEMPLATE_IRQ_TRIGGER_MODE  
#undef INTC_TEMPLATE_FIQ_TRIGGER_MODE  
#undef INTC_TEMPLATE_IRQ_TRIGGER_LEVEL 
#undef INTC_TEMPLATE_FIQ_TRIGGER_LEVEL 
#undef INTC_TEMPLATE_IRQ_BASE          
#undef INTC_TEMPLATE_IRQ_COUNT         
#undef INTC_TEMPLATE_FIQ_BASE          
#undef INTC_TEMPLATE_FIQ_COUNT         

#undef IPMODULE
#undef IPNAME

#endif
