/*
 *  linux/arch/arm/mach-faraday/timer.c
 *
 *  Faraday FTTMR010 Timer Device Driver Implementation
 *
 *  Copyright (C) 2005 Faraday Corp. (http://www.faraday-tech.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * 
 * Note
 * 
 *  As IP_COUNT might be greater than one, timer ID is computed as follows:
 *  id=0~2 : Timer 1~3 of the first FTTMR010 IP
 *  id=3~5 : Timer 1~3 of the second FTTMR010 IP
 *  ...
 *  Therefore:
 *    (id / 3) : Compute which IP
 *    (id % 3) : Compute which timer in this IP
 *  Notice:
 *    For simplicity's sake, all code does not check for invalid timer id
 * 
 * ChangeLog
 * 
 *  Luke Lee  09/14/2005  Created
 */

#include <linux/time.h>
#include <linux/timex.h>
#include <linux/types.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/module.h>
#include <asm/io.h>

#include <asm/mach/time.h>
#include <asm/irq.h>
#include <asm/param.h>

#include <asm/arch/timer.h>
#include <asm/arch/spec.h>

#define IPMODULE TIMER
#define IPNAME   FTTMR010

static unsigned setup_flag[3] = {
    TM1UPDOWN | TM1CLOCK, TM2UPDOWN | TM2CLOCK, TM3UPDOWN | TM3CLOCK
};
static unsigned enable_flag[3] = {
    TM1ENABLE | TM1OFENABLE, TM2ENABLE | TM2OFENABLE, TM3ENABLE | TM3OFENABLE,
};

/*
 * Enable a specific timer in the system
 */
void timer_fttmr010_enable(const unsigned id)
{
        unsigned timer_id = id % 3;
        volatile unsigned *tmcr = (unsigned *)(IP_va_base[id / 3]+TIMER_TMCR);
        *tmcr = (*tmcr & ~setup_flag[timer_id]) | enable_flag[timer_id];
}

/*
 * Disable a specific timer
 */
void timer_fttmr010_disable(const unsigned id)
{
        volatile unsigned *tmcr = (unsigned *)(IP_va_base[id / 3]+TIMER_TMCR);
        *tmcr &= ~enable_flag[id % 3];
}

/*
 * Set current counter value
 */
void timer_fttmr010_set_counter(unsigned id, unsigned int value)
{
        volatile unsigned *tmvalue
                = (unsigned *)(IP_va_base[id / 3] + (id%3) * TIMER_OFFSET + TIMER1_COUNT);
        *tmvalue = value;
}

/*
 * Get current counter value
 */
unsigned timer_fttmr010_get_counter(unsigned id)
{
        volatile unsigned *tmvalue
                = (unsigned *)(IP_va_base[id / 3] + (id%3) * TIMER_OFFSET + TIMER1_COUNT);
        return *tmvalue;
}

/*
 * Set counter expiration reload value
 */
void timer_fttmr010_set_reload(unsigned int id, unsigned int value)
{
        volatile unsigned *tmload
                = (unsigned *)(IP_va_base[id / 3] + (id%3) * TIMER_OFFSET + TIMER1_LOAD);
        *tmload = value;
}

/*
 * Set counter expiration reload value
 */
unsigned timer_fttmr010_get_reload(unsigned int id)
{
        volatile unsigned *tmload
                = (unsigned *)(IP_va_base[id / 3] + (id%3) * TIMER_OFFSET + TIMER1_LOAD);
        return *tmload;
}

/*
 * Map timer ID to the associated irq number.
 */
unsigned timer_fttmr010_id_to_irq(unsigned id)
{
    return IP_irq[id];
}

/*
 * The first timer in the system is used as the system clock tick
 */
static irqreturn_t timer0_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
    /* clear irq state */
    int id = 0;
    volatile unsigned *irqstate = (unsigned *)(IP_va_base[id/3]+TIMER_INTRSTATE);  
    *irqstate = 0;
   
    timer_tick(regs);
    return IRQ_HANDLED;
}

static struct irqaction timer0_irq = {
	.name		= "Timer Tick",
	.flags		= SA_INTERRUPT,
	.handler	= timer0_interrupt
};

static struct resource timer_resource = {
        .name  = "Timer 0~2",
        .start = IP_VA_BASE(0),
        .end   = IP_VA_BASE(0) + IP_VA_SIZE(0),
};

/*
 * Timer initialization, used by platform.c, called only once
 */
static void __init timer_fttmr010_init(void)
{
        /*
         * Here we initialize only the first timer in system,
         * To use any other timer, the initilization steps are similar to the following.
         */
        timer_fttmr010_set_reload(0, AHB_CLK_IN/2/HZ);
        timer_fttmr010_set_counter(0, AHB_CLK_IN/2/HZ);
        timer_fttmr010_enable( 0 );
	
        printk(KERN_INFO "FTTMR010 timer 0 installed on IRQ %d, with clock %d at %d HZ.\r\n",
	       timer_fttmr010_id_to_irq(0),AHB_CLK_IN/2, HZ);


	/* all IRQ sources enabled, clear state */
	{
	    int id = 0;
	    volatile unsigned *irqstate = (unsigned *)(IP_va_base[id/3]+TIMER_INTRSTATE);
	    volatile unsigned *irqmask = (unsigned *)(IP_va_base[id/3]+TIMER_INTRMASK);
	    
	    *irqstate = 0;
	    *irqmask = 0;
	}
	
        setup_irq(timer_fttmr010_id_to_irq(0), &timer0_irq);

        /* Register I/O address range of this timer */
        request_resource(&ioport_resource, &timer_resource);
}

/*
 * This data structure cannot be __initdata as the .offset field keep accessed by
 * arm/kernel/time.c function do_gettimeofday(struct timeval *tv)
 */
struct sys_timer platform_timer = {
        .init	= timer_fttmr010_init,
};

/*
 * Timer Export API
 */
EXPORT_SYMBOL(timer_fttmr010_enable);
EXPORT_SYMBOL(timer_fttmr010_disable);
EXPORT_SYMBOL(timer_fttmr010_set_counter);
EXPORT_SYMBOL(timer_fttmr010_get_counter);
EXPORT_SYMBOL(timer_fttmr010_set_reload);
EXPORT_SYMBOL(timer_fttmr010_get_reload);
EXPORT_SYMBOL(timer_fttmr010_id_to_irq);
