/*
 *
 *    Copyright (c) 2003 Peppercon AG
 *
 *    Module name: peppercon.c
 *
 *    Description:
 *      Architecture- / platform-specific boot-time initialization code for
 *      Peppercons remote management boards.
 *
 * Please read the COPYING file for all license details.
 */
#include <linux/config.h>
#include <linux/init.h>
#include <linux/smp.h>
#include <linux/threads.h>
#include <linux/param.h>
#include <linux/string.h>
#include <linux/pci.h>
#include <linux/rtc.h>
#include <linux/console.h>
#include <linux/bcd.h>

#include <asm/system.h>
#include <asm/pci-bridge.h>
#include <asm/machdep.h>
#include <asm/page.h>
#include <asm/time.h>
#include <asm/io.h>
#include <asm/ocp.h>
#include <asm/ibm_ocp_pci.h>
#include <asm/todc.h>

#include <syslib/ppc4xx_i2c.h>
#include <platforms/4xx/peppercon.h>

#undef DEBUG

#ifdef DEBUG
#define DBG(x...) printk(x)
#else
#define DBG(x...)
#endif

extern spinlock_t rtc_lock;

#ifdef PP_FEAT_SYSOK_BLINKING
# define GPIO_REGS_ADDR                (GPIO0_BASE)
# define GPIO_REGS_SIZE                ((size_t)0x20)

static struct timer_list status_led_timer;
static volatile u32 * gpio_regs;
static unsigned char status_led_state;
static unsigned char status_led_initialized = 0;

static int  status_led_init(void);
static void status_led_toggle(unsigned long data);
#endif /* PP_FEAT_SYSOK_BLINKING */

#ifdef CONFIG_PCI
/*
 * Some IRQs unique to Peppercon boards.
 * Used by the generic 405 PCI setup functions in ppc4xx_pci.c
 */
int __init
ppc405_map_irq(struct pci_dev *dev, unsigned char idsel, unsigned char pin)
{
    static char pci_irq_table[][4] =
	/*
	 * PCI IDSEL/INTPIN->INTLINE
	 *     A	 B	 C	 D
	 */
#ifdef PRODUCT_ICPMMD
	{
	    { 27,	27,	27,	 0 },	/* IDSEL 8 - 8 x Serial */
	    { 29,	29,	29,	 0 },	/* IDSEL 9 - 8 x Serial */
	};
	const long min_idsel = 8, max_idsel = 9, irqs_per_slot = 3;
#elif defined(LARA_KACY)
	{
	    { 0,	0,	0,	 0 },
	    { 0,	0,	0,	 0 },
	    { 0,	0,	0,	 0 },
	    { 0,	0,	0,	 0 },
	    { 0,	0,	0,	 0 },
	    { 0,	0,	0,	 0 },	
	    { 0,	0,	0,	 0 },	
	    { 30,	30,	30,	 30 },   /* IDSEL 2 - Encrption */
	    { 29,	29,	29,	 29 },   /* IDSEL 3 - Gigabit Eth */
	    { 28,	28,	28,	 28 },   /* IDSEL 4 - MiniPCI/UART */
	    { 27,	27,	27,	 27 },   /* IDSEL 5 - USB HC */
	    { 26,	26,	26,	 26 },   /* IDSEL 6 - Z7 */
	    { 25,	25,	25,	 25 },   /* IDSEL 7 - VM FPGA */
	};

	const long min_idsel = 1, max_idsel = 13, irqs_per_slot = 4;
#else
	{
	    { 31,	31,	31,	 0 },	/* IDSEL 1 - ??? */
	    { 30,	30,	30,	 0 },   /* IDSEL 2 - ??? */
	    { 29,	29,	29,	 0 },   /* IDSEL 3 - ??? */
	    {  0,	 0,	 0,	 0 },   /* IDSEL 4 - ??? */
	};
	const long min_idsel = 1, max_idsel = 4, irqs_per_slot = 3;
#endif
    return PCI_IRQ_TABLE_LOOKUP;
};
#endif /* CONFIG_PCI */

static void __init
board_setup_arch(void)
{
    ppc4xx_setup_arch();

#ifdef LARA_KACY
    ibm_ocp_set_emac(0, 0);
#else
    ibm_ocp_set_emac(0, 1);
#endif

#if defined(CONFIG_VT) && defined(CONFIG_DUMMY_CONSOLE)
    conswitchp = &dummy_con;
#endif

    /* Identify the system */
    printk("Peppercon port (C) 2001-2005 Peppercon AG\n");
}

static void __init
board_map_io(void)
{
    ppc4xx_map_io();
}

#define RTC_READ_SAVE(off, buf, count)			\
    if ((rtc_ret = rtc_read(off, buf, count))) {	\
	goto rtc_error;					\
    }
#define RTC_WRITE_SAVE(off, buf, count)			\
    if ((rtc_ret = rtc_write(off, buf, count))) {	\
	goto rtc_error;					\
    }

static long __init
pp_time_init(void)
{
    static int not_initialized = 1;

    /* Make sure clocks are running */
    if (not_initialized) {
	unsigned char century, year, mon, day, mday, hour, min, sec;
	unsigned char rtc_val = 0;
	int rtc_ret;

	i2c_init();

	RTC_READ_SAVE(X1240_STATUS, (unsigned char *)&rtc_val, 1);
               
	if (rtc_val & X1240_STATUS_BATTERY_FLAG) {
	    printk(KERN_INFO "Real Time Clock battery exhuasted, value of RTC may be incorrect\n");
	}
#if 0
	printk(KERN_INFO "RTC X1240: Status %02x\n",rtc_val);
#endif 
       
	/* unlock Register write access */
	rtc_val = X1240_STATUS_WEL;
	RTC_WRITE_SAVE(X1240_STATUS, &rtc_val, 1);
	rtc_val |= X1240_STATUS_RWEL;
	RTC_WRITE_SAVE(X1240_STATUS, &rtc_val, 1);
       
	RTC_READ_SAVE(X1240_STATUS, &rtc_val, 1);
       
	/* 
	 * start a new powered on rtc
	 * a simple write turns on the RTC
	 */
	if (rtc_val & X1240_STATUS_RTCF){
	    rtc_read(X1240_Y2K, &century, 1);
	    rtc_write(X1240_Y2K, &century, 1);
	}
       
	/* lock Register write access, leave EEPROM enabled */
	rtc_val = X1240_STATUS_WEL;
	RTC_WRITE_SAVE(X1240_STATUS, &rtc_val, sizeof(rtc_val));
       
	RTC_READ_SAVE(X1240_SC,  &sec, 1);
	RTC_READ_SAVE(X1240_MIN, &min, 1);
	RTC_READ_SAVE(X1240_HR,  &hour, 1);
	RTC_READ_SAVE(X1240_DT,  &mday, 1);
	RTC_READ_SAVE(X1240_MO,  &mon, 1);
	RTC_READ_SAVE(X1240_YR,  &year, 1);
	RTC_READ_SAVE(X1240_DW,  &day, 1); day++;
	RTC_READ_SAVE(X1240_Y2K, &century, 1);

#if 0
	printk(KERN_INFO "RTC X1240: Time: %02x:%02x:%02x Date: %1x. %02x%02x/%02x/%02x\n",
	       hour & 0x3f,min,sec,day,century,year,mon,mday);
#endif

	not_initialized = 0;
    }

#ifdef PP_FEAT_SYSOK_BLINKING
    if (status_led_init() == 0) {
	status_led_initialized = 1;
    }
    pp_status_led_blinking(1);
#endif

    return 0;

 rtc_error:
    printk(KERN_INFO "RTC X1240: Error while accessing RTC\n");
    return -1;
}

int
pp_set_rtc_time(unsigned long nowtime)
{
    struct rtc_time tm;
    unsigned char rtc_val = 0;
    unsigned int  save_year;
    unsigned char century;
    int rtc_ret;

    to_tm(nowtime, &tm);

    save_year  = tm.tm_year;
    tm.tm_year = (tm.tm_year - 1900) % 100;
    century    = (save_year - tm.tm_year) / 100;

    BIN_TO_BCD(tm.tm_sec);
    BIN_TO_BCD(tm.tm_min);
    BIN_TO_BCD(tm.tm_hour);
    BIN_TO_BCD(tm.tm_mon);
    BIN_TO_BCD(tm.tm_mday);
    BIN_TO_BCD(tm.tm_year);
    BIN_TO_BCD(century);

    spin_lock(&rtc_lock);

    /* unlock Register write access */
    rtc_val = X1240_STATUS_WEL;
    RTC_WRITE_SAVE(X1240_STATUS, &rtc_val, 1);
    rtc_val |= X1240_STATUS_RWEL;
    RTC_WRITE_SAVE(X1240_STATUS, &rtc_val, 1);

    rtc_val = tm.tm_sec  & 0x7f;
    RTC_WRITE_SAVE(X1240_SC,  &rtc_val, 1);

    rtc_val = tm.tm_min  & 0x7f;
    RTC_WRITE_SAVE(X1240_MIN, &rtc_val, 1);

    rtc_val = (tm.tm_hour & 0x3f) | X1240_HR_MIL; /* set 24 hour mode*/
    RTC_WRITE_SAVE(X1240_HR,  &rtc_val, 1);

    rtc_val = tm.tm_mday & 0x3f;
    RTC_WRITE_SAVE(X1240_DT,  &rtc_val, 1);

    rtc_val = tm.tm_mon  & 0x1f;
    RTC_WRITE_SAVE(X1240_MO,  &rtc_val, 1);

    rtc_val = tm.tm_year;
    RTC_WRITE_SAVE(X1240_YR,  &rtc_val, 1);

    //      rtc_write(X1240_DW,  &rtc_val, 1);

    rtc_val = (century   & 0x3f);
    RTC_WRITE_SAVE(X1240_Y2K, &rtc_val, 1);

    /* lock Register write access, leave EEPROM enabled */
    rtc_val = X1240_STATUS_WEL;
    RTC_WRITE_SAVE(X1240_STATUS, &rtc_val, 1);

    spin_unlock(&rtc_lock);
    return 0;

 rtc_error:
    spin_unlock(&rtc_lock);
    printk(KERN_INFO "RTC X1240: Error while accessing RTC\n");
    return -1;
}
unsigned long
pp_get_rtc_time(void)
{
    unsigned int century, year, mon, mday, hour, min, sec;
    unsigned char rtc_val;
    int rtc_ret;

    spin_lock(&rtc_lock);

    RTC_READ_SAVE(X1240_SC,  &rtc_val, 1);
    sec = (unsigned int) rtc_val & 0x7f;

    RTC_READ_SAVE(X1240_MIN, &rtc_val, 1);
    min = (unsigned int) rtc_val & 0x7f;

    RTC_READ_SAVE(X1240_HR,  &rtc_val, 1);
    hour = (unsigned int) rtc_val & 0x3f;

    RTC_READ_SAVE(X1240_DT,  &rtc_val, 1);
    mday = (unsigned int) rtc_val & 0x3f;

    RTC_READ_SAVE(X1240_MO,  &rtc_val, 1);
    mon = (unsigned int) rtc_val & 0x1f;

    RTC_READ_SAVE(X1240_YR,  &rtc_val, 1);
    year = (unsigned int) rtc_val;

    //      RTC_READ_SAVE(X1240_DW,  &rtc_val, 1);

    RTC_READ_SAVE(X1240_Y2K, &rtc_val, 1);
    century = (unsigned int) rtc_val & 0x3f;

    spin_unlock(&rtc_lock);

    BCD_TO_BIN(sec);
    BCD_TO_BIN(min);
    BCD_TO_BIN(hour);
    BCD_TO_BIN(mday);
    BCD_TO_BIN(mon);
    BCD_TO_BIN(year);
    BCD_TO_BIN(century);

    year = year + (100 * century);

#if 0
    printk(KERN_INFO "RTC X1240 gettime(): Time: %02x:%02x:%02x Date: %02x%02x/%02x/%02x\n",
	   hour & 0x3f,min,sec,century,year,mon,mday);
#endif

    return mktime(year, mon, mday, hour, min, sec);

 rtc_error:
    spin_unlock(&rtc_lock);
    printk(KERN_ALERT "RTC X1240: Error while accessing RTC\n");
    return -1;
}

void __init
platform_init(unsigned long r3, unsigned long r4, unsigned long r5,
	      unsigned long r6, unsigned long r7)
{
    ppc4xx_init(r3, r4, r5, r6, r7);
    ppc_md.setup_arch = board_setup_arch;
    ppc_md.setup_io_mappings = board_map_io;
    ppc_md.time_init = pp_time_init;
    ppc_md.set_rtc_time = pp_set_rtc_time;
    ppc_md.get_rtc_time = pp_get_rtc_time;
}

#ifdef PP_FEAT_SYSOK_BLINKING
EXPORT_SYMBOL(pp_status_led_blinking);
void
pp_status_led_blinking(unsigned char blinking)
{
    if (!status_led_initialized) return;
    
    del_timer_sync(&status_led_timer);
        
    if (blinking) {
	init_timer(&status_led_timer);
	status_led_timer.expires = jiffies + HZ/10;
	status_led_timer.data = 0;
	status_led_timer.function = status_led_toggle;
	add_timer(&status_led_timer);
    } else {
	// if not blinking, turn on unconditioally
	status_led_state = 1;
	clear_bit((31 - 9), &gpio_regs[0]);
    }
}

static int
status_led_init(void)
{
    int ret = 0;
    
    gpio_regs = ioremap_nocache(GPIO0_BASE, GPIO_REGS_SIZE);
    if (gpio_regs == NULL) {
	printk("Failed to map GPIO registers\n");
	ret = -1;
    }

    return ret;
}

static void
status_led_toggle(unsigned long data)
{
    if (status_led_state) {
	set_bit((31 - 9), &gpio_regs[0]);
    } else {
	clear_bit((31 - 9), &gpio_regs[0]);
    }

    status_led_state = (status_led_state) ? 0 : 1;
    status_led_timer.expires = jiffies + HZ/10;
    add_timer(&status_led_timer);
}
#endif
