/*
 *	Real Time Clock interface for Linux on CPE with FTRTC010
 *
 *	Based on sa1100-rtc.c by Nils Faerber
 *
 *	Based on rtc.c by Paul Gortmaker
 *	Date/time conversion routines taken from arch/arm/kernel/time.c
 *			by Linus Torvalds and Russel King
 *		and the GNU C Library
 *	( ... I love the GPL ... just take what you need! ;)
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 *
 *	0.01	2004-11-17	I-Jui Sung <ijsung@faraday-tech.com>
 *	- initial release
 */

#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/miscdevice.h>
#include <linux/init.h>
#include <linux/poll.h>
#include <linux/proc_fs.h>
#include <linux/rtc.h>
#include <asm/arch/platform/kira100.h>

#define	DRIVER_VERSION		"0.02"

#define TIMER_FREQ		3686400

#define RTC_DEF_DIVIDER		(32768 - 1)
#define RTC_DEF_TRIM		0
#define RTC_DIVIDER_ENABLE	0x80000000

/* FTRTC010 Register Offsets */
#define RTC_SEC	    0x0
#define RTC_MIN	    0x1
#define RTC_HOUR    0x2
#define RTC_DAYS    0x3
#define RTC_ALMSEC  0x4
#define RTC_ALMMIN  0x5
#define RTC_ALMHR   0x6
#define RTC_REC	    0x7
#define RTC_CR	    0x8
#define RTC_DIV	    0xe
#define RTC_REV	    0xf

/* Those are the bits from a classic RTC we want to mimic */
#define RTC_IRQF    0x80	/* any of the following 3 is active */
#define RTC_PF	    0x40
#define RTC_AF	    0x20
#define RTC_UF	    0x10

static unsigned long rtc_status;
static volatile unsigned long rtc_irq_data;
static unsigned long rtc_freq = 1;	/*FTRTC010 supports only 1Hz clock*/
static volatile u32* rtc_regs;

static struct fasync_struct *rtc_async_queue;
static DECLARE_WAIT_QUEUE_HEAD(rtc_wait);

extern spinlock_t rtc_lock;

static const unsigned char days_in_mo[] =
{31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};

#define is_leap(year) \
    ((year) % 4 == 0 && ((year) % 100 != 0 || (year) % 400 == 0))

static unsigned getRTCuptime(void)
{
    unsigned retval;
    retval = rtc_regs[RTC_SEC] & 0x3f;
    retval += 60 * (rtc_regs[RTC_MIN] & 0x3f);
    retval += 3600 * (rtc_regs[RTC_HOUR] & 0x1f);
    retval += 86400 * rtc_regs[RTC_DAYS];
    return retval;
}

/*
 * Converts seconds since 1970-01-01 00:00:00 to Gregorian date.
 */

static void decodetime (unsigned long t, struct rtc_time *tval)
{
    long days, month, year, rem;

    days = t / 86400;
    rem = t % 86400;
    tval->tm_hour = rem / 3600;
    rem %= 3600;
    tval->tm_min = rem / 60;
    tval->tm_sec = rem % 60;
    tval->tm_wday = (4 + days) % 7;

#define LEAPS_THRU_END_OF(y) ((y)/4 - (y)/100 + (y)/400)

    year = 1970 + days / 365;
    days -= ((year - 1970) * 365
	    + LEAPS_THRU_END_OF (year - 1)
	    - LEAPS_THRU_END_OF (1970 - 1));
    if (days < 0) {
	year -= 1;
	days += 365 + is_leap(year);
    }
    tval->tm_year = year - 1900;
    tval->tm_yday = days + 1;

    month = 0;
    if (days >= 31) {
	days -= 31;
	month++;
	if (days >= (28 + is_leap(year))) {
	    days -= (28 + is_leap(year));
	    month++;
	    while (days >= days_in_mo[month]) {
		days -= days_in_mo[month];
		month++;
	    }
	}
    }
    tval->tm_mon = month;
    tval->tm_mday = days + 1;
    tval->tm_isdst=0;
}
static unsigned AIE_stat=0;
static unsigned saved_second=0; /* workaround for A320C */

static irqreturn_t rtc_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
    /* clear alarm interrupt source */
    rtc_regs[RTC_ALMSEC] = 63; /* This is the only way to stop alarm in A320C */
    rtc_irq_data |= (RTC_AF|RTC_IRQF);
    rtc_irq_data += 0x100;

    /* wake up waiting process */
    wake_up_interruptible(&rtc_wait);
    kill_fasync (&rtc_async_queue, SIGIO, POLL_IN);

    return IRQ_HANDLED;
}

static int rtc_open(struct inode *inode, struct file *file)
{
    if (test_and_set_bit (1, &rtc_status))
	return -EBUSY;
    rtc_irq_data = 0;
    return 0;
}

static int rtc_release(struct inode *inode, struct file *file)
{
    spin_lock_irq (&rtc_lock);
    rtc_regs[RTC_ALMSEC] = 63; /* This is the only way to stop alarm in A320C */
    spin_unlock_irq (&rtc_lock);
    rtc_status = 0;
    return 0;
}

static int rtc_fasync (int fd, struct file *filp, int on)
{
    return fasync_helper (fd, filp, on, &rtc_async_queue);
}

static unsigned int rtc_poll(struct file *file, poll_table *wait)
{
    poll_wait (file, &rtc_wait, wait);
    return (rtc_irq_data) ? 0 : POLLIN | POLLRDNORM;
}

static loff_t rtc_llseek(struct file *file, loff_t offset, int origin)
{
    return -ESPIPE;
}

ssize_t rtc_read(struct file *file, char *buf, size_t count, loff_t *ppos)
{
    DECLARE_WAITQUEUE(wait, current);
    unsigned long data;
    ssize_t retval;
    if (count < sizeof(unsigned long))
	return -EINVAL;

    add_wait_queue(&rtc_wait, &wait);
    set_current_state(TASK_INTERRUPTIBLE);
    for (;;) {
	spin_lock_irq (&rtc_lock);
	data = rtc_irq_data;
	if (data != 0) {
	    rtc_irq_data = 0;
	    break;
	}
	spin_unlock_irq (&rtc_lock);

	if (file->f_flags & O_NONBLOCK) {
	    retval = -EAGAIN;
	    goto out;
	}

	if (signal_pending(current)) {
	    retval = -ERESTARTSYS;
	    goto out;
	}

	schedule();
    }
    spin_unlock_irq (&rtc_lock);

    data -= 0x100;	/* the first IRQ wasn't actually missed */

    retval = put_user(data, (unsigned long *)buf);
    if (!retval)
	retval = sizeof(unsigned long);

out:
    set_current_state(TASK_RUNNING);
    remove_wait_queue(&rtc_wait, &wait);
    return retval;
}


static int rtc_ioctl(struct inode *inode, struct file *file,
	unsigned int cmd, unsigned long arg)
{
    struct rtc_time tm, tm2;

    switch (cmd) {
	case RTC_AIE_OFF:
	    spin_lock_irq(&rtc_lock);
	    AIE_stat=0;
	    rtc_regs[RTC_ALMSEC] = 63;
	    rtc_irq_data = 0;
	    spin_unlock_irq(&rtc_lock);
	    return 0;
	case RTC_AIE_ON:
	    spin_lock_irq(&rtc_lock);
	    AIE_stat=1;
	    rtc_regs[RTC_ALMSEC] = saved_second;
	    rtc_irq_data = 0;
	    spin_unlock_irq(&rtc_lock);
	    return 0;
	case RTC_ALM_READ:
	    {	
		unsigned alarm_time;
		alarm_time = saved_second;
		alarm_time += 60*(rtc_regs[RTC_ALMMIN] & 0x3f);
		alarm_time += 3600*(rtc_regs[RTC_ALMHR] & 0x3f);
		alarm_time += rtc_regs[RTC_REC];
		decodetime(alarm_time, &tm);
	    }
	    break;
	case RTC_ALM_SET:
	    if (copy_from_user (&tm2, (struct rtc_time*)arg, sizeof (tm2)))
		return -EFAULT;
	    {	
		unsigned user_alarm=mktime (tm2.tm_year+1900, tm2.tm_mon + 1, tm2.tm_mday, tm2.tm_hour, tm2.tm_min, tm2.tm_sec);
		user_alarm -= rtc_regs[RTC_REC];
		decodetime(user_alarm, &tm2);
	    }
	    /*Need to calculate real alarm time=User supplied alarm time-record time*/		
	    if ((unsigned)tm2.tm_hour < 24)
		rtc_regs[RTC_ALMHR] = tm2.tm_hour;
	    if ((unsigned)tm2.tm_min < 60)
		rtc_regs[RTC_ALMMIN] = tm2.tm_min;
	    saved_second=tm2.tm_sec;
	    if(AIE_stat) {
		if ((unsigned)tm2.tm_sec < 60)
		    rtc_regs[RTC_ALMSEC] = tm2.tm_sec;
	    }
	    return 0;
	case RTC_RD_TIME:
	    {
		unsigned uptime=getRTCuptime();
		uptime += rtc_regs[RTC_REC];
		decodetime(uptime, &tm);	
	    }
	    break;
	case RTC_SET_TIME:
	    {
		unsigned uptime=getRTCuptime(), usertime;

		if (!capable(CAP_SYS_TIME))
		    return -EACCES;
		if (copy_from_user (&tm, (struct rtc_time*)arg, sizeof (tm)))
		    return -EFAULT;
		tm.tm_year += 1900;
		if (tm.tm_year < 1970 || (unsigned)tm.tm_mon >= 12 ||
			tm.tm_mday < 1 || tm.tm_mday > (days_in_mo[tm.tm_mon] +
			    (tm.tm_mon == 1 && is_leap(tm.tm_year))) ||
			(unsigned)tm.tm_hour >= 24 ||
			(unsigned)tm.tm_min >= 60 ||
			(unsigned)tm.tm_sec >= 60)
		    return -EINVAL;
		usertime=mktime (tm.tm_year, tm.tm_mon + 1, tm.tm_mday, tm.tm_hour, tm.tm_min, tm.tm_sec);
		rtc_regs[RTC_CR] &= (~0x1);
		rtc_regs[RTC_REC] = (usertime-uptime);
		rtc_regs[RTC_CR] |= 0x1;
	    }
	    return 0;

	case RTC_IRQP_READ:
	    return put_user(rtc_freq, (unsigned long *)arg);
	case RTC_IRQP_SET:
	    if (arg != 1) return -EINVAL;
	    return 0;
	case RTC_EPOCH_READ:
	    return put_user (1970, (unsigned long *)arg);
	default:
	    return -EINVAL;
    }
    return copy_to_user ((void *)arg, &tm, sizeof (tm)) ? -EFAULT : 0;
}

static struct file_operations rtc_fops = {
owner:		THIS_MODULE,
		llseek:		rtc_llseek,
		read:		rtc_read,
		poll:		rtc_poll,
		ioctl:		rtc_ioctl,
		open:		rtc_open,
		release:	rtc_release,
		fasync:		rtc_fasync,
};

static struct miscdevice ftrtc010rtc_miscdev = {
    RTC_MINOR,
    "rtc",
    &rtc_fops
};

static int rtc_read_proc(char *page, char **start, off_t off, int count, int *eof, void *data)
{
    unsigned alarm_time;
    char *p = page;
    int len;
    struct rtc_time tm;
    decodetime(getRTCuptime()+rtc_regs[RTC_REC], &tm);
    p += sprintf(p, "rtc_time\t: %02d:%02d:%02d\n"
	    "rtc_date\t: %04d-%02d-%02d\n"
	    "rtc_epoch\t: %04d\n",
	    tm.tm_hour, tm.tm_min, tm.tm_sec,
	    tm.tm_year + 1900, tm.tm_mon + 1, tm.tm_mday, 1970);

    alarm_time = saved_second;
    alarm_time += 60*(rtc_regs[RTC_ALMMIN] & 0x3f);
    alarm_time += +3600*(rtc_regs[RTC_ALMHR] & 0x3f);
    alarm_time += rtc_regs[RTC_REC];

    decodetime(alarm_time, &tm);
    p += sprintf(p, "alrm_time\t: %02d:%02d:%02d\n"
	    "alrm_date\t: N/A for CPE120\n",
	    tm.tm_hour, tm.tm_min, tm.tm_sec);
    p += sprintf(p, "alarm_IRQ\t: %s\n", AIE_stat ? "yes" : "no" );

    len = (p - page) - off;
    if (len < 0)
	len = 0;

    *eof = (len <= count) ? 1 : 0;
    *start = page + off;

    return len;
}

static int __init rtc_init(void)
{
    int ret;
    misc_register (&ftrtc010rtc_miscdev);
    create_proc_read_entry ("driver/rtc", 0, 0, rtc_read_proc, NULL);

    rtc_regs = (volatile u32*) CPE_RTC_VA_BASE;

    ret = request_irq (IRQ_RTC, rtc_interrupt, SA_INTERRUPT, "rtc Alrm", NULL);
    if (ret) {
	printk(KERN_ERR "rtc: IRQ %d already in use.\n", 17);
	goto IRQ_RTCAlrm_failed;
    }

    /* setup and enable the divider */
    rtc_regs[RTC_DIV] = (RTC_DEF_DIVIDER | RTC_DIVIDER_ENABLE);

    /* enable RTC */
    rtc_regs[RTC_CR] |= 0x1;

    printk (KERN_INFO "Faraday FTRTC010 Real Time Clock driver v" DRIVER_VERSION "\n");
    return 0;

IRQ_RTCAlrm_failed:
    remove_proc_entry ("driver/rtc", NULL);
    misc_deregister (&ftrtc010rtc_miscdev);
    return ret;
}

static void __exit rtc_exit(void)
{
    remove_proc_entry ("driver/rtc", NULL);
    misc_deregister (&ftrtc010rtc_miscdev);
}

module_init(rtc_init);
module_exit(rtc_exit);

MODULE_AUTHOR("I-Jui Sung <ijsung@faraday-tech.com>");
MODULE_DESCRIPTION("CPE FTRTC010 Realtime Clock Driver (RTC)");
