/*
  i2c-algo-fia320.c i2c driver algorithms for Faraday's FIA320 i2c core
 
  (C) 2005 Peppercon AG
  
  based on:
  
  i2c driver algorithms for Opencores I2C controllers

  Copyright (C) 2004 Microtronix Datacom Ltd
  Written by Wentao Xu <wentao@microtronix.com>
    
  This file was highly leveraged from i2c-algo-pcf.c, which was  

  --------------------------------------------------------------------
  Copyright (C) 1995-1997 Simon G. Vogl
  1998-2000 Hans Berglund

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

  With some changes from Ky�ti M�kki <kmalkki@cc.hut.fi> and 
  Frodo Looijaard <frodol@dds.nl> ,and also from Martin Bailey
  <mbailey@littlefeet-inc.com> 

  Partially rewriten by Oleg I. Vdovikin <vdovikin@jscc.ru> to handle multiple
  messages, proper stop/repstart signaling during receive,
  added detect code
  --------------------------------------------------------------------
*/

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/i2c.h>
#include <linux/i2c-algo-fia320.h>
#include <linux/i2c-slave.h>
#include <linux/i2c-fia320.h>


MODULE_AUTHOR("Peppercon AG");
MODULE_DESCRIPTION("I2C-Bus Faraday FIA320 algorithm");
MODULE_LICENSE("GPL");

MODULE_PARM(i2c_debug,"i");
MODULE_PARM_DESC(i2c_debug, "debug level - 0 off; 1 normal; 2,3 more verbose; 9 fia-protocol");

/*****************************************************************************
 * function prototypes
 ****************************************************************************/

static void i2c_fia320_algo_handle_slave(struct i2c_adapter *adap);
static void i2c_fia320_algo_handle_interrupt(struct i2c_adapter *adap);

/*****************************************************************************
 * global variables
 ****************************************************************************/

int i2c_debug = 0;
static spinlock_t      irq_lock = SPIN_LOCK_UNLOCKED;

/*****************************************************************************
 * global defines
 ****************************************************************************/

#define NDEBUG
// #undef NDEBUG
#ifndef NDEBUG
# define DEB(x) if (i2c_debug>=1) x
# define DEB2(x) if (i2c_debug>=2) x
# define DEB3(x) if (i2c_debug>=3) x
# define DEBPROTO(x) if (i2c_debug>=9) x
#else
# define DEB(x)
# define DEB2(x)
# define DEB3(x)
# define DEBPROTO(x)
#endif

#define BUS_WAIT_RETRIES	100
#define BUS_WAIT_TIMEOUT	1000

#define set_cmd(adap, val) adap->setreg(adap->hw, REG_FA320_CR, FA320_CR_ENABLE | val)
#define set_cmd_noenable(adap, val) adap->setreg(adap->hw, REG_FA320_CR, val)
#define get_sts(adap) adap->hw->isr
#define get_real_sts(adap) adap->getreg(adap->hw, REG_FA320_SR)
#define set_txr(adap, val) adap->setreg(adap->hw, REG_FA320_DR, val)
#define get_rxr(adap) adap->getreg(adap->hw, REG_FA320_DR)

#define set_sar(adap, val) adap->setreg(adap->hw, REG_FA320_SAR, val)
#define get_sar(adap) adap->getreg(adap->hw, REG_FA320_SAR)

#define get_bmr(adap) adap->getreg(adap->hw, REG_FA320_BMR)

/******************************************************************************
 * debugging into a buffer                                                    *
 ******************************************************************************/

/* Debugging the ISR destroys our timings, so we have to print the debugging
   info asynchronously. The strings are put into a big buffer. When a new
   I2C slave address is set via ioctl(), we print the collected debug info.
*/

#if DEBUG_TO_BUFFER

#define DBG_BUFFER_ENTRY_SIZE	160
#define DBG_BUFFER_SIZE		10000

static char dbg_buffer[DBG_BUFFER_SIZE][DBG_BUFFER_ENTRY_SIZE];
static spinlock_t dbg_lock = SPIN_LOCK_UNLOCKED;
static int dbg_buffer_pos = 0;

void i2c_fia_print(const char *fmt, ...) {
	unsigned long flags;
	
	spin_lock_irqsave(&dbg_lock, flags);
	if (dbg_buffer_pos < DBG_BUFFER_SIZE) {
		va_list args;
		va_start(args, fmt);
		vsnprintf(dbg_buffer[dbg_buffer_pos++], DBG_BUFFER_ENTRY_SIZE, fmt, args);
		va_end(args);
	}
	
	spin_unlock_irqrestore(&dbg_lock, flags);
}

static void i2c_fia_print_dbg(void) {
	unsigned long flags;
	int i;
	
	spin_lock_irqsave(&dbg_lock, flags);

	printk("**************** Collected %d debug messages:\n", dbg_buffer_pos);
	for (i = 0; i < dbg_buffer_pos; i++) {
		printk(dbg_buffer[i]);
	}
	printk("**************** debug messages finished\n");
	
	dbg_buffer_pos = 0;

	spin_unlock_irqrestore(&dbg_lock, flags);
}

#define dbg	i2c_fia_print

#else /* DEBUG_TO_BUFFER */

#define dbg(fmt, x...)	printk(KERN_DEBUG fmt, ##x)

#endif /* DEBUG_TO_BUFFER */

/*****************************************************************************
 * utility functions
 ****************************************************************************/

static void i2c_start(struct i2c_algo_fia320_data *adap) 
{
#if 1
    DEBPROTO(dbg("S\n"));
#else
    dbg("S\n");
#endif
    set_cmd(adap, FA320_CR_SCLEN| FA320_CR_START | FA320_CR_TBEN);
}

static void i2c_stop(struct i2c_algo_fia320_data *adap) 
{
#if 1
    DEBPROTO(dbg("P\n"));
#else
    dbg("P\n");
#endif
    adap->hw->handle_stop = 1;
    set_cmd(adap, FA320_CR_SCLEN | FA320_CR_STOP);
}

static int update_sr(struct i2c_adapter *adap) {
    struct i2c_algo_fia320_data *fia_adap = adap->algo_data;
    struct iic_fia320_hw* hw = fia_adap->hw;
    unsigned long flags;
    int status;

    spin_lock_irqsave(&irq_lock, flags);
    status = get_real_sts(fia_adap);
    hw->isr = status;
    i2c_fia320_isr(adap, 0);
    spin_unlock_irqrestore(&irq_lock, flags);

    return status;
}

static int wait_for_bb(struct i2c_adapter *adap) {
    struct i2c_algo_fia320_data *fia_adap = adap->algo_data;
    struct iic_fia320_hw* hw = fia_adap->hw;
    int status;

    DEB3(dbg("fia320-%d: probing if bus is free.\n", hw->idx));
    status = update_sr(adap);
    
    if (status & (FA320_SR_BB | FA320_SR_I2CB)) {
    	int retry = BUS_WAIT_RETRIES;

    	DEB(dbg("fia320-%d: bus not free.\n", hw->idx));
    	
    	do {
    	    udelay(BUS_WAIT_TIMEOUT);
    	    status = update_sr(adap);
    	    retry--;
    	} while (retry && (status & (FA320_SR_BB | FA320_SR_I2CB)));
    	
    	if (status & (FA320_SR_BB | FA320_SR_I2CB)) {
    	    printk(KERN_WARNING "fia320-%d: bus is still not free, resetting\n", hw->idx);
    	    fia_adap->reset(adap);
    	    status = update_sr(adap);
    	}
    }
    
    if (status & (FA320_SR_BB | FA320_SR_I2CB)) {
	printk(KERN_ERR "Timeout waiting for bus to become free\n");
	return -1;
    }
    
    DEB3(dbg("fia320-%d: bus is free.\n", hw->idx));

    return 0;
}

static int wait_while_tip(struct i2c_algo_fia320_data *adap) {
    struct iic_fia320_hw* hw = adap->hw;
    int timeout = 1;
    
    DEB3(dbg("waitwhiletip\n"));
    //dbg("s\n");
    wait_event_interruptible_timeout(hw->wait, (hw->transfer_done || hw->arbit_lost_xfer_error), timeout*HZ);
    //dbg("u\n");
    
    if (hw->transfer_done) {
    	DEB2(dbg("waitwhiletip done - isr: %x\n", hw->isr));
    	return 0;
    }
    if (hw->arbit_lost_xfer_error) {
    	dbg("waitwhiletip error - isr: %x\n", hw->isr);
    	return -1;
    
    }
    
    dbg("waitwhiletip timeout - isr: %x\n", hw->isr);
    return -1;
}

static int wait_while_slave(struct i2c_adapter *i2c_adap) {
    int timeout = 1;
    struct i2c_algo_fia320_data *adap = i2c_adap->algo_data;
    struct iic_fia320_hw* hw = adap->hw;
    struct i2c_algo_fia320_slave_data *slave_data = adap->slave_data;

    wait_event_interruptible_timeout(hw->wait, !(slave_data->slave_write_in_progress || slave_data->slave_read_in_progress), timeout*HZ);
    
    if (!(slave_data->slave_write_in_progress || slave_data->slave_read_in_progress)) {
    	DEB2(dbg("wait_while_slave done - isr: %x\n", hw->isr));
    	return 0;
    }
    
    DEB(dbg("wait_while_slave timeout - isr: %x\n", hw->isr));
    return -1;
}

/*****************************************************************************
 * interrupt handling functions
 ****************************************************************************/

void i2c_fia320_isr(struct i2c_adapter *adap, int query_sts) {
    // should be called with interrupts disabled

    struct i2c_algo_fia320_data *fia_adap = adap->algo_data;
    struct iic_fia320_hw* hw = fia_adap->hw;
    uint32_t i;

    if (query_sts) {
    	i = get_real_sts(fia_adap);
    	hw->isr = i;
    }

    DEB2(dbg("i: %x, adap: %p\n", hw->isr, hw->adap));
    i2c_fia320_algo_handle_interrupt(hw->adap);
    DEB2(dbg("i2: %x, adap: %p\n", hw->isr, hw->adap));

    //dbg("w\n");
    wake_up_interruptible(&hw->wait);
}

static void i2c_fia320_algo_handle_interrupt(struct i2c_adapter* adap)
{
    unsigned long flags;
    struct i2c_algo_fia320_data *fia_adap = adap->algo_data;
    struct iic_fia320_hw* dev = fia_adap->hw;
    struct i2c_algo_fia320_slave_data *slave_data = fia_adap->slave_data;
	
    DEB2(dbg("algo_handle_interrupt\n"));
    //dbg("s\n");
    spin_lock_irqsave(&slave_data->slave_lock, flags);

    DEB2(dbg("fia320-%d: algo_handle_interrupt: slave isr - status: %x\n", dev->idx, dev->isr));
    {
	DEB(
	    dbg("");
	    if (dev->isr & FA320_SR_START) printk("START ");
	    if (dev->isr & FA320_SR_AL) printk("AL ");
	    if (dev->isr & FA320_SR_GC) printk("GC ");
	    if (dev->isr & FA320_SR_SAM) printk("SAM ");
	    if (dev->isr & FA320_SR_STOP) printk("STOP ");
	    if (dev->isr & FA320_SR_BERR) printk("BERR ");
	    if (dev->isr & FA320_SR_DR) printk("DR ");
	    if (dev->isr & FA320_SR_DT) printk("DT ");
	    if (dev->isr & FA320_SR_BB) printk("BB ");
	    if (dev->isr & FA320_SR_I2CB) printk("I2CB ");
	    if (dev->isr & FA320_SR_ACK) printk("ACK ");
	    if (dev->isr & FA320_SR_RW) printk("RW ");
	    printk("\n");
	)
    }	    
    
    // master mode not successful...
    if ((dev->isr & (FA320_SR_AL | FA320_SR_BERR)) || ((dev->isr & (FA320_SR_SAM | FA320_SR_STOP)) == FA320_SR_SAM)) {
    	dev->arbit_lost_xfer_error = 1;
    	dev->lost_arbitration = (dev->isr & FA320_SR_AL) ? 1 : 0;
    }

    // if not in slave mode, we have to adjust some things...
    if (!slave_data->slave_write_in_progress && !slave_data->slave_read_in_progress && (dev->isr & FA320_SR_STOP) && dev->handle_stop) {
	// leave master mode after sending STOP
	DEB3(dbg("got STOP\n"));
	set_cmd(fia_adap, 0);
	dev->isr &= ~FA320_SR_STOP;
	dev->handle_stop = 0;
    }
    	
    if ((dev->isr & FA320_SR_STOP || (!slave_data->slave_write_in_progress && !slave_data->slave_read_in_progress)) &&
    	!(dev->arbit_lost_xfer_error)) {
    	if (dev->isr & FA320_SR_DT) {
    	    dev->isr &= ~FA320_SR_DT;
    	    dev->transfer_done = 1;
    	    //dbg("t\n");
    	}
    	if ((dev->isr & (FA320_SR_DR | FA320_SR_RW)) == (FA320_SR_DR | FA320_SR_RW)) {
	    dev->master_read_byte = get_rxr(fia_adap);
    	    dev->isr &= ~(FA320_SR_DR | FA320_SR_RW);
    	    dev->transfer_done = 1;
    	    //dbg("t\n");
    	}
    }
    
    i2c_fia320_algo_handle_slave(adap);

    DEB2(dbg("fia320-%d: algo_handle_interrupt: done - slave isr - status: %x\n", dev->idx, dev->isr));

    spin_unlock_irqrestore(&slave_data->slave_lock, flags);
}

/*****************************************************************************
 * master mode functions
 ****************************************************************************/

/*----- receiving bytes -----*/
static int fia_sendbytes(struct i2c_adapter *i2c_adap, const char *buf,
			 int count, int last)
{
    struct i2c_algo_fia320_data *adap = i2c_adap->algo_data;
    struct iic_fia320_hw* hw = adap->hw;
    int wrcount, timeout;
    DEB2(dbg("start fia_sendbytes, count: %d\n", count)); 
    for (wrcount=0; wrcount<count; ++wrcount) {
	DEB3(dbg("fia_sendbytes: writing %2.2X\n", buf[wrcount]&0xff));
	/* send a byte*/
	DEB3(dbg("transfering byte: %x\n", buf[wrcount]));
	//dbg("b\n");
	adap->hw->isr = 0;
	hw->transfer_done = 0;
	set_txr(adap, buf[wrcount]);
	if (last && (wrcount+1 == count)) {
	    hw->handle_stop = 1;
	    set_cmd(adap, FA320_CR_SCLEN | FA320_CR_TBEN | FA320_CR_STOP);
	} else {
	    set_cmd(adap, FA320_CR_SCLEN | FA320_CR_TBEN);
	}
	DEB3(dbg("sb: wait for tip: %08x\n", FA320_SR_DT));
	timeout = wait_while_tip(adap);
	DEB3(dbg("got tip\n"));
	//dbg("d\n");
	if (timeout) {
	    i2c_stop(adap);
	    if (hw->arbit_lost_xfer_error) {
	       printk(KERN_ERR "fia_sendbytes: error.\n");
	    } else {
	       printk(KERN_ERR "fia_sendbytes: Timeout waiting for transfer done.\n");
	    }
	    return -EREMOTEIO; /* got a better one ?? */
	}
    }
    return (wrcount);
}


/*----- sending bytes -----*/
static int fia_readbytes(struct i2c_adapter *i2c_adap, char *buf,
			 int count, int last)
{
    int i;
    struct i2c_algo_fia320_data *adap = i2c_adap->algo_data;
    struct iic_fia320_hw* hw = adap->hw;

    for (i = 0; i < count; i++) {
	adap->hw->isr = 0;
	if (i == count - 1) {
	    if (last) {
#if 1
		DEBPROTO(dbg("P\n"));
#else
		dbg("P\n");
#endif
		hw->transfer_done = 0;
		hw->handle_stop = 1;
		set_cmd(adap, FA320_CR_SCLEN | FA320_CR_TBEN | FA320_CR_STOP | FA320_CR_ACKNAK);
		if (wait_while_tip(adap)) {
		    i2c_stop(adap);
		    printk(KERN_ERR "fia_readbytes timed out 1.\n");
		    return (-1);
		}

	    }
	    else {
		hw->transfer_done = 0;
		set_cmd(adap, FA320_CR_SCLEN | FA320_CR_TBEN | FA320_CR_ACKNAK);
		if (wait_while_tip(adap)) {
		    i2c_stop(adap);
		    printk(KERN_ERR "fia_readbytes timed out 2.\n");
		    return (-1);
		}
	    }
	}
	else {
	    hw->transfer_done = 0;
	    set_cmd(adap, FA320_CR_SCLEN | FA320_CR_TBEN);
	    if (wait_while_tip(adap)) {
		i2c_stop(adap);
		printk(KERN_ERR "fia_readbytes timed out 3.\n");
		return (-1);
	    }
	}  

	buf[i] = hw->master_read_byte;
    }

    return (i);
}

/*----- SMBus quick message -----*/

/* Bus timings (in us) for bit-banging */
static struct i2c_timings {
	unsigned int hd_sta;
	unsigned int su_sto;
	unsigned int low;
	unsigned int high;
	unsigned int buf;
} timings [] = {
/* Standard mode (100 KHz) */
{
    .hd_sta= 4,
    .su_sto= 4,
    .low= 6,
    .high= 4,
    .buf= 6,
},
/* Fast mode (400 KHz) */
{
    .hd_sta = 2,
    .su_sto= 2,
    .low = 4,
    .high = 2,
    .buf= 4,
}};

static int i2c_dc_wait(struct i2c_algo_fia320_data *adap, u32 mask) {
    unsigned long x = jiffies + HZ / 28 + 2;

    while ((get_bmr(adap) & mask) != mask) {
        if (time_after(jiffies, x)) {
            return -1;
        }
	cond_resched();
    }

    return 0;
}

static int i2c_smbus_quick(struct i2c_adapter *i2c_adap, const struct i2c_msg* p) {
    struct i2c_algo_fia320_data *adap = i2c_adap->algo_data;
    
    const struct i2c_timings* t = &timings[(adap->hw->scl && adap->hw->scl >= 400000) ? 1 : 0];
    u32 mask, v, sda;
    int i, res;

    /* Only 7-bit addresses are supported */
    if (p->flags & I2C_M_TEN) {
	dbg("smbus_quick - 10 bit addresses are not supported\n");
	return -EINVAL;
    }

    dbg("smbus_quick(0x%02x)\n", p->addr);

    /* START */
    set_cmd_noenable(adap, FA320_CR_SDA_LOW);
    sda = FA320_CR_SCL_LOW | FA320_CR_SDA_LOW;
    udelay(t->hd_sta);

    /* Send address */
    v = (u8)((p->addr << 1) | ((p->flags & I2C_M_RD) ? 1 : 0));
    for (i = 0, mask = 0x80; i < 8; ++i, mask >>= 1){
	set_cmd_noenable(adap, FA320_CR_SCL_LOW | sda);
	udelay(t->low / 2);
	sda = (v & mask) ? 0 : FA320_CR_SDA_LOW;
	set_cmd_noenable(adap, FA320_CR_SCL_LOW | sda);
	udelay(t->low / 2);

	set_cmd_noenable(adap, sda);
	if (i2c_dc_wait(adap, FA320_BMR_SCL))
            goto err;
	udelay(t->high);
    }

    /* ACK */
    set_cmd_noenable(adap, FA320_CR_SCL_LOW | sda);
    udelay(t->low / 2);
    set_cmd_noenable(adap, FA320_CR_SCL_LOW);
    udelay(t->low / 2);
    set_cmd_noenable(adap, 0);

    if (i2c_dc_wait(adap, FA320_BMR_SCL))
	goto err;
    res = (get_bmr(adap) & FA320_BMR_SDA) ? -EREMOTEIO : 1;
    udelay(t->high);

    /* STOP */
    set_cmd_noenable(adap, FA320_CR_SCL_LOW | FA320_CR_SDA_LOW);
    udelay(t->low);
    set_cmd_noenable(adap, FA320_CR_SDA_LOW);

    if (i2c_dc_wait(adap, FA320_BMR_SCL))
	goto err;
    udelay(t->su_sto);
    set_cmd_noenable(adap, 0);

    udelay(t->buf);

    dbg("smbus_quick -> %s\n", res < 0 ? "NACK" : "ACK");

out:
    /* Reset interface */
    adap->reset(i2c_adap);

    return res;
err:
    dbg("smbus_quick - bus is stuck\n");
    res = -EREMOTEIO;
    goto out;
}

/*------ bitbanging to free the bus --------------*/

int i2c_fia320_clock_bus_free(struct i2c_adapter *i2c_adap)
{
    struct i2c_algo_fia320_data *adap = i2c_adap->algo_data;
    const struct i2c_timings* t = &timings[(adap->hw->scl && adap->hw->scl >= 400000) ? 1 : 0];
    int timeout = 20;

    DEB(dbg("trying to free the bus by clocking scl.\n"));
    do {
	set_cmd_noenable(adap, FA320_CR_SCL_LOW);
	udelay(t->low);
        set_cmd_noenable(adap, 0);
	udelay(t->high);
	timeout--;
    } while ((!(get_bmr(adap) & FA320_BMR_SDA)) && (timeout > 0));
    if (timeout == 0) {
	DEB(dbg("could not free bus.\n"));
	return -1;
    } else {
	DEB(dbg("freed bus. left retries = %d\n", timeout));
	return 0;
    }
}

void i2c_fia320_send_stop(struct i2c_adapter *i2c_adap)
{
    struct i2c_algo_fia320_data *adap = i2c_adap->algo_data;
    const struct i2c_timings* t = &timings[(adap->hw->scl && adap->hw->scl >= 400000) ? 1 : 0];
    udelay(t->high);
    set_cmd_noenable(adap, FA320_CR_SDA_LOW | FA320_CR_SCL_LOW);
    udelay(t->low);
    set_cmd_noenable(adap, FA320_CR_SCL_LOW);
    udelay(t->su_sto);
    set_cmd_noenable(adap, 0);
}

/*----- addressing -----*/
static inline int fia_doAddress(struct i2c_adapter *i2c_adap,
			        struct i2c_msg *msg, int retries) 
{
    struct i2c_algo_fia320_data *adap = i2c_adap->algo_data;
    struct iic_fia320_hw* hw = adap->hw;
    unsigned short flags = msg->flags;
    unsigned char addr;

    if ( (flags & I2C_M_TEN)  ) {
	return -ENOPROTOOPT;
    } else {		/* normal 7bit address	*/
	int status;
	int i=retries;
	while (i--) {
	    addr = ( msg->addr << 1 );
	    if (flags & I2C_M_RD ) {
		addr |= 1;
	    }
	    if (flags & I2C_M_REV_DIR_ADDR )
		addr ^= 1;
	    set_txr(adap, addr);
	    hw->transfer_done = 0;
	    adap->hw->isr = 0;
	    //dbg("1\n");
	    i2c_start(adap);
	    
	    /* waiting for TIP to negate */
	    if (wait_while_tip(adap)) {
		if (!hw->arbit_lost_xfer_error) {
		    //dbg("1a: 0x%02x\n", msg->addr);
		    i2c_stop(adap);
		    printk(KERN_ERR "Timeout waiting for transfer done in do_address (0x%02x).\n", msg->addr);
		    return -EREMOTEIO;
		} else if (i > 0) {
	    	    hw->arbit_lost_xfer_error = 0;
	    	    if (wait_while_slave(i2c_adap)) {
	    	    	i2c_stop(adap);
	    	    	//dbg("1b: 0x%02x\n", msg->addr);
	    	    	break;
	    	    }
		} else {
		    i2c_stop(adap);
		    //dbg("1c: 0x%02x\n", msg->addr);
		    break;
		}
	    } else {
	    	/* check ACK */
	    	status = get_sts(adap);
	    	if (!(status & FA320_SR_ACK)) {
		    //dbg("2a: 0x%02x\n", msg->addr);
		    return 0; /* success */
		} else {
		    //dbg("2b: 0x%02x\n", msg->addr);
		}
	    }
	}
	/* we tried enough times, failed to write address */		
	DEB(dbg("failed to address slave at %x\n", msg->addr));
	return -EREMOTEIO;

    }
    return 0;
}

/*----- high level xfer function -----*/
static int fia_xfer(struct i2c_adapter *i2c_adap,
		    struct i2c_msg msgs[], 
		    int num)
{
    struct i2c_algo_fia320_data *adap = i2c_adap->algo_data;
    struct iic_fia320_hw* hw = adap->hw;
    struct i2c_msg *pmsg;
    int i;
    int ret=0, status, sent = 0;
    
    /* Check for bus busy */
    int timeout;
    
    /* check for invalid messages */
    if (num > 1) {
        for (i = 0; i < num; i++) {
            if (msgs[i].len == 0) {
                dbg("invalid len %d in msg[%d]\n", msgs[i].len, i);
                return -EINVAL;
            }
        }
    }
    
    hw->arbit_lost_xfer_error = 0;
    //dbg("wbb\n");
    timeout = wait_for_bb(i2c_adap);
    if (timeout) {
	DEB2(dbg(KERN_ERR "i2c-algo-fia320.o: "
		    "Timeout waiting for BB in fia_xfer\n");)
	    return -EIO;
    }

    for (i = 0;ret >= 0 && i < num; i++) {
	pmsg = &msgs[i];
	
	DEB2(dbg("i2c-algo-fia320.o: Doing %s %d bytes to 0x%02x - %d of %d messages\n",
		    pmsg->flags & I2C_M_RD ? "read" : "write",
		    pmsg->len, pmsg->addr, i + 1, num));
	
	if (num == 1 && !pmsg->len) {
            /* Special case for I2C_SMBUS_QUICK emulation.
             * FIA320 I2C doesn't support 0-length transactions
             * so we have to emulate them using bit-banging.
             */
            ret = i2c_smbus_quick(i2c_adap, pmsg);
            break;
	}   

	ret = fia_doAddress(i2c_adap, pmsg, i2c_adap->retries);
	if (ret) return ret;
	
	/* Check RxACK (last rcvd bit - slave ack) */
	status=get_sts(adap);
	if (status & FA320_SR_ACK) {
	    i2c_stop(adap);
	    DEB2(dbg("i2c-algo-fia320.o: No RxACK(1) in fia_xfer\n"));
		return (-EREMOTEIO);
	}
    
	DEB3(dbg("i2c-algo-fia320.o: Msg %d, addr=0x%x, flags=0x%x, len=%d\n",
		    i, msgs[i].addr, msgs[i].flags, msgs[i].len);)
        
        /* Read */
	if (pmsg->flags & I2C_M_RD) {
	    /* read bytes into buffer*/
	    ret = fia_readbytes(i2c_adap, pmsg->buf, pmsg->len,
				   (i + 1 == num));
        
	    if (ret != pmsg->len) {
		DEB2(dbg("i2c-algo-fia320.o: fail: "
				"only read %d bytes.\n",ret);)
	    } else {
		DEB2(dbg("i2c-algo-fia320.o: read %d bytes.\n",ret);)
	    }
	} else { /* Write */
	    DEB3(dbg("** will call fia_sendbytes\n"));
	    ret = fia_sendbytes(i2c_adap, pmsg->buf, pmsg->len,
				   (i + 1 == num));
        
	    if (ret != pmsg->len) {
		DEB2(dbg("i2c-algo-fia320.o: fail: "
				"only wrote %d bytes.\n",ret);)
	    } else {
		DEB2(dbg("i2c-algo-fia320.o: wrote %d bytes.\n",ret);)
	    }
	}

	++sent;
    }

    return ret < 0 ? ret : sent;
}

/*****************************************************************************
 * slave mode functions
 ****************************************************************************/

/* function prototypes */
static unsigned char iic_handle_slave_write(struct i2c_adapter *adap, int is_slave_stop);
static int iic_handle_slave_write_complete(struct i2c_adapter *adap);
static unsigned char iic_handle_slave_read(struct i2c_adapter *adap, int is_slave_stop);
static void iic_handle_slave_read_complete(struct i2c_adapter *adap);

/* set own slave address */
int i2c_fia320_slave_set_address(struct i2c_adapter* i2c_adap, unsigned int address)
{
    unsigned char reg;
    struct i2c_algo_fia320_data *adap = (struct i2c_algo_fia320_data*)(i2c_adap->algo_data);
    struct iic_fia320_hw* dev = adap->hw;

    // disable the core
    set_cmd_noenable(adap, 0);

    set_sar(adap, address);

    DEB2(dbg ("set i2c slave address to 0x%x\n", address));
    reg = get_sar(adap);
    if ((reg & 0x7f) != address) {
	printk(KERN_WARNING "fia320-%d: could not set i2c slave address, result is 0x%02x",
	       dev->idx, reg);
    
    } else {
    	DEB2(dbg ("verify slave address: 0x%02x\n", reg));
    }

    // re-enable the core
    set_cmd(adap, 0);
    return 0;
}

/* after the bus is gone down, reactivate transfer */
int i2c_fia320_slave_restart_xfer(struct i2c_adapter* adap)
{
    return 0;
}

static unsigned char iic_handle_slave_write_start(struct i2c_adapter* adap) {
    struct i2c_algo_fia320_data *fia_adap = adap->algo_data;
    struct i2c_algo_fia320_slave_data *slave_data = fia_adap->slave_data;

    /* 16.5.2 (4) */
    DEB2(dbg("fia320-%d: dest i2c address in dr: %x\n", fia_adap->hw->idx, get_rxr(fia_adap)));
    
    slave_data->ignore_slave_write_byte = 1;
    set_cmd(fia_adap, FA320_CR_TBEN);
    
    return 0;
}

/* handle i2c slave write requests */
static unsigned char iic_handle_slave_write(struct i2c_adapter* adap, int is_slave_stop) {
    struct i2c_algo_fia320_data *fia_adap = adap->algo_data;
    struct i2c_algo_fia320_slave_data *slave_data = fia_adap->slave_data;
    unsigned char data = get_rxr(fia_adap);
	
    //set_cmd(fia_adap, 0);

    /* write bytes received to buffer, but skip over address byte */
    if (slave_data->ignore_slave_write_byte) {
    	slave_data->ignore_slave_write_byte = 0;

    	DEB2(dbg("fia320-%d: iic_handle_slave_write: skipping byte 0x%02x (slave_status=0x%02x)\n",
		fia_adap->hw->idx, *(slave_data->slave_write_ptr), fia_adap->hw->isr));
    } else {
    	*(slave_data->slave_write_ptr) = data;

    	DEB2(dbg("fia320-%d: iic_handle_slave_write: putting byte 0x%02x (slave_status=0x%02x)\n",
		fia_adap->hw->idx, *(slave_data->slave_write_ptr), fia_adap->hw->isr));

    	/* avoid buffer overrun here */
    	if (slave_data->slave_write_ptr < slave_data->slave_write_buffer + FIA_SLAVE_BUF_SIZE) {
	    slave_data->slave_write_ptr++;
	}

    	slave_data->slave_write_count++;
    }

    if (is_slave_stop) {
	DEB2(dbg("fia320-%d: slave write done.\n", fia_adap->hw->idx));
	slave_data->slave_write_in_progress = 0;
	iic_handle_slave_write_complete(adap);
    } else {
    	DEB2(dbg("fia320-%d: slave write NOT done.\n", fia_adap->hw->idx));
    	set_cmd(fia_adap, FA320_CR_TBEN);
    }
    return 0;
}

/* if a slave write is complete, push the data up
   to the upper i2c-slave level */
static int iic_handle_slave_write_complete(struct i2c_adapter* adap) {
    struct i2c_algo_fia320_data *fia_adap = adap->algo_data;
    struct iic_fia320_hw* dev = fia_adap->hw;
    struct i2c_algo_fia320_slave_data *slave_data = fia_adap->slave_data;
	
    int ret = 0;
    int written_bytes;
    int bytes_to_write = slave_data->slave_write_count;
	
    if (slave_data->slave_write_count <= 0) {
	return slave_data->slave_write_count;
    }
    if (bytes_to_write == 0) {
	return 0;
    }
	
    /* copy the written data to the upper I2C slave layer */
    written_bytes = iic_slave_write(adap, get_sar(fia_adap),
    	slave_data->slave_write_buffer, bytes_to_write);
    if (written_bytes < 0) {
	printk(KERN_WARNING "fia320-%d: iic_handle_slave_write_complete: error exporting data to i2c-slave layer: %d!\n",
	       dev->idx, written_bytes);
	ret = -ENOMEM;
    }
    else if (written_bytes == 0) {
	DEB2(dbg("fia320-%d: iic_handle_slave_write_complete: 0 bytes to i2c-slave layer exported!\n",
		    dev->idx));
    }
    else if (written_bytes >= bytes_to_write) {
	DEB2(dbg("fia320-%d: iic_handle_slave_write_complete: moved %d bytes to i2c-slave layer.\n",
		    dev->idx, written_bytes));
	slave_data->slave_write_ptr = slave_data->slave_write_buffer;
	slave_data->slave_write_count = 0;
    }
    else {
	// only a partial copy
	DEB2(dbg("fia320-%d: iic_handle_slave_write_complete: moved %d bytes to i2c-slave layer, %d bytes requested.\n",
		    dev->idx, written_bytes, bytes_to_write));
	slave_data->slave_write_ptr = slave_data->slave_write_buffer + written_bytes;
	slave_data->slave_write_count -= written_bytes;
    }
    
    set_cmd(fia_adap, 0);
    
    return ret;
}

/* handle first slave read interrupt of an i2c message
   fetches data from upper i2c slave level */
static void iic_handle_slave_first_read(struct i2c_adapter* adap)
{
    struct i2c_algo_fia320_data *fia_adap = adap->algo_data;
    struct i2c_algo_fia320_slave_data *slave_data = fia_adap->slave_data;


    DEB2(dbg("fia320-%d: iic_handle_slave_first_read: Fetching slave read data from i2c-slave layer.\n", fia_adap->hw->idx));
	
    slave_data->slave_read_ptr = slave_data->slave_read_buffer;
    slave_data->slave_read_count = iic_slave_read(adap, get_sar(fia_adap),
    	slave_data->slave_read_buffer, FIA_SLAVE_BUF_SIZE, 0);
        
    if (slave_data->slave_read_count > 0) {
	DEB2(dbg("fia320-%d: iic_handle_slave_first_read: Got %d bytes.\n", fia_adap->hw->idx, slave_data->slave_read_count));
    }
    else {
	DEB(dbg("fia320-%d: iic_handle_slave_first_read: cannot read bytes.\n", fia_adap->hw->idx));
    }
}

/* handle an i2c slave read request */
static unsigned char iic_handle_slave_read(struct i2c_adapter* adap, int is_slave_stop)
{
    struct i2c_algo_fia320_data *fia_adap = adap->algo_data;
    struct i2c_algo_fia320_slave_data *slave_data = fia_adap->slave_data;

    unsigned char data = 0;

    if (slave_data->slave_read_count <= 0) {
	iic_handle_slave_first_read(adap);
    }
	
    if (slave_data->slave_read_count > 0) {
	/* avoid buffer overrun here */
	if (slave_data->slave_read_ptr < slave_data->slave_read_buffer + slave_data->slave_read_count) {
	    data = *(slave_data->slave_read_ptr);
	    slave_data->slave_read_ptr++;
	}
	else {
	    /* TODO: error handline here */
	    data = 0;
	    DEB(dbg("fia320-%d: iic_handle_slave_read: no data!\n", fia_adap->hw->idx));
	}
		
	DEB2(dbg("fia320-%d: iic_handle_slave_read: sending byte 0x%02x\n", fia_adap->hw->idx, data));
	
	set_txr(fia_adap, data);
	set_cmd(fia_adap, FA320_CR_TBEN);
    } else {
	/* TODO: signal that we are not ready by not sending data */
	set_txr(fia_adap, data);
	set_cmd(fia_adap, FA320_CR_TBEN);
	DEB2(dbg("fia320-%d: iic_handle_slave_read: not ready, sending no byte\n", fia_adap->hw->idx));
    }

    if (is_slave_stop) {
	slave_data->slave_read_in_progress = 0;
	iic_handle_slave_read_complete(adap);
    }

    return 0;
}

/* slave read completed */
static void iic_handle_slave_read_complete(struct i2c_adapter* adap)
{
    struct i2c_algo_fia320_data *fia_adap = adap->algo_data;
    struct i2c_algo_fia320_slave_data *slave_data = fia_adap->slave_data;

    if (slave_data->slave_read_ptr < slave_data->slave_read_buffer + slave_data->slave_read_count) {
	DEB(dbg("fia320-%d: iic_handle_slave_read_complete: not all data was read! amount: %d, read: %d, first byte: %x\n",
		    fia_adap->hw->idx, slave_data->slave_read_count,
		    slave_data->slave_read_ptr - slave_data->slave_read_buffer,
		    slave_data->slave_read_buffer[0]));
    }
    else {
	DEB2(dbg("fia320-%d: iic_handle_slave_read_complete: read %d bytes\n",	
		fia_adap->hw->idx, slave_data->slave_read_ptr - slave_data->slave_read_buffer));
    }
	
    slave_data->slave_read_ptr = slave_data->slave_read_buffer;
    slave_data->slave_read_count = 0;
    
    set_cmd(fia_adap, 0);
}

static void i2c_fia_finish_last_transfer(struct i2c_adapter* adap)
{
    struct i2c_algo_fia320_data *fia_adap = adap->algo_data;
    struct i2c_algo_fia320_slave_data *slave_data = fia_adap->slave_data;

    if (slave_data->slave_write_in_progress) {
        if (iic_handle_slave_write_complete(adap) < 0) {
        	DEB(dbg("fia320-%d: cannot put data to i2c-slave layer, next operation will be blocked.\n",
    	        fia_adap->hw->idx));
        	slave_data->block_next_slave_operation = 1;
        }
        slave_data->slave_write_in_progress = 0;
    }
    
    if (slave_data->slave_read_in_progress) {
        iic_handle_slave_read_complete(adap);
        slave_data->slave_read_in_progress = 0;
    }
    
}

/* handle iic slave interrupt */
static void i2c_fia320_algo_handle_slave(struct i2c_adapter* adap)
{
    struct i2c_algo_fia320_data *fia_adap = adap->algo_data;
    struct iic_fia320_hw* dev = fia_adap->hw;
    struct i2c_algo_fia320_slave_data *slave_data = fia_adap->slave_data;

    int is_slave_read_start = 0;
    int is_slave_read = 0;
    int is_slave_write_start = 0;
    int is_slave_write = 0;
    int is_slave_stop =  0;
    int is_start_and_stop = 0;

    DEB2(dbg("algo_handle_slave\n"));

    if (! (
	   (dev->isr & FA320_SR_SAM) |
	   slave_data->slave_write_in_progress |
	   slave_data->slave_read_in_progress
	   )
	) {
	// not a slave call
	DEB2(dbg("Not a slave call.\n"));
	
	goto bail;
    }

    /* find out which slave operation is to be performed */
    if ((dev->isr & (FA320_SR_SAM | FA320_SR_I2CB | FA320_SR_RW)) == (FA320_SR_SAM | FA320_SR_I2CB | FA320_SR_RW)) {
	/* start of slave read */
	DEB2(dbg("slave read start: isr:%x, mask: %x\n", dev->isr, (FA320_SR_SAM | FA320_SR_I2CB | FA320_SR_RW)));
	is_slave_read = 1;
	is_slave_read_start = 1;
    } else if ((dev->isr & (FA320_SR_SAM | FA320_SR_I2CB)) == (FA320_SR_SAM | FA320_SR_I2CB)) {
	/* start of slave write */
	DEB2(dbg("fia320-%d: slave write start\n", dev->idx));
	is_slave_write_start = 1;
	if ((dev->isr & (FA320_SR_DR | FA320_SR_ACK | FA320_SR_RW)) == FA320_SR_DR) {
	    DEB2(dbg("fia320-%d: slave write\n", dev->idx));
	    is_slave_write = 1;
	}
    } else if ((dev->isr & (FA320_SR_DT | FA320_SR_RW | FA320_SR_ACK)) ==  (FA320_SR_DT | FA320_SR_RW)) {
	/* slave read */
	DEB2(dbg("fia320-%d: slave read\n", dev->idx));
	is_slave_read = 1;
    } else if ((dev->isr & (FA320_SR_SAM | FA320_SR_DR | FA320_SR_ACK | FA320_SR_RW)) == FA320_SR_DR) {
	/* slave write */
	DEB2(dbg("fia320-%d: slave write\n", dev->idx));
	is_slave_write = 1;
    }
    if (dev->isr & FA320_SR_STOP) {
	is_slave_stop = 1;
    	if (dev->isr & FA320_SR_START) {
    	    is_start_and_stop = 1;
    	}
    }
#if 1
    DEB2(
#else
    (
#endif
    dbg("fia320-%d: is_sl_r: %d, is_sl_r_start: %d, is_sl_w: %d, is_sl_w_start: %d, is_slave_stop: %d\n",
    	dev->idx, is_slave_read, is_slave_read_start, is_slave_write, is_slave_write_start, is_slave_stop));
    /* performe the requested operation */
    slave_data->block_next_slave_operation = 0;	
    if (slave_data->block_next_slave_operation) {
	/* we block the slave operation by not reading data
	   and not resetting the slave status */
    }
    else {
	if (!slave_data->slave_write_in_progress && !slave_data->slave_read_in_progress) {
	    // ignore the STOP if no slave transfer in progress
	    is_slave_stop = 0;
	}
	if (is_slave_stop && slave_data->slave_write_in_progress) {
	    slave_data->slave_write_in_progress = 0;
	    if (iic_handle_slave_write_complete(adap) < 0) {
		DEB(dbg("fia320-%d: cannot put data to i2c-slave layer, next operation will be blocked.\n",
			   dev->idx));
		slave_data->block_next_slave_operation = 1;
	    }
	    if (is_start_and_stop) {
		is_slave_write_start = 1;
		is_slave_stop = 0;
	    }
	}
	if (is_slave_stop && slave_data->slave_read_in_progress) {
	    slave_data->slave_read_in_progress= 0;
	    iic_handle_slave_read_complete(adap);
	}
	if (is_slave_write_start)  {
	    // possible repeated start
	    i2c_fia_finish_last_transfer(adap);
	    
	    slave_data->slave_write_in_progress = 1;
	    iic_handle_slave_write_start(adap);
	}
	if (is_slave_write) {
	    iic_handle_slave_write(adap, is_slave_stop);
	}
	if (is_slave_read) {
	    if (is_slave_read_start) {
	    	// possible repeated start
	    	i2c_fia_finish_last_transfer(adap);

		slave_data->slave_read_in_progress = 1;	    	
	    }
	    iic_handle_slave_read(adap, is_slave_stop);
	}
    }
    
    dev->isr &= ~(
    	FA320_SR_START | FA320_SR_GC | FA320_SR_SAM | FA320_SR_STOP | FA320_SR_DR | 
    	FA320_SR_DT | FA320_SR_ACK | FA320_SR_RW
    );
    if (is_slave_stop) {
    	dev->isr &= ~(FA320_SR_I2CB);
    }


    DEB2(dbg("fia320-%d: iic_handle_slave: done - slave isr - status: %x\n", dev->idx, dev->isr));

 bail:
    ;
}

/*****************************************************************************
 * exported algorithm data
 ****************************************************************************/

static u32 fia_func(struct i2c_adapter *adap)
{
    return I2C_FUNC_SMBUS_EMUL | I2C_FUNC_PROTOCOL_MANGLING;

}

static int iic_algo_control(struct i2c_adapter *adap,
                            unsigned int cmd, unsigned long arg)
{
    switch (cmd) {
      case I2C_SLAVE_SET_ADDRESS:
	  i2c_fia320_slave_set_address(adap, (unsigned int) arg);
#if DEBUG_TO_BUFFER
          i2c_fia_print_dbg();
#endif
	  break;
      case I2C_SLAVE_FLUSH_BUFFER:
	  i2c_fia320_slave_restart_xfer(adap);
	  break;
      default:
	  return -1;
    }
    return 0;
}


static struct i2c_algorithm fia_algo = {
    .name		= "FIA320 algorithm",
    .id			= I2C_ALGO_FIA320,
    .master_xfer	= fia_xfer,
    .functionality	= fia_func,
    .algo_control       = iic_algo_control,
};

/*****************************************************************************
 * initializing and cleanup
 ****************************************************************************/

/* 
 * registering functions to load algorithms at runtime 
 */
int i2c_fia320_add_bus(struct i2c_adapter *adap)
{
    //	struct i2c_algo_fia320_data *fia_adap = adap->algo_data;

    DEB2(dbg("hw routines registered.\n"));
    
    /* register new adapter to i2c module... */
    
    adap->id |= fia_algo.id;
    adap->algo = &fia_algo;

    adap->timeout = 100;		/* default values, should	*/
    adap->retries = 1;		/* be replaced by defines	*/

    i2c_add_adapter(adap);
    return 0;
}


int i2c_fia320_del_bus(struct i2c_adapter *adap)
{
    return i2c_del_adapter(adap);
}

int __init i2c_algo_fia320_init (void) {
    printk(KERN_INFO "Faraday FIA320 i2c algorithm module\n");

    return 0;
}


void i2c_algo_fia320_exit(void)
{
    return;
}


/*****************************************************************************
 * module stuff
 ****************************************************************************/

EXPORT_SYMBOL(i2c_fia320_add_bus);
EXPORT_SYMBOL(i2c_fia320_del_bus);
EXPORT_SYMBOL(i2c_fia320_isr);
EXPORT_SYMBOL(i2c_fia320_slave_set_address);
EXPORT_SYMBOL(i2c_fia320_slave_restart_xfer);
EXPORT_SYMBOL(i2c_fia320_clock_bus_free);
EXPORT_SYMBOL(i2c_fia320_send_stop);

#if DEBUG_TO_BUFFER
EXPORT_SYMBOL(i2c_fia_print);
#endif

module_init(i2c_algo_fia320_init);
module_exit(i2c_algo_fia320_exit);
