/*
 * drivers/i2c/i2c-ibm_iic.c
 *
 * Support for the IIC peripheral on IBM PPC 4xx
 *
 * Copyright (c) 2003, 2004 Zultys Technologies.
 * Eugene Surovegin <eugene.surovegin@zultys.com> or <ebs@ebshome.net>
 *
 * Based on original work by 
 * 	Ian DaSilva  <idasilva@mvista.com>
 *      Armin Kuster <akuster@mvista.com>
 * 	Matt Porter  <mporter@mvista.com>
 *
 *      Copyright 2000-2003 MontaVista Software Inc.
 *
 * Original driver version was highly leveraged from i2c-elektor.c
 *
 *   	Copyright 1995-97 Simon G. Vogl
 *                1998-99 Hans Berglund
 *
 *   	With some changes from Kysti Mlkki <kmalkki@cc.hut.fi> 
 *	and even Frodo Looijaard <frodol@dds.nl>
 *
 * This program is free software; you can redistribute  it and/or modify it
 * under  the terms of  the GNU General  Public License as published by the
 * Free Software Foundation;  either version 2 of the  License, or (at your
 * option) any later version.
 *
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/ioport.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <asm/irq.h>
#include <asm/io.h>
#include <linux/i2c.h>
#include <linux/i2c-id.h>
#include <linux/i2c-slave.h>
#include <asm/ocp.h>
#include <asm/ibm4xx.h>

#include "i2c-ibm_iic.h"

#define BUS_TIMEOUT		1
#define BUS_WAIT_TIME		100
#define BUS_WAIT_RETRIES	1000

#define DRIVER_VERSION "2.1 with slave mode support"

MODULE_DESCRIPTION("IBM IIC driver v" DRIVER_VERSION);
MODULE_LICENSE("GPL");

static int iic_force_poll;
module_param(iic_force_poll, bool, 0);
MODULE_PARM_DESC(iic_force_poll, "Force polling mode");

static int iic_force_fast;
module_param(iic_force_fast, bool, 0);
MODULE_PARM_DESC(iic_force_fast, "Force fast mode (400 kHz)");

static int iic_own_addr = 0x55;
MODULE_PARM(iic_own_addr, "i");
MODULE_PARM_DESC(iic_own_addr, "Our own slave address");

/* slave mode */
static void iic_slave_init(void);
static void iic_handle_slave(struct ibm_iic_private* dev,
			     unsigned char status,
			     unsigned char ext_status,
			     unsigned char slave_status,
			     unsigned char xfer_cnt);
static int iic_slave_set_address(struct i2c_adapter *adap,
				 unsigned int address);
static int iic_slave_restart_xfer(struct i2c_adapter* adapter);
static int iic_slave_init_adapter(struct i2c_adapter* adapter);
static int iic_slave_cleanup_adapter(void);

#define DBG_LEVEL 0

#ifdef DBG
#undef DBG
#endif

#ifdef DBG2
#undef DBG2
#endif

#if DBG_LEVEL > 0
#  define DBG(f,x...)	printk(KERN_DEBUG "ibm-iic" f, ##x)
#else
#  define DBG(f,x...)	((void)0)
#endif
#if DBG_LEVEL > 1
#  define DBG2(f,x...) 	DBG(f, ##x)
#else
#  define DBG2(f,x...) 	((void)0)
#endif
#if DBG_LEVEL > 2
static void dump_iic_regs(const char* header, struct ibm_iic_private* dev)
{
	volatile struct iic_regs __iomem *iic = dev->vaddr;
	printk(KERN_DEBUG "ibm-iic%d: %s\n", dev->idx, header);
	printk(KERN_DEBUG "  cntl     = 0x%02x, mdcntl = 0x%02x\n"
	       KERN_DEBUG "  sts      = 0x%02x, extsts = 0x%02x\n"
	       KERN_DEBUG "  clkdiv   = 0x%02x, xfrcnt = 0x%02x\n"
	       KERN_DEBUG "  xtcntlss = 0x%02x, directcntl = 0x%02x\n"
#if DBG_LEVEL > 3
	       KERN_DEBUG "  lmadr    = 0x%02x, hmadr  = 0x%02x\n"
	       KERN_DEBUG "  lsadr    = 0x%02x, hsadr  = 0x%02x\n"
	       KERN_DEBUG "  intmsk   = 0x%02x\n"
#endif
	       ,
	       in_8(&iic->cntl), in_8(&iic->mdcntl), in_8(&iic->sts), 
	       in_8(&iic->extsts), in_8(&iic->clkdiv), in_8(&iic->xfrcnt), 
	       in_8(&iic->xtcntlss), in_8(&iic->directcntl)
#if DBG_LEVEL > 3
	       ,
	       in_8(&iic->lmadr), in_8(&iic->hmadr),
	       in_8(&iic->lsadr), in_8(&iic->hsadr),
	       in_8(&iic->intmsk)
#endif
	       );
}
#  define DUMP_REGS(h,dev)	dump_iic_regs((h),(dev))
#else
#  define DUMP_REGS(h,dev)	((void)0)
#endif

/* Bus timings (in ns) for bit-banging */
static struct i2c_timings {
	unsigned int hd_sta;
	unsigned int su_sto;
	unsigned int low;
	unsigned int high;
	unsigned int buf;
} timings [] = {
/* Standard mode (100 KHz) */
{
	.hd_sta	= 4000,
	.su_sto	= 4000,
	.low	= 4700,
	.high	= 4000,
	.buf	= 4700,
},
/* Fast mode (400 KHz) */
{
	.hd_sta = 600,
	.su_sto	= 600,
	.low 	= 1300,
	.high 	= 600,
	.buf	= 1300,
}};

/* Enable/disable interrupt generation */
static inline void iic_interrupt_mode(struct ibm_iic_private* dev, int enable)
{
	out_8(&dev->vaddr->intmsk, enable ? INTRMSK_EIRC | INTRMSK_EIRS | INTRMSK_EIMTC | INTRMSK_EIWC | INTRMSK_EIWS : 0);
}
 
/*
 * Initialize IIC interface.
 */
static void iic_dev_init(struct ibm_iic_private* dev)
{
	volatile struct iic_regs __iomem *iic = dev->vaddr;

	DBG("%d: init\n", dev->idx);
	
	/* Clear master address */
	out_8(&iic->lmadr, 0);
	out_8(&iic->hmadr, 0);

	/* Set slave address */
	out_8(&iic->lsadr, iic_own_addr << 1);
	out_8(&iic->hsadr, 0);

	/* Clear status & extended status */
	out_8(&iic->sts, STS_SCMP | STS_IRQA);
	out_8(&iic->extsts, EXTSTS_IRQP | EXTSTS_IRQD | EXTSTS_LA
			    | EXTSTS_ICT | EXTSTS_XFRA);

	/* Set clock divider */
	out_8(&iic->clkdiv, dev->clckdiv);

	/* Clear transfer count */
	out_8(&iic->xfrcnt, 0);

	/* Clear extended control and status */
	out_8(&iic->xtcntlss, XTCNTLSS_SRC | XTCNTLSS_SRS | XTCNTLSS_SWC
			    | XTCNTLSS_SWS);

	/* Clear control register */
	out_8(&iic->cntl, 0);
	
	/* Enable interrupts if possible */
	iic_interrupt_mode(dev, dev->irq >= 0);

	/* Set mode control */
#if defined(LARA_KIMMSI) || defined(LARA_KIMSMI)
	/* disable PPC I2C slave mode, because of address conflict with MSI host */
	out_8(&iic->mdcntl, MDCNTL_FSDB | MDCNTL_FMDB //| MDCNTL_ESM
	      | MDCNTL_EINT | MDCNTL_EUBS | MDCNTL_HSCL
	      | (dev->fast_mode ? MDCNTL_FSM : 0));
#else
	out_8(&iic->mdcntl, MDCNTL_FSDB | MDCNTL_FMDB | MDCNTL_ESM
	      | MDCNTL_EINT | MDCNTL_EUBS | MDCNTL_HSCL
	      | (dev->fast_mode ? MDCNTL_FSM : 0));
#endif


	DUMP_REGS("iic_init", dev);
}

/* 
 * Reset IIC interface
 */
static void iic_dev_reset(struct ibm_iic_private* dev)
{
	volatile struct iic_regs __iomem *iic = dev->vaddr;
	int i;
	u8 dc;
	
	DBG("%d: soft reset\n", dev->idx);
	DUMP_REGS("reset", dev);
	
    	/* Place chip in the reset state */
	out_8(&iic->xtcntlss, XTCNTLSS_SRST);
	
	/* Check if bus is free */
	dc = in_8(&iic->directcntl);	
	if (!DIRCTNL_FREE(dc)){
		DBG("%d: trying to regain bus control\n", dev->idx);
	
		/* Try to set bus free state */
		out_8(&iic->directcntl, DIRCNTL_SDAC | DIRCNTL_SCC);	
	
		/* Wait until we regain bus control */
		for (i = 0; i < 100; ++i){
			dc = in_8(&iic->directcntl);
			if (DIRCTNL_FREE(dc))
				break;
			
			/* Toggle SCL line */
			dc ^= DIRCNTL_SCC;
			out_8(&iic->directcntl, dc);
			udelay(10);
			dc ^= DIRCNTL_SCC;
			out_8(&iic->directcntl, dc);
			
			/* be nice */
			cond_resched();
		}
	}
	
	/* Remove reset */
	out_8(&iic->xtcntlss, 0);
	
	/* Reinitialize interface */
	iic_dev_init(dev);
}

/*
 * Do 0-length transaction using bit-banging through IIC_DIRECTCNTL register.
 */

/* Wait for SCL and/or SDA to be high */
static int iic_dc_wait(volatile struct iic_regs __iomem *iic, u8 mask)
{
	unsigned long x = jiffies + HZ / 28 + 2;
	while ((in_8(&iic->directcntl) & mask) != mask){
		if (unlikely(time_after(jiffies, x)))
			return -1;
		cond_resched();
	}
	return 0;
}

static int iic_smbus_quick(struct ibm_iic_private* dev, const struct i2c_msg* p)
{
	volatile struct iic_regs __iomem *iic = dev->vaddr;
	const struct i2c_timings* t = &timings[dev->fast_mode ? 1 : 0];
	u8 mask, v, sda;
	int i, res;

	/* Only 7-bit addresses are supported */
	if (unlikely(p->flags & I2C_M_TEN)){
		DBG("%d: smbus_quick - 10 bit addresses are not supported\n",
			dev->idx);
		return -EINVAL;
	}

	DBG("%d: smbus_quick(0x%02x)\n", dev->idx, p->addr);

	/* Reset IIC interface */
	out_8(&iic->xtcntlss, XTCNTLSS_SRST);

	/* Wait for bus to become free */
	out_8(&iic->directcntl, DIRCNTL_SDAC | DIRCNTL_SCC);
	if (unlikely(iic_dc_wait(iic, DIRCNTL_MSDA | DIRCNTL_MSC)))
		goto err;
	ndelay(t->buf);

	/* START */
	out_8(&iic->directcntl, DIRCNTL_SCC);
	sda = 0;
	ndelay(t->hd_sta);

	/* Send address */
	v = (u8)((p->addr << 1) | ((p->flags & I2C_M_RD) ? 1 : 0));
	for (i = 0, mask = 0x80; i < 8; ++i, mask >>= 1){
		out_8(&iic->directcntl, sda);
		ndelay(t->low / 2);
		sda = (v & mask) ? DIRCNTL_SDAC : 0;
		out_8(&iic->directcntl, sda);
		ndelay(t->low / 2);

		out_8(&iic->directcntl, DIRCNTL_SCC | sda);
		if (unlikely(iic_dc_wait(iic, DIRCNTL_MSC)))
			goto err;
		ndelay(t->high);
	}

	/* ACK */
	out_8(&iic->directcntl, sda);
	ndelay(t->low / 2);
	out_8(&iic->directcntl, DIRCNTL_SDAC);
	ndelay(t->low / 2);
	out_8(&iic->directcntl, DIRCNTL_SDAC | DIRCNTL_SCC);
	if (unlikely(iic_dc_wait(iic, DIRCNTL_MSC)))
		goto err;
	res = (in_8(&iic->directcntl) & DIRCNTL_MSDA) ? -EREMOTEIO : 1;
	ndelay(t->high);

	/* STOP */
	out_8(&iic->directcntl, 0);
	ndelay(t->low);
	out_8(&iic->directcntl, DIRCNTL_SCC);
	if (unlikely(iic_dc_wait(iic, DIRCNTL_MSC)))
		goto err;
	ndelay(t->su_sto);
	out_8(&iic->directcntl, DIRCNTL_SDAC | DIRCNTL_SCC);

	ndelay(t->buf);

	DBG("%d: smbus_quick -> %s\n", dev->idx, res ? "NACK" : "ACK");
out:
	/* Remove reset */
	out_8(&iic->xtcntlss, 0);

	/* Reinitialize interface */
	iic_dev_init(dev);

	return res;
err:
	DBG("%d: smbus_quick - bus is stuck\n", dev->idx);
	res = -EREMOTEIO;
	goto out;
}

/*
 * IIC interrupt handler
 */
static irqreturn_t iic_handler(int irq, void *dev_id, struct pt_regs *regs)
{
	struct ibm_iic_private* dev = (struct ibm_iic_private*)dev_id;
	volatile struct iic_regs __iomem *iic = dev->vaddr;
	unsigned char status, ext_status, slave_status, xfer_cnt;

	status = in_8(&iic->sts);
	ext_status = in_8(&iic->extsts);
	slave_status = in_8(&iic->xtcntlss);
	xfer_cnt = in_8(&iic->xfrcnt);

	DBG2("%d: irq handler, STS = 0x%02x, EXTSTS = 0x%02x, XTCNTLSS = 0x%02x\n", 
	     dev->idx, status, ext_status, slave_status);
#if DBG_LEVEL > 2
	DBG2("%d: irq handler, CNTL = 0x%02x, MDCNTL = 0x%02x, DIRECTCNTL = 0x%02x, XFRCNT = 0x%02x\n",
	     dev->idx, in_8(&iic->cntl), in_8(&iic->mdcntl),
	     in_8(&iic->directcntl), xfer_cnt);
#endif

	/* look if we got a slave related irq */
	if (slave_status & (XTCNTLSS_SWC | XTCNTLSS_SWS | XTCNTLSS_SRS | XTCNTLSS_SRC)) {
		iic_handle_slave(dev, status, ext_status, slave_status, xfer_cnt);
	}
	
	/* Acknowledge IRQ and wakeup iic_wait_for_tc */
	out_8(&iic->sts, STS_IRQA | STS_SCMP);
	out_8(&iic->extsts, ext_status);
	out_8(&iic->sts, status);
	wake_up_interruptible(&dev->wq);
	
	return IRQ_HANDLED;
}

/*
 * Get master transfer result and clear errors if any.
 * Returns the number of actually transferred bytes or error (<0)
 */
static int iic_xfer_result(struct ibm_iic_private* dev)
{
	volatile struct iic_regs __iomem *iic = dev->vaddr;	
	
	if (unlikely(in_8(&iic->sts) & STS_ERR)){
		DBG("%d: xfer error, EXTSTS = 0x%02x\n", dev->idx, 
			in_8(&iic->extsts));
				
		/* Clear errors and possible pending IRQs */
		out_8(&iic->extsts, EXTSTS_IRQP | EXTSTS_IRQD | 
			EXTSTS_LA | EXTSTS_ICT | EXTSTS_XFRA);
			
		/* Flush master data buffer */
		out_8(&iic->mdcntl, in_8(&iic->mdcntl) | MDCNTL_FMDB);
		
		/* Is bus free?
		 * If error happened during combined xfer
		 * IIC interface is usually stuck in some strange
		 * state, the only way out - soft reset.
		 */
		if ((in_8(&iic->extsts) & EXTSTS_BCS_MASK) != EXTSTS_BCS_FREE){
			DBG("%d: bus is stuck, resetting\n", dev->idx);
			iic_dev_reset(dev);
		}
		return -EREMOTEIO;
	}
	else
		return in_8(&iic->xfrcnt) & XFRCNT_MTC_MASK;
}

/*
 * Try to abort active transfer.
 */
static void iic_abort_xfer(struct ibm_iic_private* dev)
{
	volatile struct iic_regs __iomem *iic = dev->vaddr;
	unsigned long x;
	
	DBG("%d: iic_abort_xfer\n", dev->idx);
	
	out_8(&iic->cntl, CNTL_HMT);
	
	/*
	 * Wait for the abort command to complete.
	 * It's not worth to be optimized, just poll (timeout >= 1 tick)
	 */
	x = jiffies + 2;
	while ((in_8(&iic->extsts) & EXTSTS_BCS_MASK) != EXTSTS_BCS_FREE){
		if (time_after(jiffies, x)){
			DBG("%d: abort timeout, resetting...\n", dev->idx);
			iic_dev_reset(dev);
			return;
		}
		schedule();
	}

	/* Just to clear errors */
	iic_xfer_result(dev);
}

/*
 * Wait for master transfer to complete.
 * It puts current process to sleep until we get interrupt or timeout expires.
 * Returns the number of transferred bytes or error (<0)
 */
static int iic_wait_for_tc(struct ibm_iic_private* dev){
	
	volatile struct iic_regs __iomem *iic = dev->vaddr;
	int ret = 0;
	
	if (dev->irq >= 0){
		/* Interrupt mode */
		ret = wait_event_interruptible_timeout(dev->wq, 
			!(in_8(&iic->sts) & STS_PT), dev->adap.timeout * HZ);

		if (unlikely(ret < 0))
			DBG("%d: wait interrupted\n", dev->idx);
		else if (unlikely(in_8(&iic->sts) & STS_PT)){
			DBG("%d: wait timeout\n", dev->idx);
			ret = -ETIMEDOUT;
		}
	}
	else {
		/* Polling mode */
		unsigned long x = jiffies + dev->adap.timeout * HZ;
		
		while (in_8(&iic->sts) & STS_PT){
			if (unlikely(time_after(jiffies, x))){
				DBG("%d: poll timeout\n", dev->idx);
				ret = -ETIMEDOUT;
				break;
			}
		
			if (unlikely(signal_pending(current))){
				DBG("%d: poll interrupted\n", dev->idx);
				ret = -ERESTARTSYS;
				break;
			}
			schedule();
		}	
	}
	
	if (unlikely(ret < 0))
		iic_abort_xfer(dev);
	else
		ret = iic_xfer_result(dev);
	
	DBG2("%d: iic_wait_for_tc -> %d\n", dev->idx, ret);
	
	return ret;
}

/*
 * Low level master transfer routine
 */
static int iic_xfer_bytes(struct ibm_iic_private* dev, struct i2c_msg* pm, 
			  int combined_xfer)
{
	volatile struct iic_regs __iomem *iic = dev->vaddr;
	char* buf = pm->buf;
	int i, j, loops, ret = 0;
	int len = pm->len;

	u8 cntl = (in_8(&iic->cntl) & CNTL_AMD) | CNTL_PT;
	if (pm->flags & I2C_M_RD)
		cntl |= CNTL_RW;
	
	loops = (len + 3) / 4;
	for (i = 0; i < loops; ++i, len -= 4){
		int count = len > 4 ? 4 : len;
		u8 cmd = cntl | ((count - 1) << CNTL_TCT_SHIFT);
		
		if (!(cntl & CNTL_RW))
			for (j = 0; j < count; ++j)
				out_8((void __iomem *)&iic->mdbuf, *buf++);
		
		if (i < loops - 1)
			cmd |= CNTL_CHT;
		else if (combined_xfer)
			cmd |= CNTL_RPST;
		
		DBG2("%d: xfer_bytes, %d, CNTL = 0x%02x\n", dev->idx, count, cmd);
		
		/* Start transfer */
		out_8(&iic->cntl, cmd);
		
		/* Wait for completion */
		ret = iic_wait_for_tc(dev);

		if (unlikely(ret < 0))
			break;
		else if (unlikely(ret != count)){
			DBG("%d: xfer_bytes, requested %d, transfered %d\n", 
				dev->idx, count, ret);
			
			/* If it's not a last part of xfer, abort it */
			if (combined_xfer || (i < loops - 1))
    				iic_abort_xfer(dev);
				
			ret = -EREMOTEIO;
			break;				
		}
		
		if (cntl & CNTL_RW)
			for (j = 0; j < count; ++j)
				*buf++ = in_8((void __iomem *)&iic->mdbuf);
	}
	
	return ret > 0 ? 0 : ret;
}

/*
 * Set target slave address for master transfer
 */
static inline void iic_address(struct ibm_iic_private* dev, struct i2c_msg* msg)
{
	volatile struct iic_regs __iomem *iic = dev->vaddr;
	u16 addr = msg->addr;
	
	DBG2("%d: iic_address, 0x%03x (%d-bit)\n", dev->idx, 
		addr, msg->flags & I2C_M_TEN ? 10 : 7);
	
	if (msg->flags & I2C_M_TEN){
	    out_8(&iic->cntl, CNTL_AMD);
	    out_8(&iic->lmadr, addr);
	    out_8(&iic->hmadr, 0xf0 | ((addr >> 7) & 0x06));
	}
	else {
	    out_8(&iic->cntl, 0);
	    out_8(&iic->lmadr, addr << 1);
	}
}

static inline int iic_invalid_address(const struct i2c_msg* p)
{
	return (p->addr > 0x3ff) || (!(p->flags & I2C_M_TEN) && (p->addr > 0x7f));
}

static inline int iic_address_neq(const struct i2c_msg* p1, 
				  const struct i2c_msg* p2)
{
	return (p1->addr != p2->addr) 
		|| ((p1->flags & I2C_M_TEN) != (p2->flags & I2C_M_TEN));
} 

/*
 * Generic master transfer entrypoint. 
 * Returns the number of processed messages or error (<0)
 */
static int iic_xfer(struct i2c_adapter *adap, struct i2c_msg *msgs, int num)
{
    	struct ibm_iic_private* dev = (struct ibm_iic_private*)(i2c_get_adapdata(adap));
	volatile struct iic_regs __iomem *iic = dev->vaddr;
	int i, ret = 0;
	u8 extsts;
	
	DBG2("%d: iic_xfer, %d msg(s)\n", dev->idx, num);
	
	if (!num)
		return 0;
	
	/* Check the sanity of the passed messages.
	 * Uhh, generic i2c layer is more suitable place for such code...
	 */
	if (unlikely(iic_invalid_address(&msgs[0]))){
		DBG("%d: invalid address 0x%03x (%d-bit)\n", dev->idx, 
			msgs[0].addr, msgs[0].flags & I2C_M_TEN ? 10 : 7);
		return -EINVAL;
	}		
	for (i = 0; i < num; ++i){
		if (unlikely(msgs[i].len <= 0)){
			if (num == 1 && !msgs[0].len){
				/* Special case for I2C_SMBUS_QUICK emulation.
				 * IBM IIC doesn't support 0-length transactions
				 * so we have to emulate them using bit-banging.
				 */
				return iic_smbus_quick(dev, &msgs[0]);
			}
			DBG("%d: invalid len %d in msg[%d]\n", dev->idx, 
				msgs[i].len, i);
			return -EINVAL;
		}
		if (unlikely(iic_address_neq(&msgs[0], &msgs[i]))){
			DBG("%d: invalid addr in msg[%d]\n", dev->idx, i);
			return -EINVAL;
		}
	}
	
	/* Check bus state */
	if (unlikely(((extsts = in_8(&iic->extsts)) & EXTSTS_BCS_MASK) != EXTSTS_BCS_FREE)){
		int retries = 0;
		DBG("%d: iic_xfer, bus is not free\n", dev->idx);
		
		/* There might be another operation in progress,
		 * (slave operation). Wait for it to end.
		 */
		do {
			udelay(BUS_WAIT_TIME);
			retries++;
			extsts = in_8(&iic->extsts);
		} while (((extsts & EXTSTS_BCS_MASK) != EXTSTS_BCS_FREE) && retries < BUS_WAIT_RETRIES);

		/* If the bus is still not free, reset the i2c interface */
		if ((extsts & EXTSTS_BCS_MASK) != EXTSTS_BCS_FREE) {
			printk(KERN_WARNING "%d: iic_xfer, bus is still not free, resetting\n", dev->idx);
			iic_dev_reset(dev);
		}
		
		if ((in_8(&iic->extsts) & EXTSTS_BCS_MASK) != EXTSTS_BCS_FREE){
			printk(KERN_WARNING "%d: iic_xfer, bus is still not free\n", dev->idx);
			return -EREMOTEIO;
		}
	} 
	else {
		/* Flush master data buffer (just in case) */
		out_8(&iic->mdcntl, in_8(&iic->mdcntl) | MDCNTL_FMDB);
	}
	
	/* Load slave address */
	iic_address(dev, &msgs[0]);
	
	/* Do real transfer */
    	for (i = 0; i < num && !ret; ++i)
		ret = iic_xfer_bytes(dev, &msgs[i], i < num - 1);

	return ret < 0 ? ret : num;
}

static int iic_algo_control(struct i2c_adapter *adap, 
			    unsigned int cmd, unsigned long arg)
{
	switch (cmd) {
	  case I2C_SLAVE_SET_ADDRESS:
		iic_slave_set_address(adap, (unsigned int) arg);
		break;
	  case I2C_SLAVE_FLUSH_BUFFER:
		iic_slave_restart_xfer(adap);
		break;
	  default:
		return -1;
	}
	return 0;
}

static u32 iic_func(struct i2c_adapter *adap)
{
	return I2C_FUNC_I2C | I2C_FUNC_SMBUS_EMUL | I2C_FUNC_10BIT_ADDR;
}

static struct i2c_algorithm iic_algo = {
	.name 		= "IBM IIC algorithm",
	.id   		= I2C_ALGO_OCP,
	.master_xfer 	= iic_xfer,
	.algo_control	= iic_algo_control,
	.functionality	= iic_func
};

/*
 * Calculates IICx_CLCKDIV value for a specific OPB clock frequency
 */
static inline u8 iic_clckdiv(unsigned int opb)
{
	/* Compatibility kludge, should go away after all cards
	 * are fixed to fill correct value for opbfreq.
	 * Previous driver version used hardcoded divider value 4,
	 * it corresponds to OPB frequency from the range (40, 50] MHz
	 */
	if (!opb){
		printk(KERN_WARNING "ibm-iic: using compatibility value for OPB freq,"
			" fix your board specific setup\n");
		opb = 50000000;
	}

	/* Convert to MHz */
	opb /= 1000000;
	
	if (opb < 20 || opb > 150){
		printk(KERN_CRIT "ibm-iic: invalid OPB clock frequency %u MHz\n",
			opb);
		opb = opb < 20 ? 20 : 150;
	}
	return (u8)((opb + 9) / 10 - 1);
}

/*
 * Register single IIC interface
 */
static int __devinit iic_probe(struct ocp_device *ocp){

	struct ibm_iic_private* dev;
	struct i2c_adapter* adap;
	struct ocp_func_iic_data* iic_data = ocp->def->additions;
	int ret;
	
	if (!iic_data)
		printk(KERN_WARNING"ibm-iic%d: missing additional data!\n",
			ocp->def->index);

	if (!(dev = kmalloc(sizeof(*dev), GFP_KERNEL))){
		printk(KERN_CRIT "ibm-iic%d: failed to allocate device data\n",
			ocp->def->index);
		return -ENOMEM;
	}

	memset(dev, 0, sizeof(*dev));
	dev->idx = ocp->def->index;
	ocp_set_drvdata(ocp, dev);
	
	if (!(dev->vaddr = ioremap(ocp->def->paddr, sizeof(struct iic_regs)))){
		printk(KERN_CRIT "ibm-iic%d: failed to ioremap device registers\n",
			dev->idx);
		ret = -ENXIO;
		goto fail2;
	}
	
	init_waitqueue_head(&dev->wq);

	dev->irq = iic_force_poll ? -1 : ocp->def->irq;
	if (dev->irq >= 0){
		/* Disable interrupts until we finish initialization,
		   assumes level-sensitive IRQ setup...
		 */
		iic_interrupt_mode(dev, 0);
		if (request_irq(dev->irq, iic_handler, 0, "IBM IIC", dev)){
			printk(KERN_ERR "ibm-iic%d: request_irq %d failed\n", 
				dev->idx, dev->irq);
			/* Fallback to the polling mode */	
			dev->irq = -1;
		}
	}
	
	if (dev->irq < 0)
		printk(KERN_WARNING "ibm-iic%d: using polling mode\n", 
			dev->idx);
		
	/* Board specific settings */
	dev->fast_mode = iic_force_fast ? 1 : (iic_data ? iic_data->fast_mode : 0);
	
	/* clckdiv is the same for *all* IIC interfaces, 
	 * but I'd rather make a copy than introduce another global. --ebs
	 */
	dev->clckdiv = iic_clckdiv(ocp_sys_info.opb_bus_freq);
	DBG("%d: clckdiv = %d\n", dev->idx, dev->clckdiv);
	
	/* Reset and reinitialize IIC interface */
	iic_dev_reset(dev);
	
	/* Register it with i2c layer */
	adap = &dev->adap;
	strcpy(adap->name, "IBM_IIC I2C 0");
	i2c_set_adapdata(adap, dev);
	adap->id = I2C_HW_OCP | iic_algo.id;
	adap->algo = &iic_algo;
	adap->client_register = NULL;
	adap->client_unregister = NULL;
	adap->timeout = BUS_TIMEOUT;
	adap->retries = 1;

	if ((ret = i2c_add_adapter(adap)) != 0){
		printk(KERN_CRIT "ibm-iic%d: failed to register i2c adapter\n",
			dev->idx);
		goto fail;
	}

	/* register slave mode */
	if (iic_slave_init_adapter(adap) != 0) {
		iic_slave_cleanup_adapter();
		goto fail;
	}
	
	printk(KERN_INFO "ibm-iic%d: using %s mode\n", dev->idx,
		dev->fast_mode ? "fast (400 kHz)" : "standard (100 kHz)");

	return 0;

fail:	
	if (dev->irq >= 0){
		iic_interrupt_mode(dev, 0);
		free_irq(dev->irq, dev);
	}	

	iounmap(dev->vaddr);
fail2:	
	ocp_set_drvdata(ocp, NULL);
	kfree(dev);	
	return ret;
}

/*
 * Cleanup initialized IIC interface
 */
static void __devexit iic_remove(struct ocp_device *ocp)
{
	struct ibm_iic_private* dev = (struct ibm_iic_private*)ocp_get_drvdata(ocp);
	BUG_ON(dev == NULL);
	if (i2c_del_adapter(&dev->adap)){
		printk(KERN_CRIT "ibm-iic%d: failed to delete i2c adapter :(\n",
			dev->idx);
		/* That's *very* bad, just shutdown IRQ ... */
		if (dev->irq >= 0){
		    iic_interrupt_mode(dev, 0);	
		    free_irq(dev->irq, dev);
		    dev->irq = -1;
		}
	} else {
		if (dev->irq >= 0){
		    iic_interrupt_mode(dev, 0);	
		    free_irq(dev->irq, dev);
		}
		iounmap(dev->vaddr);
		kfree(dev);
	}
}

static struct ocp_device_id ibm_iic_ids[] __devinitdata = 
{
	{ .vendor = OCP_VENDOR_IBM, .function = OCP_FUNC_IIC },
	{ .vendor = OCP_VENDOR_INVALID }
};

MODULE_DEVICE_TABLE(ocp, ibm_iic_ids);

static struct ocp_driver ibm_iic_driver =
{
	.name 		= "iic",
	.id_table	= ibm_iic_ids,
	.probe		= iic_probe,
	.remove		= __devexit_p(iic_remove),
#if defined(CONFIG_PM)
	.suspend	= NULL,
	.resume		= NULL,
#endif
};

static int __init iic_init(void)
{
	int ret;
	printk(KERN_INFO "IBM IIC driver v" DRIVER_VERSION "\n");
	if ((ret = ocp_register_driver(&ibm_iic_driver)) == 0) {
		/* Initialize slave mode */
		iic_slave_init();
	}
	return ret;
}

static void __exit iic_exit(void)
{
	ocp_unregister_driver(&ibm_iic_driver);
}

/*****************************************************************************
 * slave mode functions
 ****************************************************************************/

#define SLAVE_BUF_SIZE 512

/* slave lock object */
static spinlock_t      slave_lock = SPIN_LOCK_UNLOCKED;

/* slave write buffer */
static unsigned char * slave_write_ptr;
static unsigned int    slave_write_count;
static unsigned char   slave_write_buffer[SLAVE_BUF_SIZE];

/* slave read buffer */
static unsigned char * slave_read_ptr;
static unsigned int    slave_read_count;
static unsigned char   slave_read_buffer[SLAVE_BUF_SIZE];

/* status flags */
static int block_next_slave_operation = 0;
static int slave_write_was_blocked = 0;
static int blocked_write_msg_complete = 0;
static int slave_read_was_blocked = 0;
static int blocked_read_msg_read_cmd = 0;

/* function prototypes */
static unsigned char iic_handle_slave_write(struct ibm_iic_private* dev,
                                            unsigned char slave_status);
static int iic_handle_slave_write_complete(struct ibm_iic_private* dev);
static unsigned char iic_handle_slave_read(struct ibm_iic_private* dev,
                                           unsigned char slave_status,
                                           int read_command);

/* Initialize slave buffers */
static void iic_slave_init()
{
	unsigned long flags;

        spin_lock_irqsave(&slave_lock, flags);
        slave_write_ptr = slave_write_buffer;
        slave_write_count = 0;
        slave_read_ptr = slave_read_buffer;
        slave_read_count = 0;
        spin_unlock_irqrestore(&slave_lock, flags);
}

struct i2c_slave_adapter slave_adapt;

/* register adapter in slave layer */
static int iic_slave_init_adapter(struct i2c_adapter *adap)
{
	struct ibm_iic_private* dev = (struct ibm_iic_private*)(i2c_get_adapdata(adap));
	int ret = 0;

	slave_adapt.adapter = adap;
	slave_adapt.reactivate = iic_slave_restart_xfer;
	if ((ret = i2c_slave_add_adapter(&slave_adapt)) != 0){
		printk(KERN_CRIT "ibm-iic%d: failed to register i2c slave adapter\n",
			dev->idx);
		return ret;
	}
	
	return 0;
}

/* deregister */
static int iic_slave_cleanup_adapter()
{
	struct ibm_iic_private* dev;
	struct i2c_adapter *adap;
	int ret;
	
	adap = slave_adapt.adapter;
	dev = (struct ibm_iic_private*)(i2c_get_adapdata(adap));
	
	if ((ret = i2c_slave_remove_adapter(&slave_adapt)) != 0){
		printk(KERN_CRIT "ibm-iic%d: failed to deregister i2c slave adapter\n",
			dev->idx);
		return ret;
	}
	
	return 0;
}

/* set own slave address */
static int iic_slave_set_address(struct i2c_adapter *adap, unsigned int address)
{
	struct ibm_iic_private* dev = (struct ibm_iic_private*)(i2c_get_adapdata(adap));
	volatile struct iic_regs *iic = dev->vaddr;
	unsigned char reg;

	reg = in_8(&iic->mdcntl);
	out_8(&iic->mdcntl, reg & ~MDCNTL_ESM);
	out_8(&iic->lsadr, (address & 0x7f) << 1);
	out_8(&iic->hsadr, 0);
	out_8(&iic->mdcntl, reg | (MDCNTL_FSDB | MDCNTL_ESM));
	DBG ("%d: set i2c slave address to 0x%x\n", dev->idx, address);
	reg = in_8(&iic->lsadr);
	if (((reg >> 1) & 0x7f) != address) {
		printk(KERN_WARNING "ibm-iic%d: could not set i2c slave address, result is 0x%02x",
			dev->idx, reg);
	}
	DBG ("%d: verify slave address: 0x%02x\n", dev->idx, reg);
	
	return 0;
}

/* after the bus is gone down, reactivate transfer */
static int iic_slave_restart_xfer(struct i2c_adapter* adapter)
{
    	struct ibm_iic_private* dev = (struct ibm_iic_private*)(i2c_get_adapdata(adapter));
	volatile struct iic_regs *iic = dev->vaddr;	
	unsigned char slave_status;
	unsigned long flags;
	
        spin_lock_irqsave(&slave_lock, flags);
        
        /* the next message shall be blocked, but no blocked message read yet
           --> just deliver the last message */
        if (block_next_slave_operation && !slave_write_was_blocked && !slave_read_was_blocked) {
        	block_next_slave_operation = 0;
		if (iic_handle_slave_write_complete(dev) < 0) {
			DBG("%d: cannot put data to i2c-slave layer, next operation will be blocked.\n",
				dev->idx);
			block_next_slave_operation = 1;
			
			goto bail;
		}
        }
        else {
        	block_next_slave_operation = 0;
        }
        
        if (slave_write_was_blocked) {
        	/* deliver the last message */
		if (iic_handle_slave_write_complete(dev) < 0) {
			DBG("%d: cannot put data to i2c-slave layer, next operation will be blocked.\n",
				dev->idx);
			block_next_slave_operation = 1;
			
			goto bail;
		}
		
		/* read the next message, if the flush of the first message succeeded */
		slave_status = in_8(&iic->xtcntlss);
		
		slave_status = iic_handle_slave_write(dev, slave_status);
		
		/* deliver the next message if complete */
		if (blocked_write_msg_complete) {
			if (iic_handle_slave_write_complete(dev) < 0) {
				DBG("%d: cannot put data to i2c-slave layer, next operation will be blocked.\n",
					dev->idx);
				block_next_slave_operation = 1;
			}
		}
		
		DUMP_REGS("iic_slave_restart_xfer", dev);
		
		out_8(&iic->xtcntlss, slave_status);
        }
        
        if (slave_read_was_blocked) {
		slave_status = in_8(&iic->xtcntlss);
		
		/* don't handle the read request here, no master will read data */
		/* slave_status = iic_handle_slave_read(dev, slave_status, blocked_read_msg_read_cmd); */

		out_8(&iic->xtcntlss, slave_status);
        }

bail:
        spin_unlock_irqrestore(&slave_lock, flags);
        
        return 0;
}

/* handle i2c slave write requests */
static unsigned char iic_handle_slave_write(struct ibm_iic_private* dev,
                                            unsigned char slave_status)
{
	volatile struct iic_regs *iic = dev->vaddr;	

	/* write bytes received to buffer */
	while (slave_status & XTCNTLSS_SBDD) {
		slave_write_count++;

		*slave_write_ptr = in_8((volatile u8 *)&iic->sbbuf);

		DBG2("%d: iic_handle_slave_write: putting byte 0x%02x (slave_status=0x%02x)\n",
			dev->idx, *slave_write_ptr, slave_status);

		/* avoid buffer overrun here */
		if (slave_write_ptr < slave_write_buffer + SLAVE_BUF_SIZE) {
			slave_write_ptr++;
		}

		slave_status = in_8(&iic->xtcntlss);
	}

	return slave_status;
}

/* if a slave write is complete, push the data up
   to the upper i2c-slave level */
static int iic_handle_slave_write_complete(struct ibm_iic_private* dev)
{
	int ret = 0;
	struct i2c_adapter *adap = &dev->adap;
	volatile struct iic_regs *iic = dev->vaddr;	
	
	if (slave_write_count <= 0) {
		return slave_write_count;
	}
	
	/* copy the written data to the upper I2C slave layer */
	int written_bytes = iic_slave_write(adap, ((in_8(&iic->lsadr)) >> 1) & 0xff, slave_write_buffer, slave_write_count);
	if (written_bytes < 0) {
		printk(KERN_WARNING "ibm-iic%d: iic_handle_slave_write_complete: error exporting data to i2c-slave layer: %d!\n",
			dev->idx, written_bytes);
		ret = -ENOMEM;
	}
	else if (written_bytes == 0) {
		DBG2("%d: iic_handle_slave_write_complete: 0 bytes to i2c-slave layer exported!\n",
			dev->idx);
	}
	else if (written_bytes >= slave_write_count) {
		DBG2("%d: iic_handle_slave_write_complete: moved %d bytes to i2c-slave layer.\n",
			dev->idx, written_bytes);
		slave_write_ptr = slave_write_buffer;
		slave_write_count = 0;
	}
	else {
		// only a partial copy
		DBG2("%d: iic_handle_slave_write_complete: moved %d bytes to i2c-slave layer, %d bytes requested.\n",
			dev->idx, written_bytes, slave_write_count);
		slave_write_ptr -= written_bytes;
		slave_write_count -= written_bytes;
	}
	
	return ret;
}

/* handle first slave read interrupt of an i2c message
   fetches data from upper i2c slave level */
static void iic_handle_slave_first_read(struct ibm_iic_private* dev,
                                        int command)
{
	struct i2c_adapter *adap = &dev->adap;
	volatile struct iic_regs *iic = dev->vaddr;	

	DBG2("%d: iic_handle_slave_first_read: Fetching slave read data from i2c-slave layer.\n", dev->idx);
	
        slave_read_ptr = slave_read_buffer;
        slave_read_count = iic_slave_read(adap, ((in_8(&iic->lsadr)) >> 1) & 0xff, slave_read_buffer, SLAVE_BUF_SIZE, command);
        
        if (slave_read_count > 0) {
        	DBG2("%d: iic_handle_slave_first_read: Got %d bytes.\n", dev->idx, slave_read_count);
        }
        else {
        	DBG("%d: iic_handle_slave_first_read: cannot read bytes.\n", dev->idx);
        }
}

/* handle an i2c slave read request */
static unsigned char iic_handle_slave_read(struct ibm_iic_private* dev,
                                           unsigned char slave_status,
                                           int read_command)
{
	volatile struct iic_regs *iic = dev->vaddr;	

	int command_read = 0;
	int command = 0;
	unsigned char data = 0;
	
	while (read_command && (slave_status & XTCNTLSS_SBDD)) {
		command_read = 1;
		command <<= 8;
		command |= in_8((volatile u8 *)&iic->sbbuf) & 0xff;
		DBG2("%d: iic_handle_slave_read: read command 0x%02x\n",
			dev->idx, command);
		slave_status = in_8(&iic->xtcntlss);
	}
	
	if (!command_read) {
		command = -1;
	}
	
	if (slave_read_count <= 0) {
		iic_handle_slave_first_read(dev, command);
	}
	
	if (slave_read_count > 0) {
		/* avoid buffer overrun here */
		if (slave_read_ptr < slave_read_buffer + slave_read_count) {
			data = *slave_read_ptr;
			slave_read_ptr++;
		}
		else {
			/* TODO: error handline here */
			data = 0;
			DBG("%d: iic_handle_slave_read: no data!\n", dev->idx);
		}
		
		DBG2("%d: iic_handle_slave_read: sending byte 0x%02x\n", dev->idx, data);
		
		out_8((volatile u8 *)&iic->sbbuf, data);
	}
	else {
		/* TODO: signal that we are not ready by not sending data */
		out_8((volatile u8 *)&iic->sbbuf, 0);
		DBG2("%d: iic_handle_slave_read: not ready, sending no byte\n", dev->idx);
	}
	
	slave_status = in_8(&iic->xtcntlss);

	return slave_status;
}

/* slave read completed */
static void iic_handle_slave_read_complete(struct ibm_iic_private* dev)
{
	if (slave_read_ptr < slave_read_buffer + slave_read_count) {
		DBG2("%d: iic_handle_slave_read_complete: not all data was read! amount: %d, read: %d\n",
			dev->idx, slave_read_count, slave_read_ptr - slave_read_buffer);
	}
        else {
		DBG2("%d: iic_handle_slave_read_complete: read %d bytes\n", dev->idx, slave_read_ptr - slave_read_buffer);
	}
	
	slave_read_ptr = slave_read_buffer;
	slave_read_count = 0;
}

/* handle iic slave interrupt */
static void iic_handle_slave(struct ibm_iic_private* dev,
			     unsigned char status,
			     unsigned char ext_status,
			     unsigned char slave_status,
			     unsigned char xfer_cnt)
{
	volatile struct iic_regs *iic = dev->vaddr;	
	unsigned char count;
	unsigned long flags;
	
	int is_slave_write_request = 0;
	int is_slave_write_complete = 0;
	int is_slave_read_request = 0;
	int is_slave_read_complete = 0;
	int slave_read_get_command = 0;

	spin_lock_irqsave(&slave_lock, flags);

	count = xfer_cnt;
	count = count >> 4;
	DBG2("%d: iic_handle_slave: slave count=0x%02x\n", dev->idx, count);
	
	/* find out which slave operation is to be performed */
	
	if (slave_status & XTCNTLSS_SWS) {
		is_slave_write_request = 1;
	}
	if (slave_status & XTCNTLSS_SWC) {
#if 1
		/* I don't know how to savely detect a slave read operation,
		   so we don't use it. So we assume that a slave write indeed
		   is a slave write. */
		is_slave_write_complete = 1;
#else
		if (!(slave_status & XTCNTLSS_SDBF) && (ext_status & EXTSTS_BCS_MASK) == EXTSTS_BCS_SLAVESEL) {
#warning This works, but I do not think it is supposed to work this way!
			/* This is a very ugly hack, but is the only way
			 * to determine a slave read operation with an
			 * address byte before the read request. The
			 * driver doesn't get a "slave read needs service"
			 * interrupt in that case. Instead, a "slave write
			 * complete" interrupt occurs (with a bus in slave
			 * selected state according to EXTSTS).
			 * So if the slave buffer is not full (meaning no
			 * write transfer pending) and the bus is in
			 * slave selected state, we assume a slave read
			 * operation here.
			 */
			is_slave_read_request = 1;
			slave_read_get_command = 1;
		}
		else {
			is_slave_write_complete = 1;
		}
#endif
	}
	if (slave_status & XTCNTLSS_SRS) {
		is_slave_read_request = 1;
	}
	if (slave_status & XTCNTLSS_SRC) {
		is_slave_read_complete = 1;
	}
	
	/* reset the flags */
	slave_write_was_blocked = 0;
	blocked_write_msg_complete = 0;
	slave_read_was_blocked = 0;
	blocked_read_msg_read_cmd = 0;
	
	/* performe the requested operation */
	
	if (block_next_slave_operation) {
		/* we block the slave operation by not reading data
		   and not resetting the slave status */
		if (is_slave_write_request) {
			slave_write_was_blocked = 1;
		}
		
		if (is_slave_write_complete) {
			slave_write_was_blocked = 1;
			blocked_write_msg_complete = 1;
		}
		
		if (is_slave_read_request) {
			slave_read_was_blocked = 1;
			blocked_read_msg_read_cmd = slave_read_get_command;
		}

		slave_status &= ~(XTCNTLSS_SWC | XTCNTLSS_SWS | XTCNTLSS_SRC | XTCNTLSS_SRS);
	}
	else {
		if (is_slave_write_request || is_slave_write_complete) {
			slave_status = iic_handle_slave_write(dev, slave_status);
			if (slave_status & XTCNTLSS_SWC) {
				if (iic_handle_slave_write_complete(dev) < 0) {
					DBG("%d: cannot put data to i2c-slave layer, next operation will be blocked.\n",
						dev->idx);
					block_next_slave_operation = 1;
				}
			}
		}
		
		if (is_slave_read_request) {
			slave_status = iic_handle_slave_read(dev, slave_status, slave_read_get_command);
		}
		
		if (slave_status & XTCNTLSS_SRC) {
			iic_handle_slave_read_complete(dev);
		}
	}

	out_8(&iic->xtcntlss, slave_status);

	DBG2("%d: iic_handle_slave: done (slave_status=0x%02x)\n", dev->idx, slave_status);

	spin_unlock_irqrestore(&slave_lock, flags);
}

module_init(iic_init);
module_exit(iic_exit);
