/*
 *  linux/drivers/char/x1226-rtc.c
 *
 *  I2C Real Time Clock Client Driver for Xicor X1226 RTC/Calendar
 *
 *  Copyright 2002 MontaVista Software Inc.
 *  Author: MontaVista Software, Inc.
 *     	stevel@mvista.com or source@mvista.com
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  TODO:
 *    - implement alarm and periodic IRQ support.
 *
 */
#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/miscdevice.h>
#include <linux/fcntl.h>
#include <linux/poll.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/i2c.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/rtc.h>
#include <linux/proc_fs.h>
#include <linux/spinlock.h>
#include <linux/time.h>
#include <asm/uaccess.h>
#include <asm/system.h>

#undef DEBUG_X1226

#ifdef DEBUG_X1226
#define	dbg(fmt, args...) printk(KERN_DEBUG "%s: " fmt, __func__, ## args)
#else
#define	dbg(fmt, args...)
#endif

#define X1226_MODULE_NAME "X1226"
#define PFX X1226_MODULE_NAME
#ifndef I2C_DRIVERID_X1226
#define I2C_DRIVERID_X1226  I2C_DRIVERID_EXP0
#endif

#define err(format, arg...) printk(KERN_ERR PFX ": " format , ## arg)
#define info(format, arg...) printk(KERN_INFO PFX ": " format , ## arg)
#define warn(format, arg...) printk(KERN_WARNING PFX ": " format , ## arg)
#define emerg(format, arg...) printk(KERN_EMERG PFX ": " format , ## arg)

#define EPOCH 2000
#define SYS_EPOCH 1900

#undef BCD_TO_BIN
#define BCD_TO_BIN(val) (((val)&15) + ((val)>>4)*10)

#undef BIN_TO_BCD
#define BIN_TO_BCD(val) ((((val)/10)<<4) + (val)%10)

#define X1226_RTC_SR      0x3f
#define   RTC_SR_RTCF  (1)
#define   RTC_SR_WEL  (1<<1)
#define   RTC_SR_RWEL (1<<2)

#define X1226_RTC_BASE    0x30

/* This is an image of the RTC registers starting at offset 0x30 */
struct rtc_registers {
	unsigned char secs;  // 30
        unsigned char mins;  // 31
        unsigned char hours; // 32
        unsigned char day;   // 33
        unsigned char mon;   // 34
        unsigned char year;  // 35
        unsigned char dayofweek; // 36
        unsigned char epoch; // 37
};

#define X1226_CONTROL_DTR  0x13
#define X1226_CONTROL_ATR  0x12
#define X1226_CONTROL_INT  0x11
#define X1226_CONTROL_BL   0x10

#define DEVID_RTC          0x6F
#define DEVID_NVRAM        0x57
#define   SLAVE_READ 0x01
#ifndef I2C_M_WR
#define I2C_M_WR 0x0
#endif
#define ABITS           9
#define EESIZE          (1 << ABITS)    /* size in bytes */

#define NVSIZE          512             /* we use 512 bytes */
#define NVOFFSET        (EESIZE-NVSIZE) /* at end of EEROM */

static struct i2c_driver x1226_driver;

static int x1226_use_count = 0;

static struct i2c_client *this_client = NULL;

static int rtc_read_proc(char *page, char **start, off_t off,
                         int count, int *eof, void *data);


static int x1226_read (struct i2c_client *client,
		       u8 reg_offset, u8* buf, int len)
{
	int ret;
	u8 regbuf[2] = { 0, reg_offset };
	struct i2c_msg random_addr_read[2] = {
		{
			/* "Set Current Address" */
			client->addr,
			client->flags | I2C_M_WR,
			sizeof(regbuf),
			regbuf
		},
		{
			/* "Sequential Read" if len>1,
			   "Current Address Read" if len=1 */
			client->addr,
			client->flags | I2C_M_RD,
			len,
			buf
		}
	};

	if ((ret = i2c_transfer(client->adapter, random_addr_read, 2)) != 2) {
		ret = -ENXIO;
		dbg("i2c_transfer failed\n");
	}

	return ret;
}

static int x1226_write (struct i2c_client *client,
			u8 reg_offset, u8* buf, int len)
{
	int ret;
	u8* local_buf;
	u8 regbuf[2] = { 0, reg_offset };
	struct i2c_msg page_write = {
		client->addr,
		client->flags,
		len + sizeof(regbuf),
		NULL
	};

	if ((local_buf = (u8*)kmalloc(len + sizeof(regbuf),
				      GFP_KERNEL)) == NULL) {
		err("buffer alloc failed\n");
		return -ENOMEM;
	}

	memcpy(local_buf, regbuf, sizeof(regbuf));
	memcpy(local_buf + sizeof(regbuf), buf, len);
	page_write.buf = local_buf;
	
	if ((ret = i2c_transfer(client->adapter, &page_write, 1)) != 1) {
		ret = -ENXIO;
		dbg("i2c_transfer failed\n");
	}

	kfree(local_buf);
	return ret;
}


static int ccr_write_enable(struct i2c_client *client)
{
	u8 sr = RTC_SR_WEL;
	int ret;
	
	if ((ret = x1226_write(client, X1226_RTC_SR, &sr, 1)) < 0)
		return ret;
	sr |= RTC_SR_RWEL;
	if ((ret = x1226_write(client, X1226_RTC_SR, &sr, 1)) < 0)
		return ret;
	sr = 0;
	if ((ret = x1226_read(client, X1226_RTC_SR, &sr, 1)) < 0)
		return ret;

	sr &= (RTC_SR_RWEL | RTC_SR_RWEL);
	if (sr != (RTC_SR_RWEL | RTC_SR_RWEL)) {
		dbg("verify SR failed\n");
		return -ENXIO;
	}

	return 0;
}

static int ccr_write_disable(struct i2c_client *client)
{
	int ret;
	u8 sr = 0;
	
	if ((ret = x1226_write(client, X1226_RTC_SR, &sr, 1)) < 0)
		return ret;
	if ((ret = x1226_read(client, X1226_RTC_SR, &sr, 1)) < 0)
		return ret;
	if (sr != 0) {
		dbg("verify SR failed\n");
		return -ENXIO;
	}

	return 0;
}


static int
x1226_get_time(struct i2c_client *client, struct rtc_time *tm)
{	
	struct rtc_registers rtc;
	u32 epoch;
	int ret;
	
	/* read RTC registers */
	if ((ret = x1226_read(client, X1226_RTC_BASE, (u8*)&rtc,
			      sizeof(struct rtc_registers))) < 0) {
		dbg("couldn't read RTC\n");
		return ret;
	}

	dbg("IN: epoch=%02x, year=%02x, mon=%02x, day=%02x, hour=%02x, "
	    "min=%02x, sec=%02x\n",
	    rtc.epoch, rtc.year, rtc.mon, rtc.day, rtc.hours,
	    rtc.mins, rtc.secs);
	
	epoch  = 100 * BCD_TO_BIN(rtc.epoch);   // 19 / 20
	tm->tm_year = BCD_TO_BIN(rtc.year); // 0 - 99
	tm->tm_year += (epoch - SYS_EPOCH);
	tm->tm_mon = BCD_TO_BIN(rtc.mon);   // 1 - 12
	tm->tm_mon--;                       /* tm_mon is 0 to 11 */
	tm->tm_mday = BCD_TO_BIN(rtc.day);  // 1 - 31
	tm->tm_hour = BCD_TO_BIN(rtc.hours & ~0x80);
	if (!(rtc.hours & 0x80)) {
		// AM/PM 1-12 format, convert to MIL
		tm->tm_hour--; // 0 - 11
		if (rtc.hours & (1<<5))
			tm->tm_hour += 12; // PM
	}
	
	tm->tm_min = BCD_TO_BIN(rtc.mins);
	tm->tm_sec = BCD_TO_BIN(rtc.secs);

	dbg("OUT: year=%d, mon=%d, day=%d, hour=%d, min=%d, sec=%d\n",
	    tm->tm_year, tm->tm_mon, tm->tm_mday, tm->tm_hour,
	    tm->tm_min, tm->tm_sec);
	
	return 0;
}

static int 
x1226_set_time(struct i2c_client *client, const struct rtc_time *tm)
{
	struct rtc_registers rtc;
	int ret;

	dbg("IN: year=%d, mon=%d, day=%d, hour=%d, min=%d, sec=%d\n",
	    tm->tm_year, tm->tm_mon, tm->tm_mday, tm->tm_hour,
	    tm->tm_min, tm->tm_sec);

	rtc.epoch = BIN_TO_BCD(EPOCH/100);
	rtc.year  = BIN_TO_BCD(tm->tm_year + SYS_EPOCH - EPOCH);
	rtc.mon   = BIN_TO_BCD(tm->tm_mon + 1); /* tm_mon is 0 to 11 */
	rtc.day   = BIN_TO_BCD(tm->tm_mday);
	rtc.dayofweek = 0; // ignore day of week
	rtc.hours = BIN_TO_BCD(tm->tm_hour) | 0x80; /* 24 hour format */
	rtc.mins  = BIN_TO_BCD(tm->tm_min);
	rtc.secs  = BIN_TO_BCD(tm->tm_sec);
	
	dbg("OUT: epoch=%02x, year=%02x, mon=%02x, day=%02x, hour=%02x, "
	    "min=%02x, sec=%02x\n",
	    rtc.epoch, rtc.year, rtc.mon, rtc.day, rtc.hours,
	    rtc.mins, rtc.secs);
	
	/* write RTC registers */
	if ((ret = ccr_write_enable(client)) < 0)
		return ret;
	if ((ret = x1226_write(client, X1226_RTC_BASE, (u8*)&rtc,
			       sizeof(struct rtc_registers))) < 0) {
		dbg("couldn't write RTC\n");
		return ret;
	}
	ccr_write_disable(client);

	return 0;
}


static int
x1226_probe(struct i2c_adapter *adap)
{
	int ret;
        char stat;
	unsigned char crs[4];
	struct rtc_time dummy_tm={0,0,0, 1, 0, 100, 0, 0};
	
	if (this_client != NULL)
		return -EBUSY;
	
	this_client = kmalloc(sizeof(*this_client), GFP_KERNEL);
	if (this_client == NULL) {
		return -ENOMEM;
	}
	memset(this_client, 0, sizeof(*this_client));

	strlcpy(this_client->name, X1226_MODULE_NAME, I2C_NAME_SIZE);
	i2c_set_clientdata(this_client, NULL);
        this_client->flags      = 0;
	this_client->addr       = DEVID_RTC;
	this_client->adapter    = adap;
	this_client->driver     = &x1226_driver;

	/*
	 * use x1226_get_time() to probe for an X1226 on this bus.
	 */
	if((ret = x1226_read(this_client,X1226_RTC_SR, &stat, 1))< 0){
		kfree(this_client);
		this_client = NULL;
		return ret;
	}

	info("found X1226 on %s\n", adap->name);

        if(stat & RTC_SR_RTCF){
	  printk("X1226:Timer Not Initialized after power fail. Setting 2002/1/1/0:00\n" );
	  ret= x1226_set_time(this_client, &dummy_tm);
	}
	dbg("stat %x \n", stat);
	x1226_read(this_client,0x10, &crs[0], 1);
	x1226_read(this_client,0x11, &crs[1], 1);
	x1226_read(this_client,0x12, &crs[2], 1);
	x1226_read(this_client,0x13, &crs[3], 1);
	dbg("CTLREG:%d %x %x %x %x\n", ret, crs[0],crs[1],crs[2],crs[3]);

	if ((ret = x1226_get_time(this_client, &dummy_tm)) < 0) {
	  printk("Fetch Timer Failed\n");
	}else{
#ifdef DEBUG_X1226
	  struct rtc_time *tm = &dummy_tm;
	  dbg("OUT: year=%d, mon=%d, day=%d, hour=%d, min=%d, sec=%d\n",
	    tm->tm_year, tm->tm_mon, tm->tm_mday, tm->tm_hour,
	    tm->tm_min, tm->tm_sec);
#endif
	}

	/* attach it. */
	return i2c_attach_client(this_client);
}

static int
x1226_detach(struct i2c_client *client)
{
	i2c_detach_client(client);

	if (this_client != NULL) {
		kfree(this_client);
		this_client = NULL;
	}

	return 0;
}

int rtc_open(struct inode *minode, struct file *mfile)
{
	/*if(MOD_IN_USE)*/
	if(x1226_use_count > 0) {
		return -EBUSY;
	}
	++x1226_use_count;
	return 0;
}

int rtc_release(struct inode *minode, struct file *mfile)
{
	--x1226_use_count;
	return 0;
}

static loff_t rtc_llseek(struct file *mfile, loff_t offset, int origint)
{
	return -ESPIPE;
}

static int
x1226_command(struct i2c_client *client, unsigned int cmd, void *arg)
{
	return -EINVAL;
}

static int rtc_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
		     unsigned long arg)
{
	struct rtc_time rtc_tm;
	int ret;
	
	switch (cmd) {
	case RTC_RD_TIME:	/* Read the time/date from RTC  */
		if ((ret = x1226_get_time(this_client, &rtc_tm)) < 0)
			return ret;
		return copy_to_user((void *)arg, &rtc_tm, sizeof(rtc_tm)) ? 
			-EFAULT : 0;
	case RTC_SET_TIME:	/* Set the RTC */
		if (!capable(CAP_SYS_TIME))
			return -EACCES;

		if (copy_from_user(&rtc_tm, 
				   (struct rtc_time *) arg,
		                   sizeof(struct rtc_time))) 
			return -EFAULT;

		return x1226_set_time(this_client, &rtc_tm);
	default:
		return -EINVAL;
	}
}


static struct i2c_driver x1226_driver = {
	.owner		= THIS_MODULE,
	.name		= X1226_MODULE_NAME,
	.id		= I2C_DRIVERID_X1226,
	.flags		= I2C_DF_NOTIFY,
	.attach_adapter	= x1226_probe,
	.detach_client	= x1226_detach,
	.command	= x1226_command
};

static struct file_operations rtc_fops = {
	.owner		= THIS_MODULE,
	.llseek		= rtc_llseek,
	.ioctl		= rtc_ioctl,
	.open		= rtc_open,
	.release	= rtc_release,
};

static struct miscdevice x1226rtc_miscdev = {
	RTC_MINOR,
	"rtc",
	&rtc_fops
};

static __init int x1226_init(void)
{
	int ret;

	info("I2C based RTC driver.\n");
	ret = i2c_add_driver(&x1226_driver);
	if (ret) {
		err("Register I2C driver failed, errno is %d\n", ret);
		return ret;
	}
	ret = misc_register(&x1226rtc_miscdev);
	if (ret) {
		err("Register misc driver failed, errno is %d\n", ret);
		ret = i2c_del_driver(&x1226_driver);
		if (ret) {
			err("Unregister I2C driver failed, errno is %d\n",
			    ret);
		}
		return ret;
	}

	create_proc_read_entry("driver/rtc", 0, 0, rtc_read_proc, NULL);

	return 0;
}

static void __exit x1226_exit(void)
{
        remove_proc_entry("driver/rtc", NULL);
        misc_deregister(&x1226rtc_miscdev);
	i2c_del_driver(&x1226_driver);
}


module_init(x1226_init);
module_exit(x1226_exit);

/*
 *	Info exported via "/proc/driver/rtc".
 */

static int rtc_proc_output(char *buf)
{
	char *p;
	struct rtc_time tm;
	int ret;
	
	if ((ret = x1226_get_time(this_client, &tm)) < 0)
		return ret;

	p = buf;

	/*
	 * There is no way to tell if the luser has the RTC set for local
	 * time or for Universal Standard Time (GMT). Probably local though.
	 */
	p += sprintf(p,
		     "rtc_time\t: %02d:%02d:%02d\n"
		     "rtc_date\t: %04d-%02d-%02d\n"
		     "rtc_epoch\t: %04d\n",
		     tm.tm_hour, tm.tm_min, tm.tm_sec,
		     tm.tm_year + SYS_EPOCH, tm.tm_mon + 1,
		     tm.tm_mday, EPOCH);

	return p - buf;
}

static int rtc_read_proc(char *page, char **start, off_t off,
			 int count, int *eof, void *data)
{
	int len = rtc_proc_output(page);
	if (len <= off + count)
		*eof = 1;
	*start = page + off;
	len -= off;
	if (len > count)
		len = count;
	if (len < 0)
		len = 0;
	return len;
}

MODULE_AUTHOR("Steve Longerbeam");
MODULE_LICENSE("GPL");

