#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/i2c.h>
#include <linux/i2c-slave.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/config.h>
#include <asm/string.h>
#include <asm/io.h>

/* ------------------------------------------------------------------------- *
 * Module debugging and output stuff
 * ------------------------------------------------------------------------- */

#define DBG_LEVEL 0

#if DBG_LEVEL > 0
#  define DBG(fmt, x...)	printk(KERN_DEBUG "i2c-slave: " fmt, ##x)
#else
#  define DBG(fmt, x...)	((void)0)
#endif
#if DBG_LEVEL > 1
#  define DBG2(fmt, x...) 	DBG( fmt, ##x )
#else
#  define DBG2(fmt, x...) 	((void)0)
#endif
#if DBG_LEVEL > 2
static void
dump_iic_buf(const char* string, const char* buf, int len)
{
	int i;
	printk(KERN_DEBUG "%s: Bytes:\n", string);
	printk(KERN_DEBUG);
	for (i = 0; i < len; i++) {
		printk("0x%02x ", buf[i] & 0xff);
	}
	printk("\n");
}
#  define DUMP_BUF(s, buf,len)	{ printk(KERN_DEBUG); dump_iic_buf((s),(buf),(len)); }
#else
#  define DUMP_BUF(s, buf,len)	((void)0)
#endif

#define WARN(fmt, x...)		printk(KERN_WARNING "i2c-slave: " fmt, ##x)

/* ------------------------------------------------------------------------- *
 * Global constants and variables
 * ------------------------------------------------------------------------- */

/* the I2C adapters list */
static LIST_HEAD(slave_adapters);

/* the I2C devices list */
static LIST_HEAD(slave_devices);

/* spinlocks for access to the slave adapters and slave devices list */
static spinlock_t slave_adapters_lock = SPIN_LOCK_UNLOCKED;
static spinlock_t slave_devices_lock = SPIN_LOCK_UNLOCKED;

/* ------------------------------------------------------------------------- *
 * Initialization/Cleanup
 * ------------------------------------------------------------------------- */

#ifdef MODULE
static __init int
i2c_slave_init(void)
{
	DBG("module initialization\n");
	
	/* nothing to do at the moment */
	DBG("module initialization finished\n");
	return 0;
}

static __exit void
i2c_slave_cleanup(void)
{
	DBG("module cleanup\n");
	
	/* nothing to do at the moment */
	
	DBG("module cleanup finished\n");
}
#endif /* MODULE */

/* ------------------------------------------------------------------------- *
 * Slave adapter and slave device managing functions
 * ------------------------------------------------------------------------- */

int
i2c_slave_add_adapter(struct i2c_slave_adapter* slave_adapter)
{
	unsigned long flags;

	spin_lock_irqsave(&slave_adapters_lock, flags);
	list_add_tail(&slave_adapter->list, &slave_adapters);
	DBG("added slave adapter %s\n", slave_adapter->adapter->name);
	spin_unlock_irqrestore(&slave_adapters_lock, flags);
	return 0;
}

int
i2c_slave_remove_adapter(struct i2c_slave_adapter* slave_adapter)
{
	struct i2c_slave_adapter *slave_adapter_from_list;
	unsigned long flags;
	int removed = 0;

	spin_lock_irqsave(&slave_adapters_lock, flags);
	/* FIXME: cleanup necessary? */
	list_for_each_entry(slave_adapter_from_list, &slave_adapters, list) {
		if (slave_adapter_from_list == slave_adapter) {
			list_del(&slave_adapter->list);
			removed = 1;
			break;
		}
	}
	if (removed) DBG("removed slave adapter %s\n", slave_adapter->adapter->name);
	spin_unlock_irqrestore(&slave_adapters_lock, flags);
	return 0;
}

int
i2c_slave_add_device(struct i2c_slave_device* slave_device)
{
	unsigned long flags;

	spin_lock_irqsave(&slave_devices_lock, flags);
	list_add_tail(&slave_device->list, &slave_devices);
	DBG("added slave device %s\n", slave_device->name);
	spin_unlock_irqrestore(&slave_devices_lock, flags);
	return 0;
}

int
i2c_slave_remove_device(struct i2c_slave_device* slave_device)
{
	struct i2c_slave_device *slave_device_from_list;
	unsigned long flags;
	int removed = 0;

	spin_lock_irqsave(&slave_devices_lock, flags);
	/* FIXME: cleanup necessary? */
	list_for_each_entry(slave_device_from_list, &slave_devices, list) {
		if (slave_device_from_list == slave_device) {
			list_del(&slave_device->list);
			removed = 1;
			break;
		}
	}
	if (removed) DBG("removed slave device %s\n", slave_device->name);
	spin_unlock_irqrestore(&slave_devices_lock, flags);
	return 0;
}

static struct i2c_slave_adapter *
i2c_slave_get_from_adap(struct i2c_adapter* adapter)
{
	struct i2c_slave_adapter *slave_adapter_from_list;
	struct i2c_slave_adapter *ret = NULL;
	unsigned long flags;

	spin_lock_irqsave(&slave_adapters_lock, flags);

	list_for_each_entry(slave_adapter_from_list, &slave_adapters, list) {
		if (slave_adapter_from_list->adapter == adapter) {
			ret = slave_adapter_from_list;
			break;
		}
	}

	if (ret == NULL) {
		WARN("i2c_slave_get_from_adap adapter [%s] not found.\n", adapter->name);
	}

	spin_unlock_irqrestore(&slave_adapters_lock, flags);
	return ret;
}

/* ------------------------------------------------------------------------- *
 * Unblock functions
 * ------------------------------------------------------------------------- */

void
unblock_adapter(struct i2c_adapter* adapter)
{
	struct i2c_slave_adapter* slave_adapter_from_list;
	unsigned long flags;
	int found = 0;
	
	spin_lock_irqsave(&slave_adapters_lock, flags);

	/* search the correct slave adapter */
	list_for_each_entry(slave_adapter_from_list, &slave_adapters, list) {
		if (slave_adapter_from_list->adapter == adapter) {
			found = 1;
			break;
		}
	}
	
	if (found) {
		DBG2("Unblock apapter %s.\n", adapter->name);
		slave_adapter_from_list->reactivate(adapter);
	} else {
		WARN("%s: No adapter to unblock!\n", adapter->name);
	}
	
	spin_unlock_irqrestore(&slave_adapters_lock, flags);
}

/* ------------------------------------------------------------------------- *
 * Read and write functions
 * ------------------------------------------------------------------------- */

static void
iic_slave_search_responsible_device(struct i2c_slave_adapter *slave_adap, struct i2c_adapter* adapter,
				    unsigned int slave_addr, char* buf, int len)
{
	struct i2c_slave_device* slave_device;
	
	slave_adap->responsible_device = NULL;
	
	list_for_each_entry(slave_device, &slave_devices, list) {
		DBG2("Probing device %s\n", slave_device->name);
		if (slave_device->adapter &&
		    slave_device->adapter == adapter &&
		    slave_device->is_responsible &&
		    slave_device->is_responsible(slave_device, slave_addr, buf, len)) {
			slave_adap->responsible_device = slave_device;
			DBG2("Device %s responsible.\n", slave_device->name);
			break;
		}
#if DBG_LEVEL > 2
		else if (slave_device->adapter != adapter) {
			DBG2("Device %s not responsible, adapters do not match.\n", slave_device->name);
		} else {
			DBG2("Device %s not responsible, refused data.\n", slave_device->name);
		}
#endif
	}
}

int
iic_slave_write(struct i2c_adapter* adapter, unsigned int slave_addr, char* buf, int len)
{
	int ret = 0;
	unsigned long flags;
	struct i2c_slave_adapter *slave_adap = i2c_slave_get_from_adap(adapter);

	if (slave_adap == NULL) {
		/* we fake the return value for the driver,
		   it must not assume that an error occured */
		return len;
	}
	
	DBG2("iic_slave_write: writing %d bytes.\n", len);

	spin_lock_irqsave(&slave_devices_lock, flags);
	
	/* find the device responsible for the current data transfer */
	iic_slave_search_responsible_device(slave_adap, adapter, slave_addr, buf, len);
	
	if (slave_adap->responsible_device && slave_adap->responsible_device->slave_write) {
		ret = slave_adap->responsible_device->slave_write(slave_adap->responsible_device, buf, len);
		if (ret < 0) {
			DBG("iic_slave_write: error writing bytes.\n");
		}
		else {
			DBG2("iic_slave_write: %d bytes written.\n", len);
		}
	}
	else {
		DBG("iic_slave_write: no responsible device, discarding data!\n");
		DUMP_BUF("iic_slave_write", buf, len);
		
		/* we fake the return value for the driver,
		   it must not assume that an error occured */
		ret = len;
	}
			
	spin_unlock_irqrestore(&slave_devices_lock, flags);
	
	return ret;
}

int
iic_slave_read(struct i2c_adapter* adapter, unsigned int slave_addr, char* buf, int len, int command)
{
	int i = 0;
	int ret = 0;
	static int start = 0;
	struct i2c_slave_adapter *slave_adap = i2c_slave_get_from_adap(adapter);
	
	if (slave_adap == NULL) {
		/* we fake the return value for the driver,
		   it must not assume that an error occured */
		return len;
	}
	
	DBG2("iic_slave_read: %d reading bytes.\n", len);
	
	/* find the device responsible for the current data transfer */
	if (command >= 0) {
		unsigned char dev_buf = command & 0xff;
		iic_slave_search_responsible_device(slave_adap, adapter, slave_addr, &dev_buf, 1);
	}
	
	if (slave_adap->responsible_device && slave_adap->responsible_device->slave_read) {
		ret = slave_adap->responsible_device->slave_read(slave_adap->responsible_device, buf, len);
		if (ret < 0) {
			DBG("iic_slave_read: error reading bytes.\n");
		}
		else {
			DBG2("iic_slave_read: %d bytes read.\n", ret);
		}
	}
	else {
		DBG("iic_slave_read: no responsible device!\n");
		
		ret = len;
		
		/* write dummy bytes */
		for (i = 0; i < len; i++) {
			buf[i] = i + start;
		}
		
		DUMP_BUF("iic_slave_read", buf, len);
			
		start++;
	}

	return ret;
}

/* ------------------------------------------------------------------------- *
 * Linux kernel module stuff
 * ------------------------------------------------------------------------- */

#ifdef MODULE

MODULE_AUTHOR("thre@peppercon.de");
MODULE_DESCRIPTION("Linux kernel module for I2C slave operations");

module_init(i2c_slave_init);
module_exit(i2c_slave_cleanup);

#endif	/* MODULE */

EXPORT_SYMBOL(i2c_slave_add_adapter);
EXPORT_SYMBOL(i2c_slave_remove_adapter);
EXPORT_SYMBOL(i2c_slave_add_device);
EXPORT_SYMBOL(i2c_slave_remove_device);

EXPORT_SYMBOL(iic_slave_write);
EXPORT_SYMBOL(iic_slave_read);

EXPORT_SYMBOL(unblock_adapter);
