/*
 * MTD chip driver for SPI flash devices connected to KIRA100
 *
 * Author: Danny Baumann <danny.baumann@raritan.com>
 *
 * parts of this driver were written by Rama Bisa <RamaB@ami.com>
 *
 */

#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/init.h>
#include <linux/pci.h>
#include <linux/mtd/map.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/flashchip.h>
#include "spi_flash.h"
#include <asm/arch/apb_dma.h>
#include <asm/arch/platform/spec.h>
#include <linux/dma-mapping.h>
#include <linux/ioport.h>
#include <linux/wait.h>

/* There's no limit. It exists only to avoid realloc. */
#define MAX_SPI_CHIPS 1

#define DMA_BUF_SIZE		5100
#define SPI_READ_FRAGMENT_SIZE	4096

/* our private structure definitions */

struct spi_flash_private {
    /* MTD related members */
    int interleave;
    int numchips;
    unsigned long chipshift;
    struct flchip chips[1];

    /* Hardware related members */
    wait_queue_head_t wait;
    volatile u32 *ssp_regs;
    apb_dma_channel_data_t *rx_dma_data;
    apb_dma_channel_data_t *tx_dma_data;
    void *tx_dma_buf_virt;
    dma_addr_t tx_dma_buf_phys;
    void *rx_dma_buf_virt;
    dma_addr_t rx_dma_buf_phys;
    int trans_done;
    struct semaphore spi_txrx_lock;
};

struct spi_flash_info {
    const __u16 mfr_id;
    const __u16 dev_id;
    const char *name;
    const u_long size;
    const int numeraseregions;
    const struct mtd_erase_region_info regions[1];
};

/* function prototypes */

static int spi_init(struct spi_flash_private *dev);
static void spi_cleanup(struct spi_flash_private *dev);
static int spi_txrx (struct spi_flash_private *dev, uint8_t *inbuf, uint8_t *outbuf, int count);
static void dma_isr(void *isr_data);

static int spi_read(struct spi_flash_private *dev, loff_t adr, size_t len, u_char *buf);
static int spi_write(struct spi_flash_private *dev, loff_t adr, size_t len, const u_char *buf);
static int spi_sector_erase (struct spi_flash_private *private, unsigned long adr);
static int spi_read_id(struct spi_flash_private *dev);
static int spi_send_write_enable_command (struct spi_flash_private *private);
static uint8_t spi_read_status (struct spi_flash_private *private);

static int spi_write_bytes(struct map_info *map, struct flchip *chip,unsigned long adr, size_t len, const __u8* buf);
static int spi_flash_write(struct mtd_info *, loff_t, size_t, size_t *, const u_char *);
static int spi_flash_read(struct mtd_info *, loff_t, size_t, size_t *, u_char *);
static int spi_flash_erase(struct mtd_info *, struct erase_info *);
static void spi_flash_sync(struct mtd_info *);
static int spi_flash_suspend(struct mtd_info *);
static void spi_flash_resume(struct mtd_info *);
static void spi_flash_destroy(struct mtd_info *);
static int spi_flash_do_unlock(struct mtd_info *mtd, loff_t ofs, size_t len, int is_unlock);
static struct mtd_info *spi_flash_probe(struct map_info *map);


/* local variables */

static struct mtd_chip_driver spi_flash_chipdrv = {
    .probe = spi_flash_probe,
    .destroy = spi_flash_destroy,
    .name = "spi_probe",
    .module = THIS_MODULE
};

static const char im_name[] = "spi flash";
//the input and output buffers
static u8 out_buf[DMA_BUF_SIZE];
static u8 in_buf[DMA_BUF_SIZE];
static u8 verify_buf[DMA_BUF_SIZE];

static int initialized = 0;


/* module stuff */

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Danny Baumann <danny.baumann@raritan.com>");
MODULE_DESCRIPTION("MTD chip driver for SPI flash devices connected to KIRA100");

int __init spi_flash_init(void)
{
    register_mtd_chip_driver(&spi_flash_chipdrv);
    return 0;
}

void __exit spi_flash_exit(void)
{
    unregister_mtd_chip_driver(&spi_flash_chipdrv);
}

module_init(spi_flash_init);
module_exit(spi_flash_exit);


/* low-level routines */

static int spi_init(struct spi_flash_private *dev)
{
    int ret = 0;
    unsigned int clkdiv;

    printk (KERN_INFO "spi_init\n");	

    if (initialized) {
	printk (KERN_ERR "SPI module was already initialized\n");
	return -1;
    }

    initialized = 1;

    dev->ssp_regs = (volatile u32*)(CPE_SSP_VA_BASE);

    if (apb_dma_request_channel(NULL, NULL, SSP_TX_DMA_CHANNEL, &dev->tx_dma_data) < 0) {
	printk (KERN_ERR "Failed to request SSP TX DMA channel.\n");
	ret = -EIO;
	goto fail;
    }
    dev->tx_dma_data->hw_handshake_num = 0x6;

    if (apb_dma_request_channel(dma_isr, dev, SSP_RX_DMA_CHANNEL, &dev->rx_dma_data) < 0) {
	printk (KERN_ERR "Failed to request SSP TX DMA channel.\n");
	ret = -EIO;
	goto fail;
    }
    dev->rx_dma_data->hw_handshake_num = 0x7;

    init_waitqueue_head(&dev->wait);
    init_MUTEX (&dev->spi_txrx_lock);

    /* allocate the read and write buffers for the DMA */
    dev->tx_dma_buf_virt = pci_alloc_consistent(NULL, DMA_BUF_SIZE, &dev->tx_dma_buf_phys);
    if (dev->tx_dma_buf_virt == NULL) {
	printk (KERN_ERR "Unable to allocate DMA TX buffer.\n");
	ret = -ENOMEM;
	goto fail;
    }
    dev->rx_dma_buf_virt = pci_alloc_consistent(NULL, DMA_BUF_SIZE, &dev->rx_dma_buf_phys);
    if (NULL == dev->rx_dma_buf_virt) {
	printk (KERN_ERR "Unable to allocate DMA RX buffer.\n");
	ret = -ENOMEM;
	goto fail;
    }

    /* initialize the SSP controller */
    dev->ssp_regs[SSPCR0] = (SSP_FRFORMAT_SPI << SSPCR0_FRFORMAT_SHIFT) | (SSP_OPM_MASTERMONO << SSPCR0_OPM_SHIFT);
    clkdiv = 0x0;
    dev->ssp_regs[SSPCR1] = (7 << SSPCR1_SDL_SHIFT) | clkdiv;
    /* reset SSP core */
    dev->ssp_regs[SSPCR2] =  SSPCR2_SSPRST | SSPCR2_TXFCLR | SSPCR2_RXFCLR;
    /* enable SSP core */
    dev->ssp_regs[SSPCR2] =  SSPCR2_SSPEN;
    dev->ssp_regs[SSPICR]  = (SSP_DMA_TX_THRESHOLD << SSPICR_TFTHOD_SHIFT) | 
	(SSP_DMA_RX_THRESHOLD << SSPICR_RFTHOD_SHIFT) | SSPICR_TFDMAEN | SSPICR_RFDMAEN;

    printk(KERN_INFO "SPI module successfully loaded. (io: 0x%p)\n", dev->ssp_regs);

    return 0;

fail:
    spi_cleanup (dev);
    return ret;

}

static void spi_cleanup(struct spi_flash_private *dev)
{
    if (dev->tx_dma_data)
	apb_dma_release_channel(&dev->tx_dma_data);

    if (dev->rx_dma_data)
	apb_dma_release_channel(&dev->rx_dma_data);
}

static int spi_txrx (struct spi_flash_private *dev, uint8_t *inbuf, uint8_t *outbuf, int count)
{
    int numretries = 3;
    apb_dma_parm_t parm;

    down (&dev->spi_txrx_lock);

    if ((count > DMA_BUF_SIZE) || (inbuf == NULL)) {
	printk(KERN_ERR "SPI transaction length > DMA_BUF_SIZE\n");
	up (&dev->spi_txrx_lock);
	return -1;
    }

    memcpy(dev->tx_dma_buf_virt, inbuf, count);
    consistent_sync(dev->tx_dma_buf_virt, count, DMA_TO_DEVICE);

    while (numretries > 0) {
	numretries--;
	/* enable SSP core clear tx fifo and rx fifo */
	dev->ssp_regs[SSPCR2] =  SSPCR2_SSPEN | SSPCR2_TXFCLR | SSPCR2_RXFCLR;

	/* program APB DMA TX channel */
	parm.src = dev->tx_dma_buf_phys;
	parm.dest = CPE_SSP_BASE + (SSPDR << 2);
	parm.width = APBDMA_WIDTH_8BIT;
	parm.sctl = APBDMA_CTL_INC1;
	parm.dctl = APBDMA_CTL_FIX;
	parm.stype = APBDMA_TYPE_AHB;
	parm.dtype = APBDMA_TYPE_APB;
	parm.burst = 0;
	parm.size = count;
	parm.irq = 0;
	apb_dma_channel_add(dev->tx_dma_data, &parm);

	/* program APB DMA RX channel */
	parm.src = CPE_SSP_BASE + (SSPDR << 2);
	parm.dest = dev->rx_dma_buf_phys;
	parm.width = APBDMA_WIDTH_8BIT;
	parm.sctl = APBDMA_CTL_FIX;
	parm.dctl = APBDMA_CTL_INC1;
	parm.stype = APBDMA_TYPE_APB;
	parm.dtype = APBDMA_TYPE_AHB;
	parm.burst = 0;
	parm.size = count;
	parm.irq = 1;
	apb_dma_channel_add(dev->rx_dma_data, &parm);

	dev->trans_done = 0;	
	/* start APB DMA channels and wait for finish */
	apb_dma_channel_start(dev->tx_dma_data);
	apb_dma_channel_start(dev->rx_dma_data);

	wait_event_timeout(dev->wait, dev->trans_done != 0,  1*HZ);
	if (dev->trans_done != 0) { 
	    consistent_sync(dev->rx_dma_buf_virt, count, DMA_FROM_DEVICE);
	    memcpy(outbuf, dev->rx_dma_buf_virt, count);
	    up (&dev->spi_txrx_lock);
	    return 0;	
	}
    }

    printk (KERN_ERR "Failed to read from SPI flash.\n");
    up (&dev->spi_txrx_lock);
    return -1;
}

/* ------------------------------------------------------------------------- *
 * interrupt service routine
 * ------------------------------------------------------------------------- */

static void dma_isr(void *isr_data)
{
    struct spi_flash_private *private = (struct spi_flash_private *)isr_data;
    if (private == NULL)
	return;

    private->trans_done = 1;
    wake_up(&private->wait);
}

/* routines which implement the SPI flash protocol */

static int spi_wait_for_ready(struct spi_flash_private *dev)
{
    int retries = MAX_RETRIES;
    uint8_t status;

    while (retries > 0) {
	status = spi_read_status (dev); 
	if (status == 0) break;
	retries--;
    }

    return (retries == 0) ? -1 : 0;
}

static inline int read_one_chip(struct map_info *map, struct flchip *chip,
	loff_t adr, size_t len, u_char *buf)
{
    uint32_t bytes;
    unsigned int i = 0;
    struct spi_flash_private *dev = map->fldrv_priv;

    chip->state = FL_READY;

    while (len > 0) {
	bytes = (len > SPI_READ_FRAGMENT_SIZE) ? SPI_READ_FRAGMENT_SIZE : len;
	if (spi_read(dev, adr, bytes, &buf[i]) != 0) {
	    printk (KERN_ERR "spi_read failed in read_one_chip function\n");
	    return -1;
	}
	i += bytes;
	len -= bytes;
	adr += bytes;
    }

    return 0;

}

static int spi_read(struct spi_flash_private *dev, loff_t adr, size_t len, u_char *buf)
{

    int index, count;

    if (spi_wait_for_ready(dev) < 0) {
        printk (KERN_ERR "Timed out waiting for flash free before spi read initiated\n");
        return -1; 
    }

    index = 0;
    out_buf [index++] = OPCODE_READ;
    out_buf [index++] = adr >> 16;
    out_buf [index++] = adr >> 8;
    out_buf [index++] = adr;
    count = index + len;

    if (spi_txrx(dev, out_buf, in_buf, count) == 0) {
	memcpy (buf, &in_buf[index], len);
	return 0;
    } else {
	printk (KERN_ERR "spi_read failed\n");
	return -1;
    }
}

static int spi_write(struct spi_flash_private *dev, loff_t adr, size_t len, const u_char *buf)
{
    int index, count;

    if (spi_wait_for_ready(dev) < 0) {
        printk (KERN_ERR "Timed out waiting for flash free before spi write initiated\n");
        return -1; 
    }

    /* Send Write Enable Command */
    if (spi_send_write_enable_command(dev) != 0) {
	printk (KERN_DEBUG "Write Enable Command failed before spi sector erase initiated\n");
	/* sending write disable command is a good idea here? */
	return -1;
    }

    index = 0;
    out_buf [index++] = OPCODE_PP;
    out_buf [index++] = adr >> 16;
    out_buf [index++] = adr >> 8;
    out_buf [index++] = adr;

    memcpy (&out_buf[index], buf, len);
    memcpy (verify_buf, buf, len);
    count = index + len;
    /*  Write a page */
    if (spi_txrx (dev, out_buf, in_buf, count) == 0) {
	if (spi_wait_for_ready(dev) < 0) {
	    printk (KERN_ERR "Timed out waiting for flash free after spi write initiated\n");
	    return -1; 
	}
	/* page has been written & Read back and verify */
	index = 0;
	out_buf [index++] = OPCODE_READ;
	out_buf [index++] = adr >> 16;
	out_buf [index++] = adr >> 8;
	out_buf [index++] = adr;
	count  = index + len;
	if (spi_txrx(dev, out_buf, in_buf, count) == 0) {
	    if (memcmp(verify_buf, &in_buf[index], len) == 0) {
		return 0;
	    }
	} 

    }
    printk (KERN_ERR "spi_write failed\n");
    return -1;

}

static int spi_sector_erase (struct spi_flash_private *private, unsigned long adr)
{
    int index;

    if (spi_wait_for_ready(private) < 0) {
        printk(KERN_ERR "Timed out waiting for flash free before spi sector erase initiated\n");
        return -1; 
    }

    if (spi_send_write_enable_command(private) != 0) {
	printk(KERN_DEBUG "Write Enable Command failed before spi sector erase initiated\n");
	/* sending write disable command is a good idea here? */
	return -1;
    }

    index = 0;
    out_buf [index++] = OPCODE_SE;
    out_buf [index++] = adr >> 16;
    out_buf [index++] = adr >> 8;
    out_buf [index++] = adr;

    if (spi_txrx(private, out_buf, in_buf, index) == 0) {
	if (spi_wait_for_ready(private) < 0) {
	    printk(KERN_ERR "Timed out waiting for flash free after spi sector erase initiated\n");
	    return -1; 
	}
	return 0;
    } else {
	printk(KERN_ERR "spi_erase_sector failed\n");
	return -1;
    }

}

static int spi_read_id(struct spi_flash_private *dev)
{
    int index;
    int id;

    index = 0;
    out_buf [index++] = OPCODE_RDID;
    out_buf [index++] = 0x00;
    out_buf [index++] = 0x00;
    out_buf [index++] = 0x00;

    if (spi_txrx(dev, out_buf, in_buf, index) == 0) {
	id = ((in_buf[1] << 16) | (in_buf [2] << 8) | in_buf[3]);
	return id;
    } else {
	printk(KERN_ERR "spi_read_id failed\n");
	return -1;
    }	
}

static int spi_send_write_enable_command (struct spi_flash_private *private)
{
    int index;

    index = 0;	
    out_buf[index++] = OPCODE_WREN;
    if (spi_txrx(private, out_buf, in_buf, index) == 0) {
	return 0;
    } else {
	printk(KERN_ERR "Write Enable Command Failed\n");
	return -1;
    }
}

static uint8_t spi_read_status (struct spi_flash_private *private)
{
    int index = 0;

    out_buf[index++] = OPCODE_RDSR;

    if (spi_txrx(private, out_buf, in_buf, index+1) == 0) {
	return in_buf[index];
    } else {
	printk(KERN_DEBUG "reading spi status failed\n");
	return -1;
    }
}

/* MTD routines */

static int spi_write_bytes(struct map_info *map, struct flchip *chip,
	unsigned long adr, size_t len, const __u8* buf)
{
    struct spi_flash_private *private = map->fldrv_priv;
    uint32_t page_size, page_offset;
    uint32_t i;

    chip->state = FL_WRITING;

    /* what page do we start with? */
    page_offset = adr % FLASH_PAGESIZE;

    /* do all the bytes fit onto one page? */
    if (page_offset + len <= FLASH_PAGESIZE) {
	if (spi_write (private, adr, len, buf) != 0) {
	    printk (KERN_ERR "spi writing in the spi_write_bytes function\n");
	    return -1;
	}
    } else {
	/* the size of data remaining on the first page */
	page_size = FLASH_PAGESIZE - page_offset;
	if (spi_write (private, adr, page_size, buf) != 0){
	    printk (KERN_ERR "spi writing in the spi_write_bytes function\n");
	    return -1;
	}
	/* write everything in PAGESIZE chunks */
	for (i = page_size; i < len; i += page_size) {
	    page_size = len - i;
	    if (page_size > FLASH_PAGESIZE)
		page_size = FLASH_PAGESIZE;
	    if (spi_write (private, adr + i, page_size, buf + i) != 0) {
		printk (KERN_ERR "spi writing in the spi_write_bytes function\n");
		return -1;
	    }
	}
    }

    chip->state = FL_READY;

    return 0;

}

static int spi_flash_write(struct mtd_info *mtd, loff_t to , size_t len,
	size_t *retlen, const u_char *buf)
{

    struct map_info *map = mtd->priv;
    struct spi_flash_private *private = map->fldrv_priv;
    int chipnum;

    if (to > mtd->size) {
	printk (KERN_ERR "write address > size of spi flash\n");
	return -EINVAL;
    }

    if ((len + to) > mtd->size) {
	printk (KERN_ERR "write address + size > size of spi flash\n");
	return -EINVAL;
    }

    *retlen = 0;
    if (!len) {
	return 0;
    }

    chipnum = 0;

    if (spi_write_bytes(map, &private->chips[chipnum], to, len, buf) == 0) {
	*retlen = len;
	return 0;
    }

    return -EINVAL;
}

static int spi_flash_read(struct mtd_info *mtd, loff_t from, size_t len, size_t *retlen, u_char *buf)
{
    struct map_info *map = mtd->priv;
    struct spi_flash_private *private = map->fldrv_priv;
    unsigned long ofs;
    int chipnum;
    int ret = 0;

    if ((from + len) > mtd->size) {
	printk(KERN_WARNING "%s: read request past end of device "
		"(0x%lx)\n", map->name, (unsigned long)from + len);
	return -EINVAL;
    }

    chipnum = 0;
    ofs = from;

    *retlen = 0;
    ret = read_one_chip(map, &private->chips[chipnum], ofs, len, buf);
    if (ret == 0) 
	*retlen = len;
     else 
	ret = -EINVAL;

    return ret;
}

static inline int erase_one_block(struct map_info *map, struct flchip *chip,
	unsigned long adr, u_long size)
{
    int retval = 0;
    struct  spi_flash_private *private = map->fldrv_priv;

    chip->state = FL_ERASING;

    retval = spi_sector_erase (private,adr);

    chip->state = FL_READY;

    return (retval);
}


static int spi_flash_erase(struct mtd_info *mtd, struct erase_info *instr)
{
    struct map_info *map = mtd->priv;
    struct spi_flash_private *private = map->fldrv_priv;
    unsigned long adr, len;
    int chipnum;
    int ret = 0;
    int i;
    int first;
    struct mtd_erase_region_info *regions = mtd->eraseregions;

    if (instr->addr > mtd->size) {
	return -EINVAL;
    }

    if ((instr->len + instr->addr) > mtd->size) {
	return -EINVAL;
    }
    /* Check that both start and end of the requested erase are
     * aligned with the erasesize at the appropriate addresses.
     */

    i = 0;

    /* Skip all erase regions which are ended before the start of
       the requested erase. Actually, to save on the calculations,
       we skip to the first erase region which starts after the
       start of the requested erase, and then go back one.
       */

    while ((i < mtd->numeraseregions) &&
	    (instr->addr >= regions[i].offset)) {
	i++;
    }
    i--;
    /* OK, now i is pointing at the erase region in which this
     * erase request starts. Check the start of the requested
     * erase range is aligned with the erase size which is in
     * effect here.
     */

    if (instr->addr & (regions[i].erasesize-1)) {
	return -EINVAL;
    }
    /* Remember the erase region we start on. */

    first = i;

    /* Next, check that the end of the requested erase is aligned
     * with the erase region at that address.
     */

    while ((i < mtd->numeraseregions) &&
	    ((instr->addr + instr->len) >= regions[i].offset)) {
	i++;
    }

    /* As before, drop back one to point at the region in which
     * the address actually falls.
     */

    i--;

    if ((instr->addr + instr->len) & (regions[i].erasesize-1)) {
	return -EINVAL;
    }

    chipnum = 0;
    adr = instr->addr;
    len = instr->len;

    i = first;

    while (len) {
	ret = erase_one_block(map, &private->chips[chipnum], adr,
		regions[i].erasesize);

	if (ret) {
	    return ret;
	}
	adr += regions[i].erasesize;
	len -= regions[i].erasesize;
    }

    instr->state = MTD_ERASE_DONE;
    mtd_erase_callback(instr);

    return 0;
}

static void spi_flash_sync(struct mtd_info *mtd)
{

    //printk("spi_flash_sync(): not implemented!\n");
}

static int spi_flash_suspend(struct mtd_info *mtd)
{
    printk("spi_flash_suspend(): not implemented!\n");
    return -EINVAL;
}

static void spi_flash_resume(struct mtd_info *mtd)
{
    printk("spi_flash_resume(): not implemented!\n");
}

static void spi_flash_destroy(struct mtd_info *mtd)
{
    struct map_info *map = mtd->priv;
    struct spi_flash_private *private = map->fldrv_priv;
    kfree(private);
}

static inline void unlock_sector(struct map_info *map, unsigned long sect_addr, int unlock)
{
    printk (KERN_DEBUG "spi_flash.c: unlock_sector not implemented \n");
}

static inline int is_sector_locked(struct map_info *map, unsigned long sect_addr)
{
    printk (KERN_DEBUG "spi_flash.c: is_sector_locked not implemented \n");
    return 0;
}

static int spi_flash_do_unlock(struct mtd_info *mtd, loff_t ofs, size_t len, int is_unlock)
{
    struct map_info *map;
    struct mtd_erase_region_info *merip;
    int eraseoffset, erasesize, eraseblocks;
    int i;
    int retval = 0;
    int lock_status;

    map = mtd->priv;

    /* Pass the whole chip through sector by sector and check for each
       sector if the sector and the given interval overlap */
    for(i = 0; i < mtd->numeraseregions; i++) {
	merip = &mtd->eraseregions[i];

	eraseoffset = merip->offset;
	erasesize = merip->erasesize;
	eraseblocks = merip->numblocks;

	if (ofs > eraseoffset + erasesize)
	    continue;

	while (eraseblocks > 0) {
	    if (ofs < eraseoffset + erasesize && ofs + len > eraseoffset) {
		unlock_sector(map, eraseoffset, is_unlock);

		lock_status = is_sector_locked(map, eraseoffset);

		if (is_unlock && lock_status) {
		    printk("Cannot unlock sector at address %x length %xx\n",
			    eraseoffset, merip->erasesize);
		    retval = -1;
		} else if (!is_unlock && !lock_status) {
		    printk("Cannot lock sector at address %x length %x\n",
			    eraseoffset, merip->erasesize);
		    retval = -1;
		}
	    }
	    eraseoffset += erasesize;
	    eraseblocks --;
	}
    }
    return retval;
}

static int spi_flash_unlock(struct mtd_info *mtd, loff_t ofs, size_t len)
{
    return spi_flash_do_unlock(mtd, ofs, len, 1);
}

static int spi_flash_lock(struct mtd_info *mtd, loff_t ofs, size_t len)
{
    return spi_flash_do_unlock(mtd, ofs, len, 0);
}


/*
 * Reads JEDEC manufacturer ID and device ID and returns the index of the first
 * matching table entry (-1 if not found or alias for already found chip).
 */
static int probe_new_chip(struct mtd_info *mtd, __u32 base,
	struct flchip *chips,
	struct spi_flash_private *private,
	const struct spi_flash_info *table, int table_size)
{

    __u32 mfr_id;
    __u32 dev_id;
    int   id;
    struct map_info *map = mtd->priv;
    struct spi_flash_private temp;
    int i;

    temp.interleave = 1;
    map->fldrv_priv = &temp;

    spi_init(private);
    id = spi_read_id (private);
    mfr_id = (id >> 16);
    dev_id = id & 0x0000FFFF;

    printk(KERN_INFO "spi_flash: detected manufacturer id = 0x%04x, device id = 0x%04x\n", mfr_id, dev_id);

    for (i = 0; i < table_size; i++) {
	if ((mfr_id == table[i].mfr_id) && (dev_id == table[i].dev_id)) {
	    if (chips) {
		if (private->numchips == MAX_SPI_CHIPS) {
		    printk(KERN_WARNING "%s: Too many flash chips detected. Increase "
			    "MAX_SPI_CHIPS from %d.\n", map->name, MAX_SPI_CHIPS);

		    return -1;
		}

		chips[private->numchips].start = base;
		chips[private->numchips].state = FL_READY;
		chips[private->numchips].mutex =
		    &chips[private->numchips]._spinlock;
		private->numchips++;
	    }

	    printk("%s: Found %d x %ldMiB %s at 0x%x\n", map->name,
		    temp.interleave, (table[i].size)/(1024*1024),
		    table[i].name, base);

	    mtd->size += table[i].size * temp.interleave;
	    mtd->numeraseregions += table[i].numeraseregions;

	    break;
	}
    }

    if (i == table_size) {
	printk(KERN_DEBUG "%s: unknown flash device at 0x%x, "
		"mfr id 0x%x, dev id 0x%x\n", map->name,
		base, mfr_id, dev_id);
	map->fldrv_priv = NULL;

	return -1;
    }

    private->interleave = temp.interleave;
    return i;

}

static struct mtd_info *spi_flash_probe(struct map_info *map)
{
    static const struct spi_flash_info table[] = {
	{
	    .mfr_id = MANUFACTURER_WINBOND,
	    .dev_id = W25P40,
	    .name = "WINBOND W25P40",
	    .size = 0x80000,
	    .numeraseregions = 1,
	    .regions = {
		{ .offset = 0x000000, .erasesize = 0x10000, .numblocks = 8 },
	    }
	},
	{
	    .mfr_id = MANUFACTURER_ST,
	    .dev_id = M25P28,
	    .name = "ST M25P28",
	    .size = 0x1000000,
	    .numeraseregions = 1,
	    .regions = {
		{ .offset = 0x000000, .erasesize = 0x40000, .numblocks = 64 },
	    }
	},
	{
	    .mfr_id = MANUFACTURER_MX,
	    .dev_id = MX25L6405,
	    .name = "Macronix MX25L6405",
	    .size = 0x800000,
	    .numeraseregions = 1,
	    .regions = {
		{ .offset = 0x000000, .erasesize = 0x10000, .numblocks = 128 },
	    }
	}
    };

    struct mtd_info *mtd;
    struct flchip chips[MAX_SPI_CHIPS];
    int table_pos[MAX_SPI_CHIPS];
    struct spi_flash_private temp;
    struct spi_flash_private *private;
    u_long size;

    int i;
    int reg_idx;
    int offset;

    temp.numchips = 1;
    private = kmalloc(sizeof(*private) + (sizeof(struct flchip) *
		temp.numchips), GFP_KERNEL);
    if (!private) {
	printk(KERN_WARNING
		"%s: kmalloc failed for private structure\n", map->name);
	return NULL;
    }

    mtd = (struct mtd_info*)kmalloc(sizeof(*mtd), GFP_KERNEL);
    if (!mtd) {
	printk(KERN_WARNING
		"%s: kmalloc failed for info structure\n", map->name);
	return NULL;
    }
    memset(mtd, 0, sizeof(*mtd));
    mtd->priv = map;

    memset(&temp, 0, sizeof(temp));

    if ((table_pos[0] = probe_new_chip(mtd, 0, NULL, private, table,
		    sizeof(table)/sizeof(table[0])))
	    == -1) {
	printk(KERN_WARNING
		"%s: Found no spi compatible device at location zero\n",
		map->name);
	kfree(mtd);

	return NULL;
    }
    chips[0].start = 0;
    chips[0].state = FL_READY;

    chips[0].mutex = &chips[0]._spinlock;
    temp.numchips = 1;
    private->numchips = 1;
    temp.interleave = 1;
    for (size = mtd->size; size > 1; size >>= 1) {
	temp.chipshift++;
    }
    switch (temp.interleave) {
	case 2:
	    temp.chipshift += 1;
	    break;
	case 4:
	    temp.chipshift += 2;
	    break;
    }

    mtd->eraseregions = kmalloc(sizeof(struct mtd_erase_region_info) *
	    mtd->numeraseregions, GFP_KERNEL);
    if (!mtd->eraseregions) {
	printk(KERN_WARNING "%s: Failed to allocate "
		"memory for MTD erase region info\n", map->name);
	kfree(mtd);
	map->fldrv_priv = NULL;
	return NULL;
    }

    reg_idx = 0;
    offset = 0;
    for (i = 0; i < temp.numchips; i++) {
	int dev_size;
	int j;

	dev_size = 0;
	for (j = 0; j < table[table_pos[i]].numeraseregions; j++) {
	    mtd->eraseregions[reg_idx].offset = offset +
		(table[table_pos[i]].regions[j].offset *
		 temp.interleave);
	    mtd->eraseregions[reg_idx].erasesize =
		table[table_pos[i]].regions[j].erasesize *
		temp.interleave;
	    mtd->eraseregions[reg_idx].numblocks =
		table[table_pos[i]].regions[j].numblocks;
	    if (mtd->erasesize < mtd->eraseregions[reg_idx].erasesize) {
		mtd->erasesize = mtd->eraseregions[reg_idx].erasesize;
	    }
	    dev_size += mtd->eraseregions[reg_idx].erasesize *
		mtd->eraseregions[reg_idx].numblocks;
	    reg_idx++;
	}
	offset += dev_size;
    }

    mtd->type = MTD_NORFLASH;
    mtd->flags = MTD_CAP_NORFLASH;
    mtd->name = map->name;
    mtd->erase = spi_flash_erase;
    mtd->read = spi_flash_read;
    mtd->write = spi_flash_write;
    mtd->sync = spi_flash_sync;
    mtd->suspend = spi_flash_suspend;
    mtd->resume = spi_flash_resume;
    mtd->lock = spi_flash_lock;
    mtd->unlock = spi_flash_unlock;

    memcpy(private->chips, chips,
	    sizeof(struct flchip) * private->numchips);
    for (i = 0; i < private->numchips; i++) {
	init_waitqueue_head(&private->chips[i].wq);
	spin_lock_init(&private->chips[i]._spinlock);
    }

    map->fldrv_priv = private;
    map->bankwidth = 1; 
    map->fldrv = &spi_flash_chipdrv;
    __module_get(THIS_MODULE);

    return mtd;
}


