/*
 *      drivers/net/ftmac100.c
 *
 *      Faraday FTMAC100/110 device driver
 *
 */

#include <linux/types.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/ethtool.h>
#include <linux/mii.h>
#include <linux/pci.h>
#include <linux/delay.h>
#include <linux/workqueue.h>
#include <asm/io.h>
#include <asm/arch/pp_bd.h>
#include "ftmac100.h"

#define DRV_NAME            "FTMAC100"
#define DRV_VERSION         "09/06/06"
#define PHY_ADDR_PROBE      (-1)
#define PHY_PROBE_MIN	    0x00
#define PHY_PROBE_MAX	    0x01

MODULE_AUTHOR("daba@raritan.com and Faraday Corp.")
MODULE_DESCRIPTION("FTMAC100/110 device driver")

#if defined(PP_BOARD_KIRA) && defined(KIRA_KIMG4)
# define SHARE_MDIO
#endif

#define FTMAC100_PHY_TIMEOUT	(HZ/10)
#define FTMAC100_MDIO_DELAY	2

/* Debugging defines */
#define noDEBUG_MDIO
#define noDEBUG_LINK
#define noDEBUG_FUNC

#ifdef DEBUG_MDIO
 #define MDIO_DEBUG(fmt, args...) printk(fmt, ##args)
#else
 #define MDIO_DEBUG(fmt, args...)
#endif

#ifdef DEBUG_LINK
 #define LINK_DEBUG(fmt, args...) printk(fmt, ##args)
#else
 #define LINK_DEBUG(fmt, args...)
#endif

#ifdef DEBUG_FUNC
  #define FUNC_DEBUG(fmt, args...) printk(fmt, ##args)
#else
  #define FUNC_DEBUG(fmt, args...)
#endif

static const char version[] = "Faraday FTMAC100 Driver,(Linux Kernel 2.6) 09/06/06 - by Faraday\n";
static unsigned int trans_busy = 0;
int ignore_phy[2] = { 0, 0 }; /* FIXME: ignoring ignore_phy for now */

struct ftmac_device {
    u32 base_addr;
    u32 irq;
    u32 phy_addr;
    struct net_device *netdev;
} ftmac_devices[] = {
#if defined(SHARE_MDIO)
    {CPE_FTMAC_VA_BASE, IRQ_MAC1, 0x01},
    {CPE_FTMAC2_VA_BASE, IRQ_MAC2, 0x00}};
#else 
    {CPE_FTMAC_VA_BASE, IRQ_MAC1, PHY_ADDR_PROBE},
    {CPE_FTMAC2_VA_BASE, IRQ_MAC2, PHY_ADDR_PROBE}};
#endif

void put_mac(u32 base, u8 *mac_addr)
{
	u32 val;

	val = (mac_addr[0] << 8) | (mac_addr[1]);
	outl(val, base);
	val = (mac_addr[2] << 24) | (mac_addr[3] << 16) |
		  (mac_addr[4] << 8) | (mac_addr[5] << 0);
	outl(val, base+4);
}

void get_mac(u32 base, u8 *mac_addr)
{
	int val;

	FUNC_DEBUG("+get_mac\n");

	val = inl(base);
	mac_addr[0] = (val >> 8) & 0xff;
	mac_addr[1] = val & 0xff;

	val = inl(base+4);
	mac_addr[2] = (val >> 24) & 0xff;
	mac_addr[3] = (val >> 16) & 0xff;
	mac_addr[4] = (val >> 8) & 0xff;
	mac_addr[5] = val & 0xff;
}

void dump(u_char *data, int size, int step)
{
    int i;

    for (i = 0; i < size; i++) {
        if ((i % step) == 0) printk("\n");
	    printk("%02X ", data[i]);
    }
}

// --------------------------------------------------------------------
// 	Print the Ethernet address
// --------------------------------------------------------------------
void print_mac(char *mac_addr)
{
	int i;
	 
	printk("MACADDR: ");
	for (i = 0; i < 5; i++) {
		printk("%2.2x:", mac_addr[i]);
	}
	printk("%2.2x \n", mac_addr[5]);
}

// --------------------------------------------------------------------
// 	Finds the CRC32 of a set of bytes.
//	Again, from Peter Cammaert's code.
// --------------------------------------------------------------------
static int crc32( char * s, int length ) 
{
    /* indices */
    int perByte;
    int perBit;
    /* crc polynomial for Ethernet */
    const unsigned long poly = 0xedb88320;
    /* crc value - preinitialized to all 1's */
    unsigned long crc_value = 0xffffffff;

    FUNC_DEBUG("+crc32\n");

    for (perByte = 0; perByte < length; perByte ++) {
        unsigned char c;
	c = *(s++);
        for ( perBit = 0; perBit < 8; perBit++ ) {
	    crc_value = (crc_value>>1) ^ (((crc_value^c) & 0x01) ? poly : 0);
	    c >>= 1;
	}
    }
    return crc_value;
}

static int ftmac100_reset( struct net_device* dev )
{
    unsigned ioaddr = dev->base_addr;
    int rcount;

    //PRINTK("+ftmac100_reset:I/O addr=%X\n", ioaddr);

    outl( SW_RST_bit, ioaddr + MACCR_REG );

    /* this should pause enough for the chip to be happy */
    for (rcount = 0; (inl( ioaddr + MACCR_REG ) & SW_RST_bit) != 0; rcount++) {
        msleep_interruptible(10);
        if (rcount > 5) // Retry 5 times
            return -ENODEV;
    }

    outl( 0, ioaddr + IMR_REG );    /* Disable all interrupts */
    if (inl(ioaddr+IMR_REG)!=0)
        return -ENODEV;

    return 0;
}

/*
 . Function: ftmac100_enable
 . Purpose: let the chip talk to the outside world
 . Method:
 .	1.  Enable the transmitter
 .	2.  Enable the receiver
 .	3.  Enable interrupts
*/
static void ftmac100_enable( struct net_device *dev )
{
    unsigned int ioaddr = dev->base_addr;
    int i;
    struct ftmac100_local *lp = (struct ftmac100_local *)netdev_priv(dev);
    char mac_addr[6];

    FUNC_DEBUG("%s:ftmac100_enable\n", dev->name);

    for (i=0; i<RXDES_NUM; ++i) {
	lp->rx_descs[i].RXDMA_OWN = OWNBY_FTMAC100; // owned by FTMAC100
    }
    lp->rx_idx = 0;

    for (i=0; i<TXDES_NUM; ++i) {
	lp->tx_descs[i].TXDMA_OWN = OWNBY_SOFTWARE; // owned by software
    }
    lp->tx_idx = 0;

    /* set the MAC address */
    put_mac(ioaddr + MAC_MADR_REG, dev->dev_addr);

    get_mac(ioaddr + MAC_MADR_REG, mac_addr);
    print_mac(mac_addr);

    outl(lp->rx_descs_dma, ioaddr + RXR_BADR_REG);
    outl(lp->tx_descs_dma, ioaddr + TXR_BADR_REG);
    outl(0x00001010, ioaddr + ITC_REG);
    outl((0UL<<TXPOLL_CNT)|(0x1<<RXPOLL_CNT), ioaddr + APTC_REG);
    outl(0x1df, ioaddr + DBLAC_REG );

    /* now, enable interrupts */
    outl( AHB_ERR_bit|NORXBUF_bit|RPKT_FINISH_bit,ioaddr + IMR_REG);

    // enable trans/recv
    outl(lp->maccr_val, ioaddr + MACCR_REG );
}


/*
 . Function: ftmac100_shutdown
 . Purpose:  closes down the SMC91xxx chip.
 . Method:
 .	1. zero the interrupt mask
 .	2. clear the enable receive flag
 .	3. clear the enable xmit flags
 .
 . TODO:
 .   (1) maybe utilize power down mode.
 .	Why not yet?  Because while the chip will go into power down mode,
 .	the manual says that it will wake up in response to any I/O requests
 .	in the register space.   Empirical results do not show this working.
*/
static void ftmac100_shutdown( unsigned int ioaddr )
{
    FUNC_DEBUG("+ftmac100_shutdown\n");

    outl( 0, ioaddr + IMR_REG );
    outl( 0, ioaddr + MACCR_REG );
}

/*
 . Function: ftmac100_wait_to_send_packet( struct sk_buff * skb, struct device * )
 . Purpose:
 .    Attempt to allocate memory for a packet, if chip-memory is not
 .    available, then tell the card to generate an interrupt when it
 .    is available.
 .
 . Algorithm:
 .
 . o	if the saved_skb is not currently null, then drop this packet
 .		on the floor.  This should never happen, because of TBUSY.
 . o	if the saved_skb is null, then replace it with the current packet,
 . o	See if I can sending it now.
 . o 	(NO): Enable interrupts and let the interrupt handler deal with it.
 . o	(YES):Send it now.
*/
static int ftmac100_wait_to_send_packet( struct sk_buff * skb, struct net_device * dev )
{
    struct ftmac100_local   *lp=(struct ftmac100_local *)netdev_priv(dev);
    unsigned int            ioaddr=dev->base_addr;
    volatile TX_DESC        *cur_desc;
    int                     length;
    unsigned long           flags;

    //FUNC_DEBUG("+ftmac100_wait_to_send_packet\n");

    spin_lock_irqsave(&lp->lock, flags);
    if (skb == NULL) {
	printk("%s(%d): NULL skb???\n", __FILE__,__LINE__);
	spin_unlock_irqrestore(&lp->lock, flags);
	return 0;
    }

    cur_desc = &lp->tx_descs[lp->tx_idx];
    consistent_sync((void*)cur_desc, 4, DMA_TO_DEVICE);
    for (; cur_desc->TXDMA_OWN != OWNBY_SOFTWARE;)
	;//printk("no empty TX descriptor:0x%x:0x%x\n",(unsigned int)cur_desc,(unsigned int)cur_desc[0]);

    length = ETH_ZLEN < skb->len ? skb->len : ETH_ZLEN;
    length = min(length, TX_BUF_SIZE);

#if FTMAC100_DEBUG > 2
    printk("Transmitting Packet at 0x%08x\n",(unsigned int)cur_desc->VIR_TXBUF_BADR);
    print_packet( skb->data, length );
#endif
    if (lp->have_zerocopy) {
	cur_desc->VIR_TXBUF_BADR = (unsigned int) skb->data;
	cur_desc->TXBUF_BADR = virt_to_phys(skb->data);
	consistent_sync(skb->data, length, DMA_TO_DEVICE);
    } else {
	memcpy((char *)cur_desc->VIR_TXBUF_BADR, skb->data, length);
    }


    cur_desc->TXBUF_Size = length;
    cur_desc->LTS = 1;
    cur_desc->FTS = 1;
    cur_desc->TX2FIC = 0;
    cur_desc->TXIC = 0;
    cur_desc->TXDMA_OWN = OWNBY_FTMAC100;
    outl(0xffffffff, ioaddr + TXPD_REG);
    lp->tx_idx = (lp->tx_idx + 1) % TXDES_NUM;
    lp->stats.tx_packets++;
    dev_kfree_skb_any(skb);
    dev->trans_start = jiffies;
    spin_unlock_irqrestore(&lp->lock, flags);
    return 0;
}

void ftmac100_ringbuf_alloc(struct ftmac100_local *lp)
{
    int i;

    FUNC_DEBUG("+ftmac100_ringbuf_alloc\n");

    lp->rx_descs = pci_alloc_consistent(NULL, sizeof(RX_DESC)*RXDES_NUM, &(lp->rx_descs_dma));

    if (lp->rx_descs == NULL) {
	printk("Receive Ring Buffer allocation error\n");
	BUG();
    }
    memset((unsigned int *)lp->rx_descs, 0, sizeof(RX_DESC)*RXDES_NUM);

    lp->rx_buf=pci_alloc_consistent(NULL, RX_BUF_SIZE*RXDES_NUM, &(lp->rx_buf_dma));
    if (lp->rx_buf == NULL) {
	printk("Receive Ring Buffer allocation error\n");
	BUG();
    }

    for (i=0; i<RXDES_NUM; ++i) {
	if (lp->have_zerocopy) {
	    /* Allocated fixed size of skbuff */
	    struct sk_buff *skb = dev_alloc_skb (RX_BUF_SIZE);
	    lp->rx_skbuff[i] = skb;
	    if (skb == NULL) {
		printk (KERN_ERR
			"%s: alloc_list: allocate Rx buffer error! ",
			lp->ndev->name);
		break;
	    }
	    skb->dev = lp->ndev;	/* Mark as being used by this device. */
	    skb_reserve (skb, 2);	/* 16 byte align the IP header. */
	    /* Rubicon now supports 40 bits of addressing space. */
	    lp->rx_descs[i].RXBUF_Size = RX_BUF_SIZE;
	    lp->rx_descs[i].RXBUF_BADR = virt_to_phys(skb->tail); 
	    lp->rx_descs[i].VIR_RXBUF_BADR = (unsigned int) skb->tail;
	} else {
	    lp->rx_descs[i].RXBUF_Size = RX_BUF_SIZE;
	    lp->rx_descs[i].EDOTR = 0;	// not last descriptor
	    lp->rx_descs[i].RXBUF_BADR = lp->rx_buf_dma+RX_BUF_SIZE*i;
	    lp->rx_descs[i].VIR_RXBUF_BADR=(unsigned int)lp->rx_buf+RX_BUF_SIZE*i;
	}
    }
    lp->rx_descs[RXDES_NUM-1].EDOTR = 1; // is last descriptor

    lp->tx_descs = pci_alloc_consistent(NULL, sizeof(TX_DESC)*TXDES_NUM, &(lp->tx_descs_dma));
    if (lp->tx_descs == NULL) {
	printk("Transmit Ring Buffer allocation error\n");
	BUG();
    }
    memset((void *)lp->tx_descs,0,sizeof(TX_DESC)*TXDES_NUM);

    if (lp->have_zerocopy) {
	for (i=0; i<TXDES_NUM; ++i)	
	    lp->tx_descs[i].EDOTR = 0;

	lp->tx_descs[TXDES_NUM-1].EDOTR = 1; // is last descriptor	
    } else {
	lp->tx_buf = pci_alloc_consistent(NULL, TX_BUF_SIZE*TXDES_NUM, &(lp->tx_buf_dma));
	if (lp->tx_buf == NULL) {
	    printk("Transmit Ring Buffer allocation error\n");
	    BUG();
	}

	for (i=0; i<TXDES_NUM; ++i) {
	    lp->tx_descs[i].EDOTR = 0;	// not last descriptor
	    lp->tx_descs[i].TXBUF_BADR=lp->tx_buf_dma+TX_BUF_SIZE*i;
	    lp->tx_descs[i].VIR_TXBUF_BADR=(unsigned int)lp->tx_buf+TX_BUF_SIZE*i;
	}
	lp->tx_descs[TXDES_NUM-1].EDOTR = 1; // is last descriptor
    }
}

void ftmac100_ringbuf_free(struct ftmac100_local *priv)
{
    if (priv->rx_descs)
	pci_free_consistent(NULL, sizeof(RX_DESC)*RXDES_NUM, (void*)priv->rx_descs, (dma_addr_t)priv->rx_descs_dma);
    if (priv->rx_buf)
	pci_free_consistent(NULL, RX_BUF_SIZE*RXDES_NUM, (void*)priv->rx_buf, (dma_addr_t)priv->rx_buf_dma );
    if (priv->tx_descs)
	pci_free_consistent(NULL, sizeof(TX_DESC)*TXDES_NUM, (void*)priv->tx_descs, (dma_addr_t)priv->tx_descs_dma );
    if (priv->tx_buf)
	pci_free_consistent(NULL, TX_BUF_SIZE*TXDES_NUM, (void*)priv->tx_buf, (dma_addr_t)priv->tx_buf_dma );
    priv->rx_descs = NULL; priv->rx_descs_dma = 0;
    priv->rx_buf   = NULL; priv->rx_buf_dma   = 0;
    priv->tx_descs = NULL; priv->tx_descs_dma = 0;
    priv->tx_buf   = NULL; priv->tx_buf_dma   = 0;
}

/*----------------------------------------------------------------------
 . Function: ftmac100_probe( unsigned int ioaddr )
 .
 . Params: - dev_mdio  : net_device to use for MDIO communication  
 .         - phy_addr  : addr of PHY on MII
 .
 . Purpose:
 .	Tests to see if a given ioaddr points to an ftmac100 chip.
 .	Returns a 0 on success
 .---------------------------------------------------------------------
 */
static int __init ftmac100_probe(struct net_device *dev)
{
    int                     retval;
    struct ftmac100_local   *priv = NULL;
    struct net_device       *dev_mdio = NULL;
    int rev_reg;
    int phy_addr;
    int i;

    FUNC_DEBUG("+ftmac100_probe\n");

    rev_reg = inl(dev->base_addr + REV_REG);

    /* Now, print out the card info, in a short format.. */
    printk(KERN_INFO "%s: device at %#3x IRQ:%d, rev. %08x\n",dev->name,
	    (unsigned)dev->base_addr, dev->irq, rev_reg);

    /* Initialize priviate data */
    priv = netdev_priv(dev);
    memset(priv, 0, sizeof(struct ftmac100_local));
    spin_lock_init(&priv->lock);

    priv->have_zerocopy = ((rev_reg & REV_B3_MASK) >> REV_B3_SHIFT) != 0 ? 1 : 0;
    priv->ndev = dev;
    priv->maccr_val = FULLDUP_bit | CRC_APD_bit | MDC_SEL_bit | RCV_EN_bit
	| XMT_EN_bit | RDMA_EN_bit | XDMA_EN_bit ;
    ftmac100_ringbuf_alloc(priv);

    /* now, reset the chip, and put it into a known state */
    retval = ftmac100_reset(dev);
    if (retval) {
	printk("%s: unable to reset.\n", dev->name);
	goto err_out;
    }

    /* Fill in the fields of the device structure with ethernet values. */
    ether_setup(dev);

    /* Grab the IRQ */
    retval = request_irq(dev->irq, &ftmac100_interrupt, SA_SAMPLE_RANDOM, dev->name, dev);
    if (retval) {
	printk("%s: unable to get IRQ %d (irqval=%d).\n", dev->name, dev->irq, retval);
	goto err_out;
    }
    set_irq_type(dev->irq, IRQT_HIGH);

    /* Init the workqueue */
    INIT_WORK(&priv->rcv_tq, (void*)ftmac100_rcv, dev);

    /* Init link monitoring timer */
    init_timer(&priv->link_timer);
    priv->link_timer.function = ftmac100_link_timer;
    priv->link_timer.data = (unsigned long) priv;
    priv->timer_ticks = 0;

    /* Fill up the mii_phy structure */
    priv->phy_mii.dev = dev;
    priv->phy_mii.mdio_read = ftmac100_phy_read;
    priv->phy_mii.mdio_write = ftmac100_phy_write;

    printk("dev_base = %08x\n", dev_base);
#ifdef SHARE_MDIO
    u32 base_addr_mdio = ftmac_devices[0].base_addr;
    struct net_device *d;

    /* scan device list for the MAC to use for MDIO communication
     * PP: on boards with KIM, MDIO signal of first MAC is typically
     *     used for both PHYs, hence CPE_FTMAC_BASE for dev_mdio*/
    read_lock(&dev_base_lock);
    for(d = dev_base; d; d = d->next) {
	if (d->base_addr == base_addr_mdio) {
	    dev_mdio = d;
	    break;
	}
    }
    read_unlock(&dev_base_lock);

    if (dev->base_addr == base_addr_mdio)
	dev_mdio = dev;

    if (!dev_mdio) {
	printk("+FTCmac_probe : could not find mdio_dev with base_addr %08x\n", base_addr_mdio);
	BUG();
    }
#else
    dev_mdio = dev;
#endif

    /* search for the PHY address to use in the ftmac_devices structure */
    phy_addr = PHY_ADDR_PROBE;
    for (i=0;i<(sizeof(ftmac_devices)/sizeof(struct ftmac_device));i++) {
	if (ftmac_devices[i].base_addr == dev->base_addr) {
	    phy_addr = ftmac_devices[i].phy_addr;
	    break;
	}
    }

    /* init net_device for MDIO communication */
    priv->dev_mdio = dev_mdio;

    /* TODO add ignore_phy stuff */
    priv->phy_mii.mii_id = -1;
    if (phy_addr == PHY_ADDR_PROBE) {
	u_char addr;
	for (addr = PHY_PROBE_MIN; addr <= PHY_PROBE_MAX; addr++) {
	    if (mii_phy_probe(&priv->phy_mii, addr) == 0) {
		priv->phy_mii.mii_id = addr;
		break;
	    }
	}
    } else {
	if (mii_phy_probe(&priv->phy_mii, phy_addr) == 0) {
	    priv->phy_mii.mii_id = phy_addr;
	}
    }

    if (priv->phy_mii.mii_id == -1) {
	printk(KERN_WARNING "%s: Failed to find PHY.\n", dev->name);
	retval = -ENODEV;
	goto err_out;	
    }
    printk(KERN_INFO "%s: Found PHY on address %d\n", dev->name, priv->phy_mii.mii_id);

    /* Setup initial PHY config & startup aneg */
    if (priv->phy_mii.def->ops->init)
	priv->phy_mii.def->ops->init(&priv->phy_mii);

    netif_carrier_off(dev);

    if (priv->phy_mii.def->features & SUPPORTED_Autoneg) {
	priv->want_autoneg = 1;
	priv->advertising =
	    ADVERTISED_10baseT_Half | ADVERTISED_10baseT_Full |
	    ADVERTISED_100baseT_Half | ADVERTISED_100baseT_Full;
    }

    dev->open               = ftmac100_open;
    dev->stop               = ftmac100_close;
    dev->hard_start_xmit    = ftmac100_wait_to_send_packet;
    dev->get_stats          = ftmac100_query_statistics;
    dev->tx_timeout         = ftmac100_timeout;
    dev->do_ioctl           = ftmac100_ioctl;
#ifdef HAVE_MULTICAST
    dev->set_multicast_list = &ftmac100_set_multicast_list;
#endif

    /* set initial MAC from boot code */
    {
	bd_arm_t * bd = (bd_arm_t *) __res;
	int i;
	for (i = 0; i < 6; i++) {
	    dev->dev_addr[i] = bd->net_info.bi_enetaddr[i];
	}
    }

    ftmac100_start_link(priv, NULL);

    return 0;

err_out:
    if (priv) {
	ftmac100_ringbuf_free(priv);
    }
    return retval;
}

static void ftmac100_start_link(struct ftmac100_local *lp, struct ethtool_cmd *ep)
{
    u32 advertise;
    int autoneg;
    int forced_speed;
    int forced_duplex;

    /* Default advertise */
    advertise = lp->advertising;
    autoneg = lp->want_autoneg;
    forced_speed = lp->phy_mii.speed;
    forced_duplex = lp->phy_mii.duplex;

    /* Setup link parameters */
    if (ep) {
	if (ep->autoneg == AUTONEG_ENABLE) {
	    advertise = ep->advertising;
	    lp->advertising = advertise;
	    autoneg = 1;
	} else {
	    autoneg = 0;
	    forced_speed = ep->speed;
	    forced_duplex = ep->duplex;
	}
    }

    /* Configure PHY & start aneg */
    lp->want_autoneg = autoneg;
    if (autoneg) {
	LINK_DEBUG("%s: start link aneg, advertise: 0x%08x\n",
		lp->ndev->name, advertise);
	lp->phy_mii.def->ops->setup_aneg(&lp->phy_mii, advertise);
    } else {
	LINK_DEBUG("%s: start link forced, speed: %d, duplex: %d\n",
		lp->ndev->name, forced_speed, forced_duplex);
	lp->phy_mii.def->ops->setup_forced(&lp->phy_mii, forced_speed,
		forced_duplex);
    }

    lp->timer_ticks = 0;
    mod_timer(&lp->link_timer, jiffies + HZ);
}

/* FIXME(miba): FTMAC100 has an IRQ for link state, use it */
static void ftmac100_link_timer(unsigned long data)
{
    struct ftmac100_local *lp = (struct ftmac100_local *) data;
    int link;
    link = lp->phy_mii.def->ops->poll_link(&lp->phy_mii);
    LINK_DEBUG("%s: poll_link: %d\n", lp->ndev->name, link);

    if (link == netif_carrier_ok(lp->ndev)) {
	if (!link && lp->want_autoneg && (++lp->timer_ticks) > 10)
	    ftmac100_start_link(lp, NULL);
	goto out;
    }
    printk(KERN_INFO "%s: Link is %s\n", lp->ndev->name, link ? "Up" : "Down");
    if (link) {
	netif_carrier_on(lp->ndev);
	ftmac100_adjust_to_link(lp);
    } else {
	lp->timer_ticks = 0;
	netif_carrier_off(lp->ndev);
    }

out:
    mod_timer(&lp->link_timer, jiffies + HZ);
}

static int ftmac100_adjust_to_link(struct ftmac100_local *lp)
{
    unsigned int ioaddr = lp->ndev->base_addr;
    int full_duplex = 0, speed = 0; 

    /* Read link mode on PHY */
    if (lp->phy_mii.def->ops->read_link(&lp->phy_mii) == 0) {
	/* If an error occurred, we don't deal with it yet */
	full_duplex = (lp->phy_mii.duplex == DUPLEX_FULL);

	speed = lp->phy_mii.speed;
    }

    lp->maccr_val &= ~(FULLDUP_bit | SPEED_100_bit);

    if (full_duplex) {
	lp->maccr_val |= FULLDUP_bit;
    }

    if (speed == SPEED_100) {
	lp->maccr_val |= SPEED_100_bit;
    }

    LINK_DEBUG("%s: adjust to link, speed: %d, duplex: %d, opened: %d\n",
	    lp->ndev->name, speed, full_duplex, lp->opened);

    printk(KERN_INFO "%s: Speed: %s, %s duplex.\n",
	    lp->ndev->name,
	    speed == SPEED_100 ? "100" : "10",
	    full_duplex ? "Full" : "Half");

    outl(lp->maccr_val, ioaddr + MACCR_REG);
    return 0;
}

#if FTMAC100_DEBUG > 2
static void print_packet( unsigned char * buf, int length )
{
#if FTMAC100_DEBUG > 3
    int i;
    int remainder;
    int lines;
#endif

    //printk("Packet of length %d \n", length );

#if FTMAC100_DEBUG > 3
    lines = length / 16;
    remainder = length % 16;

    for ( i = 0; i < lines ; i ++ ) 
    {
	int cur;

	for ( cur = 0; cur < 8; cur ++ ) 
	{
	    unsigned char a, b;

	    a = *(buf ++ );
	    b = *(buf ++ );
	    printk("%02x%02x ", a, b );
	}
	printk("\n");
    }
    for ( i = 0; i < remainder/2 ; i++ ) 
    {
	unsigned char a, b;

	a = *(buf ++ );
	b = *(buf ++ );
	printk("%02x%02x ", a, b );
    }
    printk("\n");
#endif
}
#endif

static int ftmac100_phy_wait(struct net_device *dev)
{
    struct net_device *dev_mdio = ((struct ftmac100_local *)netdev_priv(dev))->dev_mdio;
    unsigned int       ioaddr   = dev_mdio->base_addr;
    unsigned long      timeout  = jiffies + FTMAC100_PHY_TIMEOUT;
    int ret = -1;
    u_int32_t phycr = 0;

    /* wait for PHY to be free */
    while (!time_after(jiffies, timeout)) {
	if (((phycr = inl(ioaddr + PHYCR_REG)) & (PHYCR_MIIWR | PHYCR_MIIRD)) == 0) {
	    break;
	}
	udelay(FTMAC100_MDIO_DELAY);
    }

    if (phycr & (PHYCR_MIIWR | PHYCR_MIIRD)) {
	MDIO_DEBUG("%s: phy_wait timeout (PHYCR=0x%08x)\n", dev->name, phycr);
	goto bail;
    }

    MDIO_DEBUG("%s: phy_wait done (PHYCR=0x%08x)\n", dev->name, phycr);

    ret = 0;

 bail:
    return ret;
}

/**
 * Read a PHY register using MII Management serial interface
 */
static int ftmac100_phy_read(struct net_device *dev, int mii_id, int reg)
{
    struct net_device *dev_mdio = ((struct ftmac100_local *)netdev_priv(dev))->dev_mdio;
    struct ftmac100_local *lp_mdio = (struct ftmac100_local *) netdev_priv(dev_mdio);
    unsigned int       ioaddr   = dev_mdio->base_addr;
    int ret = -1;
    u_int16_t val;

    spin_lock_bh(&lp_mdio->lock_mdio);
    
    MDIO_DEBUG("%s: phy_read (base=0x%08x), reg: 0x%08x\n", dev->name, ioaddr, reg);
	
    /* wait for PHY to be free */
    if (ftmac100_phy_wait(dev) != 0) {
	MDIO_DEBUG("%s: PHY read timeout #1!\n", dev->name);
	goto bail;
    }

    outl(PHYCR_MIIRD | PHYCR_REGAD(reg) | PHYCR_PHYAD(mii_id), ioaddr + PHYCR_REG);

    /* wait for PHY to complete operation */
    if (ftmac100_phy_wait(dev) != 0) {
	MDIO_DEBUG("%s: PHY read timeout #2!\n", dev->name);
	goto bail;
    }

    val = (inl(ioaddr + PHYCR_REG) & PHYCR_MIIRDATA_MASK) >> PHYCR_MIIRDATA_SHIFT;

    MDIO_DEBUG("%s: phy_read -> 0x%08x\n", dev->name, val);

    ret = val;

 bail:
    spin_unlock_bh(&lp_mdio->lock_mdio);
    return ret;
}

/**
 * Write a PHY register using MII Management serial interface
 *
 * FIXME: add a return value to indicate an error
 */
static void ftmac100_phy_write(struct net_device *dev, int mii_id, int reg, int val)
{
    struct net_device *dev_mdio = ((struct ftmac100_local *)netdev_priv(dev))->dev_mdio;
    struct ftmac100_local *lp_mdio = (struct ftmac100_local *) netdev_priv(dev_mdio);
    unsigned int ioaddr   = dev_mdio->base_addr;

    spin_lock_bh(&lp_mdio->lock_mdio);

    MDIO_DEBUG("%s: phy_write (base=0x%08x), reg: 0x%08x, val: 0x%08x\n", dev->name, ioaddr, reg, val);

    /* wait for PHY to be free */
    if (ftmac100_phy_wait(dev) != 0) {
	MDIO_DEBUG("%s: PHY write timeout #1!\n", dev->name);
	goto bail;
    }

    outl(val, ioaddr + PHYWDATA_REG);
    outl(PHYCR_MIIWR | PHYCR_REGAD(reg) | PHYCR_PHYAD(mii_id), ioaddr + PHYCR_REG);

    /* wait for PHY to complete operation */
    if (ftmac100_phy_wait(dev) != 0) {
	MDIO_DEBUG("%s: PHY write timeout #2!\n", dev->name);
	goto bail;
    }

 bail:
    spin_unlock_bh(&lp_mdio->lock_mdio);

    return;
}

/*------------------------------------------------------------
 . Configures the specified PHY using Autonegotiation.
 .-------------------------------------------------------------*/
static void ftmac100_phy_configure(struct net_device* dev)
{
    return;
}

/*
 * Open and Initialize the board
 *
 * Set up everything, reset the card, etc ..
 *
 */
static int ftmac100_open(struct net_device *dev)
{
    FUNC_DEBUG("+%s:ftmac100_open\n", dev->name);
    /* reset the hardware */
    ftmac100_reset(dev);
    ftmac100_enable(dev);

    /* Configure the PHY */
    ftmac100_phy_configure(dev);

    netif_start_queue(dev);
    return 0;
}


/*--------------------------------------------------------
 . Called by the kernel to send a packet out into the void
 . of the net.  This routine is largely based on
 . skeleton.c, from Becker.
 .--------------------------------------------------------
*/
static void ftmac100_timeout(struct net_device *dev)
{
    /* If we get here, some higher level has decided we are broken.
    There should really be a "kick me" function call instead. */
    printk(KERN_WARNING "%s: transmit timed out?\n",dev->name);

    ftmac100_reset(dev);
    ftmac100_enable(dev);
    ftmac100_phy_configure(dev);
    netif_wake_queue(dev);
    dev->trans_start = jiffies;
}


/*--------------------------------------------------------------------
 .
 . This is the main routine of the driver, to handle the net_device when
 . it needs some attention.
 .
 . So:
 .   first, save state of the chipset
 .   branch off into routines to handle each case, and acknowledge
 .	    each to the interrupt register
 .   and finally restore state.
 .
 ---------------------------------------------------------------------*/
static irqreturn_t ftmac100_interrupt(int irq, void * dev_id, struct pt_regs * regs)
{
    struct net_device   *dev=dev_id;
    unsigned int        ioaddr=dev->base_addr;
    unsigned char       status;			// interrupt status
    unsigned char       mask;			// interrupt mask
    int	                timeout;
    struct ftmac100_local *lp = (struct ftmac100_local *)netdev_priv(dev);

    if (dev == NULL) {
	printk(KERN_WARNING "%s: irq %d for unknown device.\n", dev->name, irq);
	return IRQ_NONE;
    }

    /* read the interrupt status register */
    mask = inl(ioaddr + IMR_REG);

    /* set a timeout value, so I don't stay here forever */

    //PRINTK(KERN_WARNING "%s: MASK IS %x \n", dev->name, mask);
    for (timeout=1; timeout>0; --timeout) {
	/* read the status flag, and mask it */
	status = inl(ioaddr + ISR_REG);

	status &= mask;
	if (!status )
	    break;

	if ( status & RPKT_FINISH_bit ) {
	    ftmac100_rcv(dev);
	} else if (status & NORXBUF_bit) {
	    //printk("<0x%x:NORXBUF>",status);
	    outl(mask & ~NORXBUF_bit, ioaddr + IMR_REG);
	    trans_busy = 1;
	    schedule_delayed_work(&lp->rcv_tq, 1);
	} else if (status & AHB_ERR_bit)
	    printk("<0x%x:AHB_ERR>",status);
    }
    return IRQ_HANDLED;
}

/*-------------------------------------------------------------
 .
 . ftmac100_rcv -  receive a packet from the card
 .
 . There is ( at least ) a packet waiting to be read from
 . chip-memory.
 .
 . o Read the status
 . o If an error, record it
 . o otherwise, read in the packet
 --------------------------------------------------------------
*/

static void ftmac100_rcv(void *devp)
{
    struct net_device       *dev=(struct net_device *)devp;
    struct ftmac100_local   *lp = (struct ftmac100_local *)netdev_priv(dev);
    unsigned int            ioaddr=dev->base_addr;
    int                     packet_length;
    int                     rcv_cnt;
    volatile RX_DESC        *cur_desc;
    int                     cpy_length;
    int	                    cur_idx;
    int	                    seg_length;
    int	                    have_package;
    int	                    have_frs;
    int	                    start_idx;

    //FUNC_DEBUG("+ftmac100_rcv\n");

    start_idx = lp->rx_idx;

    for (rcv_cnt=0; rcv_cnt<8 ; ++rcv_cnt) {
	packet_length = 0;
	cur_idx = lp->rx_idx;

	have_package = 0;
	have_frs = 0;
	for (; (cur_desc = &lp->rx_descs[lp->rx_idx])->RXDMA_OWN==0; ) {
	    have_package = 1;
	    lp->rx_idx = (lp->rx_idx+1)%RXDES_NUM;
	    if (cur_desc->FRS) {
		have_frs = 1;
		if (
#if defined(KIRA_LTRXG4) && defined(PRODUCT_LARAXP)
			0 /* ignore floating signal */
#else
			cur_desc->RX_ERR 
#endif
			|| cur_desc->CRC_ERR || cur_desc->FTL || cur_desc->RUNT || cur_desc->RX_ODD_NB) {
		    lp->stats.rx_errors++;  // error frame
		    break;
		}
		if (cur_desc->MULTICAST)
		    lp->stats.multicast++;
		packet_length = cur_desc->ReceiveFrameLength; // normal frame
	    }
	    if ( cur_desc->LRS ) // packet's last frame
		break;
	}
	if (have_package==0)
	    goto done;
	if (have_frs == 0)
	    lp->stats.rx_over_errors++;

	if (packet_length>0) {
	    struct sk_buff *skb;
	    unsigned char *data;

	    if (lp->have_zerocopy) {
		//DO_PRINT("Receiving Packet\r\n");
		skb_put(skb = lp->rx_skbuff[cur_idx], packet_length);
		lp->rx_skbuff[cur_idx] = NULL;
		consistent_sync(skb->data, packet_length, DMA_FROM_DEVICE);
	    } else {
		// Allocate enough memory for entire receive frame, to be safe
		skb = dev_alloc_skb(packet_length+2);

		if (skb == NULL) {
		    printk(KERN_NOTICE "%s: Low memory, packet dropped.\n", dev->name);
		    lp->stats.rx_dropped++;
		    goto done;
		}

		skb_reserve(skb, 2);   /* 16 bit alignment */
		skb->dev = dev;

		data = skb_put(skb, packet_length);
		cpy_length = 0;
		for (; cur_idx!=lp->rx_idx; cur_idx=(cur_idx+1)%RXDES_NUM) {
		    seg_length = min(packet_length - cpy_length, RX_BUF_SIZE);
		    memcpy(data+cpy_length, (char *)lp->rx_descs[cur_idx].VIR_RXBUF_BADR, seg_length);
		    cpy_length += seg_length;
		}
	    }

	    skb->protocol = eth_type_trans(skb, dev);
	    netif_rx(skb);
	    lp->stats.rx_packets++;
	}
    }

done:
    if (start_idx != lp->rx_idx) {
	struct sk_buff *skb;

	for (cur_idx = (start_idx+1) % RXDES_NUM; cur_idx != lp->rx_idx; cur_idx = (cur_idx+1) % RXDES_NUM) {
	    if (lp->have_zerocopy) {
		/* Dropped packets don't need to re-allocate */
		if (lp->rx_skbuff[cur_idx] == NULL) {
		    skb = dev_alloc_skb (RX_BUF_SIZE);
		    if (skb == NULL) {
			printk (KERN_INFO
				"%s: receive_packet: "
				"Unable to re-allocate Rx skbuff.#%d\n",
				dev->name, cur_idx);
			break;
		    }
		    lp->rx_skbuff[cur_idx] = skb;
		    skb->dev = dev;
		    /* 16 byte align the IP header */
		    skb_reserve (skb, 2);
		    lp->rx_descs[cur_idx].RXBUF_Size = RX_BUF_SIZE;	
		    lp->rx_descs[cur_idx].RXBUF_BADR = virt_to_phys(skb->tail); 
		    lp->rx_descs[cur_idx].VIR_RXBUF_BADR = (unsigned int) skb->tail;
		}
	    }
	    lp->rx_descs[cur_idx].RXDMA_OWN = 1;
	}
	if (lp->have_zerocopy) {
	    //struct sk_buff *skb;
	    /* Dropped packets don't need to re-allocate */
	    if (lp->rx_skbuff[start_idx] == NULL) {
		skb = dev_alloc_skb (RX_BUF_SIZE);
		if (skb == NULL) {
		    printk (KERN_INFO
			    "%s: receive_packet: "
			    "Unable to re-allocate Rx skbuff.#%d\n",
			    dev->name, start_idx);
		}
		lp->rx_skbuff[start_idx] = skb;
		skb->dev = dev;
		/* 2 byte align the IP header */
		skb_reserve (skb, 2);

		lp->rx_descs[start_idx].RXBUF_BADR = virt_to_phys(skb->tail); //Richard
		lp->rx_descs[start_idx].RXBUF_Size = RX_BUF_SIZE;
		lp->rx_descs[start_idx].VIR_RXBUF_BADR = (unsigned int) skb->tail;
	    }
	}
	lp->rx_descs[start_idx].RXDMA_OWN = 1;
    }
    if (trans_busy == 1) {
	outl(lp->maccr_val, ioaddr + MACCR_REG);
	outl(inl(ioaddr + IMR_REG) | NORXBUF_bit, ioaddr + IMR_REG);
    }
    return;
}


/*----------------------------------------------------
 . ftmac100_close
 .
 . this makes the board clean up everything that it can
 . and not talk to the outside world.   Caused by
 . an 'ifconfig ethX down'
 .
 -----------------------------------------------------*/
static int ftmac100_close(struct net_device *dev)
{
    struct ftmac100_local *lp = (struct ftmac100_local*) netdev_priv(dev);

    FUNC_DEBUG("+ftmac100_close\n");

    lp->opened = 0;

    netif_stop_queue(dev);

    ftmac100_shutdown( dev->base_addr );
    return 0;
}

/*------------------------------------------------------------
 . Get the current statistics.
 . This may be called with the card open or closed.
 .-------------------------------------------------------------*/
static struct net_device_stats* ftmac100_query_statistics(struct net_device *dev) 
{
    struct ftmac100_local *lp = (struct ftmac100_local *)netdev_priv(dev);

    FUNC_DEBUG("+ftmac100_query_statistics\n");

    return &lp->stats;
}


#ifdef HAVE_MULTICAST

/*
 . Function: ftmac100_setmulticast( unsigned int ioaddr, int count, dev_mc_list * adds )
 . Purpose:
 .    This sets the internal hardware table to filter out unwanted multicast
 .    packets before they take up memory.
*/

static void ftmac100_setmulticast(unsigned int ioaddr, int count, struct dev_mc_list * addrs)
{
    struct dev_mc_list	* cur_addr;
    int crc_val;

    FUNC_DEBUG("+ftmac100_setmulticast\n");

    for (cur_addr = addrs ; cur_addr!=NULL; cur_addr = cur_addr->next) {
	if (!(*cur_addr->dmi_addr & 1 ))
	    continue;
	crc_val = crc32(cur_addr->dmi_addr, 6);
	crc_val = (crc_val >> 26) & 0x3f;
	if (crc_val >= 32)
	    outl(inl(ioaddr+MAHT1_REG) | (1UL<<(crc_val-32)), ioaddr+MAHT1_REG);
	else
	    outl(inl(ioaddr+MAHT0_REG) | (1UL<<crc_val), ioaddr+MAHT0_REG);
    }
}

/*-----------------------------------------------------------
 . ftmac100_set_multicast_list
 .
 . This routine will, depending on the values passed to it,
 . either make it accept multicast packets, go into
 . promiscuous mode ( for TCPDUMP and cousins ) or accept
 . a select set of multicast packets
*/
static void ftmac100_set_multicast_list(struct net_device *dev)
{
    unsigned int ioaddr = dev->base_addr;
    struct ftmac100_local *lp = (struct ftmac100_local *)netdev_priv(dev);

    FUNC_DEBUG("+ftmac100_set_multicast_list\n");

    if (dev->flags & IFF_PROMISC)
	lp->maccr_val |= RCV_ALL_bit;
    else
	lp->maccr_val &= ~RCV_ALL_bit;

    if (!(dev->flags & IFF_ALLMULTI))
	lp->maccr_val |= RX_MULTIPKT_bit;
    else
	lp->maccr_val &= ~RX_MULTIPKT_bit;

    if (dev->mc_count) {
	lp->maccr_val |= HT_MULTI_EN_bit;
	ftmac100_setmulticast(ioaddr, dev->mc_count, dev->mc_list);
    } else
	lp->maccr_val &= ~HT_MULTI_EN_bit;

    outl(lp->maccr_val, ioaddr + MACCR_REG);
}
#endif

/**
 * ftmac100_ioctl
 * (ethtool support)
*/
static int ftmac100_ioctl(struct net_device *dev, struct ifreq *rq, int cmd)
{
    FUNC_DEBUG("FTMAC100 IOCTL 0x%08x\n", cmd);
    struct ftmac100_local *lp = (struct ftmac100_local *)netdev_priv(dev);
    struct mii_ioctl_data *mii;
    
    switch (cmd) {
	case SIOCGMIIPHY:	/* Get the address of the PHY in use. */
	    mii = if_mii(rq);
	    mii->phy_id = lp->phy_mii.mii_id;
	    return 0;
	case SIOCGMIIREG:	/* Read the specified MII register. */
	    mii = if_mii(rq);
	    mii->val_out = lp->phy_mii.mdio_read(dev, lp->phy_mii.mii_id, mii->reg_num);
	    return 0;
	case SIOCSMIIREG:	/* Write the specified MII register */
	    mii = if_mii(rq);
	    lp->phy_mii.mdio_write(dev, lp->phy_mii.mii_id, mii->reg_num, mii->val_in);
	    return 0;
	case SIOCETHTOOL:
	    return ftmac100_ethtool(dev, rq->ifr_data);
	default:
	    return -EOPNOTSUPP;
    }
}

static int ftmac100_ethtool(struct net_device *dev, void* ep_user)
{
    struct ftmac100_local *lp = (struct ftmac100_local *)netdev_priv(dev);
    struct ethtool_cmd ecmd;
    unsigned long features = lp->phy_mii.def->features;

    if (copy_from_user(&ecmd, ep_user, sizeof(ecmd))) {
	return -EFAULT;
    }

    switch(ecmd.cmd) {
	case ETHTOOL_GDRVINFO:  {
	    struct ethtool_drvinfo info;

	    memset(&info, 0, sizeof(info));
	    info.cmd = ETHTOOL_GDRVINFO;
	    strncpy(info.driver, DRV_NAME, ETHTOOL_BUSINFO_LEN);
	    strncpy(info.version, DRV_VERSION, ETHTOOL_BUSINFO_LEN);
	    info.fw_version[0] = '\0';
	    sprintf(info.bus_info, "FTMAC100");
	    info.regdump_len = 0;
	    if (copy_to_user(ep_user, &info, sizeof(info))) {
		return -EFAULT;
	    }

	    return 0;
	}

	case ETHTOOL_GSET:
	    ecmd.supported = features;
	    ecmd.port = PORT_MII;
	    ecmd.transceiver = ignore_phy[lp->phy_mii.mii_id] ? XCVR_DUMMY1 : XCVR_EXTERNAL;
	    ecmd.phy_address = lp->mii_phy_addr;

	    spin_lock_irq(&lp->lock);
	    ecmd.autoneg = lp->want_autoneg;
	    ecmd.speed = lp->phy_mii.speed;
	    ecmd.advertising = lp->phy_mii.advertising;
	    ecmd.duplex = lp->phy_mii.duplex;	  
	    /* FIXME: #warning maxtx/rxpkt must be filled out */
	    ecmd.maxtxpkt = 0;
	    ecmd.maxrxpkt = 0;
	    spin_unlock_irq(&lp->lock);

	    if (copy_to_user(ep_user, &ecmd, sizeof(ecmd))) {
	        return -EFAULT;
	    }

	    return 0;

	case ETHTOOL_SSET:
	    if (!capable(CAP_NET_ADMIN)) {
	        return -EPERM;
	    }

	    if (ecmd.autoneg != AUTONEG_ENABLE && ecmd.autoneg != AUTONEG_DISABLE) {
	       return -EINVAL;
	    }
	    if (ecmd.autoneg == AUTONEG_ENABLE && ecmd.advertising == 0) {
	        return -EINVAL;
	    }
	    if (ecmd.duplex != DUPLEX_HALF && ecmd.duplex != DUPLEX_FULL) {
	        return -EINVAL;
	    }
	    if (ecmd.autoneg == AUTONEG_DISABLE)
	        switch(ecmd.speed) {
		    case SPEED_10:
		        if (ecmd.duplex == DUPLEX_HALF && (features & SUPPORTED_10baseT_Half) == 0)
			    return -EINVAL;
			if (ecmd.duplex == DUPLEX_FULL && (features & SUPPORTED_10baseT_Full) == 0)
			    return -EINVAL;
			break;
		    case SPEED_100:
			if (ecmd.duplex == DUPLEX_HALF && (features & SUPPORTED_100baseT_Half) == 0)
			    return -EINVAL;
			if (ecmd.duplex == DUPLEX_FULL && (features & SUPPORTED_100baseT_Full) == 0)
			    return -EINVAL;
			break;
		    default:
			return -EINVAL;
		}
	    else if ((features & SUPPORTED_Autoneg) == 0)
		return -EINVAL;
	    ftmac100_start_link(lp, &ecmd);
	    return 0;

	case ETHTOOL_NWAY_RST:
	    if (!lp->want_autoneg) {
		return -EINVAL;
	    }
	    ftmac100_start_link(lp, NULL);
	    return 0;

	case ETHTOOL_GLINK: {
	    struct ethtool_value edata;
	    memset(&edata, 0, sizeof(edata));
	    edata.cmd = ETHTOOL_GLINK;
	    edata.data = netif_carrier_ok(dev);
	    if (copy_to_user(ep_user, &edata, sizeof(edata))) {
		return -EFAULT;
	    }
	    return 0;
	}
    }
    return -EOPNOTSUPP;
}


static int __init ftmac100_module_init(void)
{
    int result,id,thisresult;
    struct net_device *dev;
    int count = sizeof(ftmac_devices)/sizeof(struct ftmac_device);

    printk("%s", version);

    result = -ENODEV;
    for (id=0; id < count; id++) {
	dev = alloc_etherdev(sizeof(struct ftmac100_local));
	if (!dev) {
	    printk(KERN_ERR "Failed to allocate ethernet device");
	    return -ENODEV;
	}
	SET_MODULE_OWNER (dev);
	/* Copy the parameters from the platform specification */
	dev->base_addr = ftmac_devices[id].base_addr;
	dev->irq = ftmac_devices[id].irq;

	dev->init = ftmac100_probe;
	if ((thisresult = register_netdev(dev)) != 0) {
	    free_irq(dev->irq, dev);
	    free_netdev(dev);
	} else {
	    ftmac_devices[id].netdev = dev;
	}
	if (thisresult == 0) // any of the devices initialized, run
	    result = 0;
    }

    return result;
}


/*
 * Cleanup when module is removed with rmmod
 */
 
static void ftmac100_module_exit(void)
{
    int id;
    struct net_device *dev;
    struct ftmac100_local *priv;
    int count = sizeof(ftmac_devices)/sizeof(struct ftmac_device);

    //PRINTK("+cleanup_module\n");

    for (id=0; id < count; id++) {
	dev = ftmac_devices[id].netdev;

	if (dev==NULL)
	    continue;

	priv = (struct ftmac100_local *)netdev_priv(dev);

	ftmac100_ringbuf_free(priv);

	/* No need to check MOD_IN_USE, as sys_delete_module() checks. */
	unregister_netdev(dev);
	free_irq(dev->irq, dev);
	free_netdev(dev);
	ftmac_devices[id].netdev = NULL;
    }
}

module_init(ftmac100_module_init);
module_exit(ftmac100_module_exit);

