/*
 * Dummy PCI Hot Plug Controller Driver
 *
 * Copyright (c) 2003 Rolf Eike Beer <eike-kernel@sf-tec.de>
 *
 * Based on code from:
 *	Vladimir Kondratiev <vladimir.kondratiev@intel.com>
 *	Greg Kroah-Hartman <greg@kroah.com>
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2 of the License.
 *
 * Send feedback to <eike-kernel@sf-tec.de>
 */

/*
 * This driver will "emulate" removing PCI devices from the system.  If
 * the "power" file is written to with "0" then the specified PCI device
 * will be completely removed from the kernel.
 *
 * WARNING, this does NOT turn off the power to the PCI device.  This is
 * a "logical" removal, not a physical or electrical removal.
 *
 * Use this module at your own risk, you have been warned!
 *
 */
#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/pci.h>
#include <linux/init.h>
#include "pci_hotplug.h"
#include "../pci.h"

#define dbg(format, arg...)					\
	do {							\
		if (debug)					\
			printk(KERN_DEBUG "%s: " format,	\
				MY_NAME , ## arg); 		\
	} while (0)
#define err(format, arg...) printk(KERN_ERR "%s: " format, MY_NAME , ## arg)
#define info(format, arg...) printk(KERN_INFO "%s: " format, MY_NAME , ## arg)

/* name size which is used for entries in sysfs */
#define SLOT_NAME_SIZE	11		/* {DOMAIN}:{BUS}:{DEV} */

struct dummy_slot {
	struct list_head	node;
	struct pci_bus		*bus;
	int			devfn;
	struct pci_dev		*dev;
	struct hotplug_slot	*slot;
};

static int debug;
static int showall;
static LIST_HEAD(slot_list);

#define MY_NAME	"dummyphp"
#define DRIVER_DESC	"Dummy PCI Hot Plug Controller Driver"

MODULE_AUTHOR("Rolf Eike Beer <eike-kernel@sf-tec.de>");
MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_LICENSE("GPL v2");
module_param(debug, bool, 0644);
MODULE_PARM_DESC(debug, "Debugging mode enabled or not");
module_param(showall, bool, 0644);
MODULE_PARM_DESC(showall, "Show all PCI slots even if there is no device in it");

static int enable_slot(struct hotplug_slot *slot);
static int disable_slot(struct hotplug_slot *slot);
static int get_adapter_status(struct hotplug_slot *hotplug_slot, u8 *value);

static struct hotplug_slot_ops dummy_hotplug_slot_ops = {
	.owner			= THIS_MODULE,
	.enable_slot		= enable_slot,
	.disable_slot		= disable_slot,
	.get_adapter_status	= get_adapter_status
};

/**
 * dummy_release - free the memory of a slot
 * @slot: slot pointer to remove
 */
static void
dummy_release(struct hotplug_slot *slot)
{
	struct dummy_slot *dslot = slot->private;

	list_del(&dslot->node);
	kfree(dslot->slot->name);
	kfree(dslot->slot->info);
	kfree(dslot->slot);
	if (dslot->dev)
		pci_dev_put(dslot->dev);
	kfree(dslot);
}

/**
 * get_adapter_status - look if adapter is present in slot
 * @slot: slot to test value
 * @value: status of the adapter
 */
static int
get_adapter_status(struct hotplug_slot *slot, u8 *value)
{
	struct dummy_slot *dslot = slot->private;

	*value = pci_scan_slot(dslot->bus, dslot->devfn);
	slot->info->adapter_status = *value;

	return 0;
}

/**
 * add_slot - add a new "hotplug" slot
 * @dev: a struct pci_dev describing this slot (regardless if
 *	there is actually a device in this slot or not)
 */
static int
add_slot(const struct pci_dev *dev)
{
	struct dummy_slot *dslot;
	struct hotplug_slot *slot;
	int retval = -ENOMEM;

	dslot = kmalloc(sizeof(*dslot), GFP_KERNEL);
	if (!dslot)
		goto error;

	dslot->bus = dev->bus;
	dslot->devfn = dev->devfn;

	dslot->dev = pci_get_slot(dslot->bus, dslot->devfn);

	if (!showall && !dslot->dev) {
		retval = 0;
		goto error_dslot;
	}

	slot = kmalloc(sizeof(*slot), GFP_KERNEL);
	if (!slot)
		goto error_dslot;

	memset(slot, 0, sizeof(*slot));

	slot->info = kmalloc(sizeof(*(slot->info)), GFP_KERNEL);
	if (!slot->info)
		goto error_slot;

	memset(slot->info, 0, sizeof(struct hotplug_slot_info));

	slot->info->max_bus_speed = PCI_SPEED_UNKNOWN;
	slot->info->cur_bus_speed = PCI_SPEED_UNKNOWN;

	slot->name = kmalloc(SLOT_NAME_SIZE, GFP_KERNEL);
	if (!slot->name)
		goto error_info;
	slot->info->power_status = (dslot->dev != NULL);
	slot->info->adapter_status = slot->info->power_status;

	slot->ops = &dummy_hotplug_slot_ops;
	slot->release = &dummy_release;
	slot->private = dslot;

	snprintf(slot->name, SLOT_NAME_SIZE, "%04x:%02x:%02x",
		pci_domain_nr(dev->bus), dslot->bus->number,
		PCI_SLOT(dslot->devfn));

	retval = pci_hp_register(slot);
	if (retval) {
		err("pci_hp_register failed with error %d\n", retval);
		goto error_name;
	}

	dslot->slot = slot;
	list_add(&dslot->node, &slot_list);

	return retval;

error_name:
	kfree(slot->name);
error_info:
	kfree(slot->info);
error_slot:
	kfree(slot);
error_dslot:
	pci_dev_put(dslot->dev);
	kfree(dslot);
error:
	return retval;
}

static int scan_pci_buses(struct list_head *list);

/**
 * scan_pci_bus - add an entry for every slot on this bus
 * @bus: bus to scan
 */
static inline int
scan_pci_bus(struct pci_bus *bus)
{
	int retval;
	struct pci_dev dev;

	dev.bus = bus;
	dev.sysdata = bus->sysdata;

	for (dev.devfn = 0; dev.devfn < 0x100; dev.devfn += 8) {
		retval = add_slot(&dev);
		if (retval)
			break;
	}
	return scan_pci_buses(&bus->children);
}

/**
 * scan_pci_buses - scan this bus and all child buses for slots
 * @list: list of buses to scan
 */
static int
scan_pci_buses(struct list_head *list)
{
	int retval = 0;
	const struct list_head *l;

	list_for_each(l, list) {
		struct list_head *tmp, *next;
		struct pci_bus *b = pci_bus_b(l);
		int i = 0;

		/* scan the list of slots to see if we have already a slot on
		   the new bus registered */
		list_for_each_safe(tmp, next, &slot_list) {
			if ((list_entry(tmp, struct dummy_slot, node))->bus == b) {
				i = 1;
				break;
			}
		}
		if (!i) {
			retval = scan_pci_bus(b);
			if (retval)
				break;
		}
	}
	return retval;
}

/**
 * enable_slot - power on and enable a slot
 * @hotplug_slot: slot to enable
 */
static int
enable_slot(struct hotplug_slot *hotplug_slot)
{
	struct dummy_slot *dslot = hotplug_slot->private;
	int num, result = -ENODEV;

	dbg("%s - physical_slot = %s\n", __FUNCTION__, hotplug_slot->name);

	/* exit if device in slot is already active */
	if (dslot->dev != NULL)
		return 0;

	dslot->dev = pci_get_slot(dslot->bus, dslot->devfn);

	/* Still NULL? Well, then scan for it! */
	if (dslot->dev == NULL) {
		num = pci_scan_slot(dslot->bus, dslot->devfn);
		if (!num) {
			dbg("INFO: enable_slot called on empty slot\n");
			return result;
		}

		pci_bus_add_devices(dslot->bus);

		dslot->dev = pci_get_slot(dslot->bus, dslot->devfn);
		if (dslot->dev == NULL) {
			WARN_ON(1);
			return result;
		}
	}

	if (dslot->dev->subordinate)
		result = scan_pci_bus(dslot->dev->subordinate);
	else
		result = 0;

	return result;
}

/**
 * disable_subordinate - remove hotplug slot entry of all devices on this bus
 * @bus: PCI bus to remove (including all children)
 */
static void
disable_subordinate(struct pci_bus *bus)
{
	struct list_head *tmp, *next;

	list_for_each_safe(tmp, next, &slot_list) {
		struct dummy_slot *dslot = list_entry(tmp, struct dummy_slot, node);

		if (dslot->bus == bus) {
			if (dslot->dev != NULL)
				if (dslot->dev->subordinate)
			/* FIXME: look if this bus can be reached through a
			   different bridge. If yes, don't disable it */
					disable_subordinate(dslot->dev->subordinate);
			/* no need to disable the devices itself, this will be
			   done by pci_remove_bus_device */
			pci_hp_deregister(dslot->slot);
		}
	}
}

/**
 * disable_slot - disable any adapter in this slot
 * @slot: slot to disable
 */
static int
disable_slot(struct hotplug_slot *slot)
{
	struct dummy_slot *dslot = slot->private;
	struct pci_dev *dev = dslot->dev;
	int i;

	dbg("%s - physical_slot = %s\n", __FUNCTION__, slot->name);

	if (!dev) {
		dbg("no device in slot, exiting\n");
		return -ENODEV;
	}

	/* check if this is a PCI bridge and remove devices on
	   sub-buses first */
	if (dev->subordinate)
		disable_subordinate(dev->subordinate);

	dslot->dev = NULL;
	pci_dev_put(dev);

	/* remove the device(s) from the PCI core */
	info("slot %s removed\n", pci_name(dev));

	for (i = 0; i < 8; i++) {
		dev = pci_get_slot(dslot->bus, dslot->devfn + i);
		if (dev)
			pci_remove_bus_device(dev);
	}

	slot->info->power_status = 0;

	return 0;
}

static int __init
dummyphp_init(void)
{
	info(DRIVER_DESC "\n");

	return scan_pci_buses(&pci_root_buses);
}

static void __exit
dummyphp_exit(void)
{
	struct list_head *tmp;
	struct list_head *next;
	struct dummy_slot *dslot;

	list_for_each_safe(tmp, next, &slot_list) {
		dslot = list_entry(tmp, struct dummy_slot, node);
		pci_hp_deregister(dslot->slot);
	}
}

module_init(dummyphp_init);
module_exit(dummyphp_exit);
