/*   Copyright (C) 2004 Microtronix Datacom Ltd

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                */

#ifndef _LINUX_I2C_ALGO_ASICS_H
#define _LINUX_I2C_ALGO_ASICS_H

#include <linux/spinlock.h>
#include <linux/i2c-slave.h>

#define NO_SLAVE_ADDRESSES		8

struct i2c_asics_interface_configuration {
    int used;
    int base_offset;
    int irq;
    int irq_mux;
    int scl;
    int clock;
    uint8_t slave_addresses[NO_SLAVE_ADDRESSES]; /* currently only 7 bit slave address supported */
    char *name;
};

struct iic_asics_hw {
    int base;
    int base_offset;
    int irq;
    int irq_mux;
    /* clock and speed */
    int clock;	/* in KHZ */
    int scl;	/* in KHZ */

    volatile u32 * asics_regs;  /* mapped ASICS registers */
    volatile u32 * intmux_reg;	/* IRQ multiplex register */
    wait_queue_head_t wait;
    int pending;
    int arbitration_lost;

    /* track init state */
    int irq_registered;
    int initialized;
};

#define SLAVE_BUF_SIZE 512

typedef enum {
	I2C_ASICS_SLAVE_OP_NONE,
	I2C_ASICS_SLAVE_OP_READ,
	I2C_ASICS_SLAVE_OP_WRITE
} slave_op_t;

struct i2c_algo_asics_slave_data {
	/* slave lock object */
	spinlock_t      slave_lock;
	
	/* slave write buffer */
	unsigned char * slave_write_ptr;
	unsigned int    slave_write_count;
	unsigned char   slave_write_buffer[SLAVE_BUF_SIZE];
	
	/* slave read buffer */
	unsigned char * slave_read_ptr;
	unsigned int    slave_read_count;
	unsigned char   slave_read_buffer[SLAVE_BUF_SIZE];
	
	/* status flags */
	int block_next_slave_operation;
	slave_op_t last_slave_op;
	uint8_t last_addressed_slave;
};

typedef enum {
	I2C_ASICS_MODE_IDLE,
	I2C_ASICS_MODE_MASTER,
	I2C_ASICS_MODE_SLAVE
} asics_state_t;

struct i2c_algo_asics_data {
	struct iic_asics_hw *hw;
	struct i2c_algo_asics_slave_data *slave_data;
	struct i2c_slave_adapter *slave_adapter;
	struct i2c_asics_interface_configuration *config;
	struct i2c_adapter *i2c_adap;
	
	void (*setreg) (struct iic_asics_hw *hw, int regno, int val);
	int  (*getreg) (struct iic_asics_hw *hw, int regno);
	int  (*wait_for_tdone) (struct iic_asics_hw *hw);
	int  (*configure) (struct i2c_adapter *i2c_adap, int initial);
	
	/* local settings */
	int udelay;
	int mdelay;
	int timeout;
	
	asics_state_t current_state;
};

#define I2C_ASICS_ADAP_MAX	16

int i2c_asics_add_bus(struct i2c_adapter *);
int i2c_asics_del_bus(struct i2c_adapter *);
int i2c_asics_slave_restart_xfer(struct i2c_adapter *);
int i2c_asics_slave_set_address(struct i2c_adapter *i2c_adap, unsigned int index, unsigned int address);
int i2c_asics_handle_slave_request(struct i2c_adapter *);

/* debugging to local console or into buffer? */
#define DEBUG_TO_BUFFER	0

#if DEBUG_TO_BUFFER

void i2c_asics_print(const char *fmt, ...);

#else /* DEBUG_TO_BUFFER */

#ifdef i2c_asics_print
#undef i2c_asics_print
#endif

#define i2c_asics_print	printk

#endif /* DEBUG_TO_BUFFER */


#endif /* _LINUX_I2C_ALGO_ASICS_H */
