// cellsplit
//              This takes a .BMP or .BIN file as input, and outputs a stream of cells,
//              with 16-bit pixel data

// Includes
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>

#include "bmp.h"
#include "rle_util.h"

// Defines

// Globals
unsigned char pixelMap[MAXHEIGHT][MAXWIDTH][3];
RLEINFO rleInfo;

struct bmf_hdr bmfHdr;
struct bmi_hdr bmiHdr;
int width = -1, height = -1;

// Settable args
int cWidth = 16;
int cHeight = 16;
int isRaw = 0;
FILE *inFP;

// Functions
long little2big();
int readBmpHdr(FILE *f), readBmpPixData(FILE *f), readBinPixData(FILE *f, u_char calc_width);
void usage(), splitFrame(void);

// Main program
int main(int argc, char *argv[])
{
    int i;
    char *inFile = 0;
    char *endptr;
    int c;

    //
    // Process args
    //
    while ((c = getopt(argc, argv, "C:rw:h:")) != -1) {
	switch (c) {
	  case 'C':
	      if (optarg != NULL) {
		cWidth = strtol(optarg, &endptr, 10);
		if (cWidth <= 0 || cWidth & 7) {
		    fprintf(stderr,
			    "CELLSPLIT ERROR: bad cell width value %d\n",
			    cWidth);
		    usage();
		}
		if (*endptr != 'x') {
		    fprintf(stderr,
			    "CELLSPLIT ERROR: bad cell size format '%s'\n",
			    &argv[i][2]);
		    usage();
		}
		cHeight = strtol(endptr + 1, &endptr, 10);
		if (cHeight <= 0 || *endptr) {
		    fprintf(stderr,
			    "CELLSPLIT ERROR: bad cell height value %d\n",
			    cHeight);
		    usage();
		}
	      } else {
		  usage();
	      }
	      break;
	  case 'w':
	      if (optarg != NULL) {
		  width = strtol(optarg, NULL, 10);
	      } else {
		  usage();
	      }
	      break;
	  case 'h':
	      if (optarg != NULL) {
		  height = strtol(optarg, NULL, 10);
	      } else {
		  usage();
	      }
	      break;
	  case 'r':
	      isRaw = 1;
	      break;
	}
    }

    if (optind < argc) {
	inFile = argv[optind];
    }

//
// Read Input File
//

// Open it
    if (inFile) {
	if (!(inFP = (FILE *) fopen(inFile, "r"))) {
	    fprintf(stderr, "CELLSPLIT ERROR: can't open file '%s'\n",
		    inFile);
	    exit(-1);
	}
    } else {
	inFP = stdin;
    }
    
// Read it
    if (inFile && !strcmp(inFile + strlen(inFile) - 4, ".bin")) {
	if (readBinPixData(inFP, 1) < 0)
	    exit(-1);
    } else if (isRaw) {
	if (width == -1 || height == -1) {
	    fprintf(stderr, "CELLSPLIT ERROR: no width and height for raw data given\n\n");
	    usage();
	}
	if (readBinPixData(inFP, 0) < 0)
	    exit(-1);
    } else {
	if (readBmpHdr(inFP) < 0)
	    exit(-1);
	if (readBmpPixData(inFP) < 0)
	    exit(-1);
    }

//
// Report parameters
//
    if (!inFile)
	inFile = "STDIN";
    fprintf(stderr, ">>CELLSPLIT:\n");
    fprintf(stderr, "\tInput file: %s\n", inFile);
    fprintf(stderr, "\tFrame size: %dx%d; cell size: %dx%d\n",
	    width, height, cWidth, cHeight);

//
// Output file header
//
    strcpy(rleInfo.name, inFile);
    rleInfo.fWidth = width;
    rleInfo.fHeight = height;
    rleInfo.cWidth = cWidth;
    rleInfo.cHeight = cHeight;
    putRLEHdr(rleInfo);
//
// Output cell data
//
    splitFrame();
    return 0;
}

//
// Split a frame into dividing it into cells; output each cell.  Don't pad!
void splitFrame(void)
{
    int x, y, xx, yy;
    int cw, ch;
    unsigned char *pp;
    char *lbp;
    char linebuf[1 + MAXWIDTH * 5];
    char *hexdata = "0123456789abcdef";

// Copy each cell to its own array, then output
    for (y = 0; y < height; y += cHeight) {
	ch = (height - y) < cHeight ? height - y : cHeight;
	for (x = 0; x < width; x += cWidth) {
	    cw = (width - x) < cWidth ? width - x : cWidth;
	    printf(">>CELLDATA %d %d\n", cw, ch);
	    for (yy = 0; yy < ch; yy++) {
		lbp = linebuf;
		for (xx = 0; xx < cw; xx++) {
		    pp = pixelMap[yy + y][xx + x];
		    if (xx)
			*lbp++ = ' ';
/*
					pixel = (pp[RED]   << 11) +
							(pp[GREEN] << 5 ) +
							(pp[BLUE]       );
					*lbp++ = hexdata[(pixel>>12) & 0xf];
					*lbp++ = hexdata[(pixel>>8 ) & 0xf];
					*lbp++ = hexdata[(pixel>>4 ) & 0xf];
					*lbp++ = hexdata[(pixel    ) & 0xf];
*/

		    *lbp++ = hexdata[pp[RED] >> 1];
		    *lbp++ =
			hexdata[((pp[RED] & 1) << 3) + (pp[GREEN] >> 3)];
		    *lbp++ =
			hexdata[((pp[GREEN] & 7) << 1) + (pp[BLUE] >> 4)];
		    *lbp++ = hexdata[pp[BLUE] & 0xf];
		}
		*lbp++ = 0;
		puts(linebuf);
	    }
	    printf("END CELLDATA\n");
	}
    }
}


/*======================================================
 * File Utilities:  BMP and BIN file readers and writers
 *======================================================
 */

int readBmpHdr(FILE * fp)
{
// BMF Header
    if (fread(&bmfHdr, BMFHDRSIZE, 1, fp) != 1) {
	fprintf(stderr, "CELLSPLIT ERROR: couldn't read BMFHDR\n");
	return (-1);
    }
    if (bcmp(bmfHdr.bfType, defBmfHdr.bfType, 2)) {
	fprintf(stderr,
		"CELLSPLIT ERROR: incorrect BFTYPE field: '%c%c'\n",
		bmfHdr.bfType[0], bmfHdr.bfType[1]);
	return (-1);
    }
    if (bcmp(bmfHdr.bfOffBits, defBmfHdr.bfOffBits, 4)) {
	fprintf(stderr,
		"CELLSPLIT ERROR: incorrect BFOFFBITS field: 0x%02x%02x%02x%02x\n",
		bmfHdr.bfOffBits[3], bmfHdr.bfOffBits[2],
		bmfHdr.bfOffBits[1], bmfHdr.bfOffBits[0]);
	return (-1);
    }
// BMI Header
    if (fread(&bmiHdr, BMIHDRSIZE, 1, fp) != 1) {
	fprintf(stderr, "CELLSPLIT ERROR: couldn't read BMIHDR\n");
	return (-1);
    }
    if (bcmp(bmiHdr.biSize, defBmiHdr.biSize, 4)) {
	fprintf(stderr,
		"CELLSPLIT ERROR: incorrect BISIZE field: 0x%02x%02x%02x%02x\n",
		bmiHdr.biSize[3], bmiHdr.biSize[2], bmiHdr.biSize[1],
		bmiHdr.biSize[0]);
	return (-1);
    }
    width = little2big(bmiHdr.biWidth, 4);
    height = little2big(bmiHdr.biHeight, 4);
    if (width <= 0 || width > MAXWIDTH /*||  (width&0x7) */  ||
	height <= 0 || height > MAXHEIGHT /*|| (height&0x7) */ ) {
	fprintf(stderr, "CELLSPLIT ERROR: invalid image size: %d x %d\n",
		width, height);
	return (-1);
    }
    if (bcmp(bmiHdr.biPlanes, defBmiHdr.biPlanes, 2)) {
	fprintf(stderr,
		"CELLSPLIT ERROR: invalid BIPLANES field: 0x%02x%02x\n",
		bmiHdr.biPlanes[1], bmiHdr.biPlanes[0]);
	return (-1);
    }
// OK!
    return (0);
}

long little2big(unsigned char *cp, int size)
{
    int i;
    long v = 0;

    for (i = size - 1; i >= 0; i--)
	v = (v << 8) + cp[i];
    return (v);
}

int readBmpPixData(FILE * fp)
{
    int x, y;
    unsigned char *pp;

    for (y = height - 1; y >= 0; y--)
	for (x = 0; x < width; x++) {
	    pp = pixelMap[y][x];
	    if (fread(pp, 3, 1, fp) != 1) {
		fprintf(stderr,
			"CELLSPLIT ERROR: couldn't read pixel data at (%d,%d)\n",
			x, y);
		return (-1);
	    }
	    pp[RED] >>= 3;
	    pp[GREEN] >>= 2;
	    pp[BLUE] >>= 3;
	}
    return 0;
}

int readBinPixData(FILE * fp, u_char calc_width)
{
    int c1, c2, x, y;
    static unsigned short inBuf[MAXWIDTH * MAXHEIGHT];
    int inSize;

    for (inSize = 0;; inSize++) {
	c1 = getc(fp);
	c2 = getc(fp);
	if (c2 == EOF) {
	    break;
	}
#if __BYTE_ORDER == __BIG_ENDIAN
	c1 = (c1 << 8) | c2;
#else
	c1 += (c2 << 8);
#endif
	inBuf[inSize] = c1;
    }

    if (calc_width) {
	switch (inSize) {
	  case 640 * 480:
	      width = 640;
	      height = 480;
	      break;
	  case 800 * 600:
	      width = 800;
	      height = 600;
	      break;
	  case 1024 * 768:
	      width = 1024;
	      height = 768;
	      break;
	  case 1280 * 1024:
	      width = 1280;
	      height = 1024;
	      break;
	  case 1600 * 1200:
	      width = 1600;
	      height = 1200;
	      break;
	  default:
	      fprintf(stderr,
		      "\nCELLSPLIT ERROR: unrecognized resolution: %d pixels\n",
		      inSize);
	      return (-1);
	      break;
	}
    }
   
    inSize = 0;
    if (isRaw) {
	int i;
	
	/* hextile continuous */
	for (y = 0; y < height; y+=cHeight) {
	    for (x = 0; x < width; x+=cWidth) {
		for (i=0; i<cWidth*cHeight;i++) {
		    c1 = inBuf[inSize++];
		    pixelMap[y + (i / cWidth)][x + (i % cWidth)][RED] = (c1 & 0xF800) >> 11;
		    pixelMap[y + (i / cWidth)][x + (i % cWidth)][GREEN] = (c1 & 0x07E0) >> 5;
		    pixelMap[y + (i / cWidth)][x + (i % cWidth)][BLUE] = (c1 & 0x001F);
		}
	    }
	}
    } else {
	/* linear */
	for (y = 0; y < height; y++) {
	    for (x = 0; x < width; x++) {
		c1 = inBuf[inSize++];
		pixelMap[y][x][RED] = (c1 & 0x7C00) >> 9;
		pixelMap[y][x][GREEN] = (c1 & 0x03E0) >> 4;
		pixelMap[y][x][BLUE] = (c1 & 0x001F) << 1;
	    }
	}
    }
    return (0);
}

void usage(void)
{
    fprintf(stderr,
	    "usage: cellsplit [<infile>] [-r -w <width> -h <height>] [-C <cell_width>x<cell_height>]\n"
	    "\t<infile> must be valid .BMP or .BIN file.  If <infile> not specified,\n\t\tstandard input will be used.\n"
	    "\t***HEIGHT AND WIDTH OF INPUT MUST BE DIVISIBLE BY EIGHT***\n"
	    "\t-C specifies cell (or tile) size\n"
	    "\t-r indicates the input data is in RAW (16bpp RGB 565) format\n"
	    "\t   in this case width and height options are required\n"
	    "\t  Example: -c64x32 specifies a 16x16 cell. Default is 16x16\n");    
    exit(-1);
}
