// lrle_engine
//              This implements a 16-bit LRLE encoder, with 2d processing.
//              Input is cell data from rle_cellsplit, output is RLE code.

// Includes
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>

#include "bmp.h"
#include "rle_util.h"
#include "ccr.h"

// Defines
// RLE commands
#define RLE_COPY	1
#define RLE_WORD	2
#define RLE_END		3

// Globals
RLEINFO rleInfo;

unsigned char cellMap[MAXWIDTH][MAXWIDTH][3];

// Hard-wired args
// settable args
int rLimit = 31;
int cLimit = 31;
int rMargin_rb = 1;
int rMargin_g = 1;
int cMargin_rb = 1;
int cMargin_g = 1;
int use2d = 1;
FILE *inFP;

// Functions
void usage();
void encodeCell(), encodeFrame(), genRLE();

// Main program
int main(int argc, char *argv[])
{
    char *inFile = 0;
    char use_ccr = 0;
    unsigned long ccr;
    int c;
    
    memset(&rleInfo, 0, sizeof(rleInfo));
    
    //
    // Process args
    //
    while ((c = getopt(argc, argv, "M:L:c:12")) != -1) {
	char *endptr;
	switch (c) {
	  case '1':
	      use2d = 0;
	      break;
	  case '2':
	      use2d = 1;
	      break;
	  case 'M':
	      if (optarg != NULL) {
		  rMargin_rb = strtol(optarg, &endptr, 10);
		  if (rMargin_rb < 0 || *endptr != ',') {
		      fprintf(stderr, "ERROR: bad run margin value '%s'\n", optarg);
		      usage();
		  }
		  rMargin_g = strtol(endptr + 1, NULL, 10);
		  if (rMargin_g < 0) {
		      fprintf(stderr, "ERROR: bad run margin value '%s'\n", optarg);
		      usage();
		  }
	      } else {
		  usage();
	      }
	      break;
	  case 'C':
	      if (optarg != NULL) {
		  cMargin_rb = strtol(optarg, &endptr, 10);
		  if (cMargin_rb < 0 || *endptr != ',') {
		      fprintf(stderr, "ERROR: bad copy margin value '%s'\n", optarg);
		      usage();
		  }
		  cMargin_g = strtol(endptr + 1, NULL, 10);
		  if (cMargin_g < 0) {
		      fprintf(stderr, "ERROR: bad copy margin value '%s'\n", optarg);
		      usage();
		  }	  
	      } else {
		  usage();
	      }
	      break;
	  case 'L':
	      if (optarg != NULL) {
		  rLimit = cLimit = strtol(optarg, NULL, 10);
		  if (rLimit < 0 || rLimit > 255) {
		      fprintf(stderr, "ERROR: bad run limit value '%s'\n", optarg);
		      usage();
		  }
	      } else {
		  usage();
	      }
	      break;
	  case 'c':
	      if (optarg != NULL) {
		  ccr = strtoul(optarg, NULL, 16);
		  use_ccr = 1;
	      } else {
		  usage();
	      }
	      break;
	}
    }

    if (optind < argc) {
	inFile = argv[optind];
    }

    // override options according to the CCR register
    if (use_ccr) {
	use2d = (ccr & CCR_LRLE_LINECOPY_MASK) >> CCR_LRLE_LINECOPY_SHIFT;
	rMargin_rb = (ccr & CCR_LRLE_RMARGIN_RB_MASK) >> CCR_LRLE_RMARGIN_RB_SHIFT;
	rMargin_g = (ccr & CCR_LRLE_RMARGIN_G_MASK) >> CCR_LRLE_RMARGIN_G_SHIFT;
	cMargin_rb = (ccr & CCR_LRLE_CMARGIN_RB_MASK) >> CCR_LRLE_CMARGIN_RB_SHIFT;
	cMargin_g = (ccr & CCR_LRLE_CMARGIN_G_MASK) >> CCR_LRLE_CMARGIN_G_SHIFT;
	rLimit = cLimit = (ccr & CCR_LRLE_RUNLIMIT_MASK) >> CCR_LRLE_RUNLIMIT_SHIFT;
    }

//
// Read Input File
//

// Open it
    if (inFile) {
	if (!(inFP = (FILE *) fopen(inFile, "r"))) {
	    fprintf(stderr, "ERROR: can't open file '%s'\n", inFile);
	    exit(-1);
	}
    } else
	inFP = stdin;

//
// Read file header
//
    if (getRLEHdr(inFP, &rleInfo) < 0)
	exit(-1);

//
// Report parameters
//
    fprintf(stderr, ">>LRLE_ENGINE:\n");
    if (inFile) {
	fprintf(stderr, "\tInput file: %s\n", inFile);
	fprintf(stderr, "\tOriginal Input file: %s\n", rleInfo.name);
	fprintf(stderr, "\tFrame size: %dx%d; cell size: %dx%d\n",
		rleInfo.fWidth, rleInfo.cHeight, rleInfo.cWidth,
		rleInfo.cHeight);
    }
    fprintf(stderr,
	    "\tLRLE parameters: rMargin_rb=%d; rMargin_g=%d\n"
	    "\t                 cMargin_rb=%d; cMargin_g=%d, limit=%d; 2d=%s\n",
	    rMargin_rb, rMargin_g, cMargin_rb, cMargin_g, rLimit, (use2d ? "ON" : "OFF"));

    putRLEHdr(rleInfo);

//
// Generate RLE data
//
    encodeFrame(inFP);
    return 0;
}

/*
 *===================
 * Encoding Functions
 *===================
 */

//
// Encode a frame by dividing it into cells and encoding each cell.
// Generate RLE code in rleCode[] array.
void encodeFrame(FILE * inFP)
{
    int rv;
    int x, y;
    int cw, ch;
    char s[5 * MAXWIDTH], *sp, hd;
    unsigned long pixel;

    while ((rv = getBlockHdr(inFP, sizeof(s), s, "CELLDATA"))) {
	// parse cell header
	if (rv < 0) {
	    fprintf(stderr,
		    "LRLE_ENGINE ERROR: unexpected block type '%s'\n", s);
	    exit(-1);
	}
	if (sscanf(s, " %d %d", &cw, &ch) != 2) {
	    fprintf(stderr,
		    "LRLE_ENGINE ERROR: bad celldata header '%s'\n", s);
	    exit(-1);
	}
	// load pixel data into cellMap
	for (y = 0; y < ch; y++) {
	    // get a scanline
	    if (!(rv = getBlockData(inFP, sizeof(s), s))) {
		fprintf(stderr,
			"LRLE_ENGINE ERROR: unexpected EOF (saw only %d out of %d lines in cell\n",
			y, ch);
		exit(-1);
	    }
	    if (rv < 0) {
		fprintf(stderr,
			"LRLE_ENGINE ERROR: unexpected END (saw only %d out of %d lines in cell\n",
			y, ch);
		exit(-1);
	    }
	    // extract pixel values from scanline
	    for (x = 0, sp = s; x < cw; x++) {
		while (*sp == ' ')	// skip whitespace
		    sp++;
		if (!*sp) {	// unexpected EOL?
		    fprintf(stderr,
			    "LRLE_ENGINE ERROR: unexpected EOL (saw only %d out of %d pixels in line\n",
			    x, cw);
		    exit(-1);
		}
		// cheesy home-brew input routine because strtok() and sscanf()
		// were too slow
		pixel = 0;
		while (1) {
		    hd = toupper(*sp++);
		    if (hd >= '0' && hd <= '9')
			pixel = (pixel << 4) + hd - '0';
		    else if (hd >= 'A' && hd <= 'F')
			pixel = (pixel << 4) + 10 + hd - 'A';
		    else if (hd == 0 || hd == ' ') {
			sp--;
			break;
		    } else {
			fprintf(stderr,
				"LRLE_ENGINE ERROR: bad pixel value\n");
			exit(-1);
		    }

		    if (pixel >> 16) {
			fprintf(stderr,
				"LRLE_ENGINE ERROR: pixel value too large\n");
			exit(-1);
		    }
		}
		cellMap[y][x][BLUE] = pixel & 0x1F;
		pixel >>= 5;
		cellMap[y][x][GREEN] = pixel & 0x3F;
		pixel >>= 6;
		cellMap[y][x][RED] = pixel & 0x1F;
	    }
	}
	if (getBlockData(inFP, sizeof(s), s) >= 0 || strcmp(s, "CELLDATA")) {
	    fprintf(stderr,
		    "LRLE_ENGINE ERROR: bad or missing END to CELLDATA\n");
	    exit(-1);
	}
	// Encode the cell
	encodeCell(cw, ch);
    }
}

//
// Create the RLE code for a single cell.  Pixel data is found in 
// cellMap[0..ch-1][0..cw-1]
//
void encodeCell(int lastX, int lastY)
{
    int x, y, c;
    int outX, outY;
    unsigned char *pix, *ppix;
    static unsigned char prevLine[MAXWIDTH][3];
    int endLocalRun, endLineCopy;
    int run, beginRun, endRun, continueRun;
    int minV[3], maxV[3], runV[3];

// Initialize
    endLocalRun = endLineCopy = -1;
    run = 0;
// print cell header
    printf(">>CELLRLE %d %d\n", lastX, lastY);

// Loop through cell pixel data
    for (y = 0; y <= lastY; y++) {
	for (x = 0; x < lastX; x++) {
	    // set up some shortcuts
	    pix = cellMap[y][x];
	    ppix = prevLine[x];
	    // Do we need to end our local run?
	    if (endLocalRun == 0 && ((y == lastY) || (run == rLimit) ||
				  (pix[RED] > minV[RED] + rMargin_rb) ||
				  (pix[RED] < maxV[RED] - rMargin_rb) ||
				  (pix[GREEN] > minV[GREEN] + 2 * rMargin_g) ||
				  (pix[GREEN] < maxV[GREEN] - 2 * rMargin_g) ||
				  (pix[BLUE] > minV[BLUE] + rMargin_rb) || 
				  (pix[BLUE] < maxV[BLUE] - rMargin_rb))) {
		endLocalRun = run + 1;
	    }
	    // Do we need to end our line copy?
	    if (endLineCopy == 0 && ((y == lastY) || (run == cLimit) ||
				     (abs(pix[RED]-ppix[RED])>cMargin_rb) ||
				     (abs(pix[GREEN]-ppix[GREEN])>2*cMargin_g) ||
				     (abs(pix[BLUE] - ppix[BLUE])>cMargin_rb))) {
		endLineCopy = run + 1;
	    }
	    // Figure out what we're doing
	    endRun = endLocalRun && endLineCopy && (x || y);
	    beginRun = (endRun && (y < lastY)) || !(x || y);
	    continueRun = !endRun && !beginRun;

	    // Do end-of-run processing
	    if (endRun) {
		// print out completed run
		if (endLineCopy >= endLocalRun) {	// use line copy
		    genRLE(RLE_COPY, run);
		    outX = x;
		    outY = y;
		} else {	// use local run
		    for (c = 0; c < 3; c++)
			runV[c] = (minV[c] + maxV[c] + 1) / 2;
		    genRLE(RLE_WORD, run, runV);
		    // Copy new pixel data to prevLine: very important!!!
		    while (run-- >= 0) {
			for (c = 0; c < 3; c++)
			    prevLine[outX][c] = runV[c];
			if (++outX == lastX) {
			    outX = 0;
			    outY++;
			}
		    }
		    outX = x;
		}
	    }
	    // Initialize new run
	    if (beginRun) {
		outX = x;
		outY = y;
		run = 0;
		endLocalRun = 0;
		endLineCopy = ((y == 0) || !use2d ||
		    (abs(pix[0] - ppix[0]) > cMargin_rb) ||
		    (abs(pix[1] - ppix[1]) > 2 * cMargin_g) ||
		    (abs(pix[2] - ppix[2]) > cMargin_rb)) ? -1 : 0;
		for (c = 0; c < 3; c++)
		    minV[c] = maxV[c] = pix[c];
	    }
	    // Continue run
	    if (continueRun) {
		run++;
		if (!endLocalRun)	// update local run if necessary
		    for (c = 0; c < 3; c++) {
			if (pix[c] < minV[c])
			    minV[c] = pix[c];
			else if (pix[c] > maxV[c])
			    maxV[c] = pix[c];
		    }
	    }
	    // Get out of here if we're at end of cell
	    if (y == lastY)
		break;
	}
    }
// Error check
    if (outX) {
	fprintf(stderr, "ERROR: cell %dx%d; wrote through column %d\n",
		lastX, lastY, outX);
	fprintf(stderr, "run=%d, endLocalRun=%d, endLineCopy=%d\n",
		run, endLocalRun, endLineCopy);
	exit(-1);
    }
// Done!
    genRLE(RLE_END);
    return;
}

// Generate RLE code for the given command.
// Returns the size of the code generated.

void genRLE(int cmd, int run, int v[])
{
    switch (cmd) {
    case RLE_COPY:
	printf("LC %d\n", run);
	break;
    case RLE_WORD:
	putRun(0, 1, run, v[RED], v[GREEN], v[BLUE]);
	break;
    case RLE_END:		// supress trailing line copies
	puts("END CELLRLE");
	break;
    default:
	fprintf(stderr, "ERROR: unknown RLE command %d\n", cmd);
	exit(-1);
    }
    return;
}

void usage()
{
    fprintf(stderr,
	    "usage: lrle_engine [<infile>] [-c <CCR>] [-M <run margins>] [-C <copy margins>] -L<limit>] [-1 | -2]\n"
	    "\t<infile> must be cell data as generated by rle_cellsplit\n"
	    "\tIf <infile> is not specified, standard input will be used.\n"
	    "\t<run margins> two comma-separated numbers for red/blue and green, range [0,1] (default 1)\n"
	    "\t<copy margins> two comma-separated numbers for red/blue and green, range [0,1] (default 1)\n"
	    "\t<margin> must be in the range [0,9] (default 1)\n"
	    "\t<limit> must be in the range [0,31] (default 30)\n"	    
	    "\t-1 = enable 1d processing only\n"
	    "\t-2 = enable 2d processing (default)\n"
	    "\t<CCR> = 32bit hex value for the CCR register, overrides all other settings\n"
	    "\tStandard output is the raw RLE file\n");

    exit(-1);
}
