// rle_decode
//		This decodes RLE input (all formats) and generates a BMP file for
//		output.
//
//		One option is provided:  -p turns on "pretty" output, which means that
//		instead of a BMP file, a human-readable description of the RLE is
//		printed to standard out.  Handy for debugging!!!

// Includes
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>

#include "bmp.h"

// Defines
#define MAXHEIGHT	1200
#define MAXWIDTH	1600
#define MIN(a,b)	((a)<(b)?(a):(b))
#define MAX(a,b)	((a)>(b)?(a):(b))

// Globals
unsigned char	pixelMap[MAXHEIGHT][MAXWIDTH][3];
unsigned short	pixelArray[MAXHEIGHT][MAXWIDTH];
unsigned char	rleCode[MAXHEIGHT*MAXWIDTH*3];
int	rleSize = 0;
int pretty = 0;

struct bmf_hdr	bmfHdr;
struct bmi_hdr	bmiHdr;
int	width = 0;
int height = 0;
int depth = 0;
int isGrey = 0;
int	isCompact = 0;
char format[80];
int c;
long notify = -1;
long notifycount = 0;
int cellCount=1;

// settable args
int cWidth = 16;
int cHeight = 16;
FILE *inFP;

// Functions
int		readRLEData();
void	usage(), outputBMP(), transColor();
void	decodeRLEFrame(), genGreyTable(), genColorTable(), genPixelMap();
unsigned char *decodeRLECell();
unsigned char *decodeRLEMap(unsigned char *p, int cx, int cy, int lastX, int lastY, int gdepth);


// Main program
int
main(int argc, char *argv[]) {
	int	i;
	char *inFile = 0;

//
// Process args
//
	while ((c = getopt(argc, argv, "pn:")) != -1) {
	    switch (c) {
	      case 'p':
		  pretty = 1;
		  break;
	      case 'n':
		  if (optarg != NULL) {
		      notify = strtol(optarg, NULL, 16);
		  } else {
		      usage();
		  }
		  break;
	      default:
		  fprintf(stderr,"ERROR: unknown dash argument '%s'\n",
			  argv[i]);
		  usage();
		  break;
	    }
	}

	if (optind < argc) {
	    inFile = argv[optind];
	}
	
//	
// Read Input File
//

// Open it
	if (inFile) {
		if (!(inFP = (FILE *) fopen(inFile, "r"))) {
			fprintf(stderr, "ERROR: can't open file '%s'\n", inFile);
			exit(-1);
		  }
	  }
	else
		inFP = stdin;

// Read it
	if (readRLEData(inFP) < 0)
		exit(-1);

// Create 24-bit pixelMap
	if (isGrey)
		genGreyTable(depth);
	else {
		genColorTable(depth);
		genGreyTable(depth == 7 ? 4 : 6);
	  }
// Decode
	decodeRLEFrame();

	if (depth != 1 && depth != 2) {
	    genPixelMap();
	}

// Generate output bitmap
	if (!pretty)
		outputBMP();

	return(0);
  }

void
decodeRLEFrame() {
	int cx, cy, lastX, lastY;
	unsigned char *rlep = rleCode;
	unsigned char *lastp = &rleCode[rleSize];

	if (pretty) 		// Header + start of first cell
		printf("RLESIZE = %d\n", rleSize);
	cx = cy = 0;
	
	while (rlep != lastp) {
	// calculate bounds of cell
		lastX = MIN(cx + cWidth, width);
		lastY = MIN(cy + cHeight, height);

	// decode cell
		if (pretty)
			printf("[%-6d]CELL AT [%d,%d]\n  LINE 0:\n", (rlep-rleCode),cx,cy);

		if (depth == 1 || depth == 2) {
		    rlep = decodeRLEMap(rlep, cx, cy, lastX, lastY, depth);
		} else {
		    rlep = decodeRLECell(rlep, cx, cy, lastX, lastY);
		}

		if (pretty) {
			printf(">>END OF CELL\n");
		}
	        cellCount++;

	// determine position of next cell
		cx = lastX;
		if (cx == width) {
			cx = 0;
			cy = lastY;
			if (cy == height)
				break;
		  }
	  }
  }

inline unsigned char
getByte(unsigned char **p)
{
    FILE *out;
    
    out=pretty?stdout:stderr;
    if (notify != -1 && ++notifycount == notify) {
	fprintf(out, "NOTIFY COUNT 0x%lx REACHED in cell %d\n", notify, cellCount);
    }

    return *(*p)++;
}

unsigned char *
decodeRLEMap(unsigned char *p, int cx, int cy, int lastX, int lastY, int gdepth)
{
    int w = lastX - cx;
    int steps = 8 / depth;
    int lineBytes = w / steps;
    int lineRest  = w % steps;
    int segment;
    int k, code;
    int mask  = (1 << depth) - 1;

    while (cy < lastY) {
	for (segment = 0; segment < lineBytes; segment++) {
	    code = getByte(&p);
	    for (k = steps - 1; k >= 0; k--) {
		transColor((code & mask) | 0x8000,
			   pixelMap[cy][cx + (segment * steps) + k]);
		code >>= gdepth;
	    }
	}

	if (lineRest != 0) {
	    code = getByte(&p);
	    for (k = lineRest - 1; k >= 0; k--) {
		transColor((code & mask) | 0x8000,
			   pixelMap[cy][cx + (lineBytes * steps) + k]);
		code >>= gdepth;
	    }    
	}
	cy++;
    }
    
    return p;
}

unsigned char *
decodeRLECell(unsigned char *p, int cx, int cy, int lastX, int lastY) {
	int j;
	unsigned char color[3];
	int code;
	int	dx, dy;
	int c, run, copy, repeat;
	unsigned short prevLine[64], grey;
	unsigned short *pa;

	dx = dy = 0;
	c = -1;
	run = copy = repeat = 0;
	grey = (isGrey ? 0x8000 : 0);
	pa = &pixelArray[cy][cx];

	while (cy + dy < lastY) {
	    code = getByte(&p);
	// look at code, set up for execution
		repeat = 0;
		copy = ((code & 0xE0) == 0xE0);		// Line copy always the same format!
		if (copy) {					
			if (code == 0xFF)
			    code = (code << 8) + (run = getByte(&p));
			else
				run = (code & 0x1f);
		  }
		else if (isCompact) {				// Decode compact 3/4 bit format
			if (depth <= 3) {					// 3-bit mode
				c = (code & 7);
				run = code >> 3;
			  }
			else {								// 4-bit mode
				c = (code & 15);
				run = code >> 4;
			  }
		  }
		else if (code & 0x80) {				// Grey compression or repeat
			if (code & 0x40) {					// repeat
				run = (code & 0x1f);
				repeat = 1;
			  }
			else {								// grey pixel
				c = (code & 0x3f);
				run = 0;
				grey = 0x8000;
			  }
		  }
//XXX Experimental compact mode, used for 5-bit mode
		else if (depth < 7) {
			c = (code & 0x7C)>>(7-depth);
			run = (code & ((1<<(7-depth))-1));
			if (run == (1<<(7-depth))-1)
			    run = getByte(&p);
		  }
		else {
			if (depth == 15)
			    code = (code << 8) + getByte(&p);
			c = code;
			run = 0;
			grey = 0;
		  }

	// Do pretty printing
		if (pretty) {
			printf("\tX=%-2d ", dx);
			if (copy)
				printf("   LINE COPY 0x%02x (%d)\n", code, run+1);
			else if (repeat)
				printf("   RUN 0x%02x (%d)\n", code, run+1);
			else if (isCompact)
				printf("PIX RUN 0x%02x; [%2d] (%d)\n", code, c, run+1);
			else if (grey)
				printf("GREY 0x%02x; [%3d]\n", code, (c-grey));
			else {
				transColor(c, color);
				if (depth == 15)
					printf("PIX 0x%04x; [%3d,%3d,%3d]\n", 
						code, color[RED], color[GREEN], color[BLUE]);
				else 
					printf("PIX 0x%02x; [%3d,%3d,%3d]\n", 
						code, color[RED], color[GREEN], color[BLUE]);
			  }
		  }

	// Execute the run
		c |= grey;
		for (j = 0; j <= run; j++) {
			if (!copy)
				prevLine[dx] = pa[dx] = c;
			else
				pa[dx] = prevLine[dx];
			++dx;
			if (cx+dx == lastX) {
				dx = 0;
				dy ++;
				pa = &pixelArray[cy+dy][cx];
				if (pretty)
					printf("  LINE %d:\n", dy);
			  }
		  }
	  }
	return(p);
  }

unsigned char greyTable[64];

void
genGreyTable(int depth) {
	int i;

	for (i = 0; i < (1 << depth); i++)
		switch(depth) {
			case 1:		greyTable[i] = (i*255);			break;
			case 2:		greyTable[i] = (i* 85);			break;
			case 3:		greyTable[i] = (i* 73) / 2;		break;
			case 4:		greyTable[i] = (i* 17);			break;
			case 5:		greyTable[i] = (i* 33) / 4;		break;
			case 6:		greyTable[i] = (i* 65) / 16;	break;
			default:	greyTable[i] = 0;				break;
		  }
  }

unsigned char	c16table[16][3] = {
	{0,  0,  0  },	// black
	{0,  0,  127},	// dark red
	{0,  127,0  },	// dark green
	{0,  127,127},	// dark yellow
	{127,0,  0  },	// dark blue
	{127,0,  127},	// dark magenta
	{127,127,0  },	// dark cyan
	{127,127,127},	// dark grey
	{192,192,192},	// light grey
	{0,  0,  255},	// light red
	{0,  255,0  },	// light green
	{0,  255,255},	// light yellow
	{255,0,  0  },	// light blue
	{255,0,  255}, 	// light magenta
	{255,255,0  },	// light cyan
	{255,255,255},	// white
  };

unsigned char	c32table[32][3] = {
	{0,  0,  0  },  {127,0,  0  },  {255,0,  0  },
	{0,  127,0  },  {127,127,0  },  {255,127,0  },
	{0,  255,0  },  {127,255,0  },  {255,255,0  },
	{0,  0,  127},  {127,0,  127},  {255,0,  127},
	{0,  127,127},  {127,127,127},  {255,127,127},
	{0,  255,127},  {127,255,127},  {255,255,127},
	{0,  0,  255},  {127,0,  255},  {255,0,  255},
	{0,  127,255},  {127,127,255},  {255,127,255},
	{0,  255,255},  {127,255,255},  {255,255,255},

	{0,  0,  0  },  { 64, 64, 64},  {128,128,128},
	{192,192,192},  {255,255,255}
  };

unsigned char	colorTable[1<<15][3];

void
genColorTable(int depth) {
	int i;
	static unsigned char decode7_5[5] = { 0, 64, 128, 192, 255 };

	for (i = 0; i < (1 << depth); i++) {
		switch(depth) {
			case 4:
				colorTable[i][RED]		= c16table[i][RED];
				colorTable[i][GREEN]	= c16table[i][GREEN];
				colorTable[i][BLUE]		= c16table[i][BLUE];
			  break;
			case 5:
				colorTable[i][RED]		= c32table[i][RED];
				colorTable[i][GREEN]	= c32table[i][GREEN];
				colorTable[i][BLUE]		= c32table[i][BLUE];
			  break;
			case 7:
/*XXX DECODER FOR OLD 7-BIT MODE
				colorTable[i][RED]		= ((i & 0x60) << 1) + ((i & 0x60) >> 1);
				colorTable[i][GREEN] 	= ((i & 0x1C) << 3) + ((i & 0x18)     );
				colorTable[i][BLUE]		= ((i & 0x03) << 6) + ((i & 0x03) << 4);
*/
				if (i >= 125)	// shouldn't ever be true
					colorTable[i][RED] = colorTable[i][GREEN] = 
						colorTable[i][BLUE] = 0xff;
				else {
					colorTable[i][RED]		= decode7_5[i / 25];
					colorTable[i][GREEN]	= decode7_5[(i / 5) % 5];
					colorTable[i][BLUE]		= decode7_5[i % 5];
				  }
			  break;
			case 15:
				colorTable[i][RED]		= (((i & 0x7C00) >> 10) * 33) / 4;
				colorTable[i][GREEN]	= (((i & 0x03E0) >> 5 ) * 33) / 4;
				colorTable[i][BLUE]		= (((i & 0x001F)      ) * 33) / 4;
			  break;
			default:
				fprintf(stderr, "ERROR: color depth '%d' unsupported\n", depth);
				exit(-1);
			  break;
		  }
	  }
  }

void
transColor(int code, unsigned char *color) {
    if (code & 0x8000) { 		// grey
	color[RED] = color[GREEN] = color[BLUE] = greyTable[code & 0x7fff];
    } else {
		color[RED]		= colorTable[code][RED];
		color[GREEN]	= colorTable[code][GREEN];
		color[BLUE]		= colorTable[code][BLUE];
	  }
  }

void
genPixelMap() {
	int	x, y;
	unsigned short *pa;

	for (y = 0; y < height; y++) {
		pa = pixelArray[y];
		for (x = 0; x < width; x++)
			transColor(*pa++, pixelMap[y][x]);
	  }
  }

/*======================================================
 * File Utilities:  BMP and BIN file readers and writers
 *======================================================
 */
int
readRLEData(FILE *fp) {
	int i, c;
	char	mode, format[80];

	for (i = 0; i < 80; i++) {
		int in = getc(fp);
		if (in == EOF) {
			fprintf(stderr, "ERROR: didn't find format string\n");
			exit(-1);
		  }
		format[i] = in;
		if (format[i] == 10) {
			format[i] = 0;
			break;
		  }
	  }

	if (sscanf(format, "%d %d %c%d", &width, &height, &mode, &depth) != 4) {
		fprintf(stderr, "ERROR: missing or invalid format string in RLE stream\n");
		exit(-1);
	  }
	if (width <= 0 || width > MAXWIDTH || height <= 0 || height > MAXHEIGHT) {
		fprintf(stderr, "ERROR: height or width out of range\n");
		exit(-1);
	  }
	if ((toupper(mode) != 'G') && (toupper(mode) != 'C')) {
		fprintf(stderr, "ERROR: invalid mode '%c'\n", mode) ;
		exit(-1);
	  }

	isGrey = (toupper(mode) == 'G');
	isCompact = (toupper(mode) == mode);

	if (isGrey) {
		if (depth < 1 || depth > 6) {
			fprintf(stderr, "ERROR: unsupported greyscale depth '%d'\n", depth);
			exit(-1);
		  }
	  }
	else {
		if (depth != 4 && depth != 5 && depth != 7 && depth != 15) {
			fprintf(stderr, "ERROR: unsupported color depth '%d'\n", depth);
			exit(-1);
		  }
	  }
	fprintf((pretty ? stdout : stderr),
		"DECODE_RLE7: Input %c%d\n", mode, depth);
	rleSize = 0;
	while ((c=getc(fp)) != EOF)
		rleCode[rleSize++] = c;
	return(0);
  }

void
outputBMP() {
	int imageSize, y;

// Write the file
	bcopy(&defBmfHdr, &bmfHdr, sizeof(struct bmf_hdr));
	bcopy(&defBmiHdr, &bmiHdr, sizeof(struct bmi_hdr));
	bmiHdr.biWidth[0] = width & 0xff;
	bmiHdr.biWidth[1] = (width >> 8) & 0xff;
	bmiHdr.biHeight[0] = height & 0xff;
	bmiHdr.biHeight[1] = (height >> 8) & 0xff;

	imageSize = height * width * 3;
	bmiHdr.biSizeImage[0] = imageSize & 0xFF;
	bmiHdr.biSizeImage[1] = (imageSize >> 8) & 0xFF;
	bmiHdr.biSizeImage[1] = (imageSize >> 16) & 0xFF;
	bmiHdr.biSizeImage[1] = (imageSize >> 24) & 0xFF;
	
	fwrite(&bmfHdr, BMFHDRSIZE, 1, stdout);
	fwrite(&bmiHdr, BMIHDRSIZE, 1, stdout);
	for (y = height-1; y >= 0; y--)
		fwrite(pixelMap[y][0], 1, width*3, stdout);
  }

void
usage() {
	fprintf(stderr, "usage: ipr_decode [-p] [-n <count>]\n"
			"\tTakes RLE data on standard input, and generates BMP file to STDOUT.\n"
			"\t-p generates pretty output instead of a BMP.\n"
			"\t-n expects an byte count (hex) and notifies when\n"
			"\t   this count is reach in the input data.\n");
	exit(-1);
  }

