// rle_greycomp
//              This takes a RLE data and, when appropriate, converts certain runs to
//              grey.  Otherwise, the input is passed to the output unchanged.

// Includes
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <unistd.h>

#include "bmp.h"
#include "rle_util.h"
#include "ccr.h"

// Globals
RLEINFO rleInfo;
FILE *inFP;

// Settable parameters
int gMargin = 3;		// grey margin
int gMode = 0;			// how to generate the grey value
int gForce = 0;			// whether to force all pixels grey

// Functions
void usage(), greyCompress();

// Main program
int main(int argc, char *argv[])
{
    char *inFile = 0;
    char use_ccr = 0;
    unsigned long ccr;
    int c;
    
    //
    // Process args
    //
    while ((c = getopt(argc, argv, "G:M:c:")) != -1) {
	switch (c) {
	  case 'G':
	      if (optarg != NULL) {
		  if (optarg[0] == 'F') {
		      gForce = 1;
		  } else if (optarg[0] == 'D') {
		      gForce = -1;
		  } else {
		      gMargin = strtol(optarg, NULL, 10);
		      if (gMargin < 0) {
			  fprintf(stderr,
				  "RLE_GREYCOMP ERROR: bad grey margin value '%s'\n",
				  optarg);
			  usage();
		      }
		  }
	      } else {
		  usage();
	      }
	      break;
	  case 'M':
	      if (optarg != NULL) {
		  gMode = strtol(optarg, NULL, 10);
		  if (gMode < 0 || gMode > 1) {
		      fprintf(stderr,
			      "RLE_GREYCOMP ERROR: bad grey mode value '%s'\n",
			      optarg);
		      usage();
		  }
	      } else {
		  usage();
	      }
	      break;
	  case 'c':
	      if (optarg != NULL) {
		  ccr = strtoul(optarg, NULL, 16);
		  use_ccr = 1;
	      } else {
		  usage();
	      }
	      break;
	}
    }

    if (optind < argc) {
	inFile = argv[optind];
    }

    if (use_ccr) {
	if (ccr & CCR_LRLE_GREY_FORCE_MASK) {
	    gForce = 1;
	} else if (ccr & CCR_LRLE_GREY_DISABLE_MASK) {
	    gForce = -1;
	} else {
	    gMargin = (ccr & CCR_LRLE_GMARGIN_MASK) >> CCR_LRLE_GMARGIN_SHIFT;
	}
	if (ccr & CCR_LRLE_GREY_GREEN_MASK) {
	    gMode = 1;
	}
    }
    
//
// Open input file
//
    if (inFile) {
	if (!(inFP = (FILE *) fopen(inFile, "r"))) {
	    fprintf(stderr, "RLE_GREYCOMP ERROR: can't open file '%s'\n",
		    inFile);
	    exit(-1);
	}
    } else
	inFP = stdin;

//
// Read file header
//
    if (getRLEHdr(inFP, &rleInfo) < 0)
	exit(-1);

//
// Report parameters
//
    fprintf(stderr, ">>RLE_GREYCOMP:\n");
    if (inFile) {
	fprintf(stderr, "\tInput file: %s\n", inFile);
	fprintf(stderr, "\tFrame size: %dx%d; cell size: %dx%d\n",
		rleInfo.fWidth, rleInfo.fHeight, rleInfo.cWidth,
		rleInfo.cHeight);
    }
    if (gForce > 0)
	fprintf(stderr, "\tForce grey; ");
    else if (gForce < 0)
	fprintf(stderr, "\tDisable grey; ");
    else
	fprintf(stderr, "\tGrey Margin: %d; ", gMargin);
    fprintf(stderr, "Mode: %s\n", (gMode ? "GREEN" : "TRUE GREY"));

    putRLEHdr(rleInfo);

    greyCompress(inFP);
    return 0;
}

// Do it
void greyCompress(FILE * inFP)
{
    char s[160];
    int r, g, b, run, isGrey, isNew;
//
// Process input lines; convert to grey when appropriate
//
    while (fgets(s, sizeof(s), inFP)) {
	if (gForce >= 0 && !strncmp(s, "RUN ", 4)) {
	    if (!parseRunData(s, &isGrey, &isNew, &run, &r, &g, &b)) {
		fprintf(stderr,
			"RLE_GREYCOMP ERROR: bad run format '%s'\n", s);
		exit(-1);
	    }
	    if (gForce
		|| MAX3(2 * r, g, 2 * b) - MIN3(2 * r, g,
						2 * b) <= gMargin) {
		if (gMode == 0)
		    g = (2 * (2 * r) + 5 * g + (2 * b) + 4) / 8;
		putRun(1, 1, run, 0, g, 0);
	    } else
		putRun(0, 1, run, r, g, b);
	} else
	    fputs(s, stdout);
    }
}

void usage()
{
    fprintf(stderr,
	    "usage: rle_greycomp [<infile>] [-c <CCR>] [-G <grey margin>] [-M <mode]\n"
	    "\t<infile> must be RLE data as generated by lrle_engine\n"
	    "\tIf <infile> is not specified, standard input will be used.\n"
	    "\t<grey margin> may be a number greater >= zero, or\n"
	    "\t\t'F' ==> which forces all pixels grey\n"
	    "\t\t'D' ==> which disables grey compression\n"
	    "\t<mode> may either be 0 or 1:\n"
	    "\t\t0 ==> normal, using luminance to calculate grey value\n"
	    "\t\t1 ==> green mode, where pixel is left untouched\n"
	    "\t<CCR> = 32bit hex value for the CCR register, overrides all other settings\n");
    
    exit(-1);
}
