// rle_runaccum
//              This takes RLE data and merges consecutive runs that have the same
//              color value.

// Includes
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <unistd.h>

#include "bmp.h"
#include "rle_util.h"
#include "ccr.h"

// Globals
RLEINFO rleInfo;
FILE *inFP;

// Settable parameters
int rLimit = 31;		// limit of length of ordinary runs
int cLimit = 255;		// limit of length of line copies

// Functions
void usage(), runReaccumulate();

// Main program
int main(int argc, char *argv[])
{
    char *inFile = 0;
    char use_ccr = 0;
    int c;
    unsigned long ccr;
    
    //
    // Process args
    //
    while ((c = getopt(argc, argv, "R:C:c:")) != -1) {
	switch (c) {
	  case 'R':
	      if (optarg != NULL) {
		  rLimit = strtol(optarg, NULL, 10);
		  if (rLimit < 0 || rLimit > 255) {
		      fprintf(stderr,
			      "RLE_RUNACCUM ERROR: bad run limit value '%s'\n",
			      optarg);
		      usage();
		}
	      } else {
		  usage();
	      }
	      break;
	  case 'C':
	      if (optarg != NULL) {
		  cLimit = strtol(optarg, NULL, 10);
		  if (cLimit < 0 || cLimit > 255) {
		      fprintf(stderr,
			      "RLE_RUNACCUM ERROR: bad copy limit value '%s'\n",
			      optarg);
		      usage();
		  }
	      } else {
	      usage();
	      } 
	      break;	  
	  case 'c':
	      if (optarg != NULL) {
		  ccr = strtoul(optarg, NULL, 16);
		  use_ccr = 1;
	      } else {
		  usage();
	      }
	      break;
	}
    }

    if (optind < argc) {
	inFile = argv[optind];
    }
    
    if (use_ccr) {
	rLimit = (ccr & CCR_LRLE_REACC_RUNLIMIT_MASK)  >> CCR_LRLE_REACC_RUNLIMIT_SHIFT;
    }

//
// Open input file
//
    if (inFile) {
	if (!(inFP = (FILE *) fopen(inFile, "r"))) {
	    fprintf(stderr, "RLE_RUNACCUM ERROR: can't open file '%s'\n",
		    inFile);
	    exit(-1);
	}
    } else
	inFP = stdin;

//
// Read file header
//
    if (getRLEHdr(inFP, &rleInfo) < 0)
	exit(-1);

//
// Report parameters
//
    fprintf(stderr, ">>RLE_RUNACCUM:\n");
    if (inFile) {
	fprintf(stderr, "\tInput file: %s\n", inFile);
	fprintf(stderr, "\tFrame size: %dx%d; cell size: %dx%d\n",
		rleInfo.fWidth, rleInfo.fHeight, rleInfo.cWidth,
		rleInfo.cHeight);
    }
    fprintf(stderr, "\tRun limit=%d; copy limit=%d\n", rLimit, cLimit);
    putRLEHdr(rleInfo);

    runReaccumulate(inFP);
    return 0;
}

void runReaccumulate(FILE * inFP)
{
    char s[9999];
    int r, g, b, run, isGrey, isNew;
    int oldR, oldG, oldB, oldGrey, oldRun, force;
    int partial;

//
// Initialize
//
    oldGrey = oldRun = 0;
    oldR = oldG = oldB = -1;
//
// Process input lines
//
    while (getLine(inFP, sizeof(s), s)) {

	// Parse the incoming line
	force = 0;
	if (!strncmp(s, "RUN ", 4)) {	// parse a run
	    if (!parseRunData(s, &isGrey, &isNew, &run, &r, &g, &b)) {
		fprintf(stderr,
			"RLE_RUNACCUM ERROR: bad run format '%s'\n", s);
		exit(-1);
	    }
	} else if (!strncmp(s, "LC ", 3)) {	// parse a line copy
	    if (!parseInt(s + 3, &run, 0)) {
		fprintf(stderr, "RLE_RUNACCUM ERROR: bad LC format '%s'\n",
			s);
		exit(-1);
	    }
	    isGrey = 0;
	    r = g = b = -1;
	} else {
	    force = 1;
	    isGrey = 0;
	    r = g = b = run = -1;
	}

	// Accumulate
	if (force || isGrey != oldGrey || r != oldR || g != oldG
	    || b != oldB) {
	    // put out the old run
	    if (oldR < 0)	// line copy
		while (oldRun > 0) {
		    partial = MIN(oldRun, cLimit + 1);
		    printf("LC %d\n", partial - 1);
		    oldRun -= partial;
	    } else {		// regular run
		isNew = 1;
		while (oldRun > 0) {
		    partial = MIN(oldRun, rLimit + 1);
		    putRun(oldGrey, isNew, partial - 1, oldR, oldG, oldB);
		    oldRun -= partial;
		    isNew = 0;
		}
	    }

	    // init new run
	    oldGrey = isGrey;
	    oldR = r;
	    oldG = g;
	    oldB = b;
	    oldRun = run + 1;
	} else			// continue run
	    oldRun += run + 1;

	// If not a RUN or LC, regurgitate the line
	if (force)
	    puts(s);
    }
}

void usage()
{
    fprintf(stderr,
	    "usage: rle_runaccum [<infile>] [-c <CCR>] [-r <run limit>] [-c <copy limit>]\n"
	    "\t<infile> must be RLE data as generated by lrle_engine\n"
	    "\tIf <infile> is not specified, standard input will be used.\n"
	    "\t<run limit> may be in the range [0,255]\n"
	    "\t<copy limit> may be in the range [0,255]\n"
	    "\t<CCR> = 32bit hex value for the CCR register, overrides all other settings\n");
    exit(-1);
}
