#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "rle_util.h"

static int lineNumber;

// return the last read line number.  If "clear" is non-zero, initialize
// the line number to 0.
int lineNum(int clear)
{
    if (clear)
	lineNumber = 0;
    return (lineNumber);
}

// fetch line from input, max "n" characters, and put in "s[]".
// strip trailing whitespace.  Skip lines that are all whitespace.
//
// return non-zero if a line was found
int getLine(FILE * inFP, int n, char *s)
{
    int i;

    while (1) {
	if (!fgets(s, n, inFP))
	    return (0);
	lineNumber++;
	for (i = strlen(s) - 1; i >= 0; i--) {
	    if (!isspace(s[i]))
		return (1);
	    else
		s[i] = 0;
	}
    }
}

// look for a line starting with ">>", max "n" characters, and put in "s[]".
// strip trailing whitespace, remove the ">>" from the line.
// return 1 if a line was found
// return 0 if EOF was reached
// return -1 if error
//
// if "expect" is non-zero, then return error if the next block header is
// of a type different from "expect", and remove the type from the line.
int getBlockHdr(FILE * inFP, int n, char *s, char *expect)
{
    int len = expect ? strlen(expect) : 0;

    while (1) {
	if (!getLine(inFP, n, s))
	    return (0);
	else if (s[0] != '>' || s[1] != '>')
	    continue;
	else {
	    if (len && strncmp(s + 2, expect, len))
		return (-1);
	    strcpy(s, s + 2 + len + (s[2 + len] == ' '));
	    return (1);
	}
    }
}

// get a line of data from a block
int getBlockData(FILE * inFP, int n, char *s)
{
    if (!getLine(inFP, n, s))
	return (0);
    else if (!strncmp(s, "END ", 4)) {
	strcpy(s, s + 4);
	return (-1);
    } else
	return (1);
}

// write the RLE file header to stdout
void putRLEHdr(RLEINFO rleInfo)
{
    printf(">>FILEINFO\n");
    printf("FILENAME %s\n", rleInfo.name);
    printf("FRAME %d %d\n", rleInfo.fWidth, rleInfo.fHeight);
    printf("CELL %d %d\n", rleInfo.cWidth, rleInfo.cHeight);
    printf("END FILEINFO\n");
}

// read RLE file header
int getRLEHdr(FILE * inFP, RLEINFO * rip)
{
    char s[9999], field[9999];
    int rv;

    if (getBlockHdr(inFP, 9999, s, "FILEINFO") <= 0) {
	fprintf(stderr, "ERROR: couldn't find FILEINFO block\n");
	return (-1);
    }

    while (1) {
	rv = getBlockData(inFP, 9999, s);
	if (!rv) {
	    fprintf(stderr, "ERROR: unexpected EOF in FILEINFO block\n");
	    return (-1);
	} else if (rv < 0) {
	    if (strcmp(s, "FILEINFO")) {
		fprintf(stderr,
			"ERROR: mismatched END statement in FILEINFO block\n");
		return (-1);
	    } else
		return (1);
	} else {
	    sscanf(s, "%s ", field);
	    if (!strcmp(field, "FILENAME"))
		sscanf(s, "%*s %s", rip->name);
	    else if (!strcmp(field, "FRAME"))
		sscanf(s, "%*s %d %d", &(rip->fWidth), &(rip->fHeight));
	    else if (!strcmp(field, "CELL"))
		sscanf(s, "%*s %d %d", &(rip->cWidth), &(rip->cHeight));
	    else;
	}
    }
}

// parse a RUN
int
parseRunData(char *s, int *grey, int *new, int *run, int *r, int *g,
	     int *b)
{
    s += 4;

// check if GREY
    if (!(s = parseInt(s, grey, ' ')))
	return (0);
/*    
    *grey = 0;
    if (!strncmp(s, "GREY ", 5)) {
	s += 5;
	*grey = 1;
    }
 */
// check if new
    if (!(s = parseInt(s, new, ' ')))
	return (0);
/*    
    if (!strncmp(s, "RPT ", 4)) {
	s += 4;
	*new = 0;
    } else
	*new = 1;
*/
// read run length
    if (!(s = parseInt(s, run, ' ')))
	return (0);

// read r, g, b
    if (!(s = parseInt(s, r, ',')) ||
	!(s = parseInt(s, g, ',')) || !(s = parseInt(s, b, 0)) || *s)
	return (0);

// Everything's OK!
    return (1);
}

char *parseInt(char *s, int *val, int sep)
{
    char *sp = s;

    *val = 0;
    while (1) {
	if (*sp >= '0' && *sp <= '9')
	    *val = *val * 10 + *sp++ - '0';
	else if ((*sp == sep || *sp == '\n' || *sp == 0) && s != sp)
	    return (sp + (*sp != 0));
	else
	    return ((char *) 0);
    }
}

void putRun(int isGrey, int isNew, int run, int r, int g, int b)
{
    /*
    printf("RUN%s%s %d %d,%d,%d\n",
	   (isGrey ? " GREY" : ""), (isNew ? "" : " RPT"), run, r, g, b);
     */
    printf("RUN %d %d %d %d,%d,%d\n",
	   (isGrey ? 1 : 0), (isNew ? 1 : 0), run, r, g, b);
}
