#!/usr/bin/perl -w

use strict;

use Getopt::Std;
use eRIC_Firmware;

my $usage = "\n\n" .
"usage:\n" .
"patch_version.pl [-v <version>] [-b <buildno>] [-d <version>] [-t <tag>] [-r <src_filename>] [-W <file>] [-R <file>] <filename>\n\n" .
"  -v <version>      set version information without dots, eg 040100\n" .
"  -n <buildno>      set firmware build number (use _only_ for mk_firmware)\n" .
"  -M <version>      set minimum downgrade version, format see -v\n" .
"  -t <tag>          set firmware tag. Tags with multiple words must\n" .
"                    be passed with parantheses, eg \"Standard Edition\"\n" .
"  -r <src_filename> read all parameters from the given file" .
"  -W <file>      write firmware version string to file\n" .
"  -R <file>      read firmware version string from file\n" .
"                 setting a specific parameter directly overrides the appropriate part\n" .
"  <filename>     name of firmware file to patch (./xxx/.../libpp_firmware.so)\n" .
"\n\n";


sub print_version_info( $ ) {
  my $version_str = shift;
  my $version;
  my $buildno;
  my $downgrade_version;
  my $tag;
  
  if (($version, $buildno, $downgrade_version, $tag) = ($version_str =~ m/(.{6}).(.{5}).(.{6}).(.{64})./gs)) {
    $buildno =~ s/^ //g;
    print("Version information:\n");
    print("  Version             : $version\n");
    print("  Build No.           : $buildno\n");
    print("  Min. downgrade ver. : $downgrade_version\n");
    print("  Tag                 : $tag\n\n");
  } else {
    print("Error retrieving the new version information\n");
  }
}

my $file;
my $version_str;
my $opts  = {};

getopts('v:n:M:t:r:R:W:', $opts);

$file = shift @ARGV;  # first remaining element after getopts ...

if (!(defined $file)) {
  die ($usage . "Error: please specify a firmware file");
}

$version_str = read_pp_fw_ver($file);
if ($version_str eq "") {
  print("NOTE: could not read FW version string from $file.\n".
        "      Corresponding FW probably didn't support\n".
        "      version patching, yet -> skipping...\n");
  exit(0);
}

# read options
if (defined $opts->{r}) {
  # read all options from the given file
  $version_str = read_pp_fw_ver($opts->{r});
  if ($version_str eq "") {
    print("NOTE: could not read source FW version string from $file.\n".
          "      Version information not known, cannot patch version\n");
    exit(1);
  }
}

if (defined ${opts}->{W}) {
    my $ver_filename = ${opts}->{W};
    open FILE, ">$ver_filename" or do {
	die "Warning: can't open '$ver_filename'. ($!)\n";
    };

    print FILE "$version_str";
    close FILE;
}

if (defined ${opts}->{R}) {
    my $ver_filename = ${opts}->{R};
    open FILE, "<$ver_filename" or do {
	die "Warning: can't open '$ver_filename'. ($!)\n";
    };

    $version_str = <FILE>;

    close FILE;
}

if (defined $opts->{v}) {
  $version_str = version_str_set_version($version_str, $opts->{v});
}

if (defined $opts->{n}) {
  $version_str = version_str_set_buildno($version_str, $opts->{n});
}

if (defined $opts->{M}) {
  $version_str = version_str_set_downgrade_version($version_str, $opts->{M});
}

if (defined $opts->{t}) {
  $version_str = version_str_set_tag($version_str, $opts->{t});
}

print_version_info($version_str);

write_pp_fw_ver($file, $version_str);

