#!/bin/sh 

# stop if one command fails
set -e

usage () {
cat <<EOF
root_fw.sh -F <FLASH_MAP_NAME> (-m <FILENAME> | -u <FILENAME> | -U | -c <INSTALL_ROOT_PATH>)

Modifies the initrd root filesystem for erla devices. Main
use is for gathering all root calls in one file. Must be
called with sudo.

Options:
    -m <FILENAME>
       unzip and mount the packed root filesystem FILENAME 
       to target_root
    -u <FILENAME>
       unmount, zip and recreate the root file system file
    -U just unmount, do not save
    -c <INSTALL_ROOT_PATH>
       copy the specified path into target_root
    -F <FLASH_MAP_NAME>
       map_ramfs | map_jffs
EOF
}

# parse options
MODE=""
FLASH_MAP_NAME=""

while getopts F:m:u:Uc: opt; do
    case "$opt" in
	F)  FLASH_MAP_NAME="$OPTARG";;
	m)  MODE=mount
	    FILENAME="$OPTARG"
	    ;;
	u)  MODE=umount_save
	    FILENAME="$OPTARG";;
	U)  MODE=umount_nosave;;
	c)  MODE=copy
	    INSTALL_ROOT_PATH="$OPTARG";;
	[?])
	    echo usage
	    exit 1;;
    esac
done
shift $(($OPTIND-1))

if [ -z "$MODE" ]; then
    echo "ERROR: -m <FILENAME>, -u <FILENAME>, or -c <INSTALL_ROOT_PATH> required!"
    usage
    exit 1
fi
if [ "$FLASH_MAP_NAME" != "map_ramfs" -a "$FLASH_MAP_NAME" != "map_jffs" ]; then
    if [ -z "$FLASH_MAP_NAME" ]; then
	echo "ERROR: -F <FLASH_MAP_NAME> required!"
    else
	echo "ERROR: invalid -F <FLASH_MAP_NAME> given: '$FLASH_MAP_NAME'!"
    fi
    usage
    exit 1
fi

# Update the root partition of a firmware image with
# the compiled sourcecode from an sdk.

case "$MODE" in
    mount)
	if [ "$FLASH_MAP_NAME" = "map_jffs" ]; then
	    # unpack
	    mkdir -p target_root
	    tar -C target_root -xzf "$FILENAME"
	else
	    # unpack and mount
	    ./img2data.sh "$FILENAME" root.initrd.unpacked
	    mkdir -p target_root
	    # just in case something failed earlier on:
	    # put this in an if because it might fail and we do _not_ want it
	    # to stop script execution
	    if (umount target_root &> /dev/null) then echo ; fi
	    mount -o loop -t ext2 root.initrd.unpacked target_root
	fi
	exit 0
	;;
    umount_save)
	if [ "$FLASH_MAP_NAME" = "map_jffs" ]; then
	    # umount and repack
	    tar -C target_root -czf "$FILENAME" .
	    rm -rf target_root
	else
	    # umount and repack
	    umount target_root
	    ./data2img.sh root.initrd.unpacked "$FILENAME"
	    rm root.initrd.unpacked
	    rmdir target_root
	fi
	exit 0
	;;
    umount_nosave)
	if [ "$FLASH_MAP_NAME" = "map_jffs" ]; then
	    rm -rf target_root
	else
	    umount target_root
	    rm root.initrd.unpacked
	    rmdir target_root
	fi
	exit 0
	;;
    copy)
	# copy install_root to target_root
	cp -a "$INSTALL_ROOT_PATH" target_root
	exit 0
	;;
    *)
	echo "$0: Internal error!"
	exit 1
esac

