/***************************************************************************
 *   Copyright (C) 2004 by Ralf Guenther                                   *
 *   rgue@peppercon.de                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <sys/mman.h>
#include <errno.h>
#include <string.h>
#include <fcntl.h>
#include <ctype.h>
#include <unistd.h>

static void
usage(void)
{
	printf("Usage: peek [b|w|d] <phys mem addr> [<len>]\n");
	exit(1);
}

static void
hex_dump(unsigned char* p, unsigned int size, int phys_addr)
{
	unsigned int i, j;
	unsigned char b[16];
	for (i = 0; i < size; i += 16) {
		printf("%08x:", phys_addr + i);
		
		/* pre-cache values to avoid multiple mem reads */
		for (j = 0; j < 16; j++)
			if (i + j < size) b[j] = p[i + j];

		/* hex display */
		for (j = 0; j < 8; j++)
			if (i + j >= size) printf("   ");
			else printf(" %02x", b[j]);
		printf(" ");
		for (j = 8; j < 16; j++)
			if (i + j >= size) printf("   ");
			else printf(" %02x", b[j]);
		printf("  ");
		
		/* ascii display */
		for (j = 0; j < 8; j++)
			if (i + j >= size) printf(" ");
			else if (!isgraph(b[j])) printf(".");
			else printf("%c", b[j]);
		printf(" ");
		for (j = 8; j < 16; j++)
			if (i + j >= size) printf(" ");
			else if (!isgraph(b[j])) printf(".");
			else printf("%c", b[j]);
		printf("\n");
	}
}

static void
hex_dump_w(unsigned short* p, unsigned int size, int phys_addr)
{
	unsigned int i, j, k;
	unsigned short b[8];
	for (i = 0; i < size; i += 8) {
		printf("%08x:", phys_addr + i * 2);
		
		/* pre-cache values to avoid multiple mem reads */
		for (j = 0; j < 8; j++)
			if (i + j < size) b[j] = p[i + j];

		/* hex display */
		for (j = 0; j < 4; j++)
			if (i + j >= size) printf("         ");
			else printf(" %04x", b[j]);
		printf(" ");
		for (j = 4; j < 8; j++)
			if (i + j >= size) printf("         ");
			else printf(" %04x", b[j]);
		printf("  ");
		
		/* ascii display */
		for (j = 0; j < 4; j++) {
			if (i + j >= size) printf("    ");
			else for (k = 0; k < 2; k++) {
				char c = ((char*)(b + j))[k];
				if (!isgraph(c)) printf(".");
				else printf("%c", c);
			}
			printf(" ");
		}
		printf(" ");
		for (j = 4; j < 8; j++) {
			if (i + j >= size) printf(" ");
			else for (k = 0; k < 2; k++) {
				char c = ((char*)(b + j))[k];
				if (!isgraph(c)) printf(".");
				else printf("%c", c);
			}
			printf(" ");
		}
		printf("\n");
	}
}

static void
hex_dump_d(unsigned int* p, unsigned int size, int phys_addr)
{
	unsigned int i, j, k;
	unsigned int b[4];
	for (i = 0; i < size; i += 4) {
		printf("%08x:", phys_addr + i * 4);
		
		/* pre-cache values to avoid multiple mem reads */
		for (j = 0; j < 4; j++)
			if (i + j < size) b[j] = p[i + j];

		/* hex display */
		for (j = 0; j < 2; j++)
			if (i + j >= size) printf("         ");
			else printf(" %08x", b[j]);
		printf(" ");
		for (j = 2; j < 4; j++)
			if (i + j >= size) printf("         ");
			else printf(" %08x", b[j]);
		printf("  ");
		
		/* ascii display */
		for (j = 0; j < 2; j++) {
			if (i + j >= size) printf("    ");
			else for (k = 0; k < 4; k++) {
				char c = ((char*)(b + j))[k];
				if (!isgraph(c)) printf(".");
				else printf("%c", c);
			}
			printf(" ");
		}
		printf(" ");
		for (j = 2; j < 4; j++) {
			if (i + j >= size) printf("    ");
			else for (k = 0; k < 4; k++) {
				char c = ((char*)(b + j))[k];
				if (!isgraph(c)) printf(".");
				else printf("%c", c);
			}
			printf(" ");
		}
		printf("\n");
	}
}

int main(int argc, char *argv[])
{
	int arg = 1;
	char mode;
	unsigned long addr;
	unsigned long len;
	size_t page_size;
	unsigned long offs;
	unsigned long base;
	unsigned long size;
	int fd;
	unsigned char *p;

	if (argc < 2 || argc > 4) usage();

	mode = 'b'; if (argv[arg][0] == 'b' || argv[arg][0] == 'w' || argv[arg][0] == 'd') mode = argv[arg++][0];
	if (sscanf(argv[arg++], "%li", &addr) != 1) usage();
	len = 1; if (arg < argc) if (sscanf(argv[arg++], "%li", &len) != 1) usage();
	
	switch (mode) {
	case 'b': printf("peeking %lu byte(s) from 0x%08lx\n", len, addr); break;
	case 'w': printf("peeking %lu word(s) from 0x%08lx\n", len, addr); break;
	case 'd': printf("peeking %lu dword(s) from 0x%08lx\n", len, addr); break;
	}
	
	page_size = (size_t)sysconf(_SC_PAGESIZE);
	offs = addr % page_size;
	base = addr - offs;
	size = len + offs;

	fd = open("/dev/mem",O_RDONLY);
	if (fd == -1) { printf("Error: open /dev/mem failed: %s (%d)\n", strerror(errno), errno); exit(-1); }
	
	p = mmap(0, size, PROT_READ, MAP_SHARED, fd, (off_t)base);
	if (p == (unsigned char*)-1) { printf("Error: mmap failed: %s (%d)\n", strerror(errno), errno); exit(-1); }

	switch (mode) {
	case 'b': hex_dump(p + offs, len, addr); break;
	case 'w': hex_dump_w((short*)(p + offs), len, addr); break;
	case 'd': hex_dump_d((int*)(p + offs), len, addr); break;
	}
	
	if (-1 == munmap(p, size)) { printf("Error: munmap failed: %s (%d)\n", strerror(errno), errno); exit(-1); }
	if (-1 == close(fd)) { printf("Error: close /dev/mem failed: %s (%d)\n", strerror(errno), errno); exit(-1); }
	
	return 0;
}
