/*
 * pp_msp_proto.h - header file for the mass storage protocol (msp) version 1.2
 *
 * Uses types CARD<n> for an n-bit unsigned integer, INT<n> for an n-bit signed
 * integer (for n = 8, 16 and 32).
 *
 * All multiple byte integers are in big endian (network) order (most
 * significant byte first).  Unless noted otherwise there is no special
 * alignment of protocol structures.
 *
 *
 * Once the initial handshaking is done, all messages start with a type byte,
 * (usually) followed by message-specific data.  The order of definitions in
 * this file is as follows:
 *
 *  (1) Structures used in several types of message.
 *  (2) Structures used in the initial handshaking.
 *  (3) Message types.
 *  (4) Encoding types.
 *  (5) For each message type, the form of the data following the type byte.
 *      Sometimes this is defined by a single structure but the more complex
 *      messages have to be explained by comments.
 */

 
#ifndef __PP_MSP_PROTO_H
#define __PP_MSP_PROTO_H

#ifdef WIN32
# pragma pack(1)
# define PACKED_STRUCT
#else /* WIN32 */
# define PACKED_STRUCT __attribute__ ((packed))
#endif /* WIN32 */


/*****************************************************************************
 *
 * Protocol overview
 *
 *****************************************************************************/

/*--------------------

During the connection phase, the following scheme is used:

      Client                                    erla
----------------------------------------------------------------------
  msp_prot_init_pdu_t
  (Initialization message)
  
                                       msp_prot_ver_pdu_t
                                       (Server protocol version)
  
  msp_prot_ver_pdu_t
  (Client Protocol version)
  
  msp_login_pdu_t
  (Login)
  
                                       msp_response_pdu_t
                                       (ACK/NACK to the login)
                                       Close connection on NACK
  
  msp_rq_connection_pdu_t
  (Request for CD connection)
  
                                       msp_response_pdu_t
                                       (ACK/NACK to the connection)
                                       Close connection on NACK

*--------------------*/


/**--------------------

Protocol version history:

    1.1:    - initial MSP version
    1.2:    - support for more than one mass storage device
              (index is given in the msp_rq_connection_pdu_t message)

*--------------------*/ 

/*****************************************************************************
 *
 * Initial handshaking messages
 *
 *****************************************************************************/

#define MSP_PROTOCOL_INIT_MESSAGE	"e-RIC MSP P"

/* NOTE: The client's protocol initialization message is actually only 12 bytes long
   (see above + trailing 0), but accidently we use 19 bytes since the beginning.
   To stay compatible, we have to keep this. */
#define msp_prot_init_pdu_size 19

#define MSP_PROTOCOL_VERSION_FORMAT	"e-RIC MSP %02d.%02d\n"
#define MSP_PROTOCOL_MAJOR_VERSION	1
#define MSP_PROTOCOL_MINOR_VERSION	2

typedef char msp_prot_ver_pdu_t[17];	/* allow extra byte for null */

#define msp_prot_ver_pdu_size 16

/*****************************************************************************
 *
 * Message types
 *
 *****************************************************************************/

/* client -> server*/
//Messages start with 'MSP'
#define MSP_TYPE_LOGIN			0x00
#define MSP_TYPE_RQ_CONNECTION		0x01
#define MSP_TYPE_SEND_DATA		0x02	/* used in both directions */
#define MSP_TYPE_QUIT_CONNECTION	0x03	/* used in both directions */
#define MSP_TYPE_PING			0x04	/* used in both directions */
#define MSP_TYPE_PONG			0x05	/* used in both directions */
#define MSP_TYPE_SESSION_ID		0x06
#define MSP_TYPE_DATA_ACK		0x82    /* this was previously misplaced, so don't bother about the number */

/* server -> client */
#define MSP_TYPE_RSP_CONNECTION		0x80
#define MSP_TYPE_RQ_DATA		0x81

/*****************************************************************************
 * bidirectional message definitions
 *****************************************************************************/

#define MSP_QUIT_USER_CANCELLED		0x00
#define MSP_QUIT_DEVICE_CANCELLED	0x01

/* Send a termination message and its reason to the client/server */
typedef struct {
    u_int8_t type_8;
    u_int8_t reason_8;
} PACKED_STRUCT msp_quit_pdu_t;

typedef struct {
    u_int8_t type_8;
} PACKED_STRUCT msp_ping_pdu_t;

typedef struct {
    u_int8_t type_8;
} PACKED_STRUCT msp_pong_pdu_t;

#define MSP_DATA_OKAY		0x00	// request completed successfully
#define MSP_DATA_ERROR		0x01	// error in client (e.g. no memory)
#define MSP_DATA_NO_CD		0x02	// currently no cd inserted
#define MSP_DATA_NOT_COMPLETE	0x03	// not all requested sectors read,
					// but other sectors read successfully

/* SEND DATA */
typedef struct {
	u_int8_t		type_8;			//message type
	u_int8_t		code_8;			//everything okay or error code
	u_int16_t		actual_count_be16;	//actually read sectors
	u_int16_t		sector_size_be16;	//this might have changed (CD change)
	u_int16_t		pad;
	u_int32_t 		sector_be32;		//number of first sector (only server->client)
    /* followed by (actual_count * sector_size) bytes of data */
} PACKED_STRUCT msp_send_data_pdu_t;

/*****************************************************************************
 * server -> client message definitions
 *****************************************************************************/

//possible reason codes for connection or no connection
#define MSP_CONNECTION_RESP_OKAY		0x00	// only if connection is established
#define MSP_CONNECTION_RESP_NOT_AVAILABLE	0x01
#define MSP_CONNECTION_RESP_ALREADY_CONNECTED	0x02
#define MSP_CONNECTION_RESP_ALREADY_IMAGE	0x03
#define MSP_CONNECTION_RESP_PROTOCOL_ERROR	0x04
#define MSP_CONNECTION_RESP_AUTH_FAILED		0x05
#define MSP_CONNECTION_RESP_NO_PERMISSION	0x06
#define MSP_CONNECTION_RESP_INTERNAL_ERROR	0x07
#define MSP_CONNECTION_RESP_NO_SUCH_MS_INDEX	0x08
#define MSP_CONNECTION_RESP_WRITING_NOT_ALLOWED	0x09

/* Response_Connection */
typedef struct {
    u_int8_t 	type_8;		//type of message (mess.-ID)
    u_int8_t	ack_8;		//acknowledgement
    u_int8_t	reason_8;	//reason for connection or no connection
} PACKED_STRUCT msp_response_pdu_t;

/* Request Data */
typedef struct {
    u_int8_t 	type_8;
    u_int8_t 	pad;
    u_int16_t	count_be16;	//number of consecutive sectors
    u_int32_t 	sector_be32;	//number of first sector
} PACKED_STRUCT msp_rq_data_pdu_t;

/* server to client messages union */
typedef union {
    u_int8_t			type;
    msp_quit_pdu_t		quit;
    msp_response_pdu_t		response;
    msp_rq_data_pdu_t		rq_data;
    msp_ping_pdu_t		ping;
    msp_pong_pdu_t		pong;
    msp_send_data_pdu_t		send_data;
} msp_s2c_pdu_t;

/*****************************************************************************
 * client -> server message definitions
 *****************************************************************************/

typedef struct {
    u_int8_t	type_8;
    u_int8_t	pad;
    u_int16_t	username_len_be16;	//length of login name
    u_int16_t	password_len_be16;	//length of password
    //followed by username_len + password_len bytes
} PACKED_STRUCT msp_login_pdu_t;

typedef struct {
    u_int8_t	type_8;
} PACKED_STRUCT msp_session_id_pdu_t;

#define MSP_DISC_TYPE_CDROM	0x00
#define MSP_DISC_TYPE_FLOPPY	0x01
#define MSP_DISC_TYPE_REMOVABLE	0x02
#define MSP_DISC_TYPE_SOLID	0x03
#define MSP_DISC_NONE		0xff

/* REQUEST CONNECTION */
typedef struct {
    u_int8_t		type_8;			//message type
    u_int8_t		id_8;			//computer generated ID, maybe not used
    u_int8_t		ro_8;			//read only
    u_int8_t		disc_type_8;		//type of the device/drive
    u_int16_t		sector_size_be16;	//sector size, generally 2048 bytes
    u_int16_t		pad;
    u_int32_t		last_sector_no_be32;	//last sector number (e.g. total - 1)
} PACKED_STRUCT msp_rq_connection_pdu_v_1_1_t;

typedef struct {
    u_int8_t		type_8;			//message type
    u_int8_t		ms_index_8;		//index of mass storage device to use
    u_int8_t		id_8;			//computer generated ID, maybe not used
    u_int8_t		ro_8;			//read only
    u_int8_t		disc_type_8;		//type of the device/drive
    u_int8_t		pad;
    u_int16_t		sector_size_be16;	//sector size, generally 2048 bytes
    u_int32_t		last_sector_no_be32;	//last sector number (e.g. total - 1)
} PACKED_STRUCT msp_rq_connection_pdu_v_1_2_t;

/* ACK DATA */
typedef struct {
    u_int8_t		type_8;
    u_int8_t		ack_8;
} PACKED_STRUCT msp_data_ack_pdu_t;

/* -------------------------------------------------------------------------
 * Union of all client->server messages.
 * ------------------------------------------------------------------------- */

typedef union {
    u_int8_t				type;
    msp_login_pdu_t			login;
    msp_session_id_pdu_t		session;
    msp_rq_connection_pdu_v_1_1_t	request_connection_v_1_1;
    msp_rq_connection_pdu_v_1_2_t	request_connection_v_1_2;
    msp_send_data_pdu_t			send_data;
    msp_quit_pdu_t			quit;
    msp_ping_pdu_t			ping;
    msp_pong_pdu_t			pong;
    msp_data_ack_pdu_t			data_ack;
} msp_c2s_pdu_t;


#ifdef WIN32
# pragma pack()
#endif /* WIN32 */

#endif /* __PP_MSP_PROTO_H */
