/*
 *  Copyright (C) 2000, 2001 Const Kaplinsky.  All Rights Reserved.
 *  Copyright (C) 2000 Tridia Corporation. All Rights Reserved.
 *  Copyright (C) 1999 AT&T Laboratories Cambridge. All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

/*
 * proto.h - header file for the RFB protocol version 3.5
 *
 * Uses types CARD<n> for an n-bit unsigned integer, INT<n> for an n-bit signed
 * integer (for n = 8, 16 and 32).
 *
 * All multiple byte integers are in big endian (network) order (most
 * significant byte first).  Unless noted otherwise there is no special
 * alignment of protocol structures.
 *
 *
 * Once the initial handshaking is done, all messages start with a type byte,
 * (usually) followed by message-specific data.  The order of definitions in
 * this file is as follows:
 *
 *  (1) Structures used in several types of message.
 *  (2) Structures used in the initial handshaking.
 *  (3) Message types.
 *  (4) Encoding types.
 *  (5) For each message type, the form of the data following the type byte.
 *      Sometimes this is defined by a single structure but the more complex
 *      messages have to be explained by comments.
 *
 * 
 * Attention: 
 * Do not change bit order or values of flags without issuing a new rfb
 * version to ensure backward compatibility. Keeping the current rfb
 * version is possible only if flags are extended but not modified. This
 * file must be consistent even among different branches to please try
 * to add your changes on the head even if you are working in a release
 * branch.
 */


#ifndef __PP_RFB_PROTO_H
#define __PP_RFB_PROTO_H

#ifdef WIN32
# pragma pack(1)
# define PACKED_STRUCT
#else /* WIN32 */
# define PACKED_STRUCT __attribute__ ((packed))
#endif /* WIN32 */

/*****************************************************************************
 *
 * Structures used in several messages
 *
 *****************************************************************************/

/*-----------------------------------------------------------------------------
 * Structure used to specify a rectangle.  This structure is a multiple of 4
 * bytes so that it can be interspersed with 32-bit pixel data without
 * affecting alignment.
 */

typedef struct {
    u_int16_t x_be16;
    u_int16_t y_be16;
    u_int16_t w_be16;
    u_int16_t h_be16;
} PACKED_STRUCT rfbRectangle;

/*-----------------------------------------------------------------------------
 * Structure used to specify pixel format.
 */

typedef struct {

    u_int8_t bitsPerPixel_8;	/* 8,16,32 only */

    u_int8_t depth_8;		/* 8 to 32 */

    u_int8_t bigEndian_8;	/* True if multi-byte pixels are interpreted
				   as big endian, or if single-bit-per-pixel
				   has most significant bit of the byte
				   corresponding to first (leftmost) pixel. Of
				   course this is meaningless for 8 bits/pix */

    u_int8_t trueColour_8;	/* If false then we need a "colour map" to
				   convert pixels to RGB.  If true, xxxMax and
				   xxxShift specify bits used for red, green
				   and blue */

    /* the following fields are only meaningful if trueColour is true */

    u_int16_t redMax_be16;	/* maximum red value (= 2^n - 1 where n is the
				   number of bits used for red). Note this
				   value is always in big endian order. */

    u_int16_t greenMax_be16;	/* similar for green */

    u_int16_t blueMax_be16;	/* and blue */

    u_int8_t redShift_8;	/* number of shifts needed to get the red
				   value in a pixel to the least significant
				   bit. To find the red value from a given
				   pixel, do the following:
				   1) Swap pixel value according to bigEndian
				      (e.g. if bigEndian is false and host byte
				      order is big endian, then swap).
				   2) Shift right by redShift.
				   3) AND with redMax (in host byte order).
				   4) You now have the red value between 0 and
				      redMax. */

    u_int8_t greenShift_8;	/* similar for green */

    u_int8_t blueShift_8;	/* and blue */

    u_int8_t pad1;
    u_int16_t pad2;

} PACKED_STRUCT rfbPixelFormat;

/* -------------------------------------------------------------------------
 * structure used to specify timestamps (SAS)
 */

typedef struct {
    u_int32_t timestamp_be32;	/* UNIX timestamp */
    u_int32_t microseconds_be32;
} PACKED_STRUCT rfbTimestamp;

/*****************************************************************************
 *
 * Initial handshaking messages
 *
 *****************************************************************************/

/*-----------------------------------------------------------------------------
 * Protocol Version
 *
 * The server always sends 12 bytes to start which identifies the latest RFB
 * protocol version number which it supports.  These bytes are interpreted
 * as a string of 12 ASCII characters in the format "RFB xxx.yyy\n" where
 * xxx and yyy are the major and minor version numbers (for version 3.5
 * this is "RFB 003.005\n").
 *
 * The client then replies with a similar 12-byte message giving the version
 * number of the protocol which should actually be used (which may be different
 * to that quoted by the server).
 *
 * It is intended that both clients and servers may provide some level of
 * backwards compatibility by this mechanism.  Servers in particular should
 * attempt to provide backwards compatibility, and even forwards compatibility
 * to some extent.  For example if a client demands version 3.1 of the
 * protocol, a 3.0 server can probably assume that by ignoring requests for
 * encoding types it doesn't understand, everything will still work OK.  This
 * will probably not be the case for changes in the major version number.
 *
 * The format string below can be used in sprintf or sscanf to generate or
 * decode the version string respectively.
 */

/*****************************************************************************
 *
 * RFB Version Changelog:
 *
 * 1.01: - LARA Video contrast settings splitted in R,G,B
 * 1.02: - Compression Predictor
 * 1.03: - Mouse Wheel Support
 * 1.04: - Client Init Message contains Target Channel ID (for B52X2, else always 0)
 * 1.05: - New color depth's for Tight-Encoding: 1, 2 or 4 Bit
 * 1.06: - RC mode enter and leave messages
 * 1.07: - OSD message and blanking state for VSC devices
 * 1.08: - new video settings for VSC devices
 *       - command messages from server->client
 * 1.09: - Cache in Tight Encoding
 * 1.10: - Video Quality message added (noise filter and grabber quality)
 * 1.11: - support for video optimized encoding (raw VSC)
 * 1.12: - VSC hardware encoding (LRLE)
 *       - 32bit encoding word divided into encoding and parameter, used
 *	     hardware color format and tight subencoding currently
 *       - removed CopyRect encoding everywhere
 *       - removed alternate RFB handler
 *       - removed quality/speed setting from video quality message
 *       - added capability table
 * 1.13: - VSC2 hardware encoding: packages aligned 4-byte wise
 * 1.14: - new authentication scheme -> uses a challenge response mechanism
 *           with the session id and a random number
 *       - new ackPixelFormat message used to acknowledge a switch to a requested
 *           pixel format, for switching renderers between 8 and 16 bit in the
 *           client asynchronously
 *       - video optimized now separate flag for an encoding, not automatically 
 *           used when raw vsc is used
 * 1.15: - new intial connection handshake sequence
 *       - user login without HTTP possible (username + password plain or MD5)
 * 1.16: - SAS and SP extensions for CC-F (formerly known as SARA) project
 *       - connection flags on connection startup
 *       - flags for framebuffer update
 *       - new authentication methods: no auth, RDM session
 * 1.17: - add unit, port to init_pdu
 *       - moved client init msg to the start of the protocol (right before OSD msg)
 *       - needed for KX2 to make sure that initial connection has a target
 *       - added global property change message
 * 1.18-1.21: KX2 related stuff (developed in other branch)
 * 1.22: - connection parameter/capability table is send earlier during initial
 *         handshake to have the information about the character encoding of the
 *         following OSD messages
 *
 *****************************************************************************/

#define rfbProtocolInitString       "e-RIC AUTH="
#define rfbProtocolInitStringLen    11

#define rfbProtocolVersionFormat "e-RIC RFB %02d.%02d\n"
typedef char rfbProtocolVersionMsg[17];	/* allow extra byte for null */
#define rfbProtocolVersionMsgSize 16

#define rfbProtocolMajorVersion 1
#define rfbProtocolMinorVersion 22

/*-----------------------------------------------------------------------------
 * Client Initialisation Message
 *
 * Once the client and server are sure that they're happy to talk to one
 * another, the client sends an initialisation message.  At present this
 * message only consists of a boolean indicating whether the server should try
 * to share the desktop by leaving other clients connected, or give exclusive
 * access to this client by disconnecting all other clients.
 */

typedef struct {
    u_int8_t shared_8;
} PACKED_STRUCT rfbClientInitMsgV1_0;

typedef struct {
    u_int8_t shared_8;
    u_int8_t channelID_8;
} PACKED_STRUCT rfbClientInitMsgV1_4;

#define rfbinitMsgFlagsNone                 0
#define rfbinitMsgFlagsDoInitialSwitch      1

typedef struct {
    u_int8_t kvm_node_8;
    u_int8_t unit_8;
    u_int16_t flags_be16;
    u_int16_t port_be16;
} PACKED_STRUCT rfbClientInitMsgV1_17;

/*-----------------------------------------------------------------------------
 * Server Framebuffer Format Message
 *
 * After the client initialisation message, the server sends one of its own.
 * This tells the client the width and height of the server's framebuffer,
 * its pixel format and the name associated with the desktop.
 * 
 * This message is also sent when the screen resolution has changed.
 */

typedef struct {
    u_int8_t       type_8;
    u_int8_t       unsupported_8; /* not used in the client */
    u_int16_t      framebufferWidth_be16;
    u_int16_t      framebufferHeight_be16;
    rfbPixelFormat format;	/* the server's preferred pixel format */
} PACKED_STRUCT rfbServerFBFormatMsgV1_0;

typedef struct {
    u_int8_t       type_8;
    u_int8_t       unsupported_8; /* not used in the client */
    u_int16_t      framebufferWidth_be16;
    u_int16_t      framebufferHeight_be16;
    u_int16_t      pad2;
    rfbTimestamp   timestamp;
    rfbPixelFormat format;	/* the server's preferred pixel format */
} PACKED_STRUCT rfbServerFBFormatMsgV1_16;

/*
 * Following the server initialisation message it's up to the client to send
 * whichever protocol messages it wants.  Typically it will send a
 * SetPixelFormat message and a SetEncodings message, followed by a
 * FramebufferUpdateRequest.  From then on the server will send
 * FramebufferUpdate messages in response to the client's
 * FramebufferUpdateRequest messages.  The client should send
 * FramebufferUpdateRequest messages with incremental set to true when it has
 * finished processing one FramebufferUpdate and is ready to process another.
 * With a fast client, the rate at which FramebufferUpdateRequests are sent
 * should be regulated to avoid hogging the network.
 */



/*****************************************************************************
 *
 * Message types
 *
 *****************************************************************************/

/* server -> client */

#define rfbFramebufferUpdate		0x00
#define rfbSetColourMapEntries		0x01
#define rfbBell				0x02
#define rfbQuit				0x03
#define rfbUtf8String			0x07
#define rfbVideoSettingsS2CEvent	0x08
#define rfbKeyboardLayoutEvent		0x09
#define rfbOSDState			0x10
#define rfbVideoQualityS2CEvent		0x11
#define rfbConnectionParameterList	0x12
#define rfbAckPixelFormat		0x13

#define rfbAuthCaps			0x20
#define rfbSessionChallenge		0x21
#define rfbAuthSuccessful		0x22

#define rfbServerFBFormat		0x80
#define rfbServerRCMessage		0x83
#define rfbServerCommand		0x84

#define rfbSasEvent			0x85
#define rfbSasError			0x86

#define rfbSpTimestamp			0x88

#define rfbRCModeReply			0xA1

#define rfbSpSessionInfo                0xA2
#define rfbSpCommand			0xA3	/* used in both directions */
#define rfbSpSetSessionTimes		0xA4	/* used in both directions */

// read from file
#define rfbFileTimediff			0x40
#define rfbFileEOF			0x41
#define rfbFileKeyframeSeek		0x42
#define rfbFileIsKeyframe		0x43
#define rfbFileFormat			0x44

/* client -> server */

#define rfbSetPixelFormat		0x00
#define rfbFixColourMapEntries		0x01	/* not currently supported */
#define rfbSetEncodings			0x02
#define rfbFramebufferUpdateRequest	0x03
#define rfbKeyEvent			0x04
#define rfbPointerEvent			0x05
#define rfbScancodeEvent		0x06

#define rfbLogin			0x20
#define rfbChallengeResponse		0x21

#define rfbMouseStateEvent		0x85	// hide/show remote mouse
#define rfbMouseSyncEvent		0x86	// sync the remote mouse
#define rfbUserPropChange		0x87
#define rfbCursorChangeEvent		0x88
#define rfbKVMSwitchEvent		0x89
#define rfbVideoSettingsC2SEvent	0x90
#define rfbVideoSettingsReqEvent	0x91
#define rfbVideoRefreshRequest		0x92
#define rfbPointerRelativeEvent		0x93
#define rfbPingRequestEvent		0x94
#define rfbPingReply			0x95
#define rfbBandwidthRequestEvent	0x96
#define rfbBandwidthReply		0x97
#define rfbVideoQualityC2SEvent		0x98
#define rfbVideoQualityReqEvent		0x99
#define rfbGlobalPropChange		0x9A

#define rfbRCModeRequest		0xA0

#define rfbSpSetCimId			0xA1
#define rfbSpSetSession			0xA2
//#define rfbSpCommand			0xA3
//#define rfbSpSetSessionTimes		0xA4


/*****************************************************************************
 *
 * Authentication Methods
 *
 *****************************************************************************/

#define rfbAuthHttpSessionID		(1 << 0)
#define rfbAuthPlain			(1 << 1)
#define rfbAuthMD5			(1 << 2)
#define rfbAuthNoAuth                   (1 << 3)
#define rfbAuthRDMsession               (1 << 4)
// ...

/*****************************************************************************
 *
 * Connection Type Flags
 *
 *****************************************************************************/

#define rfbConnectionFlagSasHW			(1 << 0)	/* the CCF box client */
#define rfbConnectionFlagSasSW			(1 << 1)	/* normal RC client with forensic capabilities */
#define rfbConnectionFlagSP			(1 << 2)	/* SARA session player connection */
#define rfbConnectionFlagSasLocalOnly		(1 << 3)	/* don't show KVM-IP events */
/* bits 4 and 5 are used for keyboard method: */
#define rfbConnectionFlagKeyboardShift		4
#define rfbConnectionFlagKeyboardMask		3
# define rfbConnectionFlagGetKbd(flags)		(((flags) >> rfbConnectionFlagKeyboardShift) & rfbConnectionFlagKeyboardMask)
# define rfbConnectionFlagMakeKbd(kbd)		(((kbd) & rfbConnectionFlagKeyboardMask) << rfbConnectionFlagKeyboardShift)
#  define rfbConnectionSasKbdNormal		0
#  define rfbConnectionSasKbdNoKbd		1		/* don't record keyboard events */
#  define rfbConnectionSasKbdFilter		2		/* filter alphanumeric keys */
#  define rfbConnectionSasKbdAllFilterPb	3		/* record all with filtered playback */

/*****************************************************************************
 *
 * Encoding types
 *
 *****************************************************************************/

#define rfbEncodingRaw			0x00
#define rfbEncodingCopyRect		0x01
#define rfbEncodingRRE			0x02
#define rfbEncodingCoRRE		0x04
#define rfbEncodingHextile		0x05
#define rfbEncodingZlib			0x06
#define rfbEncodingTight		0x07
#define rfbEncodingZlibHex		0x08
#define rfbEncodingUseTightCache	0x09
#define rfbEncodingRawVSC		0x0a
#define rfbEncodingLRLESoft		0x0b
#define rfbEncodingJpeg			0x0c
#define rfbEncodingLRLEHard		0x80
#define rfbEncodingDCT			0x81

#define rfbEncodingAuto			0xff
#define rfbEncodingAutoSW		0x7f

#define rfbEncodingMask			0x000000FF
#define rfbEncodingIsHW			0x80

#define rfbEncodingParamZLIBShift	8
#define rfbEncodingParamZLIBMask	(0x0F << rfbEncodingParamZLIBShift)
#define rfbEncodingParamZLIB(x)		(x << rfbEncodingParamZLIBShift)
#define rfbEncodingParamSubencShift	12
#define rfbEncodingParamSubencMask	(0x0F << rfbEncodingParamSubencShift)
#define rfbEncodingParamSubenc(x)	(x << rfbEncodingParamSubencShift)
#define rfbEncodingParamVideoShift	16
#define rfbEncodingParamVideoMask	(0x01 << rfbEncodingParamVideoShift)
#define rfbEncodingParamVideo(x)	(x << rfbEncodingParamVideoShift)
#define rfbEncodingParamCompressShift	17	/* stream is compressed with the zlib copmress() function */
#define rfbEncodingParamCompressMask	(0x01 << rfbEncodingParamCompressShift)
#define rfbEncodingParamCompress(x)	(x << rfbEncodingParamCompressShift)

/*
 * Special encoding numbers:
 *   0xFFFFFF00 .. 0xFFFFFF0F -- encoding-specific compression levels;
 *   0xFFFFFF10 .. 0xFFFFFF1F -- mouse cursor shape data;
 *   0xFFFFFF20 .. 0xFFFFFF2F -- various protocol extensions;
 *   0xFFFFFF30 .. 0xFFFFFFDF -- not allocated yet;
 *   0xFFFFFFE0 .. 0xFFFFFFEF -- quality level for JPEG compressor;
 *   0xFFFFFFF0 .. 0xFFFFFFFF -- cross-encoding compression levels.
 */

#define rfbEncodingCompressLevel0  0xFFFFFF00
#define rfbEncodingCompressLevel1  0xFFFFFF01
#define rfbEncodingCompressLevel2  0xFFFFFF02
#define rfbEncodingCompressLevel3  0xFFFFFF03
#define rfbEncodingCompressLevel4  0xFFFFFF04
#define rfbEncodingCompressLevel5  0xFFFFFF05
#define rfbEncodingCompressLevel6  0xFFFFFF06
#define rfbEncodingCompressLevel7  0xFFFFFF07
#define rfbEncodingCompressLevel8  0xFFFFFF08
#define rfbEncodingCompressLevel9  0xFFFFFF09

#define rfbEncodingXCursor         0xFFFFFF10
#define rfbEncodingRichCursor      0xFFFFFF11

#define rfbEncodingLastRect        0xFFFFFF20
 
#define rfbEncodingQualityLevel0   0xFFFFFFE0
#define rfbEncodingQualityLevel1   0xFFFFFFE1
#define rfbEncodingQualityLevel2   0xFFFFFFE2
#define rfbEncodingQualityLevel3   0xFFFFFFE3
#define rfbEncodingQualityLevel4   0xFFFFFFE4
#define rfbEncodingQualityLevel5   0xFFFFFFE5
#define rfbEncodingQualityLevel6   0xFFFFFFE6
#define rfbEncodingQualityLevel7   0xFFFFFFE7
#define rfbEncodingQualityLevel8   0xFFFFFFE8
#define rfbEncodingQualityLevel9   0xFFFFFFE9

/*****************************************************************************
 *
 * Message definitions (client -> server)
 *
 *****************************************************************************/


/*-----------------------------------------------------------------------------
 * SetPixelFormat - tell the RFB server the format in which the client wants
 * pixels sent.
 */

typedef struct {
    u_int8_t       type_8;		/* always rfbSetPixelFormat */
    u_int8_t       pad1;
    u_int16_t      pad2;
    rfbPixelFormat format;
} PACKED_STRUCT rfbSetPixelFormatMsg;


/*-----------------------------------------------------------------------------
 * FixColourMapEntries - when the pixel format uses a "colour map", fix
 * read-only colour map entries.
 *
 *    ***************** NOT CURRENTLY SUPPORTED *****************
 */

typedef struct {
    u_int8_t  type_8;			/* always rfbFixColourMapEntries */
    u_int8_t  pad;
    u_int16_t firstColour_be16;
    u_int16_t nColours_be16;

    /* Followed by nColours * 3 * u_int16_t
       r1, g1, b1, r2, g2, b2, r3, g3, b3, ..., rn, bn, gn */

} PACKED_STRUCT rfbFixColourMapEntriesMsg;


/*-----------------------------------------------------------------------------
 * SetEncodings - tell the RFB server which encoding types we accept.  Put them
 * in order of preference, if we have any.  We may always receive raw
 * encoding, even if we don't specify it here.
 */

typedef struct {
    u_int8_t  type_8;			/* always rfbSetEncodings */
    u_int8_t  pad;
    u_int16_t nEncodings_be16;
    /* followed by nEncodings_be16 * u_int32_t encoding types */
} PACKED_STRUCT rfbSetEncodingsMsg;


/*-----------------------------------------------------------------------------
 * FramebufferUpdateRequest - request for a framebuffer update.  If incremental
 * is true then the client just wants the changes since the last update.  If
 * false then it wants the whole of the specified rectangle.
 */

typedef struct {
    u_int8_t  type_8;			/* always rfbFramebufferUpdateRequest */
    u_int8_t  incremental_8;
    u_int16_t x_be16;
    u_int16_t y_be16;
    u_int16_t w_be16;
    u_int16_t h_be16;
} PACKED_STRUCT rfbFramebufferUpdateRequestMsg;


/* ------------------------------------------------------------------------- *
 * login, client wants to create session
 * ------------------------------------------------------------------------- */

typedef struct {
    u_int8_t    type_8;
    u_int8_t    auth_method_8; /* see rfbAuth_XXXX above */
    u_int8_t    name_len_8;
    /* u_int8_t name[name_len]; - user name */
} PACKED_STRUCT rfbLoginMsgV1_15;

typedef struct {
    u_int8_t    type_8;
    u_int8_t    auth_method_8; /* see rfbAuth_XXXX above */
    u_int8_t    name_len_8;
    u_int8_t    pad;
    u_int32_t   connection_flags_be32;
    /* u_int8_t name[name_len]; - user name */
} PACKED_STRUCT rfbLoginMsgV1_16;

/* ------------------------------------------------------------------------- *
 * challenge response, client answers challenge by sending encrypted data
 * ------------------------------------------------------------------------- */

typedef struct {
    u_int8_t    type_8;
    u_int8_t    data_len_8;
    /* u_int8_t data[data_len]; - encrypted data */
} PACKED_STRUCT rfbChalRespMsg;

/*-----------------------------------------------------------------------------
 * KeyEvent - key press or release
 */

typedef struct {
    u_int8_t type_8;		/* always rfbKeyEvent */
    u_int8_t key_8;
} PACKED_STRUCT rfbKeyEventMsg;


typedef struct {
    u_int8_t  type_8;			/* always rfbKeyEvent */
    u_int8_t  down_8;			/* true if down (press), false if up */
    u_int16_t length_be16;
} PACKED_STRUCT rfbScancodeEventMsg;

// LARA Messages

/* ------------------------------------------------------------------------- *
 * MouseStateEvent - show/hide mouse cursor on host
 * ------------------------------------------------------------------------- */
typedef struct {
    u_int8_t  type_8;
    u_int8_t  state_8;
} PACKED_STRUCT rfbMousestateEventMsg;

#define rfbMouseStateToggle		0x04

/* ------------------------------------------------------------------------- *
 * MouseSyncEvent - trigger mouse/acc-table detection or resync
 * ------------------------------------------------------------------------- */
typedef struct {
    u_int8_t  type_8;
    u_int8_t  sync_type_8;
} PACKED_STRUCT rfbMousesyncEventMsg;

#define rfbMouseSyncNorm		0x00
#define rfbMouseSyncHard		0x01
#define rfbMouseSyncFast		0x02

/* ------------------------------------------------------------------------- *
 * CursorChange - save the name of the current local cursor shape
 * ------------------------------------------------------------------------- */
typedef struct {
    u_int8_t  type_8;
    u_int8_t  len_8;
    /* followed by len chars */
} PACKED_STRUCT rfbCursorChangeEventMsg;

/* ------------------------------------------------------------------------- *
 * KVMSwitch - switch the current kvm port to the given number
 * ------------------------------------------------------------------------- */
typedef struct {
    u_int8_t  type_8;
    u_int8_t  pad;
    u_int16_t port_be16;
} PACKED_STRUCT rfbKVMSwitchEventMsg;

/* ------------------------------------------------------------------------- *
 * VideoSettingsC2S - update the video/tft settings from client on LARA
 * ------------------------------------------------------------------------- */

typedef struct {
    u_int8_t  type_8;
    u_int8_t  event_8;
    u_int16_t value_be16;
} PACKED_STRUCT rfbVideoSettingsC2SMsg;

#define rfbVSBrightnessV1_0		0x00
#define rfbVSContrastV1_0		0x01
#define rfbVSBlackLevelV1_0		0x02
#define rfbVSClockV1_0			0x03
#define rfbVSPhaseV1_0			0x04
#define rfbVSXOffsetV1_0		0x05
#define rfbVSYOffsetV1_0		0x06
#define rfbVSResetAllV1_0		0x07
#define rfbVSResetCurrentV1_0		0x08
#define rfbVSSaveSettingsV1_0		0x09
#define rfbVSCancelSettingsV1_0		0x0A
#define rfbVSAutoAdjustV1_0		0x0B

#define rfbVSBrightnessV1_1		0x00
#define rfbVSContrastRedV1_1		0x01
#define rfbVSContrastGreenV1_1		0x02
#define rfbVSContrastBlueV1_1		0x03
#define rfbVSBlackLevelV1_1		0x04
#define rfbVSClockV1_1			0x05
#define rfbVSPhaseV1_1			0x06
#define rfbVSXOffsetV1_1		0x07
#define rfbVSYOffsetV1_1		0x08
#define rfbVSResetAllV1_1		0x09
#define rfbVSResetCurrentV1_1		0x0A
#define rfbVSSaveSettingsV1_1		0x0B
#define rfbVSCancelSettingsV1_1		0x0C
#define rfbVSAutoAdjustV1_1		0x0D

#define rfbVSBrightnessV1_8		0x00
#define rfbVSContrastRedV1_8		0x01
#define rfbVSContrastGreenV1_8		0x02
#define rfbVSContrastBlueV1_8		0x03
#define rfbVSClockV1_8			0x04
#define rfbVSPhaseV1_8			0x05
#define rfbVSXOffsetV1_8		0x06
#define rfbVSYOffsetV1_8		0x07
#define rfbVSResetAllV1_8		0x08
#define rfbVSResetCurrentV1_8		0x09
#define rfbVSSaveSettingsV1_8		0x0A
#define rfbVSCancelSettingsV1_8		0x0B
#define rfbVSAutoAdjustV1_8		0x0C

/* ------------------------------------------------------------------------- *
 * VideoSettingsRequest - request/refuse video settings updates from server
 * ------------------------------------------------------------------------- */

typedef struct {
    u_int8_t  type_8;
    u_int8_t  request_8;
} PACKED_STRUCT rfbVideoSettingsReqMsg;

#define rfbVSUpdateNow			0x01
#define rfbVSUpdateDeny			0x02

/* ------------------------------------------------------------------------- *
 * VideoQualityC2S - update the video quality settings from client on LARA
 * ------------------------------------------------------------------------- */

typedef struct {
    u_int8_t  type_8;
    u_int8_t  event_8;
    u_int8_t  value_8;
} PACKED_STRUCT rfbVideoQualityC2SMsg;

#define rfbVQNoiseFilter		0x01
#define rfbVQQuality			0x02

/* ------------------------------------------------------------------------- *
 * VideoQualityRequest - request video quality updates from server
 * ------------------------------------------------------------------------- */

typedef struct {
    u_int8_t  type_8;
    u_int8_t  request_8;
} PACKED_STRUCT rfbVideoQualityReqMsg;

#define rfbVQUpdateNow			0x01

/* ------------------------------------------------------------------------- *
 * UserPropChange
 * ------------------------------------------------------------------------- */

typedef struct {
    u_int8_t	type_8;
    u_int8_t	keyLength_8;
    u_int8_t	valueLength_8;
} PACKED_STRUCT rfbUserPropChangeMsg;

/* ------------------------------------------------------------------------- *
 * GlobalPropChange
 * ------------------------------------------------------------------------- */

typedef struct {
    u_int8_t	type_8;
    u_int8_t	keyLength_8;
    u_int8_t	valueLength_8;
} PACKED_STRUCT rfbGlobalPropChangeMsg;

/* ------------------------------------------------------------------------- *
 * PingReply
 * ------------------------------------------------------------------------- */

typedef struct {
    u_int8_t	type_8;
    u_int8_t	pad1;
    u_int16_t	pad2;
    u_int32_t	serial_be32;
} PACKED_STRUCT rfbPingReplyMsg;

/* ------------------------------------------------------------------------- *
 * BandwithReply
 * ------------------------------------------------------------------------- */

typedef struct {
    u_int8_t	type_8;
    u_int8_t	stage_8;
} PACKED_STRUCT rfbBandwidthReplyMsg;

#define rfbBandwidthTickStage1		0x01
#define rfbBandwidthTickStage2		0x02

/*-----------------------------------------------------------------------------
 * PointerEvent - mouse/pen move and/or button press.
 */

typedef struct {
    u_int8_t  type_8;		/* always rfbPointerEvent */
    u_int8_t  buttonMask_8;	/* bits 0-7 are buttons 1-8, 0=up, 1=down */
    u_int16_t x_be16;
    u_int16_t y_be16;
} PACKED_STRUCT rfbPointerEventMsgV1_0;

typedef struct {
    u_int8_t  type_8;		/* always rfbPointerEvent */
    u_int8_t  buttonMask_8;	/* bits 0-7 are buttons 1-8, 0=up, 1=down */
    u_int16_t x_be16;
    u_int16_t y_be16;
    u_int16_t z_be16;		/* wheel mouse data (-7 ... 8) */
} PACKED_STRUCT rfbPointerEventMsgV1_3;

#define rfbButton1Mask 0x01
#define rfbButton2Mask 0x02
#define rfbButton3Mask 0x04
#define rfbButton4Mask 0x08
#define rfbButton5Mask 0x10

/* ------------------------------------------------------------------------- *
 * rfbRCModeRequest - switch into another Remote Console Mode
 * (RDP or secondary RFB)
 * ------------------------------------------------------------------------- */
typedef struct {
    u_int8_t  type_8;
    u_int8_t  mode_8;
} PACKED_STRUCT rfbRCModeRequestMsg;

#define rfbRCModeRDPEnter		0
#define rfbRCModeRDPLeave		1
#define rfbRCModeSecRFBEnter		2
#define rfbRCModeSecRFBLeave		3

/* ------------------------------------------------------------------------- *
 * SARA replay (SP) messages
 * ------------------------------------------------------------------------- */

typedef struct {
    u_int8_t   type_8;
    u_int8_t   stop_at_end_8;
    u_int16_t  pad;
    u_int32_t  session_id_be32;
} PACKED_STRUCT rfbSpSetSessionMsg;

typedef struct {
    u_int8_t   type_8;
    u_int8_t   pad1;
    u_int16_t  pad2;
    u_int32_t  cim_id_be32;
} PACKED_STRUCT rfbSpSetCimIdMsg;


/*****************************************************************************
 *
 * Bi-directional message types
 *
 *****************************************************************************/

/*-----------------------------------------------------------------------------
 * EnableExtension - tell client/server that a particular extension is available,
 * or reconfigure an extension.  If the new_msg field is non-zero then the sender
 * is indicating that the specified protocol uses the given message number.
 * The recipient may then send rfbExtensionData messages on the given message
 * number, provided the extension configuration data is recognised.
 *
 * The length field in the ExtensionData message may be invalid, if the zeroeth
 * bit of "flags" was not set in the corresponding EnableExtensionRequest.
 * If the length field is invalid then obviously the message can't be dealt
 * with by dumb proxies - avoid this where possible.
 *
 * The first bit of the flags field indicates that the extension is a new, named
 * encoding.  This form is used by servers to tell clients the names of the
 * encodings they support.  Typically, a receiving client will then dispatch
 * an "enable encoding" message for the specified encoding number, if supported.
 */
typedef struct {
    u_int8_t  type_8;			/* always rfbEnableExtensionRequest */
    u_int8_t  new_msg_8;
    u_int8_t  flags_8;
    u_int8_t  pad1;
    u_int32_t length_be32;
    /* Followed by <length> bytes of data */
} PACKED_STRUCT rfbEnableExtensionRequestMsg;

typedef struct {
    u_int8_t  type_8;			/* always >= rfbExtensionData */
    u_int8_t  pad1;
    u_int16_t pad2;
    u_int32_t length_be32;		/* Must be correct if used */
    /* Followed by <length> bytes of data */
} PACKED_STRUCT rfbExtensionDataMsg;

/* -----------------------------------------------------------------------------
 * Bell - ring a bell on the client if it has one.
 * ------------------------------------------------------------------------- */

typedef struct {
    u_int8_t type_8;			/* always rfbBell */
} PACKED_STRUCT rfbBellMsg;


/* ------------------------------------------------------------------------- *
 * PingRequest
 * ------------------------------------------------------------------------- */

typedef struct {
    u_int8_t	type_8;
    u_int8_t	pad1;
    u_int16_t	pad2;
    u_int32_t	serial_be32;
} PACKED_STRUCT rfbPingRequestEventMsg;

/* ------------------------------------------------------------------------- *
 * BandwithRequest
 * ------------------------------------------------------------------------- */

typedef struct {
    u_int8_t	type_8;
    u_int8_t	pad;
    u_int16_t	length_be16;
    //followed by length test bytes
} PACKED_STRUCT rfbBandwidthRequestEventMsg;


/* ------------------------------------------------------------------------- *
 * SARA replay (SP) messages
 * ------------------------------------------------------------------------- */

typedef struct {
    u_int8_t  type_8;
    u_int8_t  command_8;
    union {
    	u_int16_t param16_be16;
    	struct {
    	    u_int8_t param8_8[2];
    	} PACKED_STRUCT;
    } PACKED_STRUCT;
} PACKED_STRUCT rfbSpCommandMsg;

// commands
#define rfbSpCommandPlay	0
	/* params: none */
#define rfbSpCommandPause	1
	/* params: none */
#define rfbSpCommandStop	2
	/* params: none */
#define rfbSpCommandSetSpeed	3
	/* params: param8[0]: speed (signed power of 2,
				     0 --> 1x, 2 --> 4x -3 --> 1/8x
		   param8[1]: reserved (must be set to 0) */
#define rfbSpCommandSeek	4

typedef struct {
    u_int8_t      type_8;
    u_int8_t      pad1;
    u_int16_t     pad2;
    rfbTimestamp  startTime;
    rfbTimestamp  endTime;
} PACKED_STRUCT rfbSpSetSessionTimesMsg;


/*****************************************************************************
 *
 * Server -> client message definitions
 *
 *****************************************************************************/


/*-----------------------------------------------------------------------------
 * FramebufferUpdate - a block of rectangles to be copied to the framebuffer.
 *
 * This message consists of a header giving the number of rectangles of pixel
 * data followed by the rectangles themselves.  The header is padded so that
 * together with the type byte it is an exact multiple of 4 bytes (to help
 * with alignment of 32-bit pixels):
 */

typedef struct {
    u_int8_t  type_8;			/* always rfbFramebufferUpdate */
    u_int8_t  flags_8;		/* only RFB 1.16 or newer! */
    u_int16_t nRects_be16;
    /* followed by nRects rectangles */
} PACKED_STRUCT rfbFramebufferUpdateMsg;

#define rfbFramebufferUpdateFlagTimestamp   (1 << 0)
#define rfbFramebufferUpdateFlagFullFrame   (1 << 1)

/*
 * Each rectangle of pixel data consists of a header describing the position
 * and size of the rectangle and a type word describing the encoding of the
 * pixel data, followed finally by the pixel data.  Note that if the client has
 * not sent a SetEncodings message then it will only receive raw pixel data.
 * Also note again that this structure is a multiple of 4 bytes.
 */

typedef struct {
    rfbRectangle r;
    u_int32_t encoding_be32;	/* one of the encoding types rfbEncoding... */
} PACKED_STRUCT rfbFramebufferUpdateRectHeader;

/* ------------------------------------------------------------------------- *
 * AckPixelFormat - Tell the client we accepted a pixel format from it and
 * all further fb updates are sent in this format.
 * ------------------------------------------------------------------------- */

typedef struct {
    u_int8_t  type_8;
    u_int8_t  pad1;
    u_int16_t pad2;
    rfbPixelFormat pix_fmt;
} PACKED_STRUCT rfbAckPixelFormatMsg;

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *
 * Auhtentication Capabilities - Tell the client what auth methods
 * are supported.
 * - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

typedef struct {
    u_int8_t    type_8;
    u_int8_t    auth_caps_8; /* see rfbAuthXXXX above */
} PACKED_STRUCT rfbAuthCapsMsg;

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *
 * Session Challenge - Send the client a server generated data used
 * as encryption seed.
 * - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

typedef struct {
    u_int8_t    type_8;
    u_int8_t    data_len_8;
    /* u_int8_t data[data_len] - challenge data (encryption seed) */
} PACKED_STRUCT rfbSessChalMsg;

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *
 * Auhtentication successful - just an ACK from the server
 * - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

typedef struct {
    u_int8_t    type_8;
} PACKED_STRUCT rfbAuthSuccessfulMsgV1_15;

typedef struct {
    u_int8_t   type_8;
    u_int8_t   pad1;
    u_int16_t  pad2;
    u_int32_t  connection_flags_be32;
} PACKED_STRUCT rfbAuthSuccessfulMsgV1_16;

/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 * Raw Encoding.  Pixels are sent in top-to-bottom scanline order,
 * left-to-right within a scanline with no padding in between.
 */


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *
 * Raw VSC Encoding. Pixels are sent to the client top-to-bottom in
 * VSC like hextiles. That means, the data is not sent as a linear
 * framebuffer. The client has to rearrange the data into a linear
 * order if it wants to draw it. This should be the fastest way to
 * encode the framebuffer because erla does no data translation at
 * all (only a color conversion in the 8-bit case). 16 Bit Raw VSC
 * should be fastest.
 *
 * The first byte of the encoding represents a flag (subencoding byte)
 * with the following meaning:
 *
 * rfbRawVSCIsVSC: The data is in VSC framebuffer order. If this
 *                 flag is not set, the data is sent as a normal
 *                   RAW update and the other flags are ignored.
 *
 * rfbRawVSCBigEndian: The pixel data is sent in Big Endian format
 *                     (or Little Endian if not set). The Client
 *                     has to make sure that the pixel format is
 *                     converted into the client's pixel format.
 * 
 * This encoding is only used with VSC devices!
 * - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

typedef struct {
    u_int8_t flags_8;
} PACKED_STRUCT rfbRawVscHdr;

#define rfbRawVSCIsVSC				(1 << 0)
#define rfbRawVSCUseServersPixelFormat		(1 << 1)
#define rfbRawVSCBigEndian			(1 << 2)

/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 * CopyRect Encoding.  The pixels are specified simply by the x and y position
 * of the source rectangle.
 */

typedef struct {
    u_int16_t srcX_be16;
    u_int16_t srcY_be16;
} PACKED_STRUCT rfbCopyRect;


/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 * RRE - Rise-and-Run-length Encoding.  We have an rfbRREHeader structure
 * giving the number of subrectangles following.  Finally the data follows in
 * the form [<bgpixel><subrect><subrect>...] where each <subrect> is
 * [<pixel><rfbRectangle>].
 */

typedef struct {
    u_int32_t nSubrects_be32;
} PACKED_STRUCT rfbRREHeader;


/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 * CoRRE - Compact RRE Encoding.  We have an rfbRREHeader structure giving
 * the number of subrectangles following.  Finally the data follows in the form
 * [<bgpixel><subrect><subrect>...] where each <subrect> is
 * [<pixel><rfbCoRRERectangle>].  This means that
 * the whole rectangle must be at most 255x255 pixels.
 */

typedef struct {
    u_int8_t x_8;
    u_int8_t y_8;
    u_int8_t w_8;
    u_int8_t h_8;
} PACKED_STRUCT rfbCoRRERectangle;


/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 * Hextile Encoding.  The rectangle is divided up into "tiles" of 16x16 pixels,
 * starting at the top left going in left-to-right, top-to-bottom order.  If
 * the width of the rectangle is not an exact multiple of 16 then the width of
 * the last tile in each row will be correspondingly smaller.  Similarly if the
 * height is not an exact multiple of 16 then the height of each tile in the
 * final row will also be smaller.  Each tile begins with a "subencoding" type
 * byte, which is a mask made up of a number of bits.  If the Raw bit is set
 * then the other bits are irrelevant; w*h pixel values follow (where w and h
 * are the width and height of the tile).  Otherwise the tile is encoded in a
 * similar way to RRE, except that the position and size of each subrectangle
 * can be specified in just two bytes.  The other bits in the mask are as
 * follows:
 *
 * BackgroundSpecified - if set, a pixel value follows which specifies
 *    the background colour for this tile.  The first non-raw tile in a
 *    rectangle must have this bit set.  If this bit isn't set then the
 *    background is the same as the last tile.
 *
 * ForegroundSpecified - if set, a pixel value follows which specifies
 *    the foreground colour to be used for all subrectangles in this tile.
 *    If this bit is set then the SubrectsColoured bit must be zero.
 *
 * AnySubrects - if set, a single byte follows giving the number of
 *    subrectangles following.  If not set, there are no subrectangles (i.e.
 *    the whole tile is just solid background colour).
 *
 * SubrectsColoured - if set then each subrectangle is preceded by a pixel
 *    value giving the colour of that subrectangle.  If not set, all
 *    subrectangles are the same colour, the foreground colour;  if the
 *    ForegroundSpecified bit wasn't set then the foreground is the same as
 *    the last tile.
 *
 * The position and size of each subrectangle is specified in two bytes.  The
 * Pack macros below can be used to generate the two bytes from x, y, w, h,
 * and the Extract macros can be used to extract the x, y, w, h values from
 * the two bytes.
 */

#define rfbHextileRaw			(1 << 0)
#define rfbHextileBackgroundSpecified	(1 << 1)
#define rfbHextileForegroundSpecified	(1 << 2)
#define rfbHextileAnySubrects		(1 << 3)
#define rfbHextileSubrectsColoured	(1 << 4)
#define rfbHextileZlibRaw		(1 << 5)
#define rfbHextileZlibHex		(1 << 6)

//compression types from file
#define rfbFileComprNone		0
#define rfbFileComprZlib		1
#define rfbFileComprJpeg		2

#define rfbHextilePackXY(x,y)		(((x) << 4) | (y))
#define rfbHextilePackWH(w,h)		((((w)-1) << 4) | ((h)-1))
#define rfbHextileExtractX(byte)	((byte) >> 4)
#define rfbHextileExtractY(byte)	((byte) & 0xf)
#define rfbHextileExtractW(byte)	(((byte) >> 4) + 1)
#define rfbHextileExtractH(byte)	(((byte) & 0xf) + 1)

/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 * ZLIB - zlib compression Encoding.  We have an rfbZlibHeader structure
 * giving the number of bytes to follow.  Finally the data follows in
 * zlib compressed format.
 */

typedef struct {
    u_int32_t nBytes_be32;
} PACKED_STRUCT rfbZlibHeader;


/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 * Tight Encoding.
 *
 *-- The first byte of each Tight-encoded rectangle is a "compression control
 *   byte". Its format is as follows (bit 0 is the least significant one):
 *
 *   bit 0:    if 1, then compression stream 0 should be reset;
 *   bit 1:    if 1, then compression stream 1 should be reset;
 *   bit 2:    if 1, then compression stream 2 should be reset;
 *   bit 3:    if 1, then compression stream 3 should be reset;
 *   bits 7-4: if 1000 (0x08), then the compression type is "fill",
 *             if 1001 (0x09), then the compression type is "jpeg", -> is no more supported
 *             if 1010 (0x0a), then the compression type is "1-bitFullColorBlackWhite",
 *             if 1011 (0x0b), then the compression type is "2-bitFullColorGrayscale",
 *             if 1100 (0x0c), then the compression type is "4-bitFullColorGrayscale",
 *             if 1101 (0x0d), then the compression type is "4-bitFullColor16Colors",
 *             if 1111 (0x0f), then the compression type is "fill X-bit",
 *             if 0xxx, then the compression type is "basic",
 *             values greater than 1111 are not valid.
 *
 * If the compression type is "basic", then bits 6..4 of the
 * compression control byte (those xxx in 0xxx) specify the following:
 *
 *   bits 5-4:  decimal representation is the index of a particular zlib
 *              stream which should be used for decompressing the data;
 *   bit 6:     if 1, then a "filter id" byte is following this byte.
 *
 *-- The data that follows after the compression control byte described
 * above depends on the compression type ("fill", "jpeg" or "basic").
 *
 *-- If the compression type is "fill", then the only pixel value follows, in
 * client pixel format (see NOTE 1). This value applies to all pixels of the
 * rectangle.
 *
 *-- If the compression type is "jpeg", the following data stream looks like
 * this:
 *
 *   1..3 bytes:  data size (N) in compact representation;
 *   N bytes:     JPEG image.
 *
 * Data size is compactly represented in one, two or three bytes, according
 * to the following scheme:
 *
 *  0xxxxxxx                    (for values 0..127)
 *  1xxxxxxx 0yyyyyyy           (for values 128..16383)
 *  1xxxxxxx 1yyyyyyy zzzzzzzz  (for values 16384..4194303)
 *
 * Here each character denotes one bit, xxxxxxx are the least significant 7
 * bits of the value (bits 0-6), yyyyyyy are bits 7-13, and zzzzzzzz are the
 * most significant 8 bits (bits 14-21). For example, decimal value 10000
 * should be represented as two bytes: binary 10010000 01001110, or
 * hexadecimal 90 4E.
 *
 *-- If the compression type is "basic" and bit 6 of the compression control
 * byte was set to 1, then the next (second) byte specifies "filter id" which
 * tells the decoder what filter type was used by the encoder to pre-process
 * pixel data before the compression. The "filter id" byte can be one of the
 * following:
 *
 *   0:  no filter ("copy" filter);
 *   1:  "palette" filter;
 *   2:  "gradient" filter.
 *
 *-- If bit 6 of the compression control byte is set to 0 (no "filter id"
 * byte), or if the filter id is 0, then raw pixel values in the client
 * format (see NOTE 1) will be compressed. See below details on the
 * compression.
 *
 *-- The "gradient" filter pre-processes pixel data with a simple algorithm
 * which converts each color component to a difference between a "predicted"
 * intensity and the actual intensity. Such a technique does not affect
 * uncompressed data size, but helps to compress photo-like images better. 
 * Pseudo-code for converting intensities to differences is the following:
 *
 *   P[i,j] := V[i-1,j] + V[i,j-1] - V[i-1,j-1];
 *   if (P[i,j] < 0) then P[i,j] := 0;
 *   if (P[i,j] > MAX) then P[i,j] := MAX;
 *   D[i,j] := V[i,j] - P[i,j];
 *
 * Here V[i,j] is the intensity of a color component for a pixel at
 * coordinates (i,j). MAX is the maximum value of intensity for a color
 * component.
 *
 *-- The "palette" filter converts true-color pixel data to indexed colors
 * and a palette which can consist of 2..256 colors. If the number of colors
 * is 2, then each pixel is encoded in 1 bit, otherwise 8 bits is used to
 * encode one pixel. 1-bit encoding is performed such way that the most
 * significant bits correspond to the leftmost pixels, and each raw of pixels
 * is aligned to the byte boundary. When "palette" filter is used, the
 * palette is sent before the pixel data. The palette begins with an unsigned
 * byte which value is the number of colors in the palette minus 1 (i.e. 1
 * means 2 colors, 255 means 256 colors in the palette). Then follows the
 * palette itself which consist of pixel values in client pixel format (see
 * NOTE 1).
 *
 *-- The pixel data is compressed using the zlib library. But if the data
 * size after applying the filter but before the compression is less then 12,
 * then the data is sent as is, uncompressed. Four separate zlib streams
 * (0..3) can be used and the decoder should read the actual stream id from
 * the compression control byte (see NOTE 2).
 *
 * If the compression is not used, then the pixel data is sent as is,
 * otherwise the data stream looks like this:
 *
 *   1..3 bytes:  data size (N) in compact representation;
 *   N bytes:     zlib-compressed data.
 *
 * Data size is compactly represented in one, two or three bytes, just like
 * in the "jpeg" compression method (see above).
 *
 *-- NOTE 1. If the color depth is 24, and all three color components are
 * 8-bit wide, then one pixel in Tight encoding is always represented by
 * three bytes, where the first byte is red component, the second byte is
 * green component, and the third byte is blue component of the pixel color
 * value. This applies to colors in palettes as well.
 *
 *-- NOTE 2. The decoder must reset compression streams' states before
 * decoding the rectangle, if some of bits 0,1,2,3 in the compression control
 * byte are set to 1. Note that the decoder must reset zlib streams even if
 * the compression type is "fill" or "jpeg".
 *
 *-- NOTE 3. The "gradient" filter and "jpeg" compression may be used only
 * when bits-per-pixel value is either 16 or 32, not 8.
 *
 *-- NOTE 4. The width of any Tight-encoded rectangle cannot exceed 2048
 * pixels. If a rectangle is wider, it must be split into several rectangles
 * and each one should be encoded separately.
 *
 */

#define rfbTightStreamId		0x03
#define rfbTightExplicitFilter		0x07
#define rfbTightFill			0x08
#define rfbTightCheckForXBitFullColor	0x08
#define rfbTight1BitFullColorBlackWhite	0x0a
#define rfbTight2BitFullColorGrayscale	0x0b
#define rfbTight4BitFullColorGrayscale	0x0c
#define rfbTight4BitFullColor16Colors	0x0d
#define rfbTightFillXBit		0x0f
#define rfbTightMaxSubencoding		0x0f

/* Filters to improve compression efficiency */
#define rfbTightFilterCopy		0x00
#define rfbTightFilterPalette		0x01
#define rfbTightFilterGradient		0x02

#define rfbTight1BitBlackWhite		1
#define rfbTight2BitGrayscale		2
#define rfbTight4BitGrayscale		3
#define rfbTight4Bit16Colors		4
#define rfbTight8Bit256Colors		8

#define rfbTightCacheMinComprLevel	6
#define rfbTightCacheBufferDepth	8
#define rfbTightTileDimension		16
#define rfbTightCacheTag		0x0C
#define rfbTightCacheClientIndex	0x7F
#define	rfbTightTileCached		0x80
#define	rfbTightTileNot_Cached		0x00
#define	rfbTightCacheAll		0x00
#define	rfbTightCacheNone		0x04
#define	rfbTightCacheMix		0x08

#define	rfbTightCacheMinUpdRegA		0x4000

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - *
 * LRLE Hardware Encoding in the VSC 2.0. Supports different output subencodings.
 * Each subencoding corresponds with a fixed, predefined set of encoding
 * algorithms and settings.
 * Currently supports the LRLE encoding.
 * Hardware encoding may be sent through ZLIB (see encoding field in the rect
 * header).
 * When ZLIB is active, each rect content consists of:
 *   1..3 bytes:  uncompressed size (N) of the rect, compact representation
 *   A number of portions, each consisting of:
 *     1..3 bytes: compressed size (M) of the portion, compact representation
 *     M bytes   : ZLIB compressed data
 * - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

#define rfbLRLESubenc15bitDirectLossy		0
#define rfbLRLESubenc15bitDirectLossless	1
#define rfbLRLESubenc7bitDirectLossy		2
#define rfbLRLESubenc7bitDirectLossless		3
#define rfbLRLESubenc4bitPaletteLossy		4
#define rfbLRLESubenc4bitPaletteLossless	5
#define rfbLRLESubenc4bitGreyLossy		6
#define rfbLRLESubenc4bitGreyLossless		7
#define rfbLRLESubenc3bitGreyLossy		8
#define rfbLRLESubenc3bitGreyLossless		9
#define rfbLRLESubenc2bitGreyLossy		10
#define rfbLRLESubenc2bitGreyLossless		11
#define rfbLRLESubenc1bitGreyLossy		12
#define rfbLRLESubenc1bitGreyLossless		13

#define rfbLRLESubencMax			rfbLRLESubenc1bitGreyLossless
#define rfbLRLESubencCount			(rfbLRLESubencMax + 1)

typedef struct {
    u_int32_t size_be32;
} PACKED_STRUCT rfbHwencHdr;

/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 * XCursor encoding. This is a special encoding used to transmit X-style
 * cursor shapes from server to clients. Note that for this encoding,
 * coordinates in rfbFramebufferUpdateRectHeader structure hold hotspot
 * position (r.x, r.y) and cursor size (r.w, r.h). If (w * h != 0), two RGB
 * samples are sent after header in the rfbXCursorColors structure. They
 * denote foreground and background colors of the cursor. If a client
 * supports only black-and-white cursors, it should ignore these colors and
 * assume that foreground is black and background is white. Next, two bitmaps
 * (1 bits per pixel) follow: first one with actual data (value 0 denotes
 * background color, value 1 denotes foreground color), second one with
 * transparency data (bits with zero value mean that these pixels are
 * transparent). Both bitmaps represent cursor data in a byte stream, from
 * left to right, from top to bottom, and each row is byte-aligned. Most
 * significant bits correspond to leftmost pixels. The number of bytes in
 * each row can be calculated as ((w + 7) / 8). If (w * h == 0), cursor
 * should be hidden (or default local cursor should be set by the client).
 */

typedef struct {
    u_int8_t foreRed_8;
    u_int8_t foreGreen_8;
    u_int8_t foreBlue_8;
    u_int8_t backRed_8;
    u_int8_t backGreen_8;
    u_int8_t backBlue_8;
} PACKED_STRUCT rfbXCursorColors;


/*- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 * RichCursor encoding. This is a special encoding used to transmit cursor
 * shapes from server to clients. It is similar to the XCursor encoding but
 * uses client pixel format instead of two RGB colors to represent cursor
 * image. For this encoding, coordinates in rfbFramebufferUpdateRectHeader
 * structure hold hotspot position (r.x, r.y) and cursor size (r.w, r.h).
 * After header, two pixmaps follow: first one with cursor image in current
 * client pixel format (like in raw encoding), second with transparency data
 * (1 bit per pixel, exactly the same format as used for transparency bitmap
 * in the XCursor encoding). If (w * h == 0), cursor should be hidden (or
 * default local cursor should be set by the client).
 */


/*-----------------------------------------------------------------------------
 * SetColourMapEntries - these messages are only sent if the pixel
 * format uses a "colour map" (i.e. trueColour false) and the client has not
 * fixed the entire colour map using FixColourMapEntries.  In addition they
 * will only start being sent after the client has sent its first
 * FramebufferUpdateRequest.  So if the client always tells the server to use
 * trueColour then it never needs to process this type of message.
 */

typedef struct {
    u_int8_t  type_8;			/* always rfbSetColourMapEntries */
    u_int8_t  pad;
    u_int16_t firstColour_be16;
    u_int16_t nColours_be16;

    /* Followed by nColours * 3 * u_int16_t
       r1, g1, b1, r2, g2, b2, r3, g3, b3, ..., rn, bn, gn */

} PACKED_STRUCT rfbSetColourMapEntriesMsg;


// LARA Message
/* ------------------------------------------------------------------------- *
 * VideoSettingsS2C - update the video/tft settings on clients
 * ------------------------------------------------------------------------- */
typedef struct {
    u_int8_t  type_8;
    u_int8_t  brightness_8;
    u_int8_t  contrast_8;
    u_int8_t  blacklevel_8;
    u_int16_t clock_be16;
    u_int16_t phase_be16;
    u_int16_t x_offset_be16;
    u_int16_t y_offset_be16;
    u_int16_t x_res_be16;
    u_int16_t y_res_be16;
    u_int16_t refresh_be16;
    u_int16_t y_max_offset_be16;
} PACKED_STRUCT rfbVideoSettingsS2CMsgV1_0;

typedef struct {
    u_int8_t  type_8;
    u_int8_t  brightness_8;
    u_int8_t  contrast_red_8;
    u_int8_t  contrast_green_8;
    u_int8_t  contrast_blue_8;
    u_int8_t  blacklevel_8;
    u_int16_t clock_be16;
    u_int16_t phase_be16;
    u_int16_t x_offset_be16;
    u_int16_t y_offset_be16;
    u_int16_t x_res_be16;
    u_int16_t y_res_be16;
    u_int16_t refresh_be16;
    u_int16_t y_max_offset_be16;
} PACKED_STRUCT rfbVideoSettingsS2CMsgV1_1;

typedef struct {
    u_int8_t  type_8;
    u_int8_t  pad_8;
    u_int8_t  brightness_8;
    u_int8_t  contrast_red_8;
    u_int8_t  contrast_green_8;
    u_int8_t  contrast_blue_8;
    u_int16_t clock_be16;
    u_int16_t phase_be16;
    u_int16_t x_offset_be16;
    u_int16_t y_offset_be16;
    u_int16_t x_res_be16;
    u_int16_t y_res_be16;
    u_int16_t refresh_be16;
    u_int16_t y_max_offset_be16;
} PACKED_STRUCT rfbVideoSettingsS2CMsgV1_8;

/* ------------------------------------------------------------------------- *
 * VideoRefreshRequest - request a full video update
 * ------------------------------------------------------------------------- */
typedef struct {
    u_int8_t type_8;
} PACKED_STRUCT rfbVideoRefreshRequestMsg;

/* ------------------------------------------------------------------------- *
 * VideoQualityS2C - update the video quality settings on clients
 * ------------------------------------------------------------------------- */
typedef struct {
    u_int8_t  type_8;
    u_int8_t  noise_filter_8;
    u_int8_t  quality_speed_8;
} PACKED_STRUCT rfbVideoQualityS2CMsgV1_10;

typedef struct {
    u_int8_t  type_8;
    u_int8_t  noise_filter_8;
} PACKED_STRUCT rfbVideoQualityS2CMsgV1_12;

/* ------------------------------------------------------------------------- *
 * Quit Message - exit the client
 * ------------------------------------------------------------------------- */
typedef struct {
    u_int8_t type_8;
    u_int8_t reason_8;
} PACKED_STRUCT rfbQuitMsg;

// quit reasons
#define rfbQuitNoPerm			0x01
#define rfbQuitExclAccess		0x02
#define rfbQuitRejected			0x03
#define rfbQuitNoSrvPasswd		0x04
#define rfbQuitLoopback			0x05
#define rfbQuitAuthFailed		0x06
#define rfbQuitKVMPortDenied		0x07
#define rfbQuitTooManyClients		0x08
#define rfbQuitUnexpected		0x10
#define rfbQuitBadProtoVersion		0x11
#define rfbQuitProtoError		0x12
#define rfbQuitInternalError		0x13
#define rfbQuitWrongReplayParameter	0x14
#define rfbQuitNoForcedSSLConn         	0x15

/* ------------------------------------------------------------------------- *
 * rfbKeyboardLayoutEvent - Change of Keyboard Layout
 * ------------------------------------------------------------------------- */
typedef struct {
    u_int8_t  type_8;
    u_int8_t  pad;
    u_int16_t length_be16;
} PACKED_STRUCT rfbKeyboardLayoutEventMsg;

/* ------------------------------------------------------------------------- *
 * rfbUtf8String - erla sends an UTF8 encoded message (chat)
 * ------------------------------------------------------------------------- */
typedef struct {
    u_int8_t  type_8;
    u_int8_t  pad;
    u_int16_t len_be16;
    /* followed by len bytes of the utf-8 encoded string. */
} PACKED_STRUCT rfbUtf8StringMsg;

/* ------------------------------------------------------------------------- *
 * rfbMessage - erla sends a message to be displayed
 * ------------------------------------------------------------------------- */
typedef struct {
    u_int8_t  type_8;
    u_int8_t  pad1;
    u_int16_t pad2;
    u_int32_t len_be32;
    /* followed by len bytes of the string. */
} PACKED_STRUCT rfbMessageV1_0;

typedef struct {
    u_int8_t  type_8;
    u_int8_t  pad1;
    u_int16_t pad2;
    u_int32_t len_be32;
    rfbTimestamp timestamp;
    /* followed by len bytes of the string. */
} PACKED_STRUCT rfbMessageV1_16;

/* ------------------------------------------------------------------------- *
 * rfbCommand - erla sends a command to the client
 * ------------------------------------------------------------------------- */
typedef struct {
    u_int8_t  type_8;
    u_int8_t  pad;
    u_int16_t length_command_be16;
    u_int16_t length_params_be16;
    /* followed by len bytes of the string. */
} PACKED_STRUCT rfbCommandMsg;

/* ------------------------------------------------------------------------- *
 * rfbOSDState - send OSD message and screen blanking state
 * ------------------------------------------------------------------------- */
typedef struct {
    u_int8_t  type_8;
    u_int8_t  blank_8;
    u_int16_t timeout_be16;
    u_int16_t length_be16;
    /* followed by length chars */
} PACKED_STRUCT rfbOSDStateMsgV1_7;

typedef struct {
    u_int8_t     type_8;
    u_int8_t     blank_8;
    u_int16_t    timeout_be16;
    u_int16_t    length_be16;
    u_int16_t    pad;
    rfbTimestamp timestamp;
    /* followed by length chars */
} PACKED_STRUCT rfbOSDStateMsgV1_16;

#define rfbBlankStateDontBlank 0
#define rfbBlankStateBlank 1

/* ------------------------------------------------------------------------- *
 * rfbRCModeReply - switch into another Remote Console Mode
 * (RDP or secondary RFB)
 * ------------------------------------------------------------------------- */
typedef struct {
    u_int8_t  type_8;
    u_int8_t  mode_8;
} PACKED_STRUCT rfbRCModeReplyMsg;

#define rfbRCModeRDPEntered		0
#define rfbRCModeRDPLeft		1
#define rfbRCModeRDPNotAvailable	2
#define rfbRCModeSecRFBEntered		3
#define rfbRCModeSecRFBLeft		4
#define rfbRCModeSecRFBNotAvailable	5

/* ------------------------------------------------------------------------- *
 * Capabilities/Connection parameters of the erla                                       
 * ------------------------------------------------------------------------- */

/* one parameter */
typedef struct {
    u_int8_t  name_length_8;
    u_int8_t  value_length_8;
    /* followed by <name_length> + <value_length> bytes */
} PACKED_STRUCT rfbConnectionParameter;

/* parameter table */
typedef struct {
    u_int8_t  type_8;
    u_int8_t  parameters_8;
    /* followed by <parameters_8> rfbConnectionParameter items */
} PACKED_STRUCT rfbConnectionParameterListMsg;

/* ------------------------------------------------------------------------- *
 * SAS events and errors
 * ------------------------------------------------------------------------- */

/* SAS Events */
typedef struct {
    u_int8_t        type_8;
    u_int8_t        event_type_8;
    u_int16_t       add_len_be16;	/* client can read unknown messages */
    u_int32_t       session_id_be32;
    rfbTimestamp timestamp;
    /* followed by add_len bytes other data (according to event_type) */
} PACKED_STRUCT rfbSasEventMsg;

typedef struct {
    u_int8_t        user_len_8;
    u_int8_t        ip_len_8;
    /* followed by user_len + ip_len bytes */
} PACKED_STRUCT rfbSasUserInfo;

typedef struct {
    rfbTimestamp    login_time;
    rfbSasUserInfo  session_info;
} PACKED_STRUCT rfbSasEventNewSessionMsg;

typedef struct {
    u_int32_t kvm_session_id_be32;
} PACKED_STRUCT rfbSasKvmInfo;

typedef struct {
    rfbSasKvmInfo  info;
    rfbTimestamp   session_time;
} PACKED_STRUCT rfbSasNewKvmSessionMsg;

typedef struct {
    u_int8_t       exclusive_8;
    u_int8_t       pad1;
    u_int16_t      pad2;
    rfbSasKvmInfo  kvm;
    rfbTimestamp   login_time;
} PACKED_STRUCT rfbSasEventExistingKvmSessionMsg;

typedef struct {
    u_int8_t        self_8;
    u_int8_t        pad1;
    u_int16_t       pad2;
    rfbTimestamp    login_time;
    rfbSasUserInfo  session_info;
} PACKED_STRUCT rfbSasEventExistingSessionMsg;

typedef struct {
    rfbSasKvmInfo	    kvm;
    rfbPointerEventMsgV1_3  pointer;
} PACKED_STRUCT rfbSasPointerEventMsg;

typedef struct {
    rfbSasKvmInfo         kvm;
    rfbMousesyncEventMsg  sync;
} PACKED_STRUCT rfbSasMousesyncEventMsg;

typedef struct {
    rfbSasKvmInfo   kvm;
    rfbKeyEventMsg  kbd;
} PACKED_STRUCT rfbSasKeyboardEventMsg;

typedef struct {
    u_int8_t  channel_8;
    u_int8_t  unit_8;
    u_int16_t port_be16;
} PACKED_STRUCT rfbSasKvmSwitchEventMsg;

#define rfbSasEventExistingSession		0x01
	/* data: rfbSasEventExistingSession */
#define rfbSasEventExistingKvmSession		0x02
	/* data: rfbSasEventExistingKvmSession */
#define rfbSasEventUserLoginFailure		0x03
	/* data: rfbSasUserInfo */
#define rfbSasEventUserSessionOpened		0x04
	/* data: rfbSasEventNewSession */
#define rfbSasEventUserSessionClosed		0x05
	/* data: none */
#define rfbSasEventKvmSessionOpened		0x06
	/* data: rfbSasNewKvmSession */
#define rfbSasEventKvmSessionClosed		0x07
	/* data: rfbSasKvmInfo */
#define rfbSasEventKvmExclusiveOn		0x08
	/* data: rfbSasKvmInfo */
#define rfbSasEventKvmExclusiveOff		0x09
	/* data: rfbSasKvmInfo */
#define rfbSasEventInput			0x0A
	/* data is rfbSasPointerEvent, rfbSasMousesyncEvent
	   or rfbSasKeyboardEvent, determined by the pdu's type */
#define rfbSasEventKvmSwitch			0x0B
	/* data: rfbSasKvmSwitchEvent */
#define rfbSasEventResetSessions		0x0C
	/* data: none */

/* SAS Errors */
typedef struct {
    u_int8_t type_8;
    u_int8_t errorcode_8;
} PACKED_STRUCT rfbSasErrorMsg;

#define rfbSasErrorNoInput			0x01
#define rfbSasErrorMissedInputEvent		0x02
#define rfbSasErrorMissedSessionEvent		0x03
#define rfbSasErrorMissedOtherEvent		0x04

/* ------------------------------------------------------------------------- *
 * SARA replay (SP) messages
 * ------------------------------------------------------------------------- */

/* timestamp information */
typedef struct {
    u_int8_t      type_8;
    u_int8_t      pad1;
    u_int16_t     pad2;
    rfbTimestamp  time;
} PACKED_STRUCT rfbSpTimestampMsg;

/* information about the captured session */
typedef struct {
    u_int8_t    type_8;
    u_int8_t    color_depth_8;          /* see color depths below */
    u_int8_t    keyboard_mode_8;        /* see rfbConnectionSasKbd... above */
    u_int8_t    use_jpeg_8;
    u_int8_t    jpeg_quality_8;
    u_int8_t    use_zlib_8;
    u_int8_t    zlib_level_8;
    u_int8_t    host_name_len_8;        /* length of the host name of the session */
    u_int8_t    host_ip_len_8;          /* length of the host ip of the session */
    u_int8_t    pad;
    u_int16_t   keyframe_interval_be16; /* in seconds */
    /* followed by host_name_len_8 + host_ip_len_8 bytes */
} PACKED_STRUCT rfbSpSessionInfoMsg;

#define rfbSpInfoColorDepthUnknown              0x00
#define rfbSpInfoColorDepth1BitBW               0x01
#define rfbSpInfoColorDepth2BitGrey             0x02
#define rfbSpInfoColorDepth3BitGrey             0x03
#define rfbSpInfoColorDepth4BitGrey             0x04
#define rfbSpInfoColorDepth4BitColor            0x05
#define rfbSpInfoColorDepth7BitColor            0x06
#define rfbSpInfoColorDepth15BitColor           0x07

 /*****************************************************************************
 *
 * File reading message types (to be treated as server -> client)
 * (needed in KVM Vision Viewer)
 *
 *****************************************************************************/

typedef struct {
    u_int8_t type_8;
    u_int8_t  pad1;
    u_int16_t pad2;
    u_int32_t time_be32;
} PACKED_STRUCT rfbFileTimediffMsg;

typedef struct {
    u_int8_t type_8;
} PACKED_STRUCT rfbFileEOFMsg;

typedef struct {
    u_int8_t  type_8;
    u_int8_t  pad1;
    u_int16_t pad2;
    u_int32_t offset_be32;
} PACKED_STRUCT rfbFileKeyframeSeekMsg;

typedef struct {
    u_int32_t uncompr_size_be32;
    u_int32_t compr_size_be32;
} PACKED_STRUCT rfbFileComprMsg;

typedef struct {
    u_int8_t type_8;
    u_int8_t truecolor_8;
} PACKED_STRUCT rfbFileFormatMsg;


/*****************************************************************************
 *
 * Unions of all messages
 *
 *****************************************************************************/

/*-----------------------------------------------------------------------------
 * Union of all server->client messages.
 */

typedef union {
    u_int8_t type_8;
    rfbServerFBFormatMsgV1_0 si1_0;
    rfbServerFBFormatMsgV1_16 si1_16;
    rfbFramebufferUpdateMsg fu;
    rfbOSDStateMsgV1_7 osd1_7;
    rfbOSDStateMsgV1_16 osd1_16;
    rfbMessageV1_0 msg1_0;
    rfbMessageV1_16 msg1_16;
    rfbUtf8StringMsg utf;
    rfbSetColourMapEntriesMsg scme;
    rfbBellMsg b;
    rfbEnableExtensionRequestMsg eer;
    rfbExtensionDataMsg ed;
    rfbVideoSettingsS2CMsgV1_0 vss1_0;
    rfbVideoSettingsS2CMsgV1_1 vss1_1;
    rfbVideoSettingsS2CMsgV1_8 vss1_8;
    rfbVideoRefreshRequestMsg vrr;
    rfbPingRequestEventMsg ping;
    rfbBandwidthRequestEventMsg bwr;
    rfbRCModeReplyMsg rcm;
    rfbCommandMsg cmd;
    rfbVideoQualityS2CMsgV1_10 vqs1_10;
    rfbVideoQualityS2CMsgV1_12 vqs1_12;
    rfbConnectionParameterListMsg params;
    rfbAckPixelFormatMsg apf;
    rfbAuthCapsMsg authCaps;
    rfbSessChalMsg chal;
    rfbAuthSuccessfulMsgV1_15 success1_15;
    rfbAuthSuccessfulMsgV1_16 success1_16;
    rfbSasErrorMsg se;
    rfbSasEventMsg sas;
    rfbSpSetSessionTimesMsg st;
    rfbSpTimestampMsg ts;
    rfbSpCommandMsg spc;
    rfbSpSessionInfoMsg spi;
} rfbServerToClientMsg;


/*-----------------------------------------------------------------------------
 * Union of all client->server messages.
 */

typedef union {
    u_int8_t type_8;
    rfbSetPixelFormatMsg spf;
    rfbFixColourMapEntriesMsg fcme;
    rfbSetEncodingsMsg se;
    rfbFramebufferUpdateRequestMsg fur;
    rfbKeyEventMsg ke;
    rfbPointerEventMsgV1_0 pe1_0;
    rfbPointerEventMsgV1_3 pe1_3;
    rfbEnableExtensionRequestMsg eer;
    rfbExtensionDataMsg ed;
    rfbScancodeEventMsg sem;
    rfbMousestateEventMsg mst;
    rfbMousesyncEventMsg msy;
    rfbCursorChangeEventMsg cur;
    rfbKVMSwitchEventMsg kvm;
    rfbVideoSettingsC2SMsg vsc;
    rfbVideoSettingsReqMsg vsr;
    rfbVideoQualityC2SMsg vqc;
    rfbVideoQualityReqMsg vqr;
    rfbUserPropChangeMsg upc;
    rfbGlobalPropChangeMsg gpb;
    rfbPingReplyMsg ping;
    rfbBandwidthReplyMsg bwm;
    rfbRCModeRequestMsg rcm;
    rfbLoginMsgV1_15 login1_15;
    rfbLoginMsgV1_16 login1_16;
    rfbChalRespMsg chalResp;
    rfbSpCommandMsg spc;
} rfbClientToServerMsg;

#ifdef WIN32
# pragma pack()
#endif /* WIN32 */

#endif //__PP_RFB_PROTO_H
