#include <kira/cpe.h>
#include <asm/u-boot.h>
#include <kira/symbol.h>
#include <kira/porting.h>
#include <linux/types.h>
#include <common.h>


#define FCS_DEBUG 0

#if FCS_DEBUG
# define DEB(...) printf(__VA_ARGS__)
#else
# define DEB(...)
#endif

static void
fLib_ClearInt(UINT32 irq) {
    UINT32 base = CPE_IC_BASE;
    if (irq >= 32) {
	base += 0x20; // offset for fast irq entries
	irq -= 32;
    }
    DEB("clear irq bit - @%08x: %08x\n", base+0x08, (1 << irq));
    cpe_outl(base+0x08, (1 << irq));
}

static void
fLib_SetIntTrig(UINT32 irq, int mode, int level) {
    UINT32 base = CPE_IC_BASE, v;
    int m,l;
    if (irq >= 32) {
	base += 0x20; // offset for fast irq entries
	irq -= 32;
    }
    if (mode == LEVEL) {
	m = 0;
    } else {
	m = 1;
    }
    if (level == H_ACTIVE) {
	l = 0;
    } else {
	l = 1;
    }
    v = cpe_inl(base+0x0c); /* mode */
    v &= ~(1 << irq);
    v |= (m << irq);
    DEB("set trig mode - @%08x: %08x\n", base+0x0c, v);
    cpe_outl(base+0x0c, v); /* trig mode */

    v = cpe_inl(base+0x10); /* level */
    v &= ~(1 << irq);
    v |= (l << irq);
    DEB("set trig level - @%08x: %08x\n", base+0x10, v);
    cpe_outl(base+0x10, v);
}

void 
EnableFIQ(void) {
     
    // already done
}

static void
fLib_EnableInt(UINT32 irq) {
    UINT32 base = CPE_IC_BASE, v;
    if (irq >= 32) {
	base += 0x20; // offset for fast irq entries
	irq -= 32;
    }
    v = cpe_inl(base+0x4);
    v |= (1 << irq);
    DEB("set irq enable - @%08x: %08x\n", base+0x4, v);
    cpe_outl(base+0x4, v);
}
static void
fLib_DisableInt(UINT32 irq) {
    UINT32 base = CPE_IC_BASE, v;
    if (irq >= 32) {
	base += 0x20; // offset for fast irq entries
	irq -= 32;
    }
    v = cpe_inl(base+0x4);
    v &= ~(1 << irq);
    DEB("set irq enable - @%08x: %08x\n", base+0x4, v);
    cpe_outl(base+0x4, v);
}

void set_turbo_mode(int turbo_on, UINT32 clock_div) {
    /* clock_div has already the correct format for the register */
    uint32_t m;

    fLib_ClearInt(FIQ_PMU);
    EnableFIQ();
//    fLib_ConnectInt(FIQ_PMU, null_handler);
    fLib_SetIntTrig(FIQ_PMU, LEVEL, H_ACTIVE);
    fLib_EnableInt(FIQ_PMU);

    m = cpe_inl(CPE_PMU_BASE + CPE_PMU_REG_PMODE);
    DEB("pmu pmode: %08x\n", m);
    if (turbo_on) {
	if (m & CPE_PMU_PMODE_TURBO) return; /* already in Turbo mode */
	m |= CPE_PMU_PMODE_TURBO; /* enable turbo mode */
	m &= ~CPE_PMU_PMODE_DIVAHBCLK_MASK; /* clear multiplier */
	/* on kira100 there are no different clock divider, 2x is hard wired */
	DEB("new pmu pmode: %08x\n", m);
	cpe_outl(CPE_PMU_BASE + CPE_PMU_REG_PMODE, m);
    } else {
	if (!(m & CPE_PMU_PMODE_TURBO)) return; /* already in normal mode */
	m &= ~CPE_PMU_PMODE_TURBO; /* disable turbo mode */
	m &= ~CPE_PMU_PMODE_DIVAHBCLK_MASK; /* clear multiplier */
	DEB("new pmu pmode: %08x\n", m);
	cpe_outl(CPE_PMU_BASE + CPE_PMU_REG_PMODE, m);
    }
    asm ("mov r0, #0"); /* not necessary, but to be more spec compliant */
    asm ("mcr p15, 0, r0, c7, c0, 4"); /* Wait for Interrupt - Idle mode */
    asm ("nop"); /* to here the cpu returns after the idle call */
    fLib_DisableInt(FIQ_PMU);
}



void
set_fcs_speed(unsigned int ahbc, int div, int fcs_ahbd) {
    UINT32 m;
    unsigned int pm, pn;
    unsigned int cpuclk;
    unsigned int fin = 12288;
    unsigned int real_pllclk;
    unsigned int real_ahbclk;
    
    DEB("input of set_fcs_speed: ahb: %d, mult: %d\n", ahbc, div );

    if (ahbc > CPE_MAX_AHBC/1000000) {
	printf("AHB clock (%d) must be smaller than %d\n", ahbc, CPE_MAX_AHBC/1000000);
	return;
    }

    cpuclk = ahbc * div * 1000;

    DEB("cpuclk: %d\n", cpuclk* 1000);
    
    if (! ((div == 1) || /* no turbo mode */
	   (div == 2) /* turbo mode, on kira100 only with multiplier 2 */
	   )) {
	printf("wrong parameter for div, only 1 (no turbo mode) or 2 (turbo mode x2) allowed.");
	return;
    }
    
    set_turbo_mode((div != 1) ? 1 : 0, 0); /* for kira100: clkdiv = 0 */
    

    fLib_ClearInt(FIQ_PMU);
    EnableFIQ();
    //    fLib_ConnectInt(FIQ_PMU, null_handler);
    fLib_SetIntTrig(FIQ_PMU, LEVEL, H_ACTIVE);
    fLib_EnableInt(FIQ_PMU);


    pm = 2;
    pn = (cpuclk*pm) / fin;

    DEB("req pm: %d, pn: %d\n", pm, pn);

    real_pllclk = fin * pn / pm;
    real_ahbclk = real_pllclk / div;

    printf("set clock: cpuc: %d, ahbc: %d, div: %d\n", real_pllclk, real_ahbclk, div);

#if 0
    m = cpe_inl(CPE_PMU_BASE + CPE_PMU_REG_PMCR);
    m |= (1 << 18);
    cpe_outl(CPE_PMU_BASE + CPE_PMU_REG_PMCR, m);
#endif
    
    m = cpe_inl(CPE_PMU_BASE + CPE_PMU_REG_PDLLCR0);
    m &= ~CPE_PMU_PDLLCR0_PLL1FRANG_MASK;
    m &= ~CPE_PMU_PDLLCR0_DLLFRANG_MASK;

    /* set DLL for SDRAM */

    /* #warning faraday used pllout(=cpuclk)/pm/2 here */
    if (real_pllclk/pm/2 > 100*1000) {
	m |= CPE_PMU_PDLLCR0_DLLFRANG100_300;
	DEB("set dll to 100_300\n");
    } else {
	m |= CPE_PMU_PDLLCR0_DLLFRANG20_100;
	DEB("set dll to 20_100\n");
    }
    
    if (cpuclk >= 150 * 1000) {
	DEB("set pll1frang 150-300\n");
        m |= CPE_PMU_PDLLCR0_PLL1FRANG150;
    } else if (cpuclk >=90 * 1000) {
	DEB("set pll1frang 90-150\n");
        m |= CPE_PMU_PDLLCR0_PLL1FRANG90;
    } else if (cpuclk >= 50 * 1000) {
	DEB("set pll1frang 50-90\n");
        m |= CPE_PMU_PDLLCR0_PLL1FRANG50;
    } else if (cpuclk >= 20 * 1000) {
	DEB("set pll1frang 20-50\n");
        m |= CPE_PMU_PDLLCR0_PLL1FRANG20;
    } else {
        printf("incorrect frequence request cpuclk: %d- returning\n", cpuclk);
    }
    
    m &= ~CPE_PMU_PDLLCR0_PLL1MS_MASK;
    m &= ~CPE_PMU_PDLLCR0_PLL1N_MASK;
    m |= (pm << CPE_PMU_PDLLCR0_PLL1MS_POS);
    m |= (pn << CPE_PMU_PDLLCR0_PLL1N_POS);

    DEB("will set cr0 to: 0x%08x\n", m);
    DEB("pll1frang: %x\n", (m & CPE_PMU_PDLLCR0_PLL1FRANG_MASK) >> 15);
    DEB("pll1ms: %d(%d)\n", (m & CPE_PMU_PDLLCR0_PLL1MS_MASK) >> CPE_PMU_PDLLCR0_PLL1MS_POS, pm);
    DEB("pll1ns: %d(%d)\n", (m & CPE_PMU_PDLLCR0_PLL1N_MASK) >> CPE_PMU_PDLLCR0_PLL1N_POS, pn);
 
    cpe_outl(CPE_PMU_BASE + CPE_PMU_REG_PDLLCR0, m);
    
    m = cpe_inl(CPE_PMU_BASE + CPE_PMU_REG_PMODE);
    m |= CPE_PMU_PMODE_FCS;
    cpe_outl(CPE_PMU_BASE + CPE_PMU_REG_PMODE, m);
    
    asm ("mov r0, #0"); /* not necessary, but to be more spec compliant */
    asm ("mcr p15, 0, r0, c7, c0, 4"); /* Wait for Interrupt - Idle mode */
    asm ("nop"); /* to here the cpu returns after the idle call */
    fLib_DisableInt(FIQ_PMU);
}
