/*****************************************************************************
 *  I2C Utilities Menu
 *
 *  FILE: i2c_menu.c
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential 
 * proprietary information distributed solely pursuant to a confidentiality 
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2004-2005 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <common.h>

#if defined(CFG_RARITAN_UTIL_I2C)

#include <i2c.h>
#include <check.h>
#include <i2c_menu.h>
#include <util.h>


static void print_i2c_menu( void )
{
    printf("\nI2C Utilities:");
    print_exit();
    printf("\n\t 1: Detect I2C devices");
    printf("\n\t 2: Get I2C chip address");
    printf("\n\t 3: Select I2C chip address");
    printf("\n\t 4: Read I2C Register");
    printf("\n\t 5: Write I2C Register");
    printf("\n\t 6: Probe specific I2C device");
    printf("\n\n");

    return;
}

int i2c_menu( void )
{
    char i2c_flag = 1;
    int result = 1; /* result is used for reverting back to U-boot cmds */
    unsigned long reg, val;
    int num_chars;
    unsigned long diag_opt;
    int opt_res;
    ushort addr, print_once;
    static ulong chip_addr = -1;

    while (i2c_flag) {
        print_i2c_menu();

        opt_res = menu_get_user_input(&diag_opt);
        if( opt_res > 0 ) {
            return 0;
        }
        else if( opt_res < 0 ) {
            continue;
        }

        switch( diag_opt ) {
            case I2C_MENU_EXIT:
                i2c_flag = 0;
                break;

            case I2C_MENU_PROBE:
                print_once = 0;
                for (addr = 0; addr < 0x7F; addr++) {
                    if (i2c_probe(addr) == 0) {
                        if (print_once == 0) {
                            printf("\nFound i2c device on address");
                            print_once = 1;
                            /* set chip_addr to first seen addr on bus */
                            if (chip_addr == -1) {
                                chip_addr = addr;
                            }
                        }
                        printf(" 0x%02x", addr);
                    }
                }
                if (print_once == 0) {
                    printf("\nNo i2c device detected.\n");
                }
                else {
                    printf("\n");
                }
                break;

            case I2C_MENU_GET_CHIP:
                printf("\nCurrent i2c chip address is 0x%02lx\n", chip_addr);
                break;

            case I2C_MENU_SELECT_CHIP:
                num_chars = print_prompt("\nEnter chip address: 0x", &val, 16);
                if (check_i2caddr(val) == 0) {
                    chip_addr = val;
                }
                break;

            case I2C_MENU_READ_REG:
                num_chars = print_prompt("\nEnter register address: ", &reg, 10);
                if (num_chars > 0) {
                    printf("I2C addr 0x%02x reg 0x%02x= 0x%02x\n",
                           (unsigned short)chip_addr, (unsigned short)reg,
                           i2c_reg_read(chip_addr, reg));
                }
                break;

            case I2C_MENU_WRITE_REG:
                num_chars = print_prompt("\nEnter register address: 0x",&reg,16);
                num_chars = print_prompt("Enter value to write  : 0x",&val,16);
                i2c_reg_write(chip_addr, reg, val);
                break;

            case I2C_MENU_PROBE_ONE:
                num_chars = print_prompt("\nEnter I2C address: 0x", &val, 16);
                if (num_chars > 0 && check_i2caddr(val) == 0) {
                    addr = (ushort)val;
                    if (i2c_probe(addr) == 0) {
                        printf("\nFound i2c device on address 0x%02x\n", addr);
                    }
                    else {
                        printf("\nNo i2c device detected.\n");
                    }
                }
                break;

            default:
                print_invalid();
                break;
        }
    }

    return result;
}

#endif /* (CFG_RARITAN_UTIL_I2C) */
