/*****************************************************************************
 *  Bootloader, CONFIG, and JFFS2 Images Update Utilities
 *
 *  FILE: image_update.c
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential 
 * proprietary information distributed solely pursuant to a confidentiality 
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2004-2005 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <common.h>
#include <command.h>
#include <net.h>
#include <rtc.h>
#include <jffs2/jffs2.h>
#include <flash_partition.h>
#include <util.h>


#define MEGABYTE              (1024 * 1024)

/* info for FLASH chips */
extern flash_info_t flash_info[];


#ifdef RTC_DEFINED
#define UPDATE_START          0
#define UPDATE_COMPLETE       1

static void get_update_time( char flag )
{
    struct rtc_time tm;

    rtc_get(&tm);
    if( flag == UPDATE_START ) {
        printf("Start ");
    }
    else if( flag == UPDATE_COMPLETE ) {
        printf("End   ");
    }
    printf("Time: %4d-%02d-%02d  %2d:%02d:%02d UTC\n", tm.tm_year,
           tm.tm_mon, tm.tm_mday, tm.tm_hour, tm.tm_min, tm.tm_sec);

    return;
}
#endif

s32 verify_image( u32 addr )
{
    ulong data, len, checksum;
    image_header_t header;
    image_header_t *hdr = &header;

    printf("## Checking Image at 0x%08x ...\n", addr);

    /* Copy header so we can blank CRC field for re-calculation */
    memmove(&header, (char *)addr, sizeof(image_header_t));

    if(ntohl(hdr->ih_magic) != IH_MAGIC) {
        printf("failed [Bad Magic Number].\n");
        return -1;
    }

    data = (ulong)&header;
    len  = sizeof(image_header_t);

    checksum = ntohl(hdr->ih_hcrc);
    hdr->ih_hcrc = 0;

    if(crc32 (0, (char *)data, len) != checksum) {
        printf("failed [Bad Header Checksum].\n");
        return -1;
    }

    /* for multi-file images we need the data part, too */
    print_image_hdr((image_header_t *)addr);

    data = addr + sizeof(image_header_t);
    len  = ntohl(hdr->ih_size);

    printf("   Verifying Checksum ... ");
    if(crc32 (0, (char *)data, len) != ntohl(hdr->ih_dcrc)) {
        printf ("failed [Bad Data CRC].\n");
        return -1;
    }
    printf("OK\n");

    return 0;
}

int boot_tftp_update( char *defname, int flash_bank, int sector_start, int sector_end )
{
    cmd_tbl_t *cmdtp;
    char filename[64];
    char load_addr[12] = "0x100000";
    char tmp[] = "tftpboot";
    char *argv[3];
    int argc = 3;
    uint addr;
    ulong dst = flash_info[flash_bank].start[sector_start];
    int rc;
    int num_chars;
    int do_update;

    memset(filename, 0, 64);
    num_chars = strlen(defname);
    strncpy(filename, defname, num_chars);

    printf("\nEnter filename (default - %s)", filename);
    num_chars = readline(": ");
    if (num_chars > 0) {
       memset(filename, 0, 64);
       strncpy(filename, console_buffer, num_chars);
    }
    else if (num_chars < 0) {
        /* user change mind and hit Ctrl-C */
        printf("\n");
        return 0;
    }

    num_chars = readline("Enter load address (default - 0x100000): ");
    if (num_chars > 0) {
       memset(load_addr, 0, 12);
       strncpy(load_addr, console_buffer, num_chars);
    }
    else if (num_chars < 0) {
        /* user change mind and hit Ctrl-C */
        printf("\n");
        return 0;
    }

    argv[0] = tmp;
    argv[1] = load_addr;
    argv[2] = filename;
    if ((cmdtp = find_cmd(argv[0])) == NULL) {
       return -1;
    }

    addr = simple_strtoul(load_addr, NULL, 16);

    /* check if we are copying to Flash */
    if (addr2info(dst) == NULL) {
       printf ("Destination address is not part of flash\n");
       return -2;
    }

    if ((cmdtp->cmd) (cmdtp, 0, argc, argv) != 0) {
       printf ("Failed to TFTP Boot\n");
       return -3;
    }
    else {
       printf("Finished tftp %s to address %s of size 0x%lx\n",
              filename, load_addr, NetBootFileXferSize);
    }

    /* verify the size of the image */
    if(NetBootFileXferSize > UBOOT_IMAGE_SIZE) {
        printf("ERROR: File size is too big (max 0x%x bytes)!\n",
               UBOOT_IMAGE_SIZE);
        return -4;
    }

    /* query the user to proceed or not */
    do {
        do_update = 0;
        num_chars = readline("Are you sure you want to update Bootloader (y/n): ");
        if (num_chars > 0) {
            strncpy(tmp, console_buffer, 1);
            if((tmp[0] == 'y') || (tmp[0] == 'Y')) {
                do_update = 1;
            }
            else if((tmp[0] == 'n') || (tmp[0] == 'N')) {
                return 0;
            }
        }
        else if (num_chars < 0) {
            /* user change mind and hit Ctrl-C */
            printf("\n");
            return 0;
        }
    } while( !do_update );
    
#ifdef RTC_DEFINED
    get_update_time(UPDATE_START);
#endif

    printf("Un-protecting flash address 0x%08lx...", dst);
    flash_protect_off(flash_bank, sector_start, sector_end);
    printf("done\n");

    printf("Erasing flash address 0x%08lx (sectors %d - %d)...",
           dst, sector_start, sector_end);
    if (flash_erase(&flash_info[flash_bank], sector_start, sector_end)!= 0) {
       printf("ERROR: Failed to erase a flash sector\n");
       return -4;
    }

    printf("Copy 0x%08x to Flash 0x%08lx... ", addr, dst);
    rc = flash_write ((uchar *)addr, dst, NetBootFileXferSize);
    if (rc != 0) {
       flash_perror(rc);
       return -5;
    }
    puts(" done\n");

#ifdef RTC_DEFINED
    get_update_time(UPDATE_COMPLETE);
#endif

    return 0;
}

#if defined(CFG_RARITAN_UPDATE_CONFIG)
int config_tftp_update( int cfg_part )
{
    cmd_tbl_t *cmdtp;
    char filename[128];
    char load_addr[20] = "0x100000";
    char tmp[] = "tftpboot";
    char *argv[3];
    int argc = 3;
    int sector_start;
    int sector_end;
    uint addr;
    ulong dst;
    int rc;
    int num_chars;
    int do_update;

    memset(filename, 0, 128);
    if (cfg_part == 0) {
        strncpy(filename, "config1.bin", 11);
        num_chars = readline("\nEnter filename (default - config1.bin): ");
    }
    else {
        strncpy(filename, "config2.bin", 11);
        num_chars = readline("\nEnter filename (default - config2.bin): ");
    }

    if (num_chars > 0) {
        memset(filename, 0, 128);
        strncpy(filename, console_buffer, num_chars);
    }
    else if (num_chars < 0) {
        /* user change mind and hit Ctrl-C */
        printf("\n");
        return 0;
    }

    num_chars = readline("Enter load address (default - 0x100000): ");
    if (num_chars > 0) {
       memset(load_addr, 0, 20);
       strncpy(load_addr, console_buffer, num_chars);
    }
    else if (num_chars < 0) {
        /* user change mind and hit Ctrl-C */
        printf("\n");
        return 0;
    }

    argv[0] = tmp;
    argv[1] = load_addr;
    argv[2] = filename;
    if ((cmdtp = find_cmd(argv[0])) == NULL) {
       return -1;
    }

    addr = simple_strtoul(load_addr, NULL, 16);

    if ((cmdtp->cmd) (cmdtp, 0, argc, argv) != 0) {
       printf ("Failed to TFTP Boot\n");
       return -2;
    }
    else {
       printf("Finished tftp %s to address %s of size 0x%lx\n",
              filename, load_addr, NetBootFileXferSize);
    }

    /* verify image size */
    if(NetBootFileXferSize > CONFIG1_IMAGE_SIZE) {
        printf("Image size is %ld. Allowed is %d\n", NetBootFileXferSize,
               CONFIG1_IMAGE_SIZE);
        return -3;
    }

    /* get the correct flash address to place the JFFS2 image */
    if (cfg_part == 0) {
        sector_start = CONFIG1_IMAGE_OFFSET/FLASH_SECTOR_SIZE;
        sector_end = sector_start + (CONFIG1_IMAGE_SIZE/FLASH_SECTOR_SIZE) - 1;
    }
    else {
        sector_start = CONFIG2_IMAGE_OFFSET/FLASH_SECTOR_SIZE;
        sector_end = sector_start + (CONFIG2_IMAGE_SIZE/FLASH_SECTOR_SIZE) - 1;
    }
    dst = flash_info[0].start[sector_start];
    debug("start_sect = %d (0x%08lx), end_sect = %d (0x%08lx)\n",
          sector_start, flash_info[0].start[sector_start],
          sector_end, flash_info[0].start[sector_end]);

    /* check if we are copying to Flash */
    if (addr2info(dst) == NULL) {
       printf ("Destination address is not part of flash\n");
       return -3;
    }

    /* query the user to proceed or not */
    do {
        do_update = 0;
        num_chars = readline("Are you sure you want to update CONFIG partition (y/n): ");
        if (num_chars > 0) {
            strncpy(tmp, console_buffer, 1);
            if((tmp[0] == 'y') || (tmp[0] == 'Y')) {
                do_update = 1;
            }
            else if((tmp[0] == 'n') || (tmp[0] == 'N')) {
                return 0;
            }
        }
        else if (num_chars < 0) {
            /* user change mind and hit Ctrl-C */
            printf("\n");
            return 0;
        }
    } while( !do_update );
    
#ifdef RTC_DEFINED
    get_update_time(UPDATE_START);
#endif

    printf("Erasing flash address 0x%08lx (sectors %d - %d)...",
           dst, sector_start, sector_end);
    if (flash_erase(&flash_info[0], sector_start, sector_end)!= 0) {
       printf("ERROR: Failed to erase a flash sector\n");
       return -4;
    }

    printf("Copy 0x%08x to Flash 0x%08lx... ", addr, dst);
    rc = flash_write ((uchar *)addr, dst, NetBootFileXferSize);
    if (rc != 0) {
       flash_perror(rc);
       return -5;
    }
    puts(" done\n");

#ifdef RTC_DEFINED
    get_update_time(UPDATE_COMPLETE);
#endif

    return 0;
}
#endif /* CFG_RARITAN_UPDATE_CONFIG */

/* forward declaration from kxgen2.c */
struct part_info* jffs2_part_info(int part_num);

int jffs2_tftp_update( void )
{
    cmd_tbl_t *cmdtp;
    char filename[128] = "jffs2-kx101.bin";
    char load_addr[20] = "0x100000";
    char tmp[] = "tftpboot";
    char *argv[3];
    int argc = 3;
    int sector_start;
    int sector_end;
    uint addr;
    ulong dst;
    int rc;
    int num_chars;
    int do_update;

    num_chars = readline("\nEnter filename (default - jffs2-kx101.bin): ");
    if (num_chars > 0) {
       memset(filename, 0, 128);
       strncpy(filename, console_buffer, num_chars);
    }
    else if (num_chars < 0) {
        /* user change mind and hit Ctrl-C */
        printf("\n");
        return 0;
    }

    num_chars = readline("Enter load address (default - 0x100000): ");
    if (num_chars > 0) {
       memset(load_addr, 0, 20);
       strncpy(load_addr, console_buffer, num_chars);
    }
    else if (num_chars < 0) {
        /* user change mind and hit Ctrl-C */
        printf("\n");
        return 0;
    }

    argv[0] = tmp;
    argv[1] = load_addr;
    argv[2] = filename;
    if ((cmdtp = find_cmd(argv[0])) == NULL) {
       return -1;
    }

    addr = simple_strtoul(load_addr, NULL, 16);

    if ((cmdtp->cmd) (cmdtp, 0, argc, argv) != 0) {
       printf ("Failed to TFTP Boot\n");
       return -2;
    }
    else {
       printf("Finished tftp %s to address %s of size 0x%lx\n",
              filename, load_addr, NetBootFileXferSize);
    }

    /* verify image size */
    if(NetBootFileXferSize > JFFS2_IMAGE_SIZE) {
        printf("JFFS2 Image size is larger than actual JFFS2 Partition\n");
        printf("Image size is %ld. Allowed is %d\n", NetBootFileXferSize,
               JFFS2_IMAGE_SIZE);
        return -3;
    }

    /* get the correct flash address to place the JFFS2 image */
    sector_start = JFFS2_IMAGE_OFFSET/FLASH_SECTOR_SIZE;
    sector_end = sector_start + (JFFS2_IMAGE_SIZE/FLASH_SECTOR_SIZE) - 1;
    dst = flash_info[0].start[sector_start];
    debug("start_sect = %d (0x%08lx), end_sect = %d (0x%08lx)\n",
          sector_start, flash_info[0].start[sector_start],
          sector_end, flash_info[0].start[sector_end]);

    /* check if we are copying to Flash */
    if (addr2info(dst) == NULL) {
       printf ("Destination address is not part of flash\n");
       return -3;
    }

    /* query the user to proceed or not */
    do {
        do_update = 0;
        num_chars = readline("Are you sure you want to update JFFS2 partition (y/n): ");
        if (num_chars > 0) {
            strncpy(tmp, console_buffer, 1);
            if((tmp[0] == 'y') || (tmp[0] == 'Y')) {
                do_update = 1;
            }
            else if((tmp[0] == 'n') || (tmp[0] == 'N')) {
                return 0;
            }
        }
        else if (num_chars < 0) {
            /* user change mind and hit Ctrl-C */
            printf("\n");
            return 0;
        }
    } while( !do_update );
    
#ifdef RTC_DEFINED
    get_update_time(UPDATE_START);
#endif

    printf("Erasing flash address 0x%08lx (sectors %d - %d)...",
           dst, sector_start, sector_end);
    if (flash_erase(&flash_info[0], sector_start, sector_end)!= 0) {
       printf("ERROR: Failed to erase a flash sector\n");
       return -4;
    }

    printf("Copy 0x%08x to Flash 0x%08lx... ", addr, dst);
    rc = flash_write ((uchar *)addr, dst, NetBootFileXferSize);
    if (rc != 0) {
       flash_perror(rc);
       return -5;
    }
    puts(" done\n");

#ifdef RTC_DEFINED
    get_update_time(UPDATE_COMPLETE);
#endif

    return 0;
}

int tftp_update_whole_flash( char *defname, int flash_size, int sector_start, int sector_end )
{
    cmd_tbl_t *cmdtp;
    char filename[64];
    char load_addr[12] = "0x100000";
    char tmp[] = "tftpboot";
    char *argv[3];
    int argc = 3;
    uint addr;
    ulong dst = flash_info[0].start[sector_start];
    int rc;
    int num_chars;
    int do_update;

    memset(filename, 0, 64);
    num_chars = strlen(defname);
    strncpy(filename, defname, num_chars);

    printf("\nEnter filename (default - %s)", filename);
    num_chars = readline(": ");
    if (num_chars > 0) {
       memset(filename, 0, 64);
       strncpy(filename, console_buffer, num_chars);
    }
    else if (num_chars < 0) {
        /* user change mind and hit Ctrl-C */
        printf("\n");
        return 0;
    }

    num_chars = readline("Enter load address (default - 0x100000): ");
    if (num_chars > 0) {
       memset(load_addr, 0, 12);
       strncpy(load_addr, console_buffer, num_chars);
    }
    else if (num_chars < 0) {
        /* user change mind and hit Ctrl-C */
        printf("\n");
        return 0;
    }

    argv[0] = tmp;
    argv[1] = load_addr;
    argv[2] = filename;
    if ((cmdtp = find_cmd(argv[0])) == NULL) {
       return -1;
    }

    /* check if we are copying to Flash */
    if (addr2info(dst) == NULL) {
       printf ("Destination address is not part of flash\n");
       return -2;
    }

    if ((cmdtp->cmd) (cmdtp, 0, argc, argv) != 0) {
       printf ("Failed to TFTP Boot\n");
       return -3;
    }
    else {
       printf("Finished tftp %s to address %s of size 0x%lx\n",
              filename, load_addr, NetBootFileXferSize);
    }

    /* verify the size of the image */
    addr = simple_strtoul(load_addr, NULL, 16);
    if( NetBootFileXferSize != flash_size ) {
        printf("ERROR: Flash image size is not %dMB!\n", flash_size/MEGABYTE);
        return -4;
    }

    /* should we verify the CRC32 as well? */

    /* query the user to proceed or not */
    do {
        do_update = 0;
        num_chars = readline("Are you sure you want to update the FLASH (y/n): ");
        if (num_chars > 0) {
            strncpy(tmp, console_buffer, 1);
            if((tmp[0] == 'y') || (tmp[0] == 'Y')) {
                do_update = 1;
            }
            else if((tmp[0] == 'n') || (tmp[0] == 'N')) {
                return 0;
            }
        }
        else if (num_chars < 0) {
            /* user change mind and hit Ctrl-C */
            printf("\n");
            return 0;
        }
    } while( !do_update );
    
#ifdef RTC_DEFINED
    get_update_time(UPDATE_START);
#endif

    printf("Un-protecting flash address 0x%08lx...", dst);
    flash_protect_off(0, sector_start, sector_end);
    printf("done\n");

    printf("Erasing flash address 0x%08lx (sector %d)...", dst, sector_start);
    if (flash_erase(&flash_info[0], sector_start, sector_end)!= 0) {
       printf("ERROR: Failed to erase a flash sector\n");
       return -4;
    }

    printf("Copy 0x%08x to Flash 0x%08lx... ", addr, dst);
    rc = flash_write ((uchar *)addr, dst, NetBootFileXferSize);
    if (rc != 0) {
       flash_perror(rc);
       return -5;
    }
    puts(" done\n");

#ifdef RTC_DEFINED
    get_update_time(UPDATE_COMPLETE);
#endif

    return 0;
}

void get_kernel_image_info( void )
{
    ulong offset = 0x100000;
    char *filename = "uzImage";
    int size;
    int part_num = 0;
    struct part_info *part;

    if(0 != (part=jffs2_part_info(part_num))){
        size = jffs2_1pass_load((char *)offset, part, filename);

        if (size > 0) {
            printf("\nCurrent Kernel Image:\n");
            verify_image(offset);
            return;
        }
        else {
            printf("\nCurrent Kernel Image: Not found.\n");
            return;
        }
    }

    printf("\nERROR: Active JFFS2 partition not valid\n");
}

void get_ramdisk_image_info( void )
{
    ulong offset = 0x100000;
    char *filename = "rootImage";
    int size;
    int part_num = 0;
    struct part_info *part;

    if(0 != (part=jffs2_part_info(part_num))){
        size = jffs2_1pass_load((char *)offset, part, filename);

        if (size > 0) {
            printf("\nCurrent Ramdisk Image:\n");
            verify_image(offset);
            return;
        }
        else {
            printf("\nCurrent Ramdisk Image: Not found.\n");
            return;
        }
    }

    printf("\nERROR: Active JFFS2 partition not valid\n");
}

void get_jffs2_image_info( void )
{
    ulong offset = 0x100000;
    char *filename = "info";
    char *info = (char *)offset;
    int size;
    int part_num = 0;
    struct part_info *part;

    if(0 != (part=jffs2_part_info(part_num))){
        size = jffs2_1pass_load((char *)offset, part, filename);

        if (size > 0) {
            /* terminate the string */
            info[size] = '\0';

            printf("\nCurrent JFFS2 Image Build Info:");
            printf("\n%s", info);
            return;
        }
        else {
            printf("\nCurrent JFFS2 Image Build Info: NONE");
            printf("\n 1. No JFFS2 image present\n");
            return;
        }
    }

    printf("\nERROR: Active JFFS2 partition not valid\n");
}
