/******************************************************************************
 *  Intel LXT972A PHY-specific interface functions
 *
 *  FILE: lxt972.c
 *
 ******************************************************************************
 *
 *                        COPYRIGHT (c) 2003
 *
 *                       Raritan Computer, Inc.
 *                        400 Cottontail Lane
 *                        Somerset, NJ 08837
 *                           732.764.8886
 *
 *                      All rights reserved.
 * Copying, compilation, modification, distribution or any other use 
 *   whatsoever of this material is strictly prohibited except in
 * accordance with a Software License Agreement with Raritan Computer, Inc.
 *****************************************************************************/

#include <common.h>
#include <mii_phy.h>
#include <miiphy.h>
#include <lxt972.h>


static void
mii_phy_print_regval(unsigned short reg, unsigned short val)
{
   switch (reg) {
      case PHY_BMCR:
         printf("Control Register                         (0): 0x%04x\n", val);
         break;
      case PHY_BMSR:
         printf("Status Register                          (1): 0x%04x\n", val);
         break;
      case PHY_PHYIDR1:
         printf("PHY Identification Register 1            (2): 0x%04x\n", val);
         break;
      case PHY_PHYIDR2:
         printf("PHY Identification Register 2            (3): 0x%04x\n", val);
         break;
      case PHY_ANAR:
         printf("Auto-Negotiation Advertisement Register  (4): 0x%04x\n", val);
         break;
      case PHY_ANLPAR:
         printf("Auto-Negotiation Link PA Register        (5): 0x%04x\n", val);
         break;
      case PHY_ANER:
         printf("Auto-Negotiation Expansion Register      (6): 0x%04x\n", val);
         break;
      case PHY_ANNPTR:
         printf("Auto-Negotiation Next Page Tx Register   (7): 0x%04x\n", val);
         break;
      case LXT972_ANLPRNPR:
         printf("Auto-Negotiation LPRx Next Page Register (8): 0x%04x\n", val);
         break;
      case LXT972_PCR:
         printf("Port Configuration Register             (16): 0x%04x\n", val);
         break;
      case LXT972_SR2:
         printf("Status Register 2                       (17): 0x%04x\n", val);
         break;
      case LXT972_IER:
         printf("Interrupt Enable Register               (18): 0x%04x\n", val);
         break;
      case LXT972_ISR:
         printf("Interrupt Status Register               (19): 0x%04x\n", val);
         break;
      case LXT972_LEDCR:
         printf("LED Configuration Register              (20): 0x%04x\n", val);
         break;
      case LXT972_DCR:
         printf("Digital Configuration Register          (26): 0x%04x\n", val);
         break;
      case LXT972_TCR:
         printf("Transmit Control Register               (30): 0x%04x\n", val);
         break;
      default:
         printf("Invalid Register %d\n", reg);
         break;
   }

   return;
}

void
mii_phy_read_all_regs(void)
{
   unsigned short regval, addr;

   for (addr = 0; addr < 9; addr++) {
      regval = mii_phy_read(addr);
      mii_phy_print_regval(addr, regval);
   }
   for (addr = 16; addr < 21; addr++) {
      regval = mii_phy_read(addr);
      mii_phy_print_regval(addr, regval);
   }
   regval = mii_phy_read(LXT972_DCR);
   mii_phy_print_regval(LXT972_DCR, regval);
   regval = mii_phy_read(LXT972_TCR);
   mii_phy_print_regval(LXT972_TCR, regval);

   return;
}

void
mii_phy_get_status(void)
{
   unsigned short reg, sr1;

   /* status register 1 (addr 1) */
   sr1 = mii_phy_read(PHY_BMSR);

   /* status register 2 (addr 17) */
   reg = mii_phy_read(LXT972_SR2);
   printf("Link Status      : %s\n",
          (reg & LXT972_SR2_LINK)?"link up":"link down");
   printf("10/100 Mode      : %s\n",
          (reg & LXT972_SR2_SPEED)?"100 Mbs":"10 Mbs");
   printf("Duplex Mode      : %s\n", (reg & LXT972_SR2_DM)?"Full":"Half");
   printf("Auto-Neg Mode    : %s\n", (reg & LXT972_SR2_AN)?"yes":"no");
   if (reg & LXT972_SR2_AN) {
      printf("Auto-Neg Complete: %s\n", (sr1 & PHY_BMSR_AUTN_COMP)?"yes":"no");
      /*printf("Auto-Neg Complete: %s\n", (reg & LXT972_SR2_ANC)?"yes":"no");*/
   }
   printf("Remote Fault     : %s\n", (sr1 & PHY_BMSR_RF)?"yes":"no");
   printf("Error Detected   : %s\n", (reg & LXT972_SR2_ERROR)?"yes":"no");
   printf("Transmit Status  : %s\n",
          (reg & LXT972_SR2_TS)?"tx'ing packet":"not tx'ing packet");
   printf("Receive Status   : %s\n",
          (reg & LXT972_SR2_RS)?"rx'ing packet":"not rx'ing packet");
   printf("Collision Occur  : %s\n", (reg & LXT972_SR2_CS)?"yes":"no");
   printf("Polarity         : %s\n",
          (reg & LXT972_SR2_POLAR)?"reverse":"normal");
   printf("Device Pause     : %s\n",
          (reg & LXT972_SR2_PAUSE)?"capable":"not capable");
   printf("Jabber Detected  : %s\n", (sr1 & PHY_BMSR_JD)?"yes":"no");
   printf("\n");

   return;
}

int
mii_phy_config_led_disp(ushort led, ushort disp)
{
    u8 shift;
    ushort ledcfg;

    switch(led) {
        case LXT972_LED1:
            shift = 12;
            break;
        case LXT972_LED2:
            shift = 8;
            break;
        case LXT972_LED3:
            shift = 4;
            break;
        default:
            return -1;
    }

    switch(disp) {
        case LED_DISP_SPEED:
        case LED_DISP_TX:
        case LED_DISP_RX:
        case LED_DISP_COLLISION:
        case LED_DISP_LINK:
        case LED_DISP_DUPLEX:
        case LED_DISP_TX_RX:
        case LED_DISP_TEST_ON:
        case LED_DISP_TEST_OFF:
        case LED_DISP_TEST_BLINK_FAST:
        case LED_DISP_TEST_BLINK_SLOW:
        case LED_DISP_LINK_RX:
        case LED_DISP_LINK_ACTIVITY:
        case LED_DISP_DUPLEX_CLSN:
            break;
        default:
            return -2;
    }

    ledcfg = mii_phy_read(LXT972_LEDCR);
    ledcfg &= ~(0x000F << shift);
    ledcfg |= (disp << shift);
    mii_phy_write(LXT972_LEDCR, ledcfg);

    return 0;
}

int
mii_phy_config_led_freq(ushort pulse_stretch)
{
    ushort ledcfg;

    switch(pulse_stretch) {
        case LED_STRETCH_30MS:
        case LED_STRETCH_60MS:
        case LED_STRETCH_100MS:
            break;
        default:
            return -1;
    }

    ledcfg = mii_phy_read(LXT972_LEDCR);
    ledcfg &= ~0x000C;
    ledcfg |= pulse_stretch;
    mii_phy_write(LXT972_LEDCR, ledcfg);

    return 0;
}

void
mii_phy_config_led_ps(ushort ps)
{
    ushort ledcfg;

    ledcfg = mii_phy_read(LXT972_LEDCR);
    if(ps) {
        ledcfg |= 0x0002;
    }
    else {
        ledcfg &= ~0x0002;
    }
    mii_phy_write(LXT972_LEDCR, ledcfg);

    return;
}
