/*****************************************************************************
 *  Real-Time Clock Utilities Menu
 *
 *  FILE: rtc_menu.c
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential 
 * proprietary information distributed solely pursuant to a confidentiality 
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2004-2005 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <common.h>

#if defined(CFG_RARITAN_UTIL_RTC)

#include <rtc.h>
#include <rtc_menu.h>
#include <util.h>
#include <x1226.h>


#ifdef DEBUG
static unsigned bcd2bin( uchar n )
{
    return ((((n >> 4) & 0x0F) * 10) + (n & 0x0F));
}

static unsigned char bin2bcd( unsigned int n )
{
    return (((n / 10) << 4) | (n % 10));
}
#endif

static void print_rtc_menu( void )
{
    int i = 1;

    printf("\nReal-Time Clock Utilities:");
    print_exit();
    printf("\n\t %d: Read All RTC Register", i++);
    printf("\n\t %d: Read RTC Register", i++);
    printf("\n\t %d: Write RTC Register", i++);
    printf("\n\t %d: Get RTC", i++);
    printf("\n\t %d: Set RTC", i++);
#ifdef DEBUG
    printf("\n\t %d: Test bcd2bin", i++);
    printf("\n\t %d: Test bin2bcd", i++);
#endif
    printf("\n\n");

    return;
}

int get_rtc_args( struct rtc_time *ptime, uchar alarm )
{
    int nc;
    ulong val;

    if (!alarm) {
        nc = print_prompt("\nEnter year   : ", &val, 10);
        if (nc <= 0) {
            return nc;
        }
        ptime->tm_year = val;
    }

    nc = print_prompt("Enter month  : ", &val, 10);
    if (nc <= 0) {
        return nc;
    }
    ptime->tm_mon = val;

    nc = print_prompt("Enter date   : ", &val, 10);
    if (nc <= 0) {
        return nc;
    }
    ptime->tm_mday = val;

    if (!alarm) {
        nc = print_prompt("Enter day    : ", &val, 10);
        if (nc <= 0) {
            return nc;
        }
        ptime->tm_wday = val;
    }

    nc = print_prompt("Enter hour   : ", &val, 10);
    if (nc <= 0) {
        return nc;
    }
    ptime->tm_hour = val;

    nc = print_prompt("Enter minutes: ", &val, 10);
    if (nc <= 0) {
        return nc;
    }
    ptime->tm_min = val;

    nc = print_prompt("Enter seconds: ", &val, 10);
    if (nc <= 0) {
        return nc;
    }
    ptime->tm_sec = val;

    return nc;
}

int rtc_menu( void )
{
    char rtc_flag = 1;
    int result = 1; /* result is used for reverting back to U-boot cmds */
    unsigned long reg, val;
    int num_chars;
    unsigned long diag_opt;
    int opt_res;
    char buffer;
    char rtcbuf[REG_CNT];
    struct rtc_time tm;

    while (rtc_flag) {
        print_rtc_menu();

        opt_res = menu_get_user_input(&diag_opt);
        if( opt_res > 0 ) {
            return 0;
        }
        else if( opt_res < 0 ) {
            continue;
        }

        switch( diag_opt ) {
            case RTC_MENU_EXIT:
                rtc_flag = 0;
                break;

            case RTC_MENU_READ_ALL:
                printf("\n");
                rtc_read_all(rtcbuf);
                print_pause();
                break;

            case RTC_MENU_READ:
                num_chars = print_prompt("\nEnter register addr: ", &reg, 10);
                if (num_chars > 0) {
                    if (rtc_check_regaddr(reg) == 0) {
                        rtc_print(reg, rtc_read(reg));
                    }
                    else {
                        printf("ERROR: Invalid RTC Register\n");
                    }
                }
                break;

            case RTC_MENU_WRITE:
                num_chars = print_prompt("\nEnter register addr: ", &reg, 10);
                if (num_chars > 0) {
                    if (rtc_check_regaddr(reg) == 0) {
                        num_chars = print_prompt("Enter value to write: 0x",
                                                 &val, 16);
                        buffer = (u8)(val & 0xff);
                        rtc_write(reg, buffer);
                    }
                    else {
                        printf("ERROR: Invalid RTC Register\n");
                    }
                }
                break;

            case RTC_MENU_GET:
                rtc_get(&tm);
                printf ("\nCurrent Time: %4d-%02d-%02d  %2d:%02d:%02d UTC\n",
                        tm.tm_year, tm.tm_mon, tm.tm_mday,
                        tm.tm_hour, tm.tm_min, tm.tm_sec);
                break;

            case RTC_MENU_SET:
                num_chars = get_rtc_args(&tm, 0);
                if (num_chars > 0) {
                    rtc_set(&tm);
                }
                break;

#ifdef DEBUG
            case RTC_MENU_TEST_BCD2BIN:
                num_chars = print_prompt("\nEnter bcd value: 0x", &val, 16);
                if (num_chars > 0) {
                    printf("bcd val %02lx = bin val %d (%xh)\n", val,
                           bcd2bin(val), bcd2bin(val));
                }
                break;

            case RTC_MENU_TEST_BIN2BCD:
                num_chars = get_rtc_args(&tm, 0);
                if (num_chars > 0) {
                    printf("Year %4d (bin) %x (bcd)\n", tm.tm_year,
                           bin2bcd(tm.tm_year));
                    printf("Month %02d (bin) %x (bcd)\n", tm.tm_mon,
                           bin2bcd(tm.tm_mon));
                    printf("Date %02d (bin) %x (bcd)\n", tm.tm_mday,
                           bin2bcd(tm.tm_mday));
                    printf("Day %02d (bin) %x (bcd)\n", tm.tm_wday,
                           bin2bcd(tm.tm_wday));
                    printf("Hour %2d (bin) %x (bcd)\n", tm.tm_hour,
                           bin2bcd(tm.tm_hour));
                    printf("Minutes %02d (bin) %x (bcd)\n", tm.tm_min,
                           bin2bcd(tm.tm_min));
                    printf("Seconds %02d (bin) %x (bcd)\n", tm.tm_sec,
                           bin2bcd(tm.tm_sec));
                }
                break;
#endif

            default:
                print_invalid();
                break;
        }
    }

    return result;
}

#endif /* (CFG_RARITAN_UTIL_RTC) */
