/*****************************************************************************
 *  Environment Setup Menu
 *
 *  FILE: setenv_menu.c
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential 
 * proprietary information distributed solely pursuant to a confidentiality 
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2004-2005 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#include <common.h>
#include <command.h>
#include <setenv_menu.h>
#include <util.h>


static int setenv_get( void );
static int setenv_sernum( void );
static int setenv_ethaddr( u32 port );
static int setenv_ipaddr( void );
static int setenv_serverip( void );
static int setenv_gatewayip( void );
static int setenv_netmask( void );
static int setenv_nfskernel( void );
static int setenv_rootpath( void );
static int configure_nfsboot( void );
static int configure_flashboot( void );
static int setenv_console_on( void );
static int setenv_console_off( void );

static void print_setenv_menu( void )
{
    int i = 1;

    printf("\nEnvironment Setup:");
    print_exit();
    printf("\n\t %d: Print Current Setup", i++);
    printf("\n\t %d: Set Serial Number", i++);
    printf("\n\t %d: Set MAC Address", i++);
    printf("\n\t %d: Set IP Address", i++);
    printf("\n\t %d: Set Server IP Address", i++);
    printf("\n\t %d: Save Environment", i++);
    printf("\n\n\t [ Optional Settings ]");
    printf("\n\t %d: Set Gateway IP Address", i++);
    printf("\n\t %d: Set Subnet Mask", i++);
    printf("\n\t %d: Set NFS Kernel Filename", i++);
    printf("\n\t%d: Set NFS Root Path", i++);
    printf("\n\t%d: Configure for NFS boot", i++);
    printf("\n\t%d: Configure for Flash boot", i++);
    printf("\n\t%d: Enable console for debugging", i++);
    printf("\n\n");

    return;
}

int setenv_menu( void )
{
    char setenv_flag = 1;
    int result = 1; /* result is used for reverting back to U-boot cmds */
    unsigned long diag_opt;
    int opt_res;

    while (setenv_flag) {
       print_setenv_menu();

       opt_res = menu_get_user_input(&diag_opt);
       if( opt_res > 0 ) {
           return 0;
       }
       else if( opt_res < 0 ) {
           continue;
       }

       switch( diag_opt ) {
           case SETENV_MENU_EXIT:
              setenv_flag = 0;
              break;

           case SETENV_GET_INFO:
               setenv_get();
               break;

           case SETENV_SERIALNUM:
               setenv_sernum();
               break;

           case SETENV_MAC:
               setenv_ethaddr(0);
               break;

           case SETENV_IP:
               setenv_ipaddr();
               break;

           case SETENV_SERVER:
               setenv_serverip();
               break;

           case SETENV_SAVE:
               saveenv();
               break;

           case SETENV_GATEWAY:
               setenv_gatewayip();
               break;

           case SETENV_NETMASK:
               setenv_netmask();
               break;

           case SETENV_NFSKERNEL:
               setenv_nfskernel();
               break;

           case SETENV_ROOTPATH:
               setenv_rootpath();
               break;

           case SETENV_NFSBOOT:
               printf("\nConfiguring for NFS booting......");
               if(configure_nfsboot() == 0) {
                   printf("passed.\n");
               }
               break;

           case SETENV_FLASHBOOT:
               printf("\nConfiguring for Flash booting......");
               if(configure_flashboot() == 0) {
                   printf("passed.\n");
               }
               break;

           case SETENV_CONSOLE_ON:
               setenv_console_on();
               break;

           default:
               print_invalid();
               break;
       }
    }

    return result;
}

static int setenv_get( void )
{
    char buf[128];
    int num_chars;

    printf("\nBoard Serial Number : ");
    memset(buf, 0, 128);
    num_chars = getenv_r("serial#", buf, 128);
    if(num_chars > 0) {
        printf("%s", buf);
    }
    else {
        print_not_set();
    }

    printf("\nMAC Address         : ");
    memset(buf, 0, 128);
    num_chars = getenv_r("ethaddr", buf, 128);
    if(num_chars > 0) {
        printf("%s", buf);
    }
    else {
        print_not_set();
    }

    printf("\nIP Address          : ");
    memset(buf, 0, 128);
    num_chars = getenv_r("ipaddr", buf, 128);
    if(num_chars > 0) {
        printf("%s", buf);
    }
    else {
        print_not_set();
    }

    printf("\nServer IP Address   : ");
    memset(buf, 0, 128);
    num_chars = getenv_r("serverip", buf, 128);
    if(num_chars > 0) {
        printf("%s", buf);
    }
    else {
        print_not_set();
    }

    printf("\n\n[ Optional Settings ]");
    printf("\nGateway IP Address  : ");
    memset(buf, 0, 128);
    num_chars = getenv_r("gatewayip", buf, 128);
    if(num_chars > 0) {
        printf("%s", buf);
    }
    else {
        print_not_set();
    }

    printf("\nSubnet Mask         : ");
    memset(buf, 0, 128);
    num_chars = getenv_r("netmask", buf, 128);
    if(num_chars > 0) {
        printf("%s", buf);
    }
    else {
        print_not_set();
    }

    printf("\nNFS Kernel Filename : ");
    memset(buf, 0, 128);
    num_chars = getenv_r("nfskernel", buf, 128);
    if(num_chars > 0) {
        printf("%s", buf);
    }
    else {
        print_not_set();
    }

    printf("\nNFS Root Path       : ");
    memset(buf, 0, 128);
    num_chars = getenv_r("rootpath", buf, 128);
    if(num_chars > 0) {
        printf("%s", buf);
    }
    else {
        print_not_set();
    }
    print_pause();

    return 0;
}

static int setenv_sernum( void )
{
    char sn[30];
    int num_chars;

    num_chars = readline("\nEnter Board Serial Number: ");
    if (num_chars > 0) {
        memset(sn, 0, 30);
        strncpy(sn, console_buffer, num_chars);
        setenv("serial#", sn);
    }
    else if (num_chars < 0) {
        /* user change mind and hit Ctrl-C */
        printf("\n");
        return 0;
    }

    return 0;
}

static int setenv_ethaddr( u32 port )
{
    char eth[128];
    int num_chars;

    switch(port) {
        case 0:
        case 1:
            break;
        default:
            printf("ERROR: Invalid port number\n");
            return 1;
    }

    num_chars = readline("\nEnter MAC address: ");
    if (num_chars > 0) {
        memset(eth, 0, 128);
        strncpy(eth, console_buffer, num_chars);
        if(port == 0) {
            setenv("ethaddr", eth);
        }
        else {
            setenv("eth1addr", eth);
        }
    }
    else if (num_chars < 0) {
        /* user change mind and hit Ctrl-C */
        printf("\n");
        return 0;
    }

    return 0;
}

static int setenv_ipaddr( void )
{
    char ip[128];
    int num_chars;

    num_chars = readline("\nEnter IP address: ");
    if (num_chars > 0) {
        memset(ip, 0, 128);
        strncpy(ip, console_buffer, num_chars);
        setenv("ipaddr", ip);
    }
    else if (num_chars < 0) {
        /* user change mind and hit Ctrl-C */
        printf("\n");
        return 0;
    }

    return 0;
}

static int setenv_serverip( void )
{
    char ip[128];
    int num_chars;

    num_chars = readline("\nEnter server IP address: ");
    if (num_chars > 0) {
        memset(ip, 0, 128);
        strncpy(ip, console_buffer, num_chars);
        setenv("serverip", ip);
    }
    else if (num_chars < 0) {
        /* user change mind and hit Ctrl-C */
        printf("\n");
        return 0;
    }

    return 0;
}

static int setenv_gatewayip( void )
{
    char ip[128];
    int num_chars;

    num_chars = readline("\nEnter gateway IP address: ");
    if (num_chars > 0) {
        memset(ip, 0, 128);
        strncpy(ip, console_buffer, num_chars);
        setenv("gatewayip", ip);
    }
    else if (num_chars < 0) {
        /* user change mind and hit Ctrl-C */
        printf("\n");
        return 0;
    }

    return 0;
}

static int setenv_netmask( void )
{
    char mask[128];
    int num_chars;

    num_chars = readline("\nEnter Subnet Mask: ");
    if (num_chars > 0) {
        memset(mask, 0, 128);
        strncpy(mask, console_buffer, num_chars);
        setenv("netmask", mask);
    }
    else if (num_chars < 0) {
        /* user change mind and hit Ctrl-C */
        printf("\n");
        return 0;
    }

    return 0;
}

static int setenv_nfskernel( void )
{
    char file[128];
    int num_chars;

    num_chars = readline("\nEnter NFS Kernel Filename: ");
    if (num_chars > 0) {
        memset(file, 0, 128);
        strncpy(file, console_buffer, num_chars);
        setenv("nfskernel", file);
    }
    else if (num_chars < 0) {
        /* user change mind and hit Ctrl-C */
        printf("\n");
        return 0;
    }

    return 0;
}

static int setenv_rootpath( void )
{
    char path[128];
    int num_chars;

    num_chars = readline("\nEnter NFS root path: ");
    if (num_chars > 0) {
        memset(path, 0, 128);
        strncpy(path, console_buffer, num_chars);
        setenv("rootpath", path);
    }
    else if (num_chars < 0) {
        /* user change mind and hit Ctrl-C */
        printf("\n");
        return 0;
    }

    return 0;
}

static int configure_nfsboot( void )
{
    char buf[128];
    int num_chars;
    cmd_tbl_t *cmdtp;
    char *argv0[2] = { "run", "setnfsce" };
    char *argv1[2] = { "run", "nfsboot" };

    memset(buf, 0, 128);
    num_chars = getenv_r("ipaddr", buf, 128);
    if(num_chars <= 0) {
        printf("FAILED.\n");
        printf("Error : Unable to configure for NFS booting.\n");
        printf("Reason: IP Address must be set for NFS boot.\n");
        return 1;
    }

    memset(buf, 0, 128);
    num_chars = getenv_r("serverip", buf, 128);
    if(num_chars <= 0) {
        printf("FAILED.\n");
        printf("Error : Unable to configure for NFS booting.\n");
        printf("Reason: Server's IP Address must be set for NFS boot.\n");
        return 1;
    }

    memset(buf, 0, 128);
    num_chars = getenv_r("rootpath", buf, 128);
    if(num_chars <= 0) {
        printf("FAILED.\n");
        printf("Error : Unable to configure for NFS booting.\n");
        printf("Reason: NFS Root Path must be set for NFS boot.\n");
        return 1;
    }

    memset(buf, 0, 128);
    num_chars = getenv_r("nfskernel", buf, 128);
    if(num_chars <= 0) {
        printf("FAILED.\n");
        printf("Error : Unable to configure for NFS booting.\n");
        printf("Reason: NFS Kernel file must be known for NFS boot.\n");
        return 1;
    }

    if ((cmdtp = find_cmd(argv0[0])) != NULL) {
        if ((cmdtp->cmd) (cmdtp, 0, 2, argv0) != 0) {
            printf ("Failed to run command\n");
            return 1;
        }
    }

    if ((cmdtp = find_cmd(argv1[0])) != NULL) {
        if ((cmdtp->cmd) (cmdtp, 0, 2, argv1) != 0) {
            printf ("Failed to run command\n");
            return 1;
        }
    }

    return 0;
}

static int configure_flashboot( void )
{
    cmd_tbl_t *cmdtp;
    char *argv0[2] = { "run", "setramce" };
    char *argv1[2] = { "run", "ramboot" };

    if ((cmdtp = find_cmd(argv0[0])) != NULL) {
        if ((cmdtp->cmd) (cmdtp, 0, 2, argv0) != 0) {
            printf ("Failed to run command\n");
            return 1;
        }
    }

    if ((cmdtp = find_cmd(argv1[0])) != NULL) {
        if ((cmdtp->cmd) (cmdtp, 0, 2, argv1) != 0) {
            printf ("Failed to run command\n");
            return 1;
        }
    }

    return 0;
}

static int setenv_console_on( void )
{
    cmd_tbl_t *cmdtp;
    char *argv[3] = { "setenv", "serial_debug", "1" };

    if ((cmdtp = find_cmd(argv[0])) != NULL) {
        if ((cmdtp->cmd) (cmdtp, 0, 3, argv) != 0) {
            printf("Failed to setenv console to on\n");
            return 1;
        }
    }

    return 0;
}

static int setenv_console_off( void )
{
    cmd_tbl_t *cmdtp;
    char *argv[3] = { "setenv", "serial_debug", "0" };

    if ((cmdtp = find_cmd(argv[0])) != NULL) {
        if ((cmdtp->cmd) (cmdtp, 0, 3, argv) != 0) {
            printf("Failed to setenv console to off\n");
            return 1;
        }
    }

    return 0;
}

int setenv_for_customer_use( void )
{
    if(configure_flashboot() != 0) {
        printf("Customer configuration is not attained!\n");
        return 1;
    }

    if(setenv_console_off() != 0) {
        printf("Customer configuration is not fully attained!\n");
        return 1;
    }

    /* save environment */
    saveenv();

    return 0;
}
