/*****************************************************************************
 *  Intersil XICOR X1226 Real-Time Clock with I2C interface functions
 *
 *  FILE: x1226.c
 *
 ******************************************************************************
 *
 * This source code is owned by Raritan Computer, Inc. and is confidential 
 * proprietary information distributed solely pursuant to a confidentiality 
 * agreement or other confidentiality obligation.  It is intended for
 * informational purposes only and is distributed "as is" with no support
 * and no warranty of any kind.
 *
 * Copyright @ 2004-2005 Raritan Computer, Inc. All rights reserved.
 * Reproduction of any element without the prior written consent of
 * Raritan Computer, Inc. is expressly forbidden.
 *
 *****************************************************************************/

#undef DEBUG

#include <common.h>
#include <command.h>
#include <rtc.h>
#include <i2c.h>
#include <x1226.h>


/*
 * Still need to write a handler for alarms and watchdog if it will be enabled.
 */

#if defined(CONFIG_RTC_X1226) && (CONFIG_COMMANDS & CFG_CMD_DATE)

static unsigned bcd2bin( uchar n )
{
    return ((((n >> 4) & 0x0F) * 10) + (n & 0x0F));
}

static unsigned char bin2bcd( unsigned int n )
{
    return (((n / 10) << 4) | (n % 10));
}

static int ccr_write_enable( void )
{
    u8 sr = X1226_SR_WEL;

    i2c_write(CFG_I2C_RTC_ADDR, X1226_SR, 1, &sr, 1);
    sr |= X1226_SR_RWEL;
    i2c_write(CFG_I2C_RTC_ADDR, X1226_SR, 1, &sr, 1);

    return 0;
}

static int ccr_write_disable( void )
{
    u8 sr = 0;

    i2c_write(CFG_I2C_RTC_ADDR, X1226_SR, 1, &sr, 1);

    return 0;
}

void rtc_get( struct rtc_time *tmp )
{
    uchar data[X1226_RTC_REG_CNT];

    /*
     * alen should be 2 since the X1226 expects a two byte word address
     * following the slave byte address.
     */
    i2c_read(CFG_I2C_RTC_ADDR, X1226_SC, 2, data, X1226_RTC_REG_CNT);

    tmp->tm_sec  = bcd2bin(data[X1226_SC - X1226_RTC_BASE] & 0x7F);
    tmp->tm_min  = bcd2bin(data[X1226_MN - X1226_RTC_BASE] & 0x7F);
    tmp->tm_hour = bcd2bin(data[X1226_HR - X1226_RTC_BASE] & 0x3F);
    tmp->tm_mday = bcd2bin(data[X1226_DT - X1226_RTC_BASE] & 0x3F);
    tmp->tm_mon  = bcd2bin(data[X1226_MO - X1226_RTC_BASE] & 0x1F);
    tmp->tm_wday = bcd2bin(data[X1226_DW - X1226_RTC_BASE] & 0x07);
    tmp->tm_year = (bcd2bin(data[X1226_YR - X1226_RTC_BASE]) +
                    (bcd2bin(data[X1226_Y2K - X1226_RTC_BASE]) * 100));
    tmp->tm_yday = 0;
    tmp->tm_isdst= 0;

    debug("Get DATE: %4d-%02d-%02d (wday=%d)  TIME: %2d:%02d:%02d\n",
          tmp->tm_year, tmp->tm_mon, tmp->tm_mday, tmp->tm_wday,
          tmp->tm_hour, tmp->tm_min, tmp->tm_sec);
}

void rtc_set( struct rtc_time *tmp )
{
    uchar data[X1226_RTC_REG_CNT];

    debug("Set DATE: %4d-%02d-%02d (wday=%d)  TIME: %2d:%02d:%02d\n",
          tmp->tm_year, tmp->tm_mon, tmp->tm_mday, tmp->tm_wday,
          tmp->tm_hour, tmp->tm_min, tmp->tm_sec);

    data[X1226_SC - X1226_RTC_BASE] = bin2bcd(tmp->tm_sec) & 0x7F;
    data[X1226_MN - X1226_RTC_BASE] = bin2bcd(tmp->tm_min);
    data[X1226_HR - X1226_RTC_BASE] = ((bin2bcd(tmp->tm_hour) & 0x3F) |
                                       X1226_HR_MIL);
    data[X1226_DT - X1226_RTC_BASE] = bin2bcd(tmp->tm_mday) & 0x3F;
    data[X1226_MO - X1226_RTC_BASE] = bin2bcd(tmp->tm_mon);
    data[X1226_DW - X1226_RTC_BASE] = bin2bcd(tmp->tm_wday) & 0x07;
    data[X1226_YR - X1226_RTC_BASE] = bin2bcd(tmp->tm_year % 100);
    data[X1226_Y2K - X1226_RTC_BASE] = bin2bcd(tmp->tm_year/100);

    ccr_write_enable();

    i2c_write(CFG_I2C_RTC_ADDR, X1226_SC, 2, data, X1226_RTC_REG_CNT);

    ccr_write_disable();

    return;
}

void rtc_reset( void )
{
    return;
}

void rtc_init( void )
{
    struct rtc_time tm;
    uchar  sr;

    /* probe for the presence of the device */
    if (i2c_probe(CFG_I2C_RTC_ADDR) != 0) {
        printf("\nWARNING: No RTC found.\n");
        return;
    }

    /* check if the RTC has lost power before */
    sr = rtc_read(X1226_SR);
    if( sr & X1226_SR_RTCF ){
        /*
         * Do not re-start the clock because it is not correct anymore.
         * The RTC needs to be set to the correct date/time again!
         */
        printf("********************************************\n");
        printf("* Real-Time Clock needs to be set again!!! *\n");
        printf("********************************************\n");
        return;
    }

    /* now get the current time */
    rtc_get(&tm);
    printf ("Current time is %4d-%02d-%02d  %2d:%02d:%02d UTC.\n", tm.tm_year,
            tm.tm_mon, tm.tm_mday, tm.tm_hour, tm.tm_min, tm.tm_sec);

    return;
}

uchar rtc_read( uchar reg )
{
    uchar data;

    i2c_read(CFG_I2C_RTC_ADDR, reg, 2, &data, 1);

    return (data);
}

void rtc_write( uchar reg, uchar val )
{
    ccr_write_enable();
    i2c_write(CFG_I2C_RTC_ADDR, reg, 2, &val, 1);
    ccr_write_disable();
}

int rtc_check_regaddr( uchar reg )
{
    if (reg > 0x3F) {
        debug("Invalid X1226 Register!\n");
        return -1;
    }

    /* register address is ok */
    return 0;
}

void rtc_read_all( uchar *data )
{
    /* read by sections */

    i2c_read(CFG_I2C_RTC_ADDR, X1226_SCA1, 2, (uchar *)(data + X1226_SCA0), 8);
    printf("Alarm0 Sec   (00h): 0x%02x\n", data[X1226_SCA0]);
    printf("Alarm0 Min   (01h): 0x%02x\n", data[X1226_MNA0]);
    printf("Alarm0 Hour  (02h): 0x%02x\n", data[X1226_HRA0]);
    printf("Alarm0 Date  (03h): 0x%02x\n", data[X1226_DTA0]);
    printf("Alarm0 Month (04h): 0x%02x\n", data[X1226_MOA0]);
    printf("Alarm0 Year  (05h): 0x%02x\n", data[X1226_YRA0]);
    printf("Alarm0 Day   (06h): 0x%02x\n", data[X1226_DWA0]);
    printf("Alarm0 Epoch (07h): 0x%02x\n", data[X1226_Y2K0]);

    i2c_read(CFG_I2C_RTC_ADDR, X1226_SCA1, 2, (uchar *)(data + X1226_SCA1), 8);
    printf("Alarm1 Sec   (08h): 0x%02x\n", data[X1226_SCA1]);
    printf("Alarm1 Min   (09h): 0x%02x\n", data[X1226_MNA1]);
    printf("Alarm1 Hour  (0Ah): 0x%02x\n", data[X1226_HRA1]);
    printf("Alarm1 Date  (0Bh): 0x%02x\n", data[X1226_DTA1]);
    printf("Alarm1 Month (0Ch): 0x%02x\n", data[X1226_MOA1]);
    printf("Alarm1 Year  (0Dh): 0x%02x\n", data[X1226_YRA1]);
    printf("Alarm1 Day   (0Eh): 0x%02x\n", data[X1226_DWA1]);
    printf("Alarm1 Epoch (0Fh): 0x%02x\n", data[X1226_Y2K1]);

    i2c_read(CFG_I2C_RTC_ADDR, X1226_BL, 2, (uchar *)(data + X1226_BL), 4);
    printf("BL           (10h): 0x%02x\n", data[X1226_BL]);
    printf("INT          (11h): 0x%02x\n", data[X1226_INT]);
    printf("ATR          (12h): 0x%02x\n", data[X1226_ATR]);
    printf("DTR          (13h): 0x%02x\n", data[X1226_DTR]);

    i2c_read(CFG_I2C_RTC_ADDR, X1226_SC, 2, (uchar *)(data + X1226_SC), 8);
    printf("Seconds      (30h): 0x%02x\n", data[X1226_SC]);
    printf("Minutes      (31h): 0x%02x\n", data[X1226_MN]);
    printf("Hours        (32h): 0x%02x\n", data[X1226_HR]);
    printf("Date         (33h): 0x%02x\n", data[X1226_DT]);
    printf("Month        (34h): 0x%02x\n", data[X1226_MO]);
    printf("Year         (35h): 0x%02x\n", data[X1226_YR]);
    printf("Day          (36h): 0x%02x\n", data[X1226_DW]);
    printf("Epoch        (37h): 0x%02x\n", data[X1226_Y2K]);

    i2c_read(CFG_I2C_RTC_ADDR, X1226_SR, 2, (uchar *)(data + X1226_SR), 1);
    printf("Status       (3Fh): 0x%02x\n", data[X1226_SR]);
    return;
}

void rtc_print( uchar reg, uchar val )
{
    switch( reg ) {
        /* Alarm 0 */
        case X1226_SCA0:
            printf("Alarm0 Sec   (00h): 0x%02x\n", val);
            break;
        case X1226_MNA0:
            printf("Alarm0 Min   (01h): 0x%02x\n", val);
            break;
        case X1226_HRA0:
            printf("Alarm0 Hour  (02h): 0x%02x\n", val);
            break;
        case X1226_DTA0:
            printf("Alarm0 Date  (03h): 0x%02x\n", val);
            break;
        case X1226_MOA0:
            printf("Alarm0 Month (04h): 0x%02x\n", val);
            break;
        case X1226_YRA0:
            printf("Alarm0 Year  (05h): 0x%02x\n", val);
            break;
        case X1226_DWA0:
            printf("Alarm0 Day   (06h): 0x%02x\n", val);
            break;
        case X1226_Y2K0:
            printf("Alarm0 Epoch (07h): 0x%02x\n", val);
            break;

        /* Alarm 1 */
        case X1226_SCA1:
            printf("Alarm1 Sec   (08h): 0x%02x\n", val);
            break;
        case X1226_MNA1:
            printf("Alarm1 Min   (09h): 0x%02x\n", val);
            break;
        case X1226_HRA1:
            printf("Alarm1 Hour  (0Ah): 0x%02x\n", val);
            break;
        case X1226_DTA1:
            printf("Alarm1 Date  (0Bh): 0x%02x\n", val);
            break;
        case X1226_MOA1:
            printf("Alarm1 Month (0Ch): 0x%02x\n", val);
            break;
        case X1226_YRA1:
            printf("Alarm1 Year  (0Dh): 0x%02x\n", val);
            break;
        case X1226_DWA1:
            printf("Alarm1 Day   (0Eh): 0x%02x\n", val);
            break;
        case X1226_Y2K1:
            printf("Alarm1 Epoch (0Fh): 0x%02x\n", val);
            break;

        /* Control */
        case X1226_BL:
            printf("BL           (10h): 0x%02x\n", val);
            break;
        case X1226_INT:
            printf("INT          (11h): 0x%02x\n", val);
            break;
        case X1226_ATR:
            printf("ATR          (12h): 0x%02x\n", val);
            break;
        case X1226_DTR:
            printf("DTR          (13h): 0x%02x\n", val);
            break;

        /* RTC */
        case X1226_SC:
            printf("Seconds      (30h): 0x%02x\n", val);
            break;
        case X1226_MN:   
            printf("Minutes      (31h): 0x%02x\n", val);
            break;
        case X1226_HR:
            printf("Hours        (32h): 0x%02x\n", val);
            break;
        case X1226_DT:
            printf("Date         (33h): 0x%02x\n", val);
            break;
        case X1226_MO:
            printf("Month        (34h): 0x%02x\n", val);
            break;
        case X1226_YR:
            printf("Year         (35h): 0x%02x\n", val);
            break;
        case X1226_DW:
            printf("Day          (36h): 0x%02x\n", val);
            break;
        case X1226_Y2K:
            printf("Epoch        (37h): 0x%02x\n", val);
            break;

        /* Status */
        case X1226_SR:
            printf("Status       (3Fh): 0x%02x\n", val);
            break;

        default:
            printf("Invalid Register Address (0x%x)\n", reg);
    }
    return;
}
#endif /* CONFIG_RTC_X1226 && CFG_CMD_DATE */
