/*
 * (C) Copyright 2001
 * Wolfgang Denk, DENX Software Engineering, wd@denx.de.
 *
 * Derived work: (C) Copyright 2002-2005 Peppercon AG
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <common.h>
#include "kxgen2_hw.h"
#include <asm/processor.h>
#include <asm/io.h>
#include <ppc4xx.h>
#include <mii_phy.h>
#include "mii_phy_10_100.h"
#include "lxt972.h"
#include "lcd_raritan.h"
#include "x1226.h"
#include "board_type.h"
#include "raritan_console.h"
#include "key.h"

#if (DYNAMIC_PLL == 1)
# include <i2c.h>
#endif

#define myendtick(milliseconds) (get_ticks() + (uint64_t)(milliseconds) * (get_tbclk() / 1000))

#ifdef PP_FEAT_SYSOK_BLINKING
# define PP_GPIO_STATUS_LED	9
static volatile unsigned char led_state = 0;
#endif

int board_pre_init (void)
{
    /* ------------------------------------------------------------------------
     * Interrupt controller setup
     */

    mtdcr(uicsr,  0xFFFFFFFF); /* clear all interrupts */
    mtdcr(uicer,  0x00000000); /* disable all interrupts */
    mtdcr(uiccr,  0x00000000); /* set all SMI to be non-critical */

    mtdcr(uicpr,  PP_CALC_UIC0_PR(PP_HWID)); /* set polarities */
    mtdcr(uictr,  PP_CALC_UIC0_TR(PP_HWID)); /* set edge/level characteritics */
    mtdcr(uicvcr, 0x00000001); /* set vect base=0, INT0 highest priority */
    mtdcr(uicsr,  0xFFFFFFFF); /* clear all intterrupts */

    /* ------------------------------------------------------------------------
     * GPIO setup
     */

    /* configure shared pins */
#if defined(CONFIG_405GP)
    mtdcr(cntrl0, mfdcr(cntrl0) | PP_CALC_CPC0_CR0_GPIO(PP_HWID));
    mtdcr(eirr, mfdcr(eirr) & 0x07ffffff ); /* clear UIC IRQ19, ref p221 */
    mtdcr(eirr, mfdcr(eirr) | 0x78000000 ); /* route GPIO15 to UIC IRQ19 */
#elif defined(CONFIG_405EP)
    
#endif
  
    /* set pegels */
    out32(GPIO0_OR, PP_CALC_GPIO0_OR(PP_HWID));
   
    /* set input/output behavior */
    out32(GPIO0_TCR, PP_CALC_GPIO0_TCR(PP_HWID));

    /* set open drain behavior */
    out32(GPIO0_ODR, PP_CALC_GPIO0_ODR(PP_HWID));

#if !defined(CONFIG_405EP)
    /* ------------------------------------------------------------------------
     * PHY reset
     */

    /* we dereference the start address of Chip Select 3 (PHY) */
    (void)*(volatile ulong*) (PP_CALC_EBC0_BnCR(PP_HWID,03) & 0xFFF00000);
    sync();
    udelay(1000);
#endif /* !CONFIG_405EP */

    return 0;
}

#if (DYNAMIC_PLL == 1)

#define PLL_SETTINGS_EQUAL(a0,b0,a1,b1)	(((a0 & 0x00333333) == (b0 & 0x00333333))  &&  \
					 ((a1 & 0x00F773FF) == (b1 & 0x00F773FF)))

#define RTC_EEP_I2C_ADDR	0x57
#define SPEED_GRADE_AUTO	0
#define SPEED_GRADE_SLOW   	1
#define SPEED_GRADE_FAST	2

static unsigned char
get_speed_grade_from_eeprom(void)
{
    unsigned char val[2];
    int ret = SPEED_GRADE_AUTO;
  
    if (i2c_read(RTC_EEP_I2C_ADDR, 0x0000, 2, val, 2) != 0) {
	goto bail;	
    }

    if (val[0] == 0xFF || val[0] == 0x00 ||
	(val[1] & 0x03) > SPEED_GRADE_FAST) {
	goto bail;
    }

    ret = val[1] & 0x03;

 bail:
    debug("get_speed_grade_from_eeprom %02x\n", ret);
    return ret;
}

int
board_reinit_pll(void)
{
    unsigned char speed;
    unsigned long pllmr0, pllmr1, temp;
    
    // determine speed grade we would like to see
    if ((speed = get_speed_grade_from_eeprom()) == SPEED_GRADE_AUTO) {
	speed = SPEED_GRADE_FAST; // FIXME, read GPIO in case of AUTO
    }

    pllmr0 = mfdcr(CPC0_PLLMR0);
    pllmr1 = mfdcr(CPC0_PLLMR1);
    
    if ((speed == SPEED_GRADE_SLOW && PLL_SETTINGS_EQUAL(PLLMR0_SLOW, pllmr0, PLLMR1_SLOW, pllmr1)) ||
	(speed == SPEED_GRADE_FAST && PLL_SETTINGS_EQUAL(PLLMR0_FAST, pllmr0, PLLMR1_FAST, pllmr1))) {

	return 0;
    }

    printf("Reinitializing PLL\n");
    if (speed == SPEED_GRADE_SLOW) {
	pllmr0 = PLLMR0_SLOW;
	pllmr1 = PLLMR1_SLOW;
    } else {
	pllmr0 = PLLMR0_FAST;
	pllmr1 = PLLMR1_FAST;
    }

    // reprogram the PLL and reset again
    mtdcr(CPC0_UCR, (mfdcr(CPC0_UCR) & 0xFFFF0000) | 0x00000101); // stop UART clock

    // reset and disable PLL
    temp = mfdcr(CPC0_PLLMR1) & 0x7FFFFFFF;
    mtdcr(CPC0_PLLMR1, temp);
    temp |= 0x40000000;
    mtdcr(CPC0_PLLMR1, temp);

    // set new dividers
    mtdcr(CPC0_PLLMR0, pllmr0);
    temp = (pllmr1 & 0x3FFFFFFF) | 0x40000000;
    mtdcr(CPC0_PLLMR1, temp);
 
    // clear PLL reset, wait and enable it
    temp &= 0xBFFFFFFF;
    mtdcr(CPC0_PLLMR1, temp);
    udelay(100);
    temp |= 0x80000000;
    mtdcr(CPC0_PLLMR1, temp);

    // reset core, cu later
    mtspr(dbcr0, 0x10000000);
    
    /* NOTREACHED */
    return 0;
}

#endif

#ifdef PP_FEAT_SYSOK_BLINKING
void
toggle_led_state(void)
{
    ulong iflag = disable_interrupts();
    
    /*	  
      Let the status LED on GPIO9 blink on LARA XP boards.
      Its ugly here, but its just a stupid LED for now.
      Next time we need periodic events I'll implement a
      timer queue...
    */
    unsigned long reg = in32(GPIO0_OR);
    reg &= ~(1 << (31 - PP_GPIO_STATUS_LED));
    reg |= ((led_state) << (31 - PP_GPIO_STATUS_LED));
    out32(GPIO0_OR, reg);
    led_state = (led_state) ? 0 : 1;
    
    if (iflag) enable_interrupts();
}
#endif

#if defined CFG_JFFS_CUSTOM_PART
#include <jffs2/jffs2.h>

static struct part_info part;
static int current_part = -1;

struct part_info*
jffs2_part_info(int part_num)
{
    if (part_num == 0) {
	if (current_part == part_num)
		return &part;

	memset(&part, 0, sizeof(part));
	part.offset = CFG_FLASH_BASE;
	part.size = PP_(FLASH_SIZE, PP_HWID) - CFG_MONITOR_LEN;

	/* Mark the struct as ready */
	current_part = part_num;

	return &part;
    }
    return 0;
}
#endif /* CFG_JFFS_CUSTOM_PART */

/* ------------------------------------------------------------------------- */

/*
 * Check Board Identity:
 */

unsigned char checkboard (void)
{
    unsigned char id = 0;

#ifndef USE_KX101_KIM_ON_WINGLE
    unsigned short *hw_id_addr = (unsigned short *)PP_HWID_18_CS_07_CR_BAS;
    unsigned short cs7_data;
    unsigned char hw_id;

    cs7_data = *(unsigned short *)hw_id_addr;
    hw_id = (cs7_data & 0xff00) >> 8;
    printf("HWID & Factory RESET: 0x%04x\n", cs7_data);
    printf("Board: ");
    switch( hw_id & RARITAN_HWID_VENDOR_MASK ) {
        case PEPPERCON_PRODUCT:
            printf("Peppercon ");
            break;
        case RARITAN_PRODUCT:
            printf("Raritan ");
            break;
        default:
            break;
    }

    if( hw_id & RARITAN_HWID_PRODUCT_MASK ) {
        printf("KSX");
    }
    else {
        printf("KX");
        switch( hw_id & RARITAN_HWID_MODEL_MASK ) {
            case KX432_MODEL:
                printf("432");
                break;
            case KX416_MODEL:
                printf("416");
                break;
            case KX232_MODEL:
                printf("232");
                break;
            case KX216_MODEL:
                printf("216");
                break;
            case KX132_MODEL:
                printf("132");
                break;
            case KX116_MODEL:
                printf("116");
                break;
            case KX464_MODEL:
                printf("464");
                break;
            default:
                printf("xxx");
                break;
        }
    }
    printf(" PCB %d\n", (hw_id & RARITAN_HWID_PCB_VERSION_MASK) >> RARITAN_HWID_PCB_VERSION_SHIFT);
    id = hw_id;
#else
    /*
      this doesn't work because of b0rken hardware (no bus driver)
      
      hw_id_addr = (unsigned char *) 0xF0600000;
    */

    printf ("HWID:  %02x - %s\n", PP_HWID_INT, PP_HW_DESCRIPTION(PP_HWID));
    id = PP_HWID_INT;
#endif

    return id;
}

long int initdram (int board_type)
{
    /* return the SDRAM size */
    return ((1 << (((PP_(SDRAM_B0CR, PP_HWID) >> 17) & 0x7) + 2)) * 1024 * 1024);
}

/* ------------------------------------------------------------------------- */

int testdram (void)
{
    /* TODO: XXX XXX XXX */
    printf ("test: xxx MB - ok\n");

    return (0);
}

/* ------------------------------------------------------------------------- */

int verify_password( int nchars, unsigned char *code_buffer )
{
    int i;
    unsigned char key[ACCESS_CODE_SIZE], passcode[ACCESS_CODE_SIZE], pcode;

    key[0] = KEYCODE0 & 0xff;
    key[1] = (KEYCODE1 >> 24) & 0xff;
    key[2] = (KEYCODE2 >> 16) & 0xff;
    key[3] = KEYCODE3 & 0xff;
    key[4] = (KEYCODE4 >> 8) & 0xff;
    key[5] = (KEYCODE5 >> 24) & 0xff;
    key[6] = (KEYCODE6 >> 8) & 0xff;
    key[7] = (KEYCODE7 >> 8) & 0xff;
    key[8] = (KEYCODE8 >> 16) & 0xff;
    key[9] = (KEYCODE9 >> 8) & 0xff;
    key[10] = (KEYCODE10 >> 24) & 0xff;

    if( nchars == 10 ) {
        passcode[0] = (PASSCODE0 >> 8) & 0xff;
        passcode[1] = (PASSCODE1 >> 24) & 0xff;
        passcode[2] = (PASSCODE2 >> 20) & 0xff;
        passcode[3] = (PASSCODE3 >> 12) & 0xff;
        passcode[4] = PASSCODE4 & 0xff;
        passcode[5] = (PASSCODE5 >> 4) & 0xff;
        passcode[6] = (PASSCODE6 >> 16) & 0xff;
        passcode[7] = PASSCODE7 & 0xff;
        passcode[8] = (PASSCODE8 >> 12) & 0xff;
        passcode[9] = PASSCODE9 & 0xff;
        for(i = 0; i < nchars; i++) {
            pcode = code_buffer[i] ^ key[i];
            if(pcode != passcode[i]) {
                return 0;
            }
        }
        return 1;
    }

    return 0;
}

int misc_init_r(void)
{
    int n_chan;

    /*
     * Initialize video controllers
     */
    printf("Initializing Video controller hardware..");
    if(lcd_hw_init() == LCD_SUCCESS) {
        printf("passed.\n");

        printf("Number of Video controllers found............");
        n_chan = lcd_detect();
        printf("%d.\n", n_chan);

        printf("Initializing Video controller driver....");
        if(lcd_sw_init(n_chan) == LCD_SUCCESS) {
            printf("passed.\n");
        }
    }

    /*
     * Initialize Real-Time Clock and Get current time
     */
    rtc_init();

    /* detect the PHY devices */
    mii_discover_phy();

    /* configure it for Auto-negotiation, 10/100 Mbs, Full-or-Half-duplex */
    mii_phy_set_configuration(AUTONEG_ENABLE, LINK_ALL);

#ifdef USE_KX101_KIM_ON_WINGLE
    /* configure the PHY LED3 to display link and activity status */
    mii_phy_config_led(PHY_LED3, LED_DISP_LINK_ACTIVITY, PHY_LED_OPTION1);
#endif

    return 0;
}

#if defined(CONFIG_POST)
void post_word_store (ulong a)
{
    return;
}

ulong post_word_load (void)
{
    return 0;
}

/*
 * Returns 1 if keys pressed to start the power-on long-running tests
 * Called from board_init_f().
 */
int post_hotkeys_pressed(void)
{
    return 0;
}
#endif  /* CONFIG_POST */
