#ifndef _PEPPERCON_H
#define _PEPPERCON_H

/*
 * Turn on when using a KX101 KIM module for testing instead of KX2.0
 */
/*#define USE_KX101_KIM_ON_WINGLE*/

/*
 * Processor types
 */
#define PP_CPU_405GP	1
#define PP_CPU_405EP	2
#define PP_CPU_405GPR	3

/*
 * Include hardware definitions.
 */
#ifdef USE_KX101_KIM_ON_WINGLE
#include "hw_0E.h"
#else
#include "hw_18.h"
#endif

/*
 * serial lines
 */
#define PP_SERIAL_0     0
#define PP_SERIAL_1     1

/*
 * Generic macro for getting hardware id specific values.
 *
 * NOTES: - Hardware ids are always two characters wide - possibly
 *	    with leading zeros.
 *
 *	  - The routine is wrapped to ensure that the arguments get
 *          expanded first.
 *
 *	  - The "name" parameter must not be macro expanded!
 */
#define PP_(name, hwid) __PP_PREFIX(_##name, hwid)
#define __PP_PREFIX(name, hwid) PP_HWID_##hwid##name

/*
 * Texttual description of the hardware
 *
 * NOTES: - Hardware ids are always two characters wide - possibly
 *	    with leading zeros.
 *
 *	  - The routine is wrapped to ensure that the arguments get
 *          expanded first.
 */
#define PP_HW_DESCRIPTION(hwid) __PP_HW_DESCRIPTION(hwid)
#define __PP_HW_DESCRIPTION(hwid) PP_HWID_##hwid##_DESCRIPTION

/*
 * Calculate the EBC Configuration Register (EBC0_CFG) for a given hardware id.
 *
 * NOTES: - Hardware ids are always two characters wide - possibly
 *	    with leading zeros.
 *
 *	  - The routine is wrapped to ensure that the arguments get
 *          expanded first.
 */
#define PP_CALC_EBC0_CFG(hwid) __PP_CALC_EBC0_CFG(hwid)
#define __PP_CALC_EBC0_CFG(hwid)				   \
    (((PP_HWID_##hwid##_EBC_CFG_EBTC   & 0x01) << (31-0))  \
     | ((PP_HWID_##hwid##_EBC_CFG_PTD  & 0x01) << (31-1))  \
     | ((PP_HWID_##hwid##_EBC_CFG_RTC  & 0x07) << (31-4))  \
     | ((PP_HWID_##hwid##_EBC_CFG_EMPL & 0x03) << (31-6))  \
     | ((PP_HWID_##hwid##_EBC_CFG_EMPH & 0x03) << (31-8))  \
     | ((PP_HWID_##hwid##_EBC_CFG_CSTC & 0x01) << (31-9))  \
     | ((PP_HWID_##hwid##_EBC_CFG_BPF  & 0x03) << (31-11)) \
     | ((PP_HWID_##hwid##_EBC_CFG_EMS  & 0x03) << (31-13)) \
     | ((PP_HWID_##hwid##_EBC_CFG_PME  & 0x01) << (31-14)) \
     | ((PP_HWID_##hwid##_EBC_CFG_PMT  & 0x1f) << (31-19)))

/*
 * Calculate the Peripheral Bank Access Parameters (EBC0_BnAP)
 * for a given hardware id and bank number.
 *
 * NOTES: - Hardware ids and bank numbers are always two characters
 *	    wide - possibly with leading zeros.
 *
 *	  - The routine is wrapped to ensure that the arguments get
 *          expanded first.
 */
#define PP_CALC_EBC0_BnAP(hwid, bank) __PP_CALC_EBC0_BnAP(hwid, bank)
#define __PP_CALC_EBC0_BnAP(hwid, bank)					\
    (((PP_HWID_##hwid##_CS_##bank##_AP_BME   & 0x01) << (31-0))		\
     | ((PP_HWID_##hwid##_CS_##bank##_AP_TWT & 0xff &			\
	 ((PP_HWID_##hwid##_CS_##bank##_AP_BME & 0x1) - 1)) << (31-8))	\
     | (((PP_HWID_##hwid##_CS_##bank##_AP_FWT   & 0x1f &		\
	  (0 - (PP_HWID_##hwid##_CS_##bank##_AP_BME & 0x1))) << (31-5))	\
	| ((PP_HWID_##hwid##_CS_##bank##_AP_BWT & 0x07 &		\
	    (0 - (PP_HWID_##hwid##_CS_##bank##_AP_BME & 0x1))) << (31-8))) \
     | ((PP_HWID_##hwid##_CS_##bank##_AP_CSN & 0x03) << (31-13))	\
     | ((PP_HWID_##hwid##_CS_##bank##_AP_OEN & 0x03) << (31-15))	\
     | ((PP_HWID_##hwid##_CS_##bank##_AP_WBN & 0x03) << (31-17))	\
     | ((PP_HWID_##hwid##_CS_##bank##_AP_WBF & 0x03) << (31-19))	\
     | ((PP_HWID_##hwid##_CS_##bank##_AP_TH  & 0x07) << (31-22))	\
     | ((PP_HWID_##hwid##_CS_##bank##_AP_RE  & 0x01) << (31-23))	\
     | ((PP_HWID_##hwid##_CS_##bank##_AP_SOR & 0x01) << (31-24))	\
     | ((PP_HWID_##hwid##_CS_##bank##_AP_BEM & 0x01) << (31-25))	\
     | ((PP_HWID_##hwid##_CS_##bank##_AP_PEN & 0x01) << (31-26)))

/*
 * Calculate the Peripheral Bank Configuration Register (EBC0_BnCR)
 * for a given hardware id and bank number.
 *
 * NOTES: - Hardware ids and bank numbers are always two characters
 *	    wide - possibly with leading zeros.
 *
 *	  - The routine is wrapped to ensure that the arguments get
 *          expanded first.
 */
#define PP_CALC_EBC0_BnCR(hwid, bank) __PP_CALC_EBC0_BnCR(hwid, bank)
#define __PP_CALC_EBC0_BnCR(hwid, bank)				\
    ((PP_HWID_##hwid##_CS_##bank##_CR_BAS   & 0xFFF00000)	\
     | ((PP_HWID_##hwid##_CS_##bank##_CR_BS & 0x7) << (31-14))	\
     | ((PP_HWID_##hwid##_CS_##bank##_CR_BU & 0x3) << (31-16))	\
     | ((PP_HWID_##hwid##_CS_##bank##_CR_BW & 0x3) << (31-18)))

/*
 * Calculate the UIC Polarity Register (UIC0_PR) for a given hardware id.
 *
 * NOTES: - Internal interrupts are set to positive polarity.
 *	    (PPC405GP manual 10.5.4)
 *
 *        - Hardware ids are always two characters wide - possibly
 *	    with leading zeros.
 *
 *	  - The routine is wrapped to ensure that the arguments get
 *          expanded first.
 */
#define PP_CALC_UIC0_PR(hwid) __PP_CALC_UIC0_PR(hwid)
#define __PP_CALC_UIC0_PR(hwid)						\
    (0xFFFFE000								\
     | ((PP_HWID_##hwid##_EXT_IRQ_07_HAS_POS_POLARITY & 0x1) << (31-19)) \
     | ((PP_HWID_##hwid##_EXT_IRQ_08_HAS_POS_POLARITY & 0x1) << (31-20)) \
     | ((PP_HWID_##hwid##_EXT_IRQ_09_HAS_POS_POLARITY & 0x1) << (31-21)) \
     | ((PP_HWID_##hwid##_EXT_IRQ_10_HAS_POS_POLARITY & 0x1) << (31-22)) \
     | ((PP_HWID_##hwid##_EXT_IRQ_11_HAS_POS_POLARITY & 0x1) << (31-23)) \
     | ((PP_HWID_##hwid##_EXT_IRQ_12_HAS_POS_POLARITY & 0x1) << (31-24)) \
     | ((PP_HWID_##hwid##_EXT_IRQ_00_HAS_POS_POLARITY & 0x1) << (31-25)) \
     | ((PP_HWID_##hwid##_EXT_IRQ_01_HAS_POS_POLARITY & 0x1) << (31-26)) \
     | ((PP_HWID_##hwid##_EXT_IRQ_02_HAS_POS_POLARITY & 0x1) << (31-27)) \
     | ((PP_HWID_##hwid##_EXT_IRQ_03_HAS_POS_POLARITY & 0x1) << (31-28)) \
     | ((PP_HWID_##hwid##_EXT_IRQ_04_HAS_POS_POLARITY & 0x1) << (31-29)) \
     | ((PP_HWID_##hwid##_EXT_IRQ_05_HAS_POS_POLARITY & 0x1) << (31-30)) \
     | ((PP_HWID_##hwid##_EXT_IRQ_06_HAS_POS_POLARITY & 0x1) << (31-31)))

/*
 * Calculate the UIC Trigger Register (UIC0_TR) for a given hardware id.
 *
 * NOTES: - Internal interrupts are set to level sensitive and the
 *	    external master interrupt is edge sensitive.
 *	    (PPC405GP manual 10.5.5)
 *
 *        - Hardware ids are always two characters wide - possibly
 *	    with leading zeros.
 *
 *	  - The routine is wrapped to ensure that the arguments get
 *          expanded first.
 */
#define PP_CALC_UIC0_TR(hwid) __PP_CALC_UIC0_TR(hwid)
#define __PP_CALC_UIC0_TR(hwid)						\
    (0x10000000								\
     | ((PP_HWID_##hwid##_EXT_IRQ_07_IS_EDGE_TRIGGERED & 0x1) << (31-19)) \
     | ((PP_HWID_##hwid##_EXT_IRQ_08_IS_EDGE_TRIGGERED & 0x1) << (31-20)) \
     | ((PP_HWID_##hwid##_EXT_IRQ_09_IS_EDGE_TRIGGERED & 0x1) << (31-21)) \
     | ((PP_HWID_##hwid##_EXT_IRQ_10_IS_EDGE_TRIGGERED & 0x1) << (31-22)) \
     | ((PP_HWID_##hwid##_EXT_IRQ_11_IS_EDGE_TRIGGERED & 0x1) << (31-23)) \
     | ((PP_HWID_##hwid##_EXT_IRQ_12_IS_EDGE_TRIGGERED & 0x1) << (31-24)) \
     | ((PP_HWID_##hwid##_EXT_IRQ_00_IS_EDGE_TRIGGERED & 0x1) << (31-25)) \
     | ((PP_HWID_##hwid##_EXT_IRQ_01_IS_EDGE_TRIGGERED & 0x1) << (31-26)) \
     | ((PP_HWID_##hwid##_EXT_IRQ_02_IS_EDGE_TRIGGERED & 0x1) << (31-27)) \
     | ((PP_HWID_##hwid##_EXT_IRQ_03_IS_EDGE_TRIGGERED & 0x1) << (31-28)) \
     | ((PP_HWID_##hwid##_EXT_IRQ_04_IS_EDGE_TRIGGERED & 0x1) << (31-29)) \
     | ((PP_HWID_##hwid##_EXT_IRQ_05_IS_EDGE_TRIGGERED & 0x1) << (31-30)) \
     | ((PP_HWID_##hwid##_EXT_IRQ_06_IS_EDGE_TRIGGERED & 0x1) << (31-31)))

/*
 * Calculate the GPIO part of the Chip Control Register 0 (CPC0_CR0)
 * for a given hardware id.
 *
 * NOTES: - PPC405EP should not use this routine since a different
 *	    configuration mechanism is used.
 *
 *	  - Hardware ids are always two characters wide - possibly
 *	    with leading zeros.
 *
 *	  - The routine is wrapped to ensure that the arguments get
 *          expanded first.
 */
#if PP_(CPU_TYPE, PP_HWID) != PP_CPU_405EP
# define PP_CALC_CPC0_CR0_GPIO(hwid) __PP_CALC_CPC0_CR0_GPIO(hwid)
# define __PP_CALC_CPC0_CR0_GPIO(hwid)				\
    (((PP_HWID_##hwid##_GPIO_10_IS_GPIO & 0x1) << (31-5))	\
     | ((PP_HWID_##hwid##_GPIO_11_IS_GPIO & 0x1) << (31-6))	\
     | ((PP_HWID_##hwid##_GPIO_12_IS_GPIO & 0x1) << (31-7))	\
     | ((PP_HWID_##hwid##_GPIO_13_IS_GPIO & 0x1) << (31-8))	\
     | ((PP_HWID_##hwid##_GPIO_14_IS_GPIO & 0x1) << (31-9))	\
     | ((PP_HWID_##hwid##_GPIO_15_IS_GPIO & 0x1) << (31-10))	\
     | ((PP_HWID_##hwid##_GPIO_16_IS_GPIO & 0x1) << (31-11))	\
     | ((PP_HWID_##hwid##_GPIO_17_IS_GPIO & 0x1) << (31-12))	\
     | ((PP_HWID_##hwid##_GPIO_18_IS_GPIO & 0x1) << (31-13))	\
     | ((PP_HWID_##hwid##_GPIO_19_IS_GPIO & 0x1) << (31-14))	\
     | ((PP_HWID_##hwid##_GPIO_20_IS_GPIO & 0x1) << (31-15))	\
     | ((PP_HWID_##hwid##_GPIO_21_IS_GPIO & 0x1) << (31-16))	\
     | ((PP_HWID_##hwid##_GPIO_22_IS_GPIO & 0x1) << (31-17))	\
     | ((PP_HWID_##hwid##_GPIO_23_IS_GPIO & 0x1) << (31-18)))
#endif /* !PP_CPU_405EP */

/*
 * Calculate the GPIO Output Select Registers (GPIO0_OSRH and GPIO0_OSRL)
 * for a given hardware id.
 *
 * NOTES: - Only PPC405EP should use this routine since a different
 *	    configuration mechanism is used on other processors.
 *
 *	  - Hardware ids are always two characters wide - possibly
 *	    with leading zeros.
 *
 *	  - The routine is wrapped to ensure that the arguments get
 *          expanded first.
 */
#if PP_(CPU_TYPE, PP_HWID) == PP_CPU_405EP
# define PP_CALC_GPIO0_OSRH(hwid) __PP_CALC_GPIO0_OSRH(hwid)
# define __PP_CALC_GPIO0_OSRH(hwid)					\
    ((((PP_HWID_##hwid##_GPIO_00_IS_GPIO & 0x1) ^ 0x1) << (31-1))	\
     | (((PP_HWID_##hwid##_GPIO_01_IS_GPIO & 0x1) ^ 0x1) << (31-3))	\
     | (((PP_HWID_##hwid##_GPIO_02_IS_GPIO & 0x1) ^ 0x1) << (31-5))	\
     | (((PP_HWID_##hwid##_GPIO_03_IS_GPIO & 0x1) ^ 0x1) << (31-7))	\
     | (((PP_HWID_##hwid##_GPIO_04_IS_GPIO & 0x1) ^ 0x1) << (31-9))	\
     | (((PP_HWID_##hwid##_GPIO_05_IS_GPIO & 0x1) ^ 0x1) << (31-11))	\
     | (((PP_HWID_##hwid##_GPIO_06_IS_GPIO & 0x1) ^ 0x1) << (31-13))	\
     | (((PP_HWID_##hwid##_GPIO_07_IS_GPIO & 0x1) ^ 0x1) << (31-15))	\
     | (((PP_HWID_##hwid##_GPIO_08_IS_GPIO & 0x1) ^ 0x1) << (31-17))	\
     | (((PP_HWID_##hwid##_GPIO_09_IS_GPIO & 0x1) ^ 0x1) << (31-19))	\
     | (((PP_HWID_##hwid##_GPIO_10_IS_GPIO & 0x1) ^ 0x1) << (31-21))	\
     | (((PP_HWID_##hwid##_GPIO_11_IS_GPIO & 0x1) ^ 0x1) << (31-23))	\
     | (((PP_HWID_##hwid##_GPIO_12_IS_GPIO & 0x1) ^ 0x1) << (31-25))	\
     | (((PP_HWID_##hwid##_GPIO_13_IS_GPIO & 0x1) ^ 0x1) << (31-27))	\
     | (((PP_HWID_##hwid##_GPIO_14_IS_GPIO & 0x1) ^ 0x1) << (31-29))	\
     | (((PP_HWID_##hwid##_GPIO_15_IS_GPIO & 0x1) ^ 0x1) << (31-31)))
# define PP_CALC_GPIO0_OSRL(hwid) __PP_CALC_GPIO0_OSRL(hwid)
# define __PP_CALC_GPIO0_OSRL(hwid)					\
    ((((PP_HWID_##hwid##_GPIO_16_IS_GPIO & 0x1) ^ 0x1) << (31-1))	\
     | (((PP_HWID_##hwid##_GPIO_17_IS_GPIO & 0x1) ^ 0x1) << (31-3))	\
     | (((PP_HWID_##hwid##_GPIO_18_IS_GPIO & 0x1) ^ 0x1) << (31-5))	\
     | (((PP_HWID_##hwid##_GPIO_19_IS_GPIO & 0x1) ^ 0x1) << (31-7))	\
     | (((PP_HWID_##hwid##_GPIO_20_IS_GPIO & 0x1) ^ 0x1) << (31-9))	\
     | (((PP_HWID_##hwid##_GPIO_21_IS_GPIO & 0x1) ^ 0x1) << (31-11))	\
     | (((PP_HWID_##hwid##_GPIO_22_IS_GPIO & 0x1) ^ 0x1) << (31-13))	\
     | (((PP_HWID_##hwid##_GPIO_23_IS_GPIO & 0x1) ^ 0x1) << (31-15))	\
     | (((PP_HWID_##hwid##_GPIO_24_IS_GPIO & 0x1) ^ 0x1) << (31-17))	\
     | (((PP_HWID_##hwid##_GPIO_25_IS_GPIO & 0x1) ^ 0x1) << (31-19))	\
     | (((PP_HWID_##hwid##_GPIO_26_IS_GPIO & 0x1) ^ 0x1) << (31-21))	\
     | (((PP_HWID_##hwid##_GPIO_27_IS_GPIO & 0x1) ^ 0x1) << (31-23))							\
     | (((PP_HWID_##hwid##_GPIO_28_IS_GPIO & 0x1) ^ 0x1) << (31-25))							\
     | (((PP_HWID_##hwid##_GPIO_29_IS_GPIO & 0x1) ^ 0x1) << (31-27))							\
     | (((PP_HWID_##hwid##_GPIO_30_IS_GPIO & 0x1) ^ 0x1) << (31-29))							\
     | (((PP_HWID_##hwid##_GPIO_31_IS_GPIO & 0x1) ^ 0x1) << (31-31)))
#endif /* PP_CPU_405EP */

/*
 * Determine if a GPIO pin needs to be configured as output. On the
 * PPC405EP we have to respect the alternate PIN configuration as
 * shown in the PPC405EP user manual under section 23.6.1.
 */
#if PP_(CPU_TYPE, PP_HWID) == PP_CPU_405EP
# define _PP_GPIO_IS_OUTPUT(hwid, gpio)				\
    (((PP_HWID_##hwid##_GPIO_##gpio##_IS_GPIO & 0x1)		\
      & (PP_HWID_##hwid##_GPIO_##gpio##_IS_OUTPUT & 0x1))	\
     | (~(PP_HWID_##hwid##_GPIO_##gpio##_IS_GPIO & 0x1)		\
	& (PP_GPIO_##gpio##_TCR_DEFAULT & 0x1)))
#else
# define _PP_GPIO_IS_OUTPUT(hwid, gpio)				\
    ((PP_HWID_##hwid##_GPIO_##gpio##_IS_GPIO & 0x1) &		\
     (PP_HWID_##hwid##_GPIO_##gpio##_IS_OUTPUT & 0x1))
#endif

/*
 * The defaults for the GPIO Three-State Control Register (GPIO0_TCR)
 * on a PPC405EP.
 */
#if PP_(CPU_TYPE, PP_HWID) == PP_CPU_405EP
# define PP_GPIO_00_TCR_DEFAULT	1
# define PP_GPIO_01_TCR_DEFAULT	1
# define PP_GPIO_02_TCR_DEFAULT	1
# define PP_GPIO_03_TCR_DEFAULT	1
# define PP_GPIO_04_TCR_DEFAULT	1
# define PP_GPIO_05_TCR_DEFAULT	1
# define PP_GPIO_06_TCR_DEFAULT	1
# define PP_GPIO_07_TCR_DEFAULT	1
# define PP_GPIO_08_TCR_DEFAULT	1
# define PP_GPIO_09_TCR_DEFAULT	1
# define PP_GPIO_10_TCR_DEFAULT	1
# define PP_GPIO_11_TCR_DEFAULT	1
# define PP_GPIO_12_TCR_DEFAULT	1
# define PP_GPIO_13_TCR_DEFAULT	1
# define PP_GPIO_14_TCR_DEFAULT	1
# define PP_GPIO_15_TCR_DEFAULT	1
# define PP_GPIO_16_TCR_DEFAULT	1
# define PP_GPIO_17_TCR_DEFAULT	0
# define PP_GPIO_18_TCR_DEFAULT	0
# define PP_GPIO_19_TCR_DEFAULT	0
# define PP_GPIO_20_TCR_DEFAULT	0
# define PP_GPIO_21_TCR_DEFAULT	0
# define PP_GPIO_22_TCR_DEFAULT	0
# define PP_GPIO_23_TCR_DEFAULT	0
# define PP_GPIO_24_TCR_DEFAULT	0
# define PP_GPIO_25_TCR_DEFAULT	0
# define PP_GPIO_26_TCR_DEFAULT	0
# define PP_GPIO_27_TCR_DEFAULT	1
# define PP_GPIO_28_TCR_DEFAULT	0
# define PP_GPIO_29_TCR_DEFAULT	1
# define PP_GPIO_30_TCR_DEFAULT	0
# define PP_GPIO_31_TCR_DEFAULT	0
#endif

/*
 * Calculate the GPIO Three-State Control Register (GPIO0_TCR)
 * for a given hardware id.
 *
 * NOTES: - Hardware ids are always two characters wide - possibly
 *	    with leading zeros.
 *
 *	  - The routine is wrapped to ensure that the arguments get
 *          expanded first.
 */
#define PP_CALC_GPIO0_TCR(hwid) __PP_CALC_GPIO0_TCR(hwid)
#define __PP_CALC_GPIO0_TCR(hwid)				\
    (_PP_GPIO0_TCR_405EP_EXT(hwid)				\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 01) & 0x1) << (31-1))		\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 02) & 0x1) << (31-2))		\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 03) & 0x1) << (31-3))		\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 04) & 0x1) << (31-4))		\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 05) & 0x1) << (31-5))		\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 06) & 0x1) << (31-6))		\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 07) & 0x1) << (31-7))		\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 08) & 0x1) << (31-8))		\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 09) & 0x1) << (31-9))		\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 10) & 0x1) << (31-10))	\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 11) & 0x1) << (31-11))	\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 12) & 0x1) << (31-12))	\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 13) & 0x1) << (31-13))	\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 14) & 0x1) << (31-14))	\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 15) & 0x1) << (31-15))	\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 16) & 0x1) << (31-16))	\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 17) & 0x1) << (31-17))	\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 18) & 0x1) << (31-18))	\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 19) & 0x1) << (31-19))	\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 20) & 0x1) << (31-20))	\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 21) & 0x1) << (31-21))	\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 22) & 0x1) << (31-22))	\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 23) & 0x1) << (31-23)))
/* PPC405_EP extensions */
#if PP_(CPU_TYPE, PP_HWID) == PP_CPU_405EP
# define _PP_GPIO0_TCR_405EP_EXT(hwid)				\
    (((_PP_GPIO_IS_OUTPUT(hwid, 00) & 0x1) << (31-0))		\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 24) & 0x1) << (31-24))	\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 25) & 0x1) << (31-25))	\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 26) & 0x1) << (31-26))	\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 27) & 0x1) << (31-27))	\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 28) & 0x1) << (31-28))	\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 29) & 0x1) << (31-29))	\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 30) & 0x1) << (31-30))	\
     | ((_PP_GPIO_IS_OUTPUT(hwid, 31) & 0x1) << (31-31)))
#else
# define _PP_GPIO0_TCR_405EP_EXT(hwid) 0x0
#endif

/*
 * Calculate the GPIO Open Drain Register (GPIO0_ODR) for a given hardware id.
 *
 * NOTES: - Hardware ids are always two characters wide - possibly
 *	    with leading zeros.
 *
 *	  - The routine is wrapped to ensure that the arguments get
 *          expanded first.
 */
#define PP_CALC_GPIO0_ODR(hwid) __PP_CALC_GPIO0_ODR(hwid)
#define __PP_CALC_GPIO0_ODR(hwid)					\
    (_PP_GPIO0_ODR_405EP_EXT(hwid)					\
     | ((PP_HWID_##hwid##_GPIO_01_IS_OPEN_DRAIN & 0x1) << (31-1))	\
     | ((PP_HWID_##hwid##_GPIO_02_IS_OPEN_DRAIN & 0x1) << (31-2))	\
     | ((PP_HWID_##hwid##_GPIO_03_IS_OPEN_DRAIN & 0x1) << (31-3))	\
     | ((PP_HWID_##hwid##_GPIO_04_IS_OPEN_DRAIN & 0x1) << (31-4))	\
     | ((PP_HWID_##hwid##_GPIO_05_IS_OPEN_DRAIN & 0x1) << (31-5))	\
     | ((PP_HWID_##hwid##_GPIO_06_IS_OPEN_DRAIN & 0x1) << (31-6))	\
     | ((PP_HWID_##hwid##_GPIO_07_IS_OPEN_DRAIN & 0x1) << (31-7))	\
     | ((PP_HWID_##hwid##_GPIO_08_IS_OPEN_DRAIN & 0x1) << (31-8))	\
     | ((PP_HWID_##hwid##_GPIO_09_IS_OPEN_DRAIN & 0x1) << (31-9))	\
     | ((PP_HWID_##hwid##_GPIO_10_IS_OPEN_DRAIN & 0x1) << (31-10))	\
     | ((PP_HWID_##hwid##_GPIO_11_IS_OPEN_DRAIN & 0x1) << (31-11))	\
     | ((PP_HWID_##hwid##_GPIO_12_IS_OPEN_DRAIN & 0x1) << (31-12))	\
     | ((PP_HWID_##hwid##_GPIO_13_IS_OPEN_DRAIN & 0x1) << (31-13))	\
     | ((PP_HWID_##hwid##_GPIO_14_IS_OPEN_DRAIN & 0x1) << (31-14))	\
     | ((PP_HWID_##hwid##_GPIO_15_IS_OPEN_DRAIN & 0x1) << (31-15))	\
     | ((PP_HWID_##hwid##_GPIO_16_IS_OPEN_DRAIN & 0x1) << (31-16))	\
     | ((PP_HWID_##hwid##_GPIO_17_IS_OPEN_DRAIN & 0x1) << (31-17))	\
     | ((PP_HWID_##hwid##_GPIO_18_IS_OPEN_DRAIN & 0x1) << (31-18))	\
     | ((PP_HWID_##hwid##_GPIO_19_IS_OPEN_DRAIN & 0x1) << (31-19))	\
     | ((PP_HWID_##hwid##_GPIO_20_IS_OPEN_DRAIN & 0x1) << (31-20))	\
     | ((PP_HWID_##hwid##_GPIO_21_IS_OPEN_DRAIN & 0x1) << (31-21))	\
     | ((PP_HWID_##hwid##_GPIO_22_IS_OPEN_DRAIN & 0x1) << (31-22))	\
     | ((PP_HWID_##hwid##_GPIO_23_IS_OPEN_DRAIN & 0x1) << (31-23)))
/* PPC405_EP extensions */
#if PP_(CPU_TYPE, PP_HWID) == PP_CPU_405EP
# define _PP_GPIO0_ODR_405EP_EXT(hwid)					\
    (((PP_HWID_##hwid##_GPIO_00_IS_OPEN_DRAIN & 0x1) << (31-0))		\
     | ((PP_HWID_##hwid##_GPIO_24_IS_OPEN_DRAIN & 0x1) << (31-24))	\
     | ((PP_HWID_##hwid##_GPIO_25_IS_OPEN_DRAIN & 0x1) << (31-25))	\
     | ((PP_HWID_##hwid##_GPIO_26_IS_OPEN_DRAIN & 0x1) << (31-26))	\
     | ((PP_HWID_##hwid##_GPIO_27_IS_OPEN_DRAIN & 0x1) << (31-27))	\
     | ((PP_HWID_##hwid##_GPIO_28_IS_OPEN_DRAIN & 0x1) << (31-28))	\
     | ((PP_HWID_##hwid##_GPIO_29_IS_OPEN_DRAIN & 0x1) << (31-29))	\
     | ((PP_HWID_##hwid##_GPIO_30_IS_OPEN_DRAIN & 0x1) << (31-30))	\
     | ((PP_HWID_##hwid##_GPIO_31_IS_OPEN_DRAIN & 0x1) << (31-31)))
#else
# define _PP_GPIO0_ODR_405EP_EXT(hwid) 0x0
#endif

/*
 * Calculate the GPIO Output Register (GPIO0_OR) for a given hardware id.
 *
 * NOTES: - Hardware ids are always two characters wide - possibly
 *	    with leading zeros.
 *
 *	  - The routine is wrapped to ensure that the arguments get
 *          expanded first.
 */
#define PP_CALC_GPIO0_OR(hwid) __PP_CALC_GPIO0_OR(hwid)
#define __PP_CALC_GPIO0_OR(hwid)					\
    (_PP_GPIO0_OR_405EP_EXT(hwid)					\
     | ((PP_HWID_##hwid##_GPIO_01_OUTPUT_VALUE & 0x1) << (31-1))	\
     | ((PP_HWID_##hwid##_GPIO_02_OUTPUT_VALUE & 0x1) << (31-2))	\
     | ((PP_HWID_##hwid##_GPIO_03_OUTPUT_VALUE & 0x1) << (31-3))	\
     | ((PP_HWID_##hwid##_GPIO_04_OUTPUT_VALUE & 0x1) << (31-4))	\
     | ((PP_HWID_##hwid##_GPIO_05_OUTPUT_VALUE & 0x1) << (31-5))	\
     | ((PP_HWID_##hwid##_GPIO_06_OUTPUT_VALUE & 0x1) << (31-6))	\
     | ((PP_HWID_##hwid##_GPIO_07_OUTPUT_VALUE & 0x1) << (31-7))	\
     | ((PP_HWID_##hwid##_GPIO_08_OUTPUT_VALUE & 0x1) << (31-8))	\
     | ((PP_HWID_##hwid##_GPIO_09_OUTPUT_VALUE & 0x1) << (31-9))	\
     | ((PP_HWID_##hwid##_GPIO_10_OUTPUT_VALUE & 0x1) << (31-10))	\
     | ((PP_HWID_##hwid##_GPIO_11_OUTPUT_VALUE & 0x1) << (31-11))	\
     | ((PP_HWID_##hwid##_GPIO_12_OUTPUT_VALUE & 0x1) << (31-12))	\
     | ((PP_HWID_##hwid##_GPIO_13_OUTPUT_VALUE & 0x1) << (31-13))	\
     | ((PP_HWID_##hwid##_GPIO_14_OUTPUT_VALUE & 0x1) << (31-14))	\
     | ((PP_HWID_##hwid##_GPIO_15_OUTPUT_VALUE & 0x1) << (31-15))	\
     | ((PP_HWID_##hwid##_GPIO_16_OUTPUT_VALUE & 0x1) << (31-16))	\
     | ((PP_HWID_##hwid##_GPIO_17_OUTPUT_VALUE & 0x1) << (31-17))	\
     | ((PP_HWID_##hwid##_GPIO_18_OUTPUT_VALUE & 0x1) << (31-18))	\
     | ((PP_HWID_##hwid##_GPIO_19_OUTPUT_VALUE & 0x1) << (31-19))	\
     | ((PP_HWID_##hwid##_GPIO_20_OUTPUT_VALUE & 0x1) << (31-20))	\
     | ((PP_HWID_##hwid##_GPIO_21_OUTPUT_VALUE & 0x1) << (31-21))	\
     | ((PP_HWID_##hwid##_GPIO_22_OUTPUT_VALUE & 0x1) << (31-22))	\
     | ((PP_HWID_##hwid##_GPIO_23_OUTPUT_VALUE & 0x1) << (31-23)))
#define LAN_GPIO6	0x02000000	/* i.e. 0x1 << (31-6) */
/* PPC405_EP extensions */
#if PP_(CPU_TYPE, PP_HWID) == PP_CPU_405EP
# define _PP_GPIO0_OR_405EP_EXT(hwid)					\
    (((PP_HWID_##hwid##_GPIO_00_OUTPUT_VALUE & 0x1) << (31-0))		\
     | ((PP_HWID_##hwid##_GPIO_24_OUTPUT_VALUE & 0x1) << (31-24))	\
     | ((PP_HWID_##hwid##_GPIO_25_OUTPUT_VALUE & 0x1) << (31-25))	\
     | ((PP_HWID_##hwid##_GPIO_26_OUTPUT_VALUE & 0x1) << (31-26))	\
     | ((PP_HWID_##hwid##_GPIO_27_OUTPUT_VALUE & 0x1) << (31-27))	\
     | ((PP_HWID_##hwid##_GPIO_28_OUTPUT_VALUE & 0x1) << (31-28))	\
     | ((PP_HWID_##hwid##_GPIO_29_OUTPUT_VALUE & 0x1) << (31-29))	\
     | ((PP_HWID_##hwid##_GPIO_30_OUTPUT_VALUE & 0x1) << (31-30))	\
     | ((PP_HWID_##hwid##_GPIO_31_OUTPUT_VALUE & 0x1) << (31-31)))
#else
# define _PP_GPIO0_OR_405EP_EXT(hwid) 0x0
#endif

/*
 * Get the PLLMR(0|1) configuration register values
 * for a given hardware id.
 */
#define PP_PLLMR0_DEFAULT(hwid) _PP_PLLMR0_DEFAULT(hwid)
#define _PP_PLLMR0_DEFAULT(hwid) __PP_PLLMR0_DEFAULT(hwid)
#define __PP_PLLMR0_DEFAULT(hwid)		\
    PP_HWID_##hwid##_PLLMR0_DEFAULT
#define PP_PLLMR1_DEFAULT(hwid) _PP_PLLMR1_DEFAULT(hwid)
#define _PP_PLLMR1_DEFAULT(hwid) __PP_PLLMR1_DEFAULT(hwid)
#define __PP_PLLMR1_DEFAULT(hwid)		\
    PP_HWID_##hwid##_PLLMR1_DEFAULT

#define PP_DYNAMIC_PLL(hwid) __PP_DYNAMIC_PLL(hwid)
#define __PP_DYNAMIC_PLL(hwid)		\
    PP_HWID_##hwid##_DYNAMIC_PLL

# define PP_PLLMR0_SLOW(hwid) __PP_PLLMR0_SLOW(hwid)
# define __PP_PLLMR0_SLOW(hwid)		\
    PP_HWID_##hwid##_PLLMR0_SLOW
# define PP_PLLMR1_SLOW(hwid) __PP_PLLMR1_SLOW(hwid)
# define __PP_PLLMR1_SLOW(hwid)		\
    PP_HWID_##hwid##_PLLMR1_DEFAULT
# define PP_PLLMR0_FAST(hwid) __PP_PLLMR0_FAST(hwid)
# define __PP_PLLMR0_FAST(hwid)		\
    PP_HWID_##hwid##_PLLMR0_FAST
# define PP_PLLMR1_FAST(hwid) __PP_PLLMR1_FAST(hwid)
# define __PP_PLLMR1_FAST(hwid)		\
    PP_HWID_##hwid##_PLLMR1_FAST

/*
 * Get the big physical memory area option for the kernel
 * for a given hardware id.
 *
 * NOTES: - Hardware ids are always two characters wide - possibly
 *	    with leading zeros.
 *
 *	  - The routine is wrapped more than once to stringify
 *          the result of the actual macro expansion
 */
#define PP_BIGPHYSAREA_OPT(hwid)			\
    PP_STRINGIFY_EXPANDED(__PP_BIGPHYSAREA_OPT(hwid))
#define __PP_BIGPHYSAREA_OPT(hwid)			\
    bigphysarea=PP_HWID_##hwid##_BIGPHYSAREA_PAGES

/*
 * Stringify a parameter (with or without macro expansion first).
 */
#define PP_STRINGIFY_EXPANDED(param) PP_STRINGIFY_UNEXPANDED(param)
#define PP_STRINGIFY_UNEXPANDED(param) #param

#endif /* _PEPPERCON_H */
