/*
 * (C) Copyright 2002
 * Sysgo Real-Time Solutions, GmbH <www.elinos.com>
 * Marius Groeger <mgroeger@sysgo.de>
 *
 * (C) Copyright 2002
 * Sysgo Real-Time Solutions, GmbH <www.elinos.com>
 * Alex Zuepke <azu@sysgo.de>
 *
 * (C) Copyright 2002
 * Gary Jennejohn, DENX Software Engineering, <gj@denx.de>
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <asm/u-boot.h>
#include <asm/proc/ptrace.h>
#include <common.h>

#include "arm940t.h"

#include "kira/porting.h"
#undef IRQ_MODE
#undef FIQ_MODE

#include <asm/ptrace.h>

static volatile unsigned long *timer_regs  = (unsigned long *)CPE_TIMER_BASE;
static volatile unsigned long *irq_regs  = (unsigned long *)CPE_IC_BASE;

extern void reset_cpu(ulong addr);
static void do_timer_irq(void);

#define EXT_CLK				32768
#define TIMER_LOAD_VAL(clock, freq)	((clock) / (freq))

#ifdef CONFIG_USE_IRQ
/* enable IRQ interrupts */
void enable_interrupts (void)
{
    unsigned long temp;
    __asm__ __volatile__("mrs %0, cpsr\n"
			 "bic %0, %0, #0xc0\n"
			 "msr cpsr_c, %0"
			 : "=r" (temp)
			 :
			 : "memory");
}

typedef struct 
{
    UINT32 TimerValue;
    UINT32 TimerLoad;
    UINT32 TimerMatch1;
    UINT32 TimerMatch2;
}fLib_TimerReg;

fLib_TimerReg *TimerBase[] ={0, (fLib_TimerReg *) CPE_TIMER_BASE,
			     (fLib_TimerReg *)(CPE_TIMER_BASE + 0x10),
			     (fLib_TimerReg *)(CPE_TIMER_BASE + 0x20)};

#define TIMER1_LOAD_VAL 0x80000000

/* 
 * disable IRQ/FIQ interrupts
 * returns true if interrupts had been enabled before we disabled them
 */
int disable_interrupts (void)
{
    unsigned long old,temp;
    __asm__ __volatile__("mrs %0, cpsr\n"
			 "orr %1, %0, #0xc0\n"
			 "msr cpsr_c, %1"
			 : "=r" (old), "=r" (temp)
			 :
			 : "memory");
    return (old & 0x80) == 0;
}
#else
void enable_interrupts (void)
{
    return;
}
int disable_interrupts (void)
{
    return 0;
}
#endif



void bad_mode(void)
{
    panic("Resetting CPU ...\n");
    reset_cpu(0);
}

void show_regs(struct pt_regs * regs)
{
    unsigned long flags;
const char *processor_modes[]=
{ "USER_26", "FIQ_26" , "IRQ_26" , "SVC_26" , "UK4_26" , "UK5_26" , "UK6_26" , "UK7_26" ,
  "UK8_26" , "UK9_26" , "UK10_26", "UK11_26", "UK12_26", "UK13_26", "UK14_26", "UK15_26",
  "USER_32", "FIQ_32" , "IRQ_32" , "SVC_32" , "UK4_32" , "UK5_32" , "UK6_32" , "ABT_32" ,
  "UK8_32" , "UK9_32" , "UK10_32", "UND_32" , "UK12_32", "UK13_32", "UK14_32", "SYS_32"
};
    
    flags = condition_codes(regs);
    
    printf("pc : [<%08lx>]    lr : [<%08lx>]\n"
	   "sp : %08lx  ip : %08lx  fp : %08lx\n",
	   instruction_pointer(regs),
	   regs->ARM_lr, regs->ARM_sp,
	   regs->ARM_ip, regs->ARM_fp);
    printf("r10: %08lx  r9 : %08lx  r8 : %08lx\n",
	   regs->ARM_r10, regs->ARM_r9,
	   regs->ARM_r8);
    printf("r7 : %08lx  r6 : %08lx  r5 : %08lx  r4 : %08lx\n",
	   regs->ARM_r7, regs->ARM_r6,
	   regs->ARM_r5, regs->ARM_r4);
    printf("r3 : %08lx  r2 : %08lx  r1 : %08lx  r0 : %08lx\n",
	   regs->ARM_r3, regs->ARM_r2,
	   regs->ARM_r1, regs->ARM_r0);
    printf("Flags: %c%c%c%c",
	   flags & CC_N_BIT ? 'N' : 'n',
	   flags & CC_Z_BIT ? 'Z' : 'z',
	   flags & CC_C_BIT ? 'C' : 'c',
	   flags & CC_V_BIT ? 'V' : 'v');
    printf("  IRQs %s  FIQs %s  Mode %s%s\n",
	   interrupts_enabled(regs) ? "on" : "off",
	   fast_interrupts_enabled(regs) ? "on" : "off",
	   processor_modes[processor_mode(regs)],
	   thumb_mode(regs) ? " (T)" : "");
}

void do_undefined_instruction(struct pt_regs *pt_regs)
{
    printf("undefined instruction\n");
    show_regs(pt_regs);
    bad_mode();
}

void do_software_interrupt(struct pt_regs *pt_regs)
{
    printf("software interrupt\n");
    show_regs(pt_regs);
    bad_mode();
}

void do_prefetch_abort(struct pt_regs *pt_regs)
{
    printf("prefetch abort\n");
    show_regs(pt_regs);
    bad_mode();
}

void do_data_abort(struct pt_regs *pt_regs)
{
    printf("data abort\n");
    show_regs(pt_regs);
    bad_mode();
}

void do_not_used(struct pt_regs *pt_regs)
{
    printf("not used\n");
    show_regs(pt_regs);
    bad_mode();
}

void do_fiq(struct pt_regs *pt_regs)
{
    printf("fast interrupt request\n");
    show_regs(pt_regs);
    bad_mode();
}

void do_irq(struct pt_regs *pt_regs)
{
    u_long status = irq_regs[KIRA_INTC_IRQ_SOURCE];

    if (status & KIRA_IRQ_TIMER) {
	do_timer_irq();
    } else {
	printf("interrupt request (status = 0x%08x)\n", status);
	show_regs(pt_regs);
	bad_mode();
    }    
}

static void do_timer_irq()
{
#if defined(PP_FEAT_SYSOK_BLINKING)
    if (timer_regs[KIRA_TMR_IRQ_STATE] & KIRA_TMR_IRQ_TM2_OF) {
	toggle_led_state();
	timer_regs[KIRA_TMR_IRQ_STATE] &= ~KIRA_TMR_IRQ_TM2_OF;
    }
#endif
}

// --------------------------------------------------------------------
//	warning:
//		timer = 1, 2, 3
// --------------------------------------------------------------------
UINT32 Read_Timer_Counter(UINT32 timer)
{
    volatile fLib_TimerReg *Timer = TimerBase[timer];    

    return Timer->TimerValue;
}


void Set_Timer_AutoReloadValue(UINT32 timer, UINT32 value)
{
    volatile fLib_TimerReg *Timer = TimerBase[timer];

    Timer->TimerLoad = value;
}

static ulong timestamp;
static ulong lastdec;

int interrupt_init (void)
{
    Set_Timer_AutoReloadValue(1, TIMER1_LOAD_VAL);
    Set_Timer_AutoReloadValue(2, TIMER_LOAD_VAL(EXT_CLK, 10));

    irq_regs[KIRA_INTC_IRQ_CLEAR] = 0xFFFFFFFF; /* clear irqs */

    timer_regs[KIRA_TMR_CONTROL] = KIRA_TMR_CR_TM1_ENABLE;

#if defined(PP_FEAT_SYSOK_BLINKING)
    timer_regs[KIRA_TMR_CONTROL] |= KIRA_TMR_CR_TM2_ENABLE | KIRA_TMR_CR_TM2_CLOCK | KIRA_TMR_CR_TM2_OF;
#endif

    /* mask out all IRQ source other than TM2 overflow */
    timer_regs[KIRA_TMR_IRQ_MASK] =
	KIRA_TMR_IRQ_TM1_MATCH1 | KIRA_TMR_IRQ_TM1_MATCH2 | KIRA_TMR_IRQ_TM1_OF |
	KIRA_TMR_IRQ_TM2_MATCH1 | KIRA_TMR_IRQ_TM2_MATCH2 |
	KIRA_TMR_IRQ_TM3_MATCH1 | KIRA_TMR_IRQ_TM3_MATCH2 | KIRA_TMR_IRQ_TM3_OF;
    
    /* set timer irq to level trigger, active high and enable it */
    irq_regs[KIRA_INTC_IRQ_MODE] &= ~KIRA_IRQ_TIMER;
    irq_regs[KIRA_INTC_IRQ_LEVEL] &= ~KIRA_IRQ_TIMER;
    irq_regs[KIRA_INTC_IRQ_ENABLE] |= KIRA_IRQ_TIMER;
    
    return 0;
}

void interrupt_cleanup (void)
{
    /* disable timer interrupt */
    irq_regs[KIRA_INTC_IRQ_ENABLE] &= ~KIRA_IRQ_TIMER;

    /* stop all timers */
    timer_regs[KIRA_TMR_CONTROL] &=
	~(KIRA_TMR_CR_TM1_ENABLE | KIRA_TMR_CR_TM2_ENABLE | KIRA_TMR_CR_TM3_ENABLE);
	    
    /* cleanup all timer IRQs sources */
    timer_regs[KIRA_TMR_IRQ_MASK] =
    	KIRA_TMR_IRQ_TM1_MATCH1 | KIRA_TMR_IRQ_TM1_MATCH2 | KIRA_TMR_IRQ_TM1_OF |
	KIRA_TMR_IRQ_TM2_MATCH1 | KIRA_TMR_IRQ_TM2_MATCH2 | KIRA_TMR_IRQ_TM2_OF |
	KIRA_TMR_IRQ_TM3_MATCH1 | KIRA_TMR_IRQ_TM3_MATCH2 | KIRA_TMR_IRQ_TM3_OF;
}

/*
 * timer without interrupts
 */

void reset_timer(void)
{
    reset_timer_masked();
}

ulong get_timer (ulong base)
{
    return get_timer_masked() - base;
}

void set_timer (ulong t)
{
    timestamp = t;
}

void udelay(unsigned long usec)
{
    ulong tmo;

    tmo = usec / 1000;
    tmo *= CFG_HZ;
    tmo /= 1000;

    tmo += get_timer(0);

    while(get_timer_masked() < tmo)
    {
      	/*NOP*/;
    }
}

void reset_timer_masked(void)
{
    /* reset time */
    lastdec = Read_Timer_Counter(1);
    timestamp = 0;
}


// --------------------------------------------------------------------
//	1. o timestamp
//	2. lastdec = now
// --------------------------------------------------------------------
ulong get_timer_masked(void)
{
    ulong now = Read_Timer_Counter(1);;

    if (lastdec >= now)
    {
        /* normal mode */
        timestamp += lastdec - now;
    } else {
        /* we have an overflow ... */
        timestamp += lastdec + TIMER1_LOAD_VAL - now;
    }
    lastdec = now;

    return timestamp;
}


void udelay_masked(unsigned long usec)
{
    ulong tmo;

    tmo = usec / 1000;
    tmo *= CFG_HZ;
    tmo /= 1000;

    reset_timer_masked();

    while(get_timer_masked() < tmo)
    {
      	/*NOP*/;
    }
}

