#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/reboot.h>
#include <sys/socket.h>
#include <fcntl.h>
#include <string.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <pp/base.h>
#include <pp/firmware.h>
#include <pp/smtp.h>
#include <pp/powerswitch.h>
#include <liberic_misc.h>
#include <liberic_notify.h>
#include <pp/um.h>
#include <lara.h>
#include <liberic_config.h>

#ifdef PP_FEAT_IPMI_SERVER
#include <pp/ipmi.h>
#endif


#define TABLE_SIZE 5
#define ERIC_1_2_FIRST_MAC "\xFE\x00\x00\x42\x00\x3F"
#define DEBUG 0

#ifdef DEBUG
# define D(fmt, args...)	{ fprintf(stderr, fmt, ##args); }
#else
# define D(fmt, args...)	{ }
#endif

/* ------------------ function declarations -------------------- */

/* functions for reading/writing values, called from generated code */
void erla_init(void);
void erla_read_octetstr_firmwareversion(char* octetstr, size_t *var_len);
void erla_read_octetstr_serialnumber(char* octetstr, size_t *var_len);
void erla_read_octetstr_ip(char* octetstr, size_t *var_len);
void erla_read_octetstr_netmask(char* octetstr, size_t *var_len);
void erla_read_octetstr_gateway(char* octetstr, size_t *var_len);
void erla_read_octetstr_mac(char* octetstr, size_t *var_len);
void erla_read_long_checkhostpower(long* longret);
int  erla_write_commit_resetboard(void);
int  erla_write_commit_hostreset(void);
int  erla_write_commit_hostpower(int long_press);
void erla_read_long_hardwarerev(long* longret);

void
erla_init(void)
{
    static int initialized = 0;
    char *kbd, *ptr, *kme, *ttyspeed;
    int err, dd, ttys;

    if (!initialized) {
	if (pp_base_init("snmpd", LOG_NOT_SILENT) != 0) {
	    pp_log("Initializing base-library failed.\n");
	    exit(1);
	}
    
	if (eric_config_init(FLUSH_IN_FOREGROUND) != 0) {
	    pp_log("Initializing config-library failed.\n");
	    exit(1);
	}
     
        if (pp_cfg_init(PP_CD_FNAME_DEFAULT, PP_CD_OEM_SKEL_FNAME_DEFAULT) != 0) {
            pp_log("Initializing cfg-library failed.\n");
            exit(1);
        }
    
	if (pp_um_init() != 0) {
	    pp_log("Initializing um-library failed.\n");
	    exit(1);
	}
    
	if (eric_misc_init() != 0) {
	    pp_log("Initializing misc-library failed.\n");
	    exit(1);
	}

	if (pp_firmware_init() != 0) {
            pp_log("Initializing firmware-library failed.\n");
	    exit(-1);
	}

#if defined(PP_FEAT_POWERSWITCH)	
	if (pp_power_switch_init() != 0) {
	    pp_log("Initializing power switch-library failed.\n");
	    exit(1);
	}
#endif	

	if (eric_notify_init(NOTIFY_INIT_WITH_HOST_TRAPS) != 0) {
	    pp_log("Initializing notify-library failed.\n");
	    exit(1);
	}

#ifdef PP_FEAT_IPMI_SERVER
	if (pp_ipmi_init() != 0) {
	    pp_log("Initializing libpp_ipmi failed.\n");
	    exit(1);
	}
#endif

	initialized = 1;
    }
}

#ifdef PP_FEAT_IPMI_SERVER
/* backend for the power control */
int ipmi_chassis_power_control(int command) {
    pp_ipmi_return_t ipmi_ret;
    pp_ipmi_parameter_t ipmi_param;
    int ret, err;

    memset(&ipmi_ret, 0, sizeof(ipmi_ret));
    memset(&ipmi_param, 0, sizeof(ipmi_param));
    ipmi_param.data.chassis_power = command;

    ret = pp_ipmi_send_command_pipe(PP_IPMI_CMD_CHASSIS, PP_IPMI_CHASSIS_SUBCMD_POWER_CONTROL, &ipmi_param, &ipmi_ret, &err, NULL, PP_BMC_LOOPI_SNMPD);
    if (ret) {
        pp_log("Error in sending Power Control command (%s).\n", pp_ipmi_get_error_string(err));
    }

    pp_ipmi_cleanup_ret(&ipmi_ret);
    return ret;
}

int ipmi_chassis_power_status() {
    pp_ipmi_return_t ipmi_ret;
    pp_ipmi_parameter_t ipmi_param;
    int ret, err;
    int val = 3;  // error/unknown

    memset(&ipmi_ret, 0, sizeof(ipmi_ret));
    memset(&ipmi_param, 0, sizeof(ipmi_param));

    ret = pp_ipmi_send_command_pipe(PP_IPMI_CMD_CHASSIS, PP_IPMI_CHASSIS_SUBCMD_POWER_STATUS, &ipmi_param, &ipmi_ret, &err, NULL, PP_BMC_LOOPI_SNMPD);
    if (ret) {
        pp_log("Error in sending Power Control command (%s).\n", pp_ipmi_get_error_string(err));
    } else {
        if (ipmi_ret.data.chassis_power_status.status == PP_IPMI_CHASSIS_POWER_STATUS_ON) {
            val = 1;  // hasPower
        } else if (ipmi_ret.data.chassis_power_status.status == PP_IPMI_CHASSIS_POWER_STATUS_OFF) {
            val = 2;  // hasNoPower
        }
    }

    pp_ipmi_cleanup_ret(&ipmi_ret);
    return val;
}
#endif

void
erla_read_octetstr_firmwareversion(char* octetstr, size_t *var_len)
{
    if (pp_firmware_erla_version[0] != '\0') {
	snprintf(octetstr, SPRINT_MAX_LEN, "%c%c.%c%c.%c%c",
		 pp_firmware_erla_version[0], pp_firmware_erla_version[1],
		 pp_firmware_erla_version[2], pp_firmware_erla_version[3],
		 pp_firmware_erla_version[4], pp_firmware_erla_version[5]);
    } else {
	snprintf(octetstr, SPRINT_MAX_LEN, "generic error");
    }
    *var_len = strlen(octetstr);
}

void
erla_read_octetstr_serialnumber(char* octetstr, size_t *var_len)
{
    char *option;

    if (PP_SUCCED(pp_cfg_get_nodflt(&option, "serial"))) {
	snprintf(octetstr, SPRINT_MAX_LEN, "%s", option);
	free(option);
    } else {
	snprintf(octetstr, SPRINT_MAX_LEN, "n/a");
    }
    *var_len = strlen(octetstr);
}  

void
erla_read_octetstr_ip(char* octetstr, size_t *var_len)
{
    struct in_addr inp;
    char *option;

    if (PP_SUCCED(pp_cfg_get_nodflt(&option, "network.ipaddr"))) {
	inet_aton(option, &inp);
	memcpy(octetstr, &inp.s_addr, sizeof(inp.s_addr));
	free(option);
    } else {
	inet_aton("0.0.0.0", &inp);
	memcpy(octetstr, &inp.s_addr, sizeof(inp.s_addr));
    }
    *var_len = 4;
}

void
erla_read_octetstr_netmask(char* octetstr, size_t *var_len)
{
    struct in_addr inp;
    char *option;

    if (PP_SUCCED(pp_cfg_get_nodflt(&option, "network.netmask"))) {
	inet_aton(option, &inp);
	memcpy(octetstr, &inp.s_addr, sizeof(inp.s_addr));
	free(option);
    } else {
	inet_aton("0.0.0.0", &inp);
	memcpy(octetstr, &inp.s_addr, sizeof(inp.s_addr));
    }
    *var_len = 4;
}

void
erla_read_octetstr_gateway(char* octetstr, size_t *var_len)
{
    struct in_addr inp;
    char *option;    

    if (PP_SUCCED(pp_cfg_get_nodflt(&option, "network.gateway"))) {
	inet_aton(option, &inp);
	memcpy(octetstr, &inp.s_addr, sizeof(inp.s_addr));
	free(option);
    } else {
	inet_aton("0.0.0.0", &inp);
	memcpy(octetstr, &inp.s_addr, sizeof(inp.s_addr));
    }
    *var_len = 4;
}

void
erla_read_octetstr_mac(char* octetstr, size_t *var_len)
{
    char * if_name;

    pp_cfg_get(&if_name, "network.eth_interface");
    if (pp_get_mac(if_name, octetstr) == -1) {
	memset(octetstr, 0, 6);
    }
    free(if_name);
    *var_len = 6;
}

void
erla_read_long_checkhostpower(long* longret)
{
#ifdef PP_FEAT_IPMI_SERVER
    *longret = ipmi_chassis_power_status();
#else
    int hostPower;

    if (eric_misc_get_host_power(&hostPower) != -1) {
	*longret = hostPower ? 1 : 2;
    } else {
	*longret = 3;
    }
#endif
}

int
erla_write_commit_resetboard()
{
    eric_misc_trigger_board_reset(0);
    return SNMP_ERR_NOERROR;
}

int
erla_write_commit_hostreset()
{
    int ret = SNMP_ERR_NOERROR;

#ifdef PP_FEAT_IPMI_SERVER
    if (ipmi_chassis_power_control(PP_IPMI_CHASSIS_RESET)) {
        ret = SNMP_ERR_GENERR;
    }
#else
    if (PP_FAILED(pp_power_login(PP_POWER_PORT_ID_INTERNAL))
        || PP_FAILED(pp_power_reset(0, 0, PP_POWER_PORT_ID_INTERNAL)))
    {
       ret = SNMP_ERR_GENERR;
    }
    pp_power_logout(PP_POWER_PORT_ID_INTERNAL);
#endif

    return ret;
}

int
erla_write_commit_hostpower(int long_press)
{
    int ret = SNMP_ERR_NOERROR;

#ifdef PP_FEAT_IPMI_SERVER
    if (ipmi_chassis_power_status() == 1) {
        ipmi_chassis_power_control(PP_IPMI_CHASSIS_POWER_DOWN);
    } else {
        ipmi_chassis_power_control(PP_IPMI_CHASSIS_POWER_UP);
    }
#else
    if (PP_FAILED(pp_power_login(PP_POWER_PORT_ID_INTERNAL))
	|| PP_FAILED(pp_power_switch(0, 0, long_press, 0, PP_POWER_PORT_ID_INTERNAL))) {
	ret = SNMP_ERR_GENERR;
    }
    pp_power_logout(PP_POWER_PORT_ID_INTERNAL);
#endif

    return SNMP_ERR_NOERROR;
}

void
erla_read_long_hardwarerev(long* longret)
{
    unsigned char u8_ret;

    ioctl(eric_fd, ERICIOCGETHARDWAREREVISION, &u8_ret);
    *longret = u8_ret;
}
