/**
 * file bmc.c
 *
 * Description: Global functions of BMC (IPMI server)
 *
 * (c) 2005 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 * 
 * TODO: make loopi,serial work on i386
 */
#include <pp/cfg.h>

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>
#include <pp/base.h>
#include <pp/selector.h>
#include <pp/bmc/ipmi_chan.h>
#include <pp/bmc/bmc.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/bmc_core.h>
#include <pp/bmc/bmc_config.h>
#include <pp/bmc/lan_serv.h>
#include <pp/bmc/loopi_adapter.h>
#include <pp/bmc/host_sensors.h>
#include "bmc_dev_oem_pp_rpc.h"

#if defined(PP_FEAT_IPMI_SERVER_SER_CHAN)
#include <pp/bmc/serial_adapter.h>
#endif

#if defined(PP_FEAT_IPMI_SERVER_SCSI_CHAN)
#include <pp/bmc/scsi_ipmi.h>
#endif

#if defined(PP_FEAT_IPMI_SERVER_PRIMARY_IPMB_CHAN) || defined(PP_FEAT_IPMI_SERVER_SECONDARY_IPMB_CHAN)
#include <pp/bmc/ipmb_adapter.h>
#endif

#if defined (PP_FEAT_IPMI_SERVER_SMI_CHAN)
#include <pp/bmc/smi_adapter.h>
#endif

#include <pp/bmc/bmc_health.h>
#include <pp/bmc/host_hardware.h>
#include <pp/bmc/host_board_id.h>

#include "bmc_debug.h"

/* callbacks for sensor consumers */
#include "bmc_dev_sensor.h"
#include <pp/bmc/host_sensors.h>


static pthread_t bmc_thread;

static void* bmc_thread_func(void* ctx UNUSED) {
    pp_select_loop();
    return NULL;
}

static int bmc_term_sf(void* ctx UNUSED) {
    pp_select_interrupt();
    return 0;
}

static int bmc_post_init(void)
{
#if defined (PP_FEAT_IPMI_THRESHOLD_PERSIST)
    pp_bmc_dev_sensor_init_thresholds();
#endif

#ifdef PRODUCT_PDU
    pp_bmc_dev_oem_pp_rpc_recover_pwr_state();   
#endif

#if !defined(PP_BMC_LOCAL_TEST) && defined(PP_FEAT_IPMI_SERVER_SMI_CHAN) && !defined(PP_FEAT_BMC_OEMCMDS_ONLY)
    pp_bmc_smi_mode_t smi_mode = pp_bmc_smi_get_mode();
    return pp_bmc_hardware_setup_interface_id(smi_mode);
#else /* PP_BMC_LOCAL_TEST */
    return PP_SUC;
#endif /* PP_BMC_LOCAL_TEST */
}

/*
 * initialize all channel adapters.
 *
 * they will implicitly register with the msg-router of the bmc
 *
 * also, they will implicitely register with the one and only
 * selector provided by libpp_base
 */
static int bmc_channel_init(void) {

#if !defined(PP_BMC_LOCAL_TEST)

# if defined(PP_FEAT_IPMI_SERVER_SMI_CHAN)
    if (PP_ERR == pp_bmc_smi_init()) {
        BMC_HEALTH_REPORT_FAULT(BMC_HEALTH_SI_CHAN);
        pp_bmc_log_pwarn("failed to init System Interface channel adapter");
    }
# endif /* PP_FEAT_IPMI_SERVER_SMI_CHAN */

    if (PP_ERR == pp_bmc_loopi_init(PP_BMC_LOOPI_ERIC)) {
        BMC_HEALTH_REPORT_FAULT(BMC_HEALTH_LOOPI_CHAN);
        pp_bmc_log_pwarn("failed to init LOOP channel adapter");
    }

    if (PP_ERR == pp_bmc_loopi_init(PP_BMC_LOOPI_SNMPD)) {
        BMC_HEALTH_REPORT_FAULT(BMC_HEALTH_LOOPI_CHAN);
        pp_bmc_log_pwarn("failed to init PP_BMC_LOOPI_SNMPD channel adapter");
    }

#if defined(PP_FEAT_BAYTECH_2_IPMI)
    if (PP_ERR == pp_bmc_loopi_init(PP_BMC_LOOPI_BAYTECH)) {
        BMC_HEALTH_REPORT_FAULT(BMC_HEALTH_LOOPI_CHAN);
       pp_bmc_log_pwarn("failed to init PP_BMC_LOOPI_BAYTECH channel adapter");
    }
#endif

#if defined(PP_FEAT_IPMI_SERVER_PRIMARY_IPMB_CHAN)
    if (PP_ERR == pp_bmc_ipmb_init(IPMI_CHAN_PRIMARY_IPMB)) {
        BMC_HEALTH_REPORT_FAULT(BMC_HEALTH_PRIM_IPMB_CHAN);
        pp_bmc_log_pwarn("failed to init PRIMARY_IPMB channel adapter");
    }
#endif

#if defined(PP_FEAT_IPMI_SERVER_SECONDARY_IPMB_CHAN)
    if (PP_ERR == pp_bmc_ipmb_init(IPMI_CHAN_SECDARY_IPMB)) {
        BMC_HEALTH_REPORT_FAULT(BMC_HEALTH_SECO_IPMB_CHAN);
        pp_bmc_log_pwarn("failed to init SECONDARY_IPMB channel adapter");
    }
#endif

#if defined(PP_FEAT_IPMI_SERVER_SER_CHAN)
    /* initialize serial before lanserv                   *
     * as SOL will need some lowlevel serial services     */
    if (PP_ERR == pp_bmc_serial_init()) {
        BMC_HEALTH_REPORT_FAULT(BMC_HEALTH_SERIAL_CHAN);
        pp_bmc_log_pwarn("failed to init Serial channel adapter");
    }
#endif /* PP_FEAT_IPMI_SERVER_SER_CHAN */

#if defined(PP_FEAT_IPMI_SERVER_SCSI_CHAN)
    if (PP_ERR == pp_bmc_scsi_ipmi_init()) {
	BMC_HEALTH_REPORT_FAULT(BMC_HEALTH_SCSI_CHAN);
	pp_bmc_log_pwarn("failed to init IPMI-over-SCSI channel adapter");
    }
#endif /* PP_FEAT_IPMI_SERVER_SCSI_CHAN */
#endif /* !PP_BMC_LOCAL_TEST */

    /* sol_init within lanserv_init will check whether   *
     * serial init went OK and skip SOL if neccessary    */
    if (PP_ERR == pp_bmc_lanserv_init()) {
        BMC_HEALTH_REPORT_FAULT(BMC_HEALTH_LAN_CHAN);
        pp_bmc_log_pwarn("failed to init LAN channel adapter");
    }

    /* we should eventually return PP_ERR if we have no working channels */
    return PP_SUC;
}

#if !defined (PP_FEAT_BMC_OEMCMDS_ONLY)
/* Attention: don't call this function while sensor scanner is running */
int pp_bmc_sensor_topo_parse_new(void) {
    pp_bmc_host_board_id_t* bid;
    char *id_str = NULL, *topofile;
    pp_strstream_t topofile_stream = PP_STRSTREAM_INITIALIZER;
    int ret = PP_ERR;

    /* try to determine topology file name */
    bid = pp_bmc_host_board_id_get();
    if (bid->id_type != HOST_BOARD_ID_TYPE_UNDEFINED) {
	id_str = pp_bmc_host_board_id_to_filename(bid);
    } else {
	id_str = pp_bmc_host_board_id_initial_filename();
	if (id_str == NULL) {
	    pp_bmc_log_error("[BMCcore] can't determine platform id (yet)");
	    goto bailout;
	}
	pp_bmc_log_info("[BMCcore] using initial topology '%s'", id_str);
    }

#if defined (PP_BMC_LOCAL_TEST)
    pp_strappendf(&topofile_stream, "./%s", id_str);
#else
    pp_strappendf(&topofile_stream, "/etc/bmc/%s", id_str);
#endif
    topofile = pp_strstream_buf(&topofile_stream);

    if (PP_ERR == pp_bmc_sensor_topology_parse(topofile,
					       pp_bmc_dev_sensor_add_object, 
					       pp_bmc_host_sensor_create,
					       NULL)) {
	pp_bmc_log_perror("[BMCcore] failed to parse sensor topology '%s'",
			  topofile);
	goto bailout;
    }
    ret = PP_SUC;

 bailout:
    free(id_str);
    pp_strstream_free(&topofile_stream);
    return ret;
}
#endif

int pp_bmc_init(void)
{
    pp_bmc_log_info("PP-BMC v-0.0.1 starting up");

    /* initialize selector */
    if (pp_select_init() == PP_ERR) {
	pp_bmc_log_pfatal("[BMCcore] pp_select_init failed");
	return PP_ERR;
    }

#if !defined (PP_FEAT_BMC_OEMCMDS_ONLY)
    /* initialize the hardware lib */
    if (pp_bmc_hardware_init() == PP_ERR) {
        pp_bmc_log_warn("[BMCcore] failed to init hardware");
        /* is this fatal ? */
    }
#endif

    /* do some sel debug initializations, remove for final version */
#ifndef PP_BUILD_TYPE_FINAL
    bmc_sel_debug_I();
#endif
        
    /* init core, including all devices */
    if (pp_bmc_core_init() == PP_ERR) {
        pp_bmc_log_pfatal("[BMCcore] failed to init BMC core");
        return PP_ERR;
    }

#if !defined (PP_FEAT_BMC_OEMCMDS_ONLY)
    /* initialize sensor lib */
    if (pp_bmc_sensor_topology_init()) {
	pp_bmc_log_pwarn("[BMCcore] failed to init sensor topology");
	return PP_ERR;
    }

    /* try to parse topo in case we already know our board */
    if (pp_bmc_sensor_topo_parse_new() == PP_ERR) {
	pp_bmc_log_warn("[BMCcore] sensor initialization skipped");	
    }
#endif   
   
    /* do some sel debug initializations, remove for final version */
#ifndef PP_BUILD_TYPE_FINAL
    bmc_sel_debug_II();
#endif
    
    /* now we are ready to serve requests - initialize channels */
    if (bmc_channel_init() == PP_ERR) {
        pp_bmc_log_fatal("[BMCcore] channel initialization failed");
        return PP_ERR;
    }

    /* if any specific initialisation left todo, now is the time */
    if (bmc_post_init() == PP_ERR) {
	pp_bmc_log_fatal("[BMCcore] post initialization failed");
	return PP_ERR;
    }
    
    return PP_SUC;
}

void pp_bmc_cleanup(void)
{
#if !defined (PP_FEAT_BMC_OEMCMDS_ONLY)    
    /* stop sensor scanning */
    pp_bmc_sensor_scanner_stop();

    /* cleanup channels */
    pp_bmc_lanserv_cleanup();
#endif    
#if !defined(PP_BMC_LOCAL_TEST)
#if defined(PP_FEAT_IPMI_SERVER_SER_CHAN)
    pp_bmc_serial_cleanup();
#endif /* PP_FEAT_IPMI_SERVER_SER_CHAN */
#if defined(PP_FEAT_IPMI_SERVER_SECONDARY_IPMB_CHAN)
    pp_bmc_ipmb_cleanup(IPMI_CHAN_SECDARY_IPMB);
#endif
#if defined(PP_FEAT_IPMI_SERVER_PRIMARY_IPMB_CHAN)
    pp_bmc_ipmb_cleanup(IPMI_CHAN_PRIMARY_IPMB);
#endif
#if defined(PP_FEAT_BAYTECH_2_IPMI)
    pp_bmc_loopi_cleanup(PP_BMC_LOOPI_BAYTECH);
#endif
    pp_bmc_loopi_cleanup(PP_BMC_LOOPI_SNMPD);
    pp_bmc_loopi_cleanup(PP_BMC_LOOPI_ERIC);
#if defined(PP_FEAT_IPMI_SERVER_SMI_CHAN)
    pp_bmc_smi_cleanup();
#endif /* PP_FEAT_IPMI_SERVER_SMI_CHAN */
#endif /* !PP_BMC_LOCAL_TEST */
    
    /* cleanup the core */
    pp_bmc_core_cleanup();

#if !defined (PP_FEAT_BMC_OEMCMDS_ONLY)
    pp_bmc_hardware_cleanup();
    
    /* clean up the topology, all objects should be deleted by now */
    pp_bmc_sensor_topology_cleanup();
#endif

    pp_select_cleanup();
    
    pp_bmc_log_info("PP-BMC terminated normally");
}

int pp_bmc_start(void)
{
    /* start main tread */
    if (pthread_create(&bmc_thread, NULL, bmc_thread_func, 0) < 0)
	return PP_ERR;

#if !defined (PP_FEAT_BMC_OEMCMDS_ONLY)
    /* start sensor scanner */
    if (PP_FAILED(pp_bmc_sensor_scanner_start())) {
	pp_bmc_log_pwarn("[BMCcore] failed to initiate sensor scanner");
    } else {
	pp_bmc_log_info("[BMCcore] sensor scanning initiated");
    }
#endif

    return PP_SUC;
}

int pp_bmc_stop(void)
{
#if !defined (PP_FEAT_BMC_OEMCMDS_ONLY)
    /* stop sensor scanner */
    pp_bmc_sensor_scanner_stop();
#endif

    /* stop main thread */
    pp_select_add_sf(bmc_term_sf, NULL);

    /* wait for thread completion */
    if (pthread_join(bmc_thread, NULL) < 0) return PP_ERR;
    
    return PP_SUC;
}

#if defined(PP_FEAT_IPMI_SERVER_FKT_INJECT)
/*
 * ATTENTION: tbr: this function is funny, I do confess, but useful.However it
 * may not really comply to BMC architectural design and it will break once BMC
 * runs in a different process context compared to the function calling this.
 * So use with care and only if you really don't want to go through
 * libpp_ipmi ;-)
 */
typedef enum bmc_run_fkt_state_e {
    RF_FREE,
    RF_BUSSY,
    RF_LOST_PATIENCE,
    RF_DONE,
} bmc_run_fkt_state_t;

typedef struct bmc_run_fkt_desc_s {
    bmc_run_fkt_state_t state;
    pthread_mutex_t mtx;
    pthread_cond_t cond;
    pp_select_sf_hndl_t fkt;
    void* ctx;
} bmc_run_fkt_desc_t;

static bmc_run_fkt_desc_t rf_desc = {
    .state = RF_FREE,
    .mtx = PTHREAD_ERRORCHECK_MUTEX_INITIALIZER_NP,
    .cond = PTHREAD_COND_INITIALIZER,
};

static int bmc_run_sf(void* ctx) {
    bmc_run_fkt_desc_t* fdesc = (bmc_run_fkt_desc_t*)ctx;
    fdesc->fkt(fdesc->ctx);
    MUTEX_LOCK(&fdesc->mtx);
    assert(fdesc->state == RF_BUSSY || fdesc->state == RF_LOST_PATIENCE);
    if (fdesc->state == RF_LOST_PATIENCE) {
	fdesc->state = RF_FREE;
    } else {
	fdesc->state = RF_DONE;
    }
    MUTEX_UNLOCK(&fdesc->mtx);
    pthread_cond_broadcast(&fdesc->cond);
    return PP_SUC;
}

int pp_bmc_run_fkt(pp_select_sf_hndl_t fkt, void* ctx) {
    struct timespec to;
    int err, ret = PP_ERR;

    /* aquire bmc_run_fkt slot, may timeout */
    MUTEX_LOCK(&rf_desc.mtx);
    while (rf_desc.state != RF_FREE) {
	to = pp_get_wakeup_time(200/*ms*/);
	if (0 != (err = pthread_cond_timedwait(&rf_desc.cond, &rf_desc.mtx,
					       &to))){
	    errno = err;
	    goto bailout;
	}
    }

    /* got it, mac, fill rf_desc structure with data.. */
    rf_desc.fkt = fkt;
    rf_desc.ctx = ctx;

    /* ..and schedule it */
    rf_desc.state = RF_BUSSY;
    pp_select_add_sf(bmc_run_sf, &rf_desc);

    /* wait for completion, we may loose patience */
    while (rf_desc.state != RF_DONE) {
	to = pp_get_wakeup_time(200/*ms*/);
	if (0 != (err = pthread_cond_timedwait(&rf_desc.cond, &rf_desc.mtx,
					       &to))){
	    rf_desc.state = RF_LOST_PATIENCE;
	    errno = err;
	    goto bailout;
	}
    }
    rf_desc.state = RF_FREE;
    ret = PP_SUC;
 bailout:
    MUTEX_UNLOCK(&rf_desc.mtx);
    return ret;
}
#endif /* PP_FEAT_IPMI_SERVER_FKT_INJECT */
