/**
 * file bmc_core.c
 *
 * Description: BMC Core
 *
 * (c) 2004 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 */

#include <pp/base.h>
#include <pp/dlist.h>
#include <pp/rb_tree.h>
#include <pp/atomic_write.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/bmc/ipmi_chan.h>
#include <pp/bmc/ipmi_cmd.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/ipmi_sess.h>
#include <pp/bmc/bmc_core.h>
#include <pp/bmc/bmc_router.h>
#include <pp/bmc/bmc_config.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/utils.h>
#include <pp/bmc/user_manager.h>
#include <pp/bmc/session_manager.h>
#include <pp/bmc/bmc_health.h>

#include "bmc_dev_app.h"
#include "bmc_dev_sel.h"
#include "bmc_dev_fru.h"
#include "bmc_dev_sdrr.h"
#include "bmc_dev_sensor.h"
#include "bmc_dev_kvm.h"
#include "bmc_msg_tracker.h"
#include "bmc_dev_app_watchdog.h"
#include "bmc_dev_pef.h"
#include "bmc_dev_app_session.h"
#include "bmc_dev_chassis.h"
#include "bmc_dev_event.h"
#include "event_receiver.h"
#include "receive_message_queue.h"
#include "bmc_dev_event_msg_buff.h"
#include "bmc_dev_oem_pp.h"
#include "bmc_dev_oem_pp_rpc.h"
#include "bmc_dev_oem_pp_selftest.h"
#include "bmc_dev_oem_supermicro.h"

#ifdef LARA_KIMMSI
#include "bmc_dev_oem_msi.h"
#endif /* LARA_KIMMSI */

#ifdef PP_FEAT_OPMA_HW
#include "bmc_dev_oem_amd.h"
#endif /* PP_FEAT_OPMA_HW */

#ifdef PRODUCT_ICPMMD
#include "bmc_dev_oem_icp.h"
#endif /* PRODUCT_ICPMMD */

/*
 *  Static Data
 */
static void tree_del_ptr(pp_mallocator_t *al UNUSED, void *p) { free(p); }

typedef struct {
    int (*cmd_hdlr)(imsg_t*);
    unsigned int min_data_size;
    unsigned char min_priv_level;
} cmd_recvr_entry_t;

#define MAKE_KEY(netfn, cmd) \
    ((void*)(long)((((unsigned int)netfn) << 8) | (cmd)))
static pp_rb_tree_t *cmd_recvr_tree = NULL;

/* SDR of BMC, will be added to SDRR during init */
unsigned char bmc_sdr[] = {
    0,        // id lsb
    0,        // ID msb
    0x51,     // version
    0x12,     // sdrr type
    19,       // length
    0x20,     // bmc address
    0,        // channel number      
    0,
    0x8E,     // device capabilities
    0, 0, 0,  // reserved
    0, 0, 0,  // entity / instance / oem
    0x07,     // length of id string
    'K', 'I', 'M', ' ', 'B',  'M', 'C', '\0' // id string "KIM BMC"
}; 
    
/*
 * Implementation
 */
static int bmc_redirect_lun(int lun, imsg_t* imsg)
{
    /* LUN 10b: redirect to SI */
    if (lun == 2) {
#if defined (PP_FEAT_IPMI_SERVER_SMI_CHAN)
        if (bmc_receive_queue_add(imsg) == PP_ERR) {
            /* queue full */
            /* unclear if this is correct, eventually 0xD3 is better */
            return pp_bmc_router_resp_err(imsg, IPMI_ERR_OUT_OF_SPACE);
        }
        return PP_SUC;
#else
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_DEST_UNAVAILABLE);
#endif
    }
    
    /* other LUNs are not supported */
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_COMMAND_INVALID_FOR_THIS_LUN);
}

#undef CMD_WATCHDOG
#ifdef CMD_WATCHDOG
const char* cmd_wd_stage;
static imsg_t* cmd_wd_imsg;
static time_t cmd_wd_time;
static pthread_t cmd_wd_thread;

static void* cmd_wd_thread_func(void* ctx UNUSED) {
    while (cmd_wd_time != -1) {
        sleep(3);
        if (cmd_wd_time > 0) {
            time_t now = time(NULL);
            if (now > cmd_wd_time  + 10) {
                pp_bmc_log_error("[BMCcore] BMC not responing since %d secs while processing cmd "
                                "(netfn/cmd 0x%02x/0x%02x, chan %d) in stage '%s'",
                                now - cmd_wd_time, cmd_wd_imsg->netfn,
                                cmd_wd_imsg->cmd, cmd_wd_imsg->chan, cmd_wd_stage);
            }
        }
    }
    return NULL;
}
#endif

/* Check whether the user is allowed to send this command through this channel */
static int msg_type_enabled(imsg_t* imsg)
{
#if defined(PP_FEAT_BMC_OEMCMDS_ONLY)
    (void)imsg;
    //in the OEM-commands-only case we don't want to have a sophisticated 
    //user management, so just return true
    return 1;
#else /* !PP_FEAT_BMC_OEMCMDS_ONLY */
    if (!imsg->session || !imsg->session->uid || !imsg->session->chan) {
	return 1;
    }

    /* these commands are always available (see IPMIv2 24.6) */
    if (imsg->netfn == IPMI_NETFN_APP) {
	if ( imsg->cmd == IPMI_CMD_SET_SESSION_PRIVILEGE
	  || imsg->cmd == IPMI_CMD_CLOSE_SESSION
	  || imsg->cmd == IPMI_CMD_DEACTIVATE_PAYLOAD
	  || imsg->cmd == IPMI_CMD_GET_PAYLOAD_ACTIVATION_STATUS
	  || imsg->cmd == IPMI_CMD_GET_CHANNEL_PAYLOAD_VERSION
	  || imsg->cmd == IPMI_CMD_GET_CHANNEL_OEM_PAYLOAD_INFO
	  || imsg->cmd == IPMI_CMD_SUSPEND_RESUME_PAYLOAD_ENCRYPTION ) {
	    return 1;
	}
    }

    /* IPMI messaging enabled? */
    if (pp_bmc_user_msg_enabled(imsg->session->uid, imsg->session->chan)) {
	return 1;
    }

    /* SOL-related commands; SOL enabled?*/
    if (imsg->netfn == IPMI_NETFN_KVM) {
	if ( imsg->cmd == IPMI_CMD_SOL_ACTIVATING
	  || imsg->cmd == IPMI_CMD_SOL_SET_SOL_CFG
	  || imsg->cmd == IPMI_CMD_SOL_GET_SOL_CFG ) {
	    return pp_bmc_user_may_use_sol(imsg->session->uid);
	}
    }

    return 0;
#endif /* !PP_FEAT_BMC_OEMCMDS_ONLY */
}

static int bmc_dispatch_cmd(imsg_t* imsg)
{
    const cmd_recvr_entry_t *entry;
    assert(cmd_recvr_tree);
        
    /* find entry */
    entry = pp_rb_tree_search(cmd_recvr_tree,
				      MAKE_KEY(imsg->netfn, imsg->cmd));
    if (entry) {
        int ret;

	if (!msg_type_enabled(imsg))
	    return pp_bmc_router_resp_err(imsg,
				IPMI_ERR_INSUFFICIENT_PRIVILEGE_LEVEL);

        /* check privilege level */
        if ((entry->min_priv_level != IPMI_PRIV_UNSPEC) &&
	    (imsg->priv_level < entry->min_priv_level))
            return pp_bmc_router_resp_err(imsg,
				IPMI_ERR_INSUFFICIENT_PRIVILEGE_LEVEL);

        /* check body size */
        if (imsg->data_size < entry->min_data_size)
	    // ICTS 6.01 wants to see a return code C7 "invalid" here
	    // instead of 0xC6 "truncated" for test (SMS)->System
	    // Interface Support -> get channel cipher suites.
            return pp_bmc_router_resp_err(imsg, IPMI_ERR_REQ_DATA_LEN_INVALID);

        /* dispatch cmd */
//        return entry->cmd_hdlr(imsg);
#ifdef CMD_WATCHDOG
        cmd_wd_imsg = imsg;
        cmd_wd_stage = "";
        cmd_wd_time = time(NULL);
#endif
        ret = entry->cmd_hdlr(imsg);
#ifdef CMD_WATCHDOG
        cmd_wd_time = 0;
#endif
        return ret;
    }

    pp_bmc_log_debug("[BMCcore] got unknown IPMI cmd "
		     "(netfn/cmd 0x%02x/0x%02x, chan %d)",
		     imsg->netfn, imsg->cmd, imsg->chan);

    // not handled: send error response
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_CMD);
}

static int bmc_handle_resp(imsg_t* imsg)
{
#if !defined (PP_FEAT_BMC_OEMCMDS_ONLY)
    return pp_bmc_mtrack_handle(imsg);
#else /* PP_FEAT_BMC_OEMCMDS_ONLY */
    (void)imsg;
    return PP_SUC;
#endif /* PP_FEAT_BMC_OEMCMDS_ONLY */
}


/*
 *  Interface
 */

int pp_bmc_core_init() {
    
    pp_bmc_log_debug("[BMCcore] init");

#ifdef CMD_WATCHDOG
    cmd_wd_time = 0;
    pthread_create(&cmd_wd_thread, NULL, cmd_wd_thread_func, 0);
#endif
    
    /*
     * The initialization sequence here is not properly ordered any
     * more. Its necessary anyway to meet initialization dependencies.
     * I will try to clean this up as soon as possible - geo.
     */
    
    /* create flash dir if not already there */
    if (-1 == mkdir_recurse(PP_BMC_FLASH_ROOT, S_RWXRWXRWX)
    && errno != EEXIST) {
        pp_bmc_log_perror("[BMCcore] failed to access/create flash dir");
        return PP_ERR;
    }
    
    /* create cmd receiver tree  */
    cmd_recvr_tree = pp_rb_tree_int_new();

    /* init router */
    if (PP_ERR == pp_bmc_router_init()) return PP_ERR;

#if defined (PP_FEAT_IPMI_SERVER_SMI_CHAN) /* no smi-queue if no smi */
    bmc_receive_message_queue_init();
    bmc_dev_event_msg_buf_init();
#endif /* PP_FEAT_IPMI_SERVER_SMI_CHAN */

#if !defined (PP_FEAT_BMC_OEMCMDS_ONLY)
    /* init message tracker */
    pp_bmc_mtrack_init();
#endif
    
    /* init user and session manager */
    pp_bmc_user_manager_init();
    pp_bmc_session_manager_init();

    if (PP_ERR == pp_fa_at_init()) return PP_ERR;

#if !defined (PP_FEAT_BMC_OEMCMDS_ONLY)
    if (PP_ERR == pp_bmc_dev_sdrr_init())
        { BMC_HEALTH_REPORT_FAULT(BMC_HEALTH_SDRR_DEV); return PP_ERR; }
    if (PP_ERR == pp_bmc_dev_sensor_init()) 
        { BMC_HEALTH_REPORT_FAULT(BMC_HEALTH_SENSOR_DEV); return PP_ERR; }
#endif
    
    /* init devices */
    if (PP_ERR == pp_bmc_dev_app_init())
        { BMC_HEALTH_REPORT_FAULT(BMC_HEALTH_APP_DEV); return PP_ERR; }
#if !defined (PP_FEAT_BMC_OEMCMDS_ONLY)
    if (PP_ERR == pp_bmc_dev_sel_init())
        { BMC_HEALTH_REPORT_FAULT(BMC_HEALTH_SEL_ACCESS); return PP_ERR; }
    if (PP_ERR == pp_bmc_event_receiver_init())
        { BMC_HEALTH_REPORT_FAULT(BMC_HEALTH_SEL_ACCESS); return PP_ERR; }
    if (PP_ERR == pp_bmc_dev_pef_init())
        { BMC_HEALTH_REPORT_FAULT(BMC_HEALTH_PEF_DEV); return PP_ERR; }
    if (PP_ERR == pp_bmc_dev_fru_init())
        { BMC_HEALTH_REPORT_FAULT(BMC_HEALTH_FRU_ACCESS); return PP_ERR; }
#endif	
#if defined (PP_FEAT_IPMI_SERVER_WATCHDOG)
    if (PP_ERR == pp_bmc_dev_watchdog_init()) 
        { BMC_HEALTH_REPORT_FAULT(BMC_HEALTH_WATCHDOG_DEV); return PP_ERR; }
#endif
#if !defined (PP_FEAT_BMC_OEMCMDS_ONLY)
    if (PP_ERR == pp_bmc_dev_kvm_init())
        { BMC_HEALTH_REPORT_FAULT(BMC_HEALTH_KVM_DEV); return PP_ERR; }
#endif	
    if (PP_ERR == pp_bmc_dev_session_init())
        { BMC_HEALTH_REPORT_FAULT(BMC_HEALTH_SESSION_DEV); return PP_ERR; }
#if defined (PP_FEAT_IPMI_SERVER_CHASSIS_CMDS)
    if (PP_ERR == pp_bmc_dev_chassis_init())
        { BMC_HEALTH_REPORT_FAULT(BMC_HEALTH_CHASSIS_DEV); return PP_ERR; }
#endif
#if !defined (PP_FEAT_BMC_OEMCMDS_ONLY)
    if (PP_ERR == pp_bmc_dev_event_init())
        { BMC_HEALTH_REPORT_FAULT(BMC_HEALTH_EVENT_DEV); return PP_ERR; }
#endif	
    if (PP_ERR == pp_bmc_dev_oem_pp_init())
        { BMC_HEALTH_REPORT_FAULT(BMC_HEALTH_OEM_PP_DEV); return PP_ERR; }
    if (PP_ERR == pp_bmc_dev_oem_pp_selftest_init())
        { BMC_HEALTH_REPORT_FAULT(BMC_HEALTH_OEM_PP_SELFTEST_DEV); return PP_ERR; }
#ifdef LARA_KIMMSI
    if (PP_ERR == pp_bmc_dev_oem_msi_init())
        { BMC_HEALTH_REPORT_FAULT(BMC_HEALTH_OEM_MSI_DEV); return PP_ERR; }
#endif
#ifdef PP_FEAT_OPMA_HW
    if (PP_ERR == pp_bmc_dev_oem_amd_init())
        { BMC_HEALTH_REPORT_FAULT(BMC_HEALTH_OEM_AMD_DEV); return PP_ERR; }
#endif
#ifdef PRODUCT_ICPMMD
    if (PP_ERR == pp_bmc_dev_oem_icp_init())
        { BMC_HEALTH_REPORT_FAULT(BMC_HEALTH_OEM_ICP_DEV); return PP_ERR; }
#endif
#ifdef PRODUCT_PDU
    if (PP_ERR == pp_bmc_dev_oem_pp_rpc_init())
        { BMC_HEALTH_REPORT_FAULT(BMC_HEALTH_OEM_PP_RPC_DEV);return PP_ERR;}
#endif
#ifdef PRODUCT_SMIDC
    if (PP_ERR == pp_bmc_dev_oem_supermicro_init())
        { BMC_HEALTH_REPORT_FAULT(BMC_HEALTH_OEM_SUPERMICRO_DEV); return PP_ERR; }
#endif
#if !defined (PP_FEAT_BMC_OEMCMDS_ONLY)
    pp_bmc_sdrr_add_sdr_transiently((ipmi_sdr_header_t*)bmc_sdr);
#endif
    pp_bmc_log_info("[BMCcore] BMC started");
    return PP_SUC;
}

void pp_bmc_core_cleanup()
{
    /* shutdown */
#ifdef PRODUCT_SMIDC
    pp_bmc_dev_oem_supermicro_cleanup();
#endif
#ifdef PRODUCT_PDU
    pp_bmc_dev_oem_pp_rpc_cleanup();
#endif
#ifdef PRODUCT_ICPMMD
    pp_bmc_dev_oem_icp_cleanup();
#endif
#ifdef PP_FEAT_OPMA_HW
    pp_bmc_dev_oem_amd_cleanup();
#endif
#ifdef LARA_KIMMSI
    pp_bmc_dev_oem_msi_cleanup();
#endif
    pp_bmc_dev_oem_pp_selftest_cleanup();
    pp_bmc_dev_oem_pp_cleanup();
#if !defined (PP_FEAT_BMC_OEMCMDS_ONLY)
    pp_bmc_dev_event_cleanup();
#endif    
#if defined (PP_FEAT_IPMI_SERVER_CHASSIS_CMDS)
    pp_bmc_dev_chassis_cleanup();
#endif
    pp_bmc_dev_session_cleanup();
#if !defined (PP_FEAT_BMC_OEMCMDS_ONLY)
    pp_bmc_dev_kvm_cleanup();
#endif    
#if defined(PP_FEAT_IPMI_SERVER_WATCHDOG)
    pp_bmc_dev_watchdog_cleanup();
#endif
#if !defined (PP_FEAT_BMC_OEMCMDS_ONLY)
    pp_bmc_dev_sensor_cleanup();
    pp_bmc_dev_sdrr_cleanup();
    pp_bmc_dev_fru_cleanup();
    pp_bmc_dev_pef_cleanup();
    pp_bmc_event_receiver_cleanup();
    pp_bmc_dev_sel_cleanup();
    pp_bmc_dev_app_cleanup();
#endif    
    pp_fa_at_cleanup();

#if !defined (PP_FEAT_BMC_OEMCMDS_ONLY)
    pp_bmc_mtrack_cleanup();
#endif    
    pp_bmc_router_cleanup();

#if defined (PP_FEAT_IPMI_SERVER_SMI_CHAN)
    bmc_receive_message_queue_cleanup();
    bmc_dev_event_msg_buf_cleanup();
#endif

#if !defined (PP_FEAT_BMC_OEMCMDS_ONLY)
    pp_bmc_session_manager_cleanup();
    pp_bmc_user_manager_cleanup();
#endif

    pp_rb_tree_destroy(cmd_recvr_tree); cmd_recvr_tree = NULL;

#ifdef CMD_WATCHDOG
    cmd_wd_time = -1; // signal termination
    pthread_join(cmd_wd_thread, NULL);
#endif

    pp_bmc_log_info("[BMCcore] BMC shut down");
}

int pp_bmc_core_handle_msg(imsg_t* imsg)
{
    pp_bmc_imsg_dump(PP_BMC_LOG_DEBUG, "[BMCcore]", imsg);

    if ((imsg->session != NULL) & (imsg->priv_level > IPMI_PRIV_UNSPEC)) {
        /* this is a valid message, keep the session alive */
        pp_bmc_session_refresh(imsg->session);
    }

    if (IPMI_IS_RESPONSE(imsg->netfn)) {
        /* redirect responses to other LUNs */
        if (imsg->rq_lun != 0) return bmc_redirect_lun(imsg->rq_lun, imsg);
        
        /* response tracking */
        return bmc_handle_resp(imsg);
    } else {
        /* redirect requests to other LUNs */
        if (imsg->rs_lun != 0) return bmc_redirect_lun(imsg->rs_lun, imsg);
        
        /* real IPMI command */
        return bmc_dispatch_cmd(imsg);
    }
}

int pp_bmc_core_reg_cmd(unsigned char netfn, unsigned char cmd,
                        int min_data_size, int min_priv_level,
                        int (*cmd_hdlr)(imsg_t*))
{
    cmd_recvr_entry_t *entry;
    
    assert(cmd_recvr_tree); /* must be init */

    /* entry already exsits? */
    if (NULL != pp_rb_tree_search(cmd_recvr_tree, MAKE_KEY(netfn, cmd))) {
        pp_bmc_log_error("[BMCcore] can't register command twice (netfn 0x%02x/cmd 0x%02x)", netfn, cmd);
        return PP_ERR;
    }
    
    /* build entry */
    entry = malloc(sizeof(cmd_recvr_entry_t));
    entry->min_data_size  = min_data_size;
    entry->min_priv_level = min_priv_level;
    entry->cmd_hdlr       = cmd_hdlr;
    
    /* insert it */
    pp_rb_tree_insert(cmd_recvr_tree, MAKE_KEY(netfn, cmd), entry,
                      tree_del_ptr, 0);
    return PP_SUC;
}

int pp_bmc_core_unreg_cmd(unsigned char netfn, unsigned char cmd)
{
    void *entry;
    assert(cmd_recvr_tree); /* must be init */
    
    /* find entry */
    entry = pp_rb_tree_search(cmd_recvr_tree, MAKE_KEY(netfn, cmd));
        
    /* not found */
    if (!entry) {
        pp_bmc_log_error("[BMCcore] unregister command failed (netfn 0x%02x/cmd 0x%02x)", netfn, cmd);
        return PP_ERR;
    }

    /* use extra variable instead of (void**)(&entry) to avoid obscure compiler warnings */    
    void* entryp = (void*)entry;
    /* remove it */
    pp_rb_tree_remove(cmd_recvr_tree, MAKE_KEY(netfn, cmd), &entryp);

    free(entry);
    
    return PP_SUC;
}

int pp_bmc_core_reg_cmd_tab(const dev_cmd_entry_t* tab)
{
    for (; tab->cmd_hndlr; tab++)
        if (PP_ERR == pp_bmc_core_reg_cmd(tab->netfn, tab->cmd,
                                          tab->min_data_size, tab->min_priv_level,
                                          tab->cmd_hndlr))
            return PP_ERR;                           

    return PP_SUC;                                  
}

int pp_bmc_core_unreg_cmd_tab(const dev_cmd_entry_t* tab)
{
    for (; tab->cmd_hndlr; tab++)
        if (PP_ERR == pp_bmc_core_unreg_cmd(tab->netfn, tab->cmd))
            return PP_ERR;

    return PP_SUC;                                  
}
