/**
 * bmc_dev_app_watchdog.c
 *
 * BMC Watchdog device
 *
 * (c) 2004 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#include <malloc.h>
#include "pp/base.h"
#include "pp/dlist.h"
#include "pp/selector.h"

#include "pp/bmc/bmc_watchdog.h"
#include "pp/bmc/ipmi_cmd.h"
#include "pp/bmc/ipmi_err.h"
#include "pp/bmc/ipmi_msg.h"
#include "pp/bmc/ipmi_chan.h"
#include "pp/bmc/ipmi_sess.h"
#include "pp/bmc/ipmi_bits.h"
#include "pp/bmc/ipmi_sdr.h"
#include "pp/bmc/bmc_imsg.h"
#include "pp/bmc/bmc_core.h"
#include "pp/bmc/bmc_event.h"
#include "pp/bmc/bmc_router.h"
#include "pp/bmc/debug.h"
#include "pp/bmc/host_power.h"
#include "pp/bmc/host_sensors.h"

#include "bmc_dev_app_watchdog.h"
#include "bmc_dev_sdrr.h"
#include "nc_interrupt.h"

#define WD_ACTION_NONE            0     // b000
#define WD_ACTION_HARD_RESET      1     // b001;
#define WD_ACTION_POWER_DOWN      2     // b010;
#define WD_ACTION_POWER_CYCLE     3     // b011;

#define WD_INTTYPE_NONE           0     // b000;
#define WD_INTTYPE_SMI            1     // b001;
#define WD_INTTYPE_NMI            2     // b010;
#define WD_INTTYPE_MSG_INT        3     // b011;

#define WD_USE_UNDEFINED          0     // b000;       // this means 'reserved' in the spec, but it seems reasonable, somewhere we have to start
#define WD_USE_BIOS_FRB2          1     // b001;
#define WD_USE_BIOS_POST          2     // b010;
#define WD_USE_OS_LOAD            3     // b011;
#define WD_USE_SMS_OS             4     // b100;
#define WD_USE_OEM                5     // b101;

typedef struct watchdog_globals_t {
    int countdown;               // in 100ms,  countdown = 256*cd_msb + cd_lsb;
    int cd_msb; int cd_lsb;      // countdown most significant byte and least sig. byte
                                 // stored for convenience reasons, only valid if countdown > 0
    int pretimeout_interval;     // in 1s
    int pretimeout_reached;
    int running;                 // holds selector-timerhandle if the watchdog is running, -1 if wd is stopped
    int pre_running;             // holds selector-handle for pretimeout
    int interrupt_handle;        // holds selector-handle for smi or nmi timeout deactivation
    
    int action;                  // what to do when timer runs out
    int preIntType;              // the type of pretimeout-interrupt handling (none, SMI, NMI, ...)
    
    int timerUse;
    
    // timer use flags
    int BIOS_FRB2;
    int BIOS_POST;
    int OS_LOAD;
    int SMS_OS;
    int OEM;
    
    int noLog;                   // do not log the next timeout
} watchdog_globals_t;

static watchdog_globals_t watchdog_globals;
static watchdog_globals_t* wdg = &watchdog_globals;


static ipmi_sdr_eventonly_sensor_t watchdog_sdr = {
    header: {
        id_le16: 0,     /* reserved to be used by SDRR */
        version: 0x51,
        type:    0x03,
        length:  sizeof(ipmi_sdr_eventonly_sensor_t)-sizeof(ipmi_sdr_header_t)
    },
    key: {
        owner_id:   0x20,
        lun:        0,
        channel:    IPMI_CHAN_PRIMARY_IPMB,
        sensor_num: IPMI_SENSOR_NUMBER_STATIC_WATCHDOG
    },
    entity: {
        id:         IPMI_ENTITY_ID_MGMT_CONTROLLER_FIRMWARE,
        instance:   0x1,
        is_logical: 0
    },
    
    sensor_type: IPMI_SENSOR_TYPE_WATCHDOG_2, 
    event_type: IPMI_EVENT_READING_TYPE_SENSOR_SPECIFIC,
    
    share: {
        count:      0,
        mod_type:   0,  // not sure if this is correct
        sens_dir:   0,
        mod_offset: 0,
        entity_inst:0
    }, 
    
    oem: 0,
    
    id: {
        length: 16, 
        code:   IPMI_STR_ASCII8,
        string: "watchdog"
    }
};


/********************************************************************
 * Debug helper
 */
#if 0 
 static void dumpWatchdogGlobals() {
     printf("countdown: %d (msb=%d, lsb=%d); Handler = %d\n",
	    wdg->countdown, wdg->cd_msb, wdg->cd_lsb, wdg->running);
     printf("pretimeout: %d; Handler = %d\n",
	    wdg->pretimeout_interval, wdg->pre_running);
     printf("pretimeout_reached: %d\n", wdg->pretimeout_reached);
     
     printf("action when wd runs out:    %.2x\n", wdg->action);
     printf("action when pretimer is up: %.2x\n", wdg->preIntType);
     printf("actual timer use: %d\n", wdg->timerUse);
     
     printf("use-expiration flags \n");
     printf("BIOS_FRB2: %d\n", wdg->BIOS_FRB2);
     printf("BIOS_POST: %d\n", wdg->BIOS_POST);
     printf("OS_LOAD: %d\n", wdg->OS_LOAD);
     printf("SMS_OS: %d\n", wdg->SMS_OS);
     printf("OEM: %d\n", wdg->OEM);
     printf("don't log: %d\n", wdg->noLog);
 }
#endif

/********************************************************************
 * misc. helpers
 */

// interrupt id to deassert is in ctx. simple but crude ...
static int interrupt_timeout_handler(const int item_id UNUSED, void* ctx) {
    pp_tp_gpio_act_t* interrupt_actor;
    wdg->interrupt_handle = -1;

    // deassert interrupt
    interrupt_actor = pp_bmc_host_sensor_get_gpio_actor((int)ctx);
    if (interrupt_actor == NULL) {
        pp_bmc_log_error("[watchdog] interrupt actor disappeared");
        return PP_ERR;
    }
    
    interrupt_actor->set_gpio(interrupt_actor, 0);
    return PP_SUC;
}

static void interrupt_initiate(int actor) {
    pp_tp_gpio_act_t* interrupt_actor;
    interrupt_actor = pp_bmc_host_sensor_get_gpio_actor(actor);
    
    if (interrupt_actor == NULL) {
        pp_bmc_log_warn("[watchdog] interrupt actor not registered");
        return;
    }
    
    if (wdg->interrupt_handle == -1) {
        // initiate interrupt
        interrupt_actor->set_gpio(interrupt_actor, 1);
        wdg->interrupt_handle = pp_select_add_to(20, 0, interrupt_timeout_handler, (void*)actor);
    } else {
        // another interrupt (SMI/NMI) is in progress - very strange - ignore this one ...
        pp_bmc_log_warn("[watchdog] another interrupt is in progress - ignoring");
    }
}

/********************************************************************i
 * Watchdog callback handling
 */

static vector_t callbacks;

int pp_bmc_dev_watchdog_add_callback(pp_bmc_dev_watchdog_cb_t cb)
{
    unsigned int i;
    for (i = 0; i < vector_size(&callbacks); i++) {
	pp_bmc_dev_watchdog_cb_t callback = vector_get(&callbacks, i);
	if (callback == cb) return PP_ERR;
    }
    vector_add(&callbacks, cb);
    return PP_SUC;
}

int pp_bmc_dev_watchdog_rem_callback(pp_bmc_dev_watchdog_cb_t cb)
{
    unsigned int i;
    for (i = 0; i < vector_size(&callbacks); i++) {
	pp_bmc_dev_watchdog_cb_t callback = vector_get(&callbacks, i);
	if (callback == cb) {
	    vector_remove(&callbacks, i);
	    return PP_SUC;
	}
    }
    return PP_ERR;
}

static void watchdog_callback(pp_bmc_dev_watchdog_event_t event)
{
    unsigned int i;
    for (i = 0; i < vector_size(&callbacks); i++) {
	pp_bmc_dev_watchdog_cb_t callback = vector_get(&callbacks, i);
	callback(event);
    }
}

/********************************************************************
 * Watchdog command handlers
 */

static int watchdog_timeout_handler(const int item_id UNUSED,
				    void* ctx UNUSED)
{
    unsigned char offset = 0;
    unsigned char data2 = 0;

    watchdog_callback(PP_BMC_DEV_WATCHDOG_TIMEOUT);

    wdg->running = -1;
    
    /* clear pretimeout interrupt if set */
    bmc_interrupt_flag_set(BMC_INTERRUPT_WATCHDOG_PRETIMEOUT, 0);
    
    // update timer use flags
    if (wdg->timerUse == WD_USE_BIOS_FRB2) {
        wdg->BIOS_FRB2 = 1;
        data2 = 1;
    }
    if (wdg->timerUse == WD_USE_BIOS_POST) {
        wdg->BIOS_POST = 1;
        data2 = 2;
    }
    if (wdg->timerUse == WD_USE_OS_LOAD) {
        wdg->OS_LOAD = 1;
        data2 = 3;
    }
    if (wdg->timerUse == WD_USE_SMS_OS) {
        wdg->SMS_OS = 1;
        data2 = 4;
    }
    if (wdg->timerUse == WD_USE_OEM) {
        wdg->OEM = 1;
        data2 = 4;
    }
    
    // pp_bmc_log_debug("[watchdog] timeout - initiating action %d - TODO", wdg->action);

    switch (wdg->action) {
    case WD_ACTION_NONE:
        pp_bmc_log_notice("[watchdog] timeout without action_none");
        break;
    case WD_ACTION_HARD_RESET:
        offset = 0x01;
        pp_bmc_host_power_control(HOST_POWER_CONTROL_RESET, HOST_POWER_CONTROL_SOURCE_WATCHDOG, 0);
        break;
    case WD_ACTION_POWER_DOWN:
        offset = 0x02;
        pp_bmc_host_power_control(HOST_POWER_CONTROL_OFF, HOST_POWER_CONTROL_SOURCE_WATCHDOG, 0);
        break;
    case WD_ACTION_POWER_CYCLE:
        offset = 0x03;
        pp_bmc_host_power_control(HOST_POWER_CONTROL_CYCLE, HOST_POWER_CONTROL_SOURCE_WATCHDOG, 0);
        break;
    default:
        pp_bmc_log_error("[watchdog] timeout - unknown action");
        break;
    }

    offset |= 0xb0;  // data2 sensor specific, data3 unspecified
    if (wdg->noLog == 0) {
        /* create an event with the watchdog event-only sensor number */
        pp_bmc_receive_event(0, 0x20, 
                             0, IPMI_CHAN_PRIMARY_IPMB, 
                             IPMI_SENSOR_TYPE_WATCHDOG_2, IPMI_SENSOR_NUMBER_STATIC_WATCHDOG,
                             0, IPMI_EVENT_READING_TYPE_SENSOR_SPECIFIC,
                             offset, data2, 0);
    }
    
    wdg->noLog = 0;  // reset noLog if set
    
    return PP_SUC;
}

static int watchdog_pre_timeout_handler(const int item_id UNUSED,
					void* ctx UNUSED)
{
    wdg->pre_running = -1;
    pp_bmc_log_info("[watchdog] pre-timeout");

    watchdog_callback(PP_BMC_DEV_WATCHDOG_PRE_TIMEOUT);

    switch (wdg->preIntType) {
        case WD_INTTYPE_NONE:
            // nothing to happen
            pp_bmc_log_debug("[watchdog] pretimeout - no action defined");
            break;
        case WD_INTTYPE_SMI:
            // SMI interrupt - use SMI actor
            pp_bmc_log_debug("[watchdog] pretimeout - generating SMI");
            interrupt_initiate(PP_ACTOR_SMI);
            break;
        case WD_INTTYPE_NMI:
            // NMI interrupt - use NMI actor
            pp_bmc_log_debug("[watchdog] pretimeout - generating NMI");
            interrupt_initiate(PP_ACTOR_NMI);
            break;
        case WD_INTTYPE_MSG_INT:
            // messaging interrupt, use bmc_interrupt_flag_set
            pp_bmc_log_debug("[watchdog] pretimeout - message interrupt");
            bmc_interrupt_flag_set(BMC_INTERRUPT_WATCHDOG_PRETIMEOUT, 1);
            // Note: this is a flag that will be set until the real interrupt goes off
            break;
        default:
            // misconfiguration, should not happen
            pp_bmc_log_error("[watchdog] pretimeout - illegal preIntType");
            break;
    };
    
    return PP_SUC;
}

static int app_watchdog_reset(imsg_t* imsg) {
    int rv;

    watchdog_callback(PP_BMC_DEV_WATCHDOG_RESET);

    // Stop the timers if we're running ...
    if (wdg->running != -1) {
        if (wdg->pretimeout_reached) {
            // we cannot reset the watchdog once the pretimeout has been reached ...
            // (use setWatchdog to renew)
            return pp_bmc_router_resp_err(imsg, IPMI_ERR_NOT_SUPPORTED_IN_PRESENT_STATE);
        }
	
        // stop existing timers
        if (wdg->running != -1) {
	    rv = pp_select_remove_to(wdg->running);
	    wdg->running = -1;
	    if (rv != PP_SUC) {
	        pp_bmc_log_error("[watchdog] could not remove (existing) watchdog timer");
	        return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
	    }
	}
	if (wdg->pre_running != -1) {
	    rv = pp_select_remove_to(wdg->pre_running);
	    wdg->pre_running = -1;
	    if (rv != PP_SUC) {
	        pp_bmc_log_error("[watchdog] could not remove (existing) watchdog pre-timer");
	        return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
	    }
	}
    } // stop existing timers
    
    // start timers
    if (wdg->countdown == -1) {   // this should only apply if we're starting for the first time
        pp_bmc_log_notice("[watchdog] attempt to start uninitialized timer");
        return pp_bmc_router_resp_err(imsg, 0x80);  // command-specific: un-initialized watchdog
    }
    // start pretimer
    if (wdg->pretimeout_interval > -1) {
        // update pre-timeout
        int pretime = wdg->countdown * 100 - wdg->pretimeout_interval * 1000;
        if (pretime < 0) pretime = 0;
        wdg->pre_running = pp_select_add_to(pretime, 0, watchdog_pre_timeout_handler, NULL);
        if (wdg->pre_running == -1) {
            pp_bmc_log_error("[watchdog] could not add pre-timer to selector");
            return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
        }
	//pp_bmc_log_debug("[watchdog] pre-timer started");
    }
    // start timer
    wdg->running = pp_select_add_to(wdg->countdown * 100, 0, watchdog_timeout_handler, NULL);
    if (wdg->running == -1) {
        pp_bmc_log_error("[watchdog] could not add timer to selector");
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
    }
    //pp_bmc_log_debug("[watchdog] timer started");

    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

static int app_watchdog_set(imsg_t* imsg) {
    char* msgdata;
    int noStop=0;

    msgdata = imsg->data;

    wdg->pretimeout_reached = 0;
    
    // Byte 1 (in spec)
    wdg->noLog = (msgdata[0] & (0x1 << 7)) >> 7;
    noStop = (msgdata[0] & (0x1 << 6)) >> 6;
    wdg->timerUse = msgdata[0] & 0x7;

    watchdog_callback(PP_BMC_DEV_WATCHDOG_SET);

    // stop timers, we will restart them later if noStop is specified
    if (wdg->running != -1) {
	if (pp_select_remove_to(wdg->running) != PP_SUC) {
	    wdg->running = -1;
	    pp_bmc_log_error("[watchdog] could not remove (existing) watchdog timer");
	    return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
	}
	wdg->running = -1;
    } else {
        if (noStop == 1) {
	    // do not restart timers if they are not running ...
	    noStop = 0;
	}
    }
    if (wdg->pre_running != -1) {
	if (pp_select_remove_to(wdg->pre_running) != PP_SUC) {
	    wdg->pre_running = -1;
	    pp_bmc_log_error("[watchdog] could not remove (existing) watchdog pre-timer");
	    return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
	}
	wdg->pre_running = -1;
    }

    // Byte 2
    wdg->preIntType = (msgdata[1] & (0x7 << 4)) >> 4;
    wdg->action = msgdata[1] & (0x7);
    
    // Byte 3
    wdg->pretimeout_interval = msgdata[2];
    
    // Byte 4
    if ((msgdata[3] & (0x1 << 5)) != 0)
        wdg->OEM = 0;
    if ((msgdata[3] & (0x1 << 4)) != 0)
        wdg->SMS_OS = 0;
    if ((msgdata[3] & (0x1 << 3)) != 0)
        wdg->OS_LOAD = 0;
    if ((msgdata[3] & (0x1 << 2)) != 0)
        wdg->BIOS_POST = 0;
    if ((msgdata[3] & (0x1 << 1)) != 0)
        wdg->BIOS_FRB2 = 0;
    
    // Byte 5+6
    wdg->cd_lsb = msgdata[4];
    wdg->cd_msb = msgdata[5];
    wdg->countdown = (wdg->cd_msb * 256) + wdg->cd_lsb;
    
    // Restart the timer if noStop=1
    if (noStop==1) {
        int pretime = wdg->countdown * 100 - wdg->pretimeout_interval * 1000;
        if (pretime < 0) pretime = 0;
        wdg->pre_running = pp_select_add_to(pretime, 0, watchdog_pre_timeout_handler, NULL);
        if (wdg->pre_running == -1) {
            pp_bmc_log_error("[watchdog] could not add pre-timer to selector");
            return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
        }
	wdg->running = pp_select_add_to(wdg->countdown*100, 0, watchdog_timeout_handler, NULL);
	if (wdg->running == -1) {
	    pp_bmc_log_error("[watchdog] could not add timer to selector");
	    return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
	}
    }
    
    //pp_bmc_log_debug("[watchdog] set()");
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

static int app_watchdog_get(imsg_t* imsg) {
    char* msgdata;

    int timeleft=0;
    if (wdg->running > -1) {
        timeleft = pp_select_remaining_time(wdg->running);
	if (timeleft == PP_ERR) {
	    pp_bmc_log_error("[watchdog] cannot get remaining time for active timeout");
	    return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
        }
    }
    
    msgdata = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, 8);
    
    // Byte 2 (see spec)
    if (wdg->noLog != 0)
        msgdata[0] |= (0x1 <<7);
    if (wdg->running != -1)
        msgdata[0] |= (0x1 <<6);
    msgdata[0] |= wdg->timerUse;

    // Byte 3
    msgdata[1] |= wdg->preIntType <<4;
    msgdata[1] |= wdg->action;
    
    // Byte 4
    if (wdg->pretimeout_interval > -1)
        msgdata[2] = wdg->pretimeout_interval;

    // Byte 5
    if (wdg->OEM == 1)
        msgdata[3] |= (0x1 << 5);
    if (wdg->SMS_OS == 1)
        msgdata[3] |= (0x1 << 4);
    if (wdg->OS_LOAD == 1)
        msgdata[3] |= (0x1 << 3);
    if (wdg->BIOS_POST == 1)
        msgdata[3] |= (0x1 << 2);
    if (wdg->BIOS_FRB2 == 1)
        msgdata[3] |= (0x1 << 1);

    // Byte 6+7
    // no need to recalc the lsb and msb from an int
    if (wdg->countdown > -1) {
	msgdata[4] = wdg->cd_lsb;
	msgdata[5] = wdg->cd_msb;
    }
    
    // Byte 8+9
    if (wdg->running > -1) {
	timeleft = timeleft / 100;   // now we're at 100ms scale
	assert(timeleft < 257*256);  // we should not have more time left than we can initially configure ...
	msgdata[7] = (timeleft / 256);
	msgdata[6] = timeleft - (msgdata[7]*256);
    }

    //pp_bmc_log_debug("[watchdog] get()");
    return pp_bmc_router_send_msg(imsg);
}


/********************************************************************
 * command registration information
 */

static const dev_cmd_entry_t watchdog_cmd_tab[] = {
    {
        app_watchdog_reset,
        IPMI_NETFN_APP, IPMI_CMD_WATCHDOG_RESET,
        0, IPMI_PRIV_OPERATOR
    },
    {
        app_watchdog_set,
        IPMI_NETFN_APP, IPMI_CMD_WATCHDOG_SET,
        6, IPMI_PRIV_OPERATOR
    },
    {
        app_watchdog_get,
        IPMI_NETFN_APP, IPMI_CMD_WATCHDOG_GET,
        0, IPMI_PRIV_USER
    },
    { NULL, 0, 0, 0, 0 }
};

/********************************************************************
 * Watchdog c'tor/d'tor
 */

int pp_bmc_dev_watchdog_init()
{
    wdg->countdown = -1;      
    wdg->cd_msb = 0;
    wdg->cd_lsb = 0;
    wdg->pretimeout_interval = -1;
    wdg->pretimeout_reached = 0;
    wdg->running = -1;
    wdg->pre_running = -1;
    wdg->interrupt_handle = -1;
    
    wdg->action = WD_ACTION_NONE;
    wdg->preIntType = WD_INTTYPE_NONE;

    wdg->timerUse = WD_USE_UNDEFINED;
    wdg->BIOS_FRB2 = 0;
    wdg->BIOS_POST = 0;
    wdg->OS_LOAD = 0;
    wdg->SMS_OS = 0;
    wdg->OEM = 0;
    
    wdg->noLog = 0;
    
    /* register all entries of cmd tab */
    if (PP_ERR == pp_bmc_core_reg_cmd_tab(watchdog_cmd_tab)) return PP_ERR;

    // allocate watchdog_sdr because it gets free()d by clear-sdr.
    pp_bmc_sdrr_add_sdr_transiently(&watchdog_sdr.header);

    // init callback vector
    vector_new(&callbacks, 3, NULL);

    pp_bmc_log_info("[watchdog] device started");
    return PP_SUC;
}

void pp_bmc_dev_watchdog_cleanup()
{
    /* unregister all entries of cmd tab */
    pp_bmc_core_unreg_cmd_tab(watchdog_cmd_tab);

    vector_delete(&callbacks);

    pp_bmc_log_info("[watchdog] device shut down");
}
