/**
 * bmc_dev_event_msg_buff.c
 *
 * The file implements the event message buffer/queue.
 * 
 * Note: I assume that the event message buffer in the
 * specification contains only one event. Thus buffer full
 * means the same as 'there are events in the buffer'. Our
 * buffer is bigger but we implement the same functionality,
 * so our 'buffer full' flag will be set as soon as there
 * are events in the buffer (geo).
 * 
 * (c) 2005 Peppercon AG, Yiche Hsu  <yiche@peppercon.com>
 */
 

#include <stdlib.h>
#include <string.h>

#include "pp/base.h"

#include "pp/bmc/ipmi_cmd.h"
#include "pp/bmc/ipmi_sess.h"
#include "pp/bmc/ipmi_err.h"
#include "pp/bmc/bmc_imsg.h"
#include "pp/bmc/bmc_core.h"
#include "pp/bmc/bmc_router.h"
#include "pp/bmc/debug.h"
#include "pp/bmc/utils.h"

#include "event_receiver.h"
#include "bmc_dev_event_msg_buff.h"
#include "nc_interrupt.h"


#define EVENT_MSG_VER_10 0x03
#define EVENT_MSG_VER_15 0x04
#define EVENT_MSG_VER_20 0x04

#define EVENT_MSG_BUF_SIZE 20

static sel_entry_t event_msg_buf[EVENT_MSG_BUF_SIZE];
static unsigned char event_msg_rd_ptr = 0;
static unsigned char event_msg_wr_ptr = 0;
static unsigned char event_msg_num=0;

static unsigned char event_msg_buf_enable = 1;


/**
 * Add a event message to buff 
 */
void bmc_dev_event_msg_buf_add(sel_entry_t* e) {
    if (event_msg_buf_enable == 0) {
        /* event message buffer is disabled, drop event */
        return;
    }

    if (event_msg_num >= EVENT_MSG_BUF_SIZE) {
//        pp_bmc_log_debug("[APP] Event message buffer is full - dropped message");
        return;
    }
    else {
        /* add event */
        event_msg_buf[event_msg_wr_ptr] = *e;
	//pp_bmc_log_debug("[APP] Add Event message buffer: p->sys.timestamp=0x%02x", e->sys.timestamp);
        event_msg_wr_ptr++;
	if (event_msg_wr_ptr == EVENT_MSG_BUF_SIZE) {
            event_msg_wr_ptr = 0;
        }
	event_msg_num++;

        if (event_msg_num == 1) {
            /* transition from empty to full */
            bmc_interrupt_flag_set(BMC_INTERRUPT_EVENT_MSG_BUFFER, 1);
        }
    }

    return;
}

/**
 * Read a event message buffer. Reads the oldest event from the internal
 * buffer. The event is removed from the list and a reference to the
 * element is returned. This reference must not be freed and will only
 * be valid while no other operations on the event message buffer take
 * place. 
 */
static const sel_entry_t* bmc_dev_event_msg_buf_read(void) {
    /* should we check here if event message buffer is enabled ? */
    unsigned char tmp_ptr = 0;
    
    if (event_msg_num > 0) {
        tmp_ptr = event_msg_rd_ptr;
	event_msg_rd_ptr++;
        if (event_msg_rd_ptr == EVENT_MSG_BUF_SIZE) {
            event_msg_rd_ptr = 0;
        }
 	event_msg_num--;
	//pp_bmc_log_debug("[APP] Read Event message buffer: sys.timestamp=0x%02x", event_msg_buf[tmp_ptr].sys.timestamp);
        
        if (event_msg_num == 0) {
            /* transition from full to empty */
            bmc_interrupt_flag_set(BMC_INTERRUPT_EVENT_MSG_BUFFER, 0);
        }
        
        return (&event_msg_buf[tmp_ptr]);
    }
    else {
        /* buffer is empty */
        //pp_bmc_log_debug("[APP] No event message to read");
        return NULL;
    }
}

/**
 * Clear the event message buffer.
 */
void bmc_dev_event_msg_buf_clear() {
    /* clear event message buffer (reseting the ringbuf */
    /* is sufficient, no mem needs to be freed)         */
    event_msg_rd_ptr = 0;
    event_msg_wr_ptr = 0;
    event_msg_num=0;

    bmc_interrupt_flag_set(BMC_INTERRUPT_EVENT_MSG_BUFFER, 0);
}


/*********************************************************************
 * Read event message buffer IPMI command
 */

static int app_cmd_read_event_msg_buff(imsg_t* imsg) {
     const sel_entry_t *rs;
     
     /* dont check this for local builds, hard to debug */
#ifdef PP_BUILD_TYPE_FINAL
     /* command allowed from system interface only */
     if (imsg->chan != IPMI_CHAN_SI) {
         return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_CMD);
     }
#endif
     
     rs = bmc_dev_event_msg_buf_read();
     if (rs == NULL) {
         pp_bmc_log_debug("[APP] read_event_msg_buff command: buffer is empty");
         return pp_bmc_router_resp_err(imsg, IPMI_ERR_EVENT_MSG_BUFFER_EMPTY);
     }
     pp_bmc_log_debug("[APP] Response Event message buffer: rs->sys.timestamp=0x%02x", le32_to_cpu(rs->sys.timestamp_le32));
     
     return pp_bmc_router_resp_msg(imsg, IPMI_ERR_SUCCESS, rs, sizeof(sel_entry_t));
}

void bmc_dev_event_msg_buf_enable_set(unsigned char value) {
    event_msg_buf_enable = value;
    // flush the buffer if disabled ?
}       

unsigned char bmc_dev_event_msg_buf_enable_get() {
    return event_msg_buf_enable;
}


/********************************************************************
 * Event message buffer device command table
 */

static const dev_cmd_entry_t event_msg_buf_cmd_tab[] = {
    {
        .cmd_hndlr = app_cmd_read_event_msg_buff,
        .netfn = IPMI_NETFN_APP, .cmd = IPMI_CMD_READ_EVENT_MSG_BUFF,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_UNSPEC
    },
    { .cmd_hndlr = NULL }
};

/********************************************************************
 * Event message buffer device c'tor/d'tor
 */

int bmc_dev_event_msg_buf_init()
{
    /* clear event message buffer */
    event_msg_rd_ptr = 0;
    event_msg_wr_ptr = 0;
    event_msg_num=0;

    /* register all entries of cmd tab */
    if (PP_ERR == pp_bmc_core_reg_cmd_tab(event_msg_buf_cmd_tab)) return PP_ERR;
    pp_bmc_log_info("[APP] event message buffer device started");
    return PP_SUC;
}

void bmc_dev_event_msg_buf_cleanup()
{
    /* unregister all entries of cmd tab */
    pp_bmc_core_unreg_cmd_tab(event_msg_buf_cmd_tab);
    pp_bmc_log_info("[APP] event message buff device shut down");
}

