/**
 * bmc_dev_oem_supermicro.c
 *
 * Description: BMC SUPERMICRO-OEM Device
 *
 * (c) 2006 Supermicro <edwardp@supermicro.com> 3/31/06
 * 
 */

#include <pp/base.h>
#include <pp/cfg.h>
#include <pp/selector.h>
#include <liberic_misc.h>

#include <pp/bmc/ipmi_cmd.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/bmc/ipmi_msg.h>
#include <pp/bmc/ipmi_sess.h>
#include <pp/bmc/bmc_imsg.h>
#include <pp/bmc/bmc_core.h>
#include <pp/bmc/bmc_router.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/host_hardware.h>
#include <pp/bmc/host_power.h>
#include <pp/bmc/host_graceful_ops.h>
#include <pp/bmc/host_i2c_access.h>
// added by Edward
#include <pp/bmc/bmc_fru.h>
#include <pp/i2c.h>
#include <fcntl.h>

#include "bmc_dev_oem_supermicro.h"

typedef struct {
    unsigned short board_id;
    unsigned char  sub_type;
    unsigned char  cpu_number;
    unsigned char  reserved[4];
} mainboard_info;

extern char fru0_access_lock_flag;


/********************************************************************
 * Device command handlers
 */
// OEM command #2
/**
 * BIOS release SMbus, so BMC can access it.
 */
static int oem_supermicro_cmd_smb_release(imsg_t* imsg)
{
    if (PP_FAILED(pp_bmc_host_i2c_access_enable())) {
	pp_bmc_log_warn("[OEM-SUPERMICRO] failed to inform BMC of BIOS released I2C access");
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_COULD_NOT_PROVIDE_RESP);
    }

    pp_bmc_log_info("[OEM-SUPERMICRO] SMB RELEASE");
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}
// OEM command #3
/**
 * To clear the chassis intrusion flag
 */
static int oem_supermicro_cmd_clear_chassis_intrusion(imsg_t* imsg)
{
    int fd;
    pp_bmc_log_info("[OEM-SUPERMICRO] CLEAR CHASSIS INTRUSION FLAG");
    
    // need to have clear_chassis_instrusion() function 	
    if(PP_FAILED(pp_bmc_hardware_clr_intrusion_flag())) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_COULD_NOT_PROVIDE_RESP);
    }

    if ((fd = open("/dev/i2c-faraday-1",O_RDWR)) < 0) {
	/* ERROR HANDLING; you can check errno to see what went wrong */
	pp_bmc_log_error("Error Opening I2C device");
    }
    if (ioctl(fd, I2C_SLAVE_FORCE, 0x2f) < 0) {
	    /* ERROR HANDLING; you can check errno to see what went wrong */
	    pp_bmc_log_error("IOCTL I2C device ERROR");
    }	
    i2c_smbus_write_byte_data(fd, 0x49, 0x80);
    close(fd);


    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}
// OEM command #4
/*
 * User can send command for graceful power control.
 * 1. graceful shutdown 2. graceful reset 3. graceful powercycle
 */
#define IPMI_SUB_CMD_SUPERMICRO_SHUTDOWN   0x01
#define IPMI_SUB_CMD_SUPERMICRO_RESET      0x02
#define IPMI_SUB_CMD_SUPERMICRO_POWERCYCLE 0x03

static int oem_supermicro_cmd_graceful_power(imsg_t* imsg)
{
    int op = -1;
    switch (imsg->data[0]) {
        case IPMI_SUB_CMD_SUPERMICRO_SHUTDOWN:   op = HOST_POWER_CONTROL_OFF; break;
        case IPMI_SUB_CMD_SUPERMICRO_RESET:      op = HOST_POWER_CONTROL_RESET; break;
        case IPMI_SUB_CMD_SUPERMICRO_POWERCYCLE: op = HOST_POWER_CONTROL_CYCLE; break;
    }

    if (op == -1) {
        pp_bmc_log_warn("[OEM-SUPERMICRO] unknown graceful power operation %d", imsg->data[0]);
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }

    if (PP_FAILED(pp_bmc_host_graceful_ops_execute(op, imsg->chan))) {
        pp_bmc_log_warn("[OEM-SUPERMICRO] initiation of graceful power operation %d failed", imsg->data[0]);
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
    }

    pp_bmc_log_info("[OEM-SUPERMICRO]  USER SEND GRACEFUL POWER COMMAND %d", imsg->data[0]);

    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

// OEM command #5
/*
 * BIOS request SMbus access permission, so BIOS can access it.
 */
static int oem_supermicro_cmd_smb_request(imsg_t* imsg)
{
    if (PP_FAILED(pp_bmc_host_i2c_access_disable())) {
	pp_bmc_log_warn("[OEM-SUPERMICRO] failed to inform BMC of BIOS requestI2C access");
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_COULD_NOT_PROVIDE_RESP);
    }

    pp_bmc_log_info("[OEM-SUPERMICRO] BIOS REQUEST SMBUS ACCESS");
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

// OEM command #6
/*
 * Command for lock/unlock the write FRU cap
 */
static int oem_supermicro_cmd_write_fru_cap(imsg_t* imsg)
{
   fru0_access_lock_flag = imsg->data[0] ? 1 : 0;
   ////pp_bmc_log_info("[OEM-SUPERMICRO] WRITE FRU CAP =  %d", fru0_access_lock_flag);
   return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

// OEM command #7
/*
 * Get SMBus status
 */
static int oem_supermicro_cmd_smb_status(imsg_t* imsg)
{
    unsigned char* rs_data;

    pp_bmc_log_info("[OEM-SUPERMICRO] GET SMBUS STATUS");

    rs_data = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, 1);
    rs_data[0] = pp_bmc_host_i2c_access_get_state() ? 1 : 0;

    return pp_bmc_router_send_msg(imsg);
}
// OEM command #0f
/*
 * Get Chassis status 
 */
static int oem_supermicro_cmd_get_chassis_status(imsg_t* imsg)
{
    unsigned char* rs_data;
    int fd;
    unsigned char value;

    pp_bmc_log_info("[OEM-SUPERMICRO] GET CHASSIS STATUS");

    if ((fd = open("/dev/i2c-faraday-1",O_RDWR)) < 0) {
	/* ERROR HANDLING; you can check errno to see what went wrong */
	pp_bmc_log_error("[OEM-SUPERMICRO] Error Opening I2C device");
    }
    if (ioctl(fd, I2C_SLAVE_FORCE, 0x2f) < 0) {
	    /* ERROR HANDLING; you can check errno to see what went wrong */
	    pp_bmc_log_error("[OEM-SUPERMICRO] IOCTL I2C device ERROR");
    }	
    i2c_smbus_write_byte_data(fd, 0x49, 0x80);
    value = (i2c_smbus_read_byte_data(fd, 0x44));// & 0x40) >> 6;
    close(fd);

    rs_data = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, 1);
    rs_data[0] = value;

    return pp_bmc_router_send_msg(imsg);
}

/** triggers a board reset until successful */
static int trigger_board_reset_to(int id UNUSED, void* ctx UNUSED) {
    if (eric_misc_trigger_board_reset(0) == 0) {
        // success
    } else {
        // try again in 10 seconds
        pp_select_add_to(10000, 0, trigger_board_reset_to, NULL);
    }
    return PP_SUC;
}

// OEM command #0x20
static int oem_supermicro_cmd_set_hw_info(imsg_t* imsg)
{
    mainboard_info msg;
    unsigned short board_id;
    unsigned short board_subtype;
    unsigned short cpu_number;

    msg.board_id = ((imsg->data[1] << 8) | (imsg->data[0]));
    msg.sub_type = imsg->data[2];
    msg.cpu_number = imsg->data[3];

    pp_bmc_log_info("[OEM-SUPERMICRO] SET HW INFO boardid=%04x, subtype=%d, cpu#=%d",
                    msg.board_id, msg.sub_type, msg.cpu_number);

    /* store the values if they differ from existing values */
    // get existing values, use defaults (0) in case of error
    pp_cfg_get_ushort(&board_id, "ipmi.oem.supermicro.board_id");
    pp_cfg_get_ushort(&board_subtype, "ipmi.oem.supermicro.board_subtype");
    pp_cfg_get_ushort(&cpu_number, "ipmi.oem.supermicro.cpu_number");

    if ( (board_id != msg.board_id) ||
         (board_subtype != msg.sub_type) ||
         (cpu_number != msg.cpu_number) )
    {
        // values changed
        if (! ( PP_SUCCED(pp_cfg_set_ushort(msg.board_id, "ipmi.oem.supermicro.board_id")) &&
                PP_SUCCED(pp_cfg_set_ushort(msg.sub_type, "ipmi.oem.supermicro.board_subtype")) &&
                PP_SUCCED(pp_cfg_set_ushort(msg.cpu_number, "ipmi.oem.supermicro.cpu_number")) &&
                PP_SUCCED(pp_cfg_save(DO_FLUSH)) ) )
        {
            // couldn't set the values, inform user, ignore
            pp_bmc_log_error("[OEM-SUPERMICRO] could not write HW info to config system");
        }
        
        // schedule a board reset. restarting eric only would be better but then we could infer with fw-updates
        pp_select_add_to(10000, 0, trigger_board_reset_to, NULL);  // 10 secs to allow proper config-flush
    } else {
        // debug only ...
        //pp_bmc_log_info("don't write values ...");
    }

    /* process number of CPUs (forward to conditions, ...) */
    // This is currently unused. Number of CPUs is _not_ sent by BIOS

    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

// OEM command #0x21
static int oem_supermicro_cmd_get_hw_info(imsg_t* imsg)
{
    unsigned short board_id;
    unsigned short board_subtype;
    unsigned short cpu_number;
    char* res;

    // get values, use defaults (0) in case of error
    pp_cfg_get_ushort(&board_id, "ipmi.oem.supermicro.board_id");
    pp_cfg_get_ushort(&board_subtype, "ipmi.oem.supermicro.board_subtype");
    pp_cfg_get_ushort(&cpu_number, "ipmi.oem.supermicro.cpu_number");

    pp_bmc_log_info("[OEM-SUPERMICRO] SET HW INFO id=%d, cpu#=%d, cpu type=%d", \
                     board_id, board_subtype, \
                     cpu_number);

    res = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, 8);

    res[0] = ((board_id >> 8) & 0xff);
    res[1] = (board_id & 0xff);
    res[2] = board_subtype;
    res[3] = cpu_number;
    
    // reserver 4 ~ 7 for others
    memset(&(imsg->data[4]), 0, 4);

    return pp_bmc_router_send_msg(imsg);
}


    
/********************************************************************
 * Device command table
 */

static const dev_cmd_entry_t oem_supermicro_cmd_tab[] = {
    {   // cmd = 0x02
        .cmd_hndlr = oem_supermicro_cmd_smb_release,
        .netfn = IPMI_NETFN_OEM_SUPERMICRO,
	.cmd = IPMI_CMD_SUPERMICRO_SMB_RELEASE,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_ADMIN
    },
    {   // cmd = 0x03
        .cmd_hndlr = oem_supermicro_cmd_clear_chassis_intrusion,
        .netfn = IPMI_NETFN_OEM_SUPERMICRO,
	.cmd = IPMI_CMD_SUPERMICRO_CLEAR_CHASSIS_INTRUSION,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_ADMIN
    },
    {   // cmd = 0x04
        .cmd_hndlr = oem_supermicro_cmd_graceful_power,
        .netfn = IPMI_NETFN_OEM_SUPERMICRO,
	.cmd = IPMI_CMD_SUPERMICRO_GRACEFUL_POWER,
        .min_data_size = 1, .min_priv_level = IPMI_PRIV_ADMIN
    },
    {   // cmd = 0x05
        .cmd_hndlr = oem_supermicro_cmd_smb_request,
        .netfn = IPMI_NETFN_OEM_SUPERMICRO,
	.cmd = IPMI_CMD_SUPERMICRO_SMB_REQUEST,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_ADMIN
    },
    {   // cmd = 0x06
        .cmd_hndlr = oem_supermicro_cmd_write_fru_cap,
        .netfn = IPMI_NETFN_OEM_SUPERMICRO, 
        .cmd = IPMI_CMD_SUPERMICRO_WRITE_FRU_CAP,
        .min_data_size = 1, .min_priv_level = IPMI_PRIV_ADMIN
    },
    {  // cmd = 0x07
        .cmd_hndlr = oem_supermicro_cmd_smb_status,
        .netfn = IPMI_NETFN_OEM_SUPERMICRO, 
        .cmd = IPMI_CMD_SUPERMICRO_SMB_STATUS,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_ADMIN
    },
    {  // cmd = 0x0F
        .cmd_hndlr = oem_supermicro_cmd_get_chassis_status,
        .netfn = IPMI_NETFN_OEM_SUPERMICRO, 
        .cmd = IPMI_CMD_SUPERMICRO_GET_CHASSIS_STATUS,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_ADMIN 
    },
    {  // cmd = 0x20
	.cmd_hndlr = oem_supermicro_cmd_set_hw_info,
        .netfn = IPMI_NETFN_OEM_SUPERMICRO,
        .cmd = IPMI_CMD_SUPERMICRO_SET_HW_INFO,
        .min_data_size = 8,
        .min_priv_level = IPMI_PRIV_ADMIN
    },
    {  // cmd = 0x21
        .cmd_hndlr = oem_supermicro_cmd_get_hw_info,
        .netfn = IPMI_NETFN_OEM_SUPERMICRO,
        .cmd = IPMI_CMD_SUPERMICRO_GET_HW_INFO,
        .min_data_size = 0,
        .min_priv_level = IPMI_PRIV_ADMIN
    },

    { .cmd_hndlr = NULL }
};

/********************************************************************
 * Device c'tor/d'tor
 */

int pp_bmc_dev_oem_supermicro_init(void)
{
    /* register all entries of cmd tab */
    if (PP_ERR == pp_bmc_core_reg_cmd_tab(oem_supermicro_cmd_tab)) return PP_ERR;

    pp_bmc_log_info("[OEM-SUPERMICRO] device started");
    return PP_SUC;
}

void pp_bmc_dev_oem_supermicro_cleanup(void)
{
    /* unregister all entries of cmd tab */
    pp_bmc_core_unreg_cmd_tab(oem_supermicro_cmd_tab);

    pp_bmc_log_info("[OEM-SUPERMICRO] device shut down");
}
