/**
 * bmc_dev_pef_nv.c
 *
 * Platform event filter related non volatile storage routines,
 * data structure definitions. May implement caching and rollback.
 * 
 * Implements data structures and access functions for
 * - event filter table
 * - alert policy table
 * - alert string table
 * - (alert queue entry)
 * - configuration
 *
 * (c) 2004 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#include <string.h>
#include "pp/base.h"

#include "pp/bmc/debug.h"
#include "pp/bmc/utils.h"
#include "pp/bmc/bmc_nv_storage.h"

#include "bmc_dev_pef_nv.h"


typedef struct {
    char magic[16];
    unsigned int ver;
} pef_file_header_t;

const unsigned int pefHeaderVersion = 0x00100004; // Version 001.00.004
const char* pefHeaderMagic = "bmc_pef_magic";

/**
 * File layout of file 'pef'
 * - header
 * - configuration
 * - event_filter_entry_t[EFT_SIZE]
 * - alert_policy_entry_t[APT_SIZE]
 * - alert_string_entry_t[AST_SIZE]
 */

/**
 * Open the pef nv_storage and check if the files header is correct.
 * Internal file-offset will be set after the header.
 * @param fd pointer to filedescriptor (set to -1 on error)
 * @returns PP_SUC if file is a correct sel
 * @returns PP_ERR if file is not a correct pef storage or could not be opened
 */
static int open_check(int* fd) {
    if (PP_FAILED(bmc_nv_open(fd, "pef"))) {
        pp_bmc_log_error("[PEF] could not open persistent storage");
        goto failed;
    }

    pef_file_header_t header;
    if (bmc_nv_read(*fd, &header, sizeof(pef_file_header_t)) != sizeof(pef_file_header_t)) {
        pp_bmc_log_error("[PEF] could not read file header from persistent storage");
        goto failed;
    }

    if (header.ver != pefHeaderVersion) {
        pp_bmc_log_error("[PEF] could not read file, version does not match pef version");
        goto failed;
    }

    if (strncmp(header.magic,pefHeaderMagic, 16) != 0) {
        pp_bmc_log_error("[PEF] could not read file, wrong magic number");
        goto failed;
    }

    return PP_SUC;

failed:
    if (*fd >= 0) {
        bmc_nv_close(*fd);
        *fd = -1;
    }
    return PP_ERR;
}

/**
 * get the pef configuration. 
 * The get operation may use a memory cache (but does not in the moment)
 * @returns PP_ERR or PP_SUC
 */
int pef_get_cfg(pef_configuration_t* config) {
    int fd;
    if (PP_FAILED(open_check(&fd))) return PP_ERR;

    // config resides directly after header, so there is no need to seek
    if (bmc_nv_read(fd, config, sizeof(pef_configuration_t)) != sizeof(pef_configuration_t)) {
        pp_bmc_log_error("[PEF] could not read configuration from persistent storage");
        bmc_nv_close(fd);
        return PP_ERR;
    }

    bmc_nv_close(fd);
    return PP_SUC;
}


/**
 * set the pef configuration.
 * The set operation is directly written to nv_storage.
 * @returns PP_ERR or PP_SUC
 */
int pef_set_cfg(pef_configuration_t* config) {
    int fd;
    if (PP_FAILED(open_check(&fd))) return PP_ERR;

    // config resides directly after header, so there is no need to seek
    if (bmc_nv_write(fd, config, sizeof(pef_configuration_t)) != sizeof(pef_configuration_t)) {
        pp_bmc_log_error("[PEF] could not write configuration to persistent storage");
        bmc_nv_close(fd);
        return PP_ERR;
    }

    bmc_nv_close(fd);
    return PP_SUC;
}

/**
 * get an entry of the event filter table. 
 * The operation may use a memory cache (but does not in the moment)
 * @returns PP_ERR or PP_SUC
 */
int pef_get_event_filter(unsigned char idx, event_filter_entry_t* filter) {
    int fd;
    if (PP_FAILED(open_check(&fd))) return PP_ERR;

    // seek
    int offset = 0;
    offset = offset + sizeof(pef_configuration_t);
    offset = offset + sizeof(event_filter_entry_t) * idx;
    bmc_nv_lseek(fd, offset, SEEK_CUR);

    if (bmc_nv_read(fd, filter, sizeof(event_filter_entry_t)) != sizeof(event_filter_entry_t)) {
        pp_bmc_log_error("[PEF] could not read event filter %d from persistent storage", idx);
        bmc_nv_close(fd);
        return PP_ERR;
    }

    bmc_nv_close(fd);
    return PP_SUC;
}

/**
 * set an entry of the event filter table
 * The set operation is directly written to nv_storage.
 * @returns PP_ERR or PP_SUC
 */
int pef_set_event_filter(unsigned char idx, event_filter_entry_t* filter) {
    int fd;
    if (PP_FAILED(open_check(&fd))) return PP_ERR;

    // seek
    int offset = 0;
    offset = offset + sizeof(pef_configuration_t);
    offset = offset + sizeof(event_filter_entry_t) * idx;
    bmc_nv_lseek(fd, offset, SEEK_CUR);

    if (bmc_nv_write(fd, filter, sizeof(event_filter_entry_t)) != sizeof(event_filter_entry_t)) {
        pp_bmc_log_error("[PEF] could not write event filter %d to persistent storage", idx);
        bmc_nv_close(fd);
        return PP_ERR;
    }

    bmc_nv_close(fd);
    return PP_SUC;
}

/**
 * get an entry of the alert policy table. 
 * The operation may use a memory cache (but does not in the moment)
 * @returns PP_ERR or PP_SUC
 */
int pef_get_alert_policy(unsigned char idx, alert_policy_entry_t* alert_policy) {
    int fd;
    if (PP_FAILED(open_check(&fd))) return PP_ERR;

    // seek
    int offset = 0;
    offset = offset + sizeof(pef_configuration_t);
    offset = offset + sizeof(event_filter_entry_t) * EFT_SIZE;
    offset = offset + sizeof(alert_policy_entry_t) * idx;
    bmc_nv_lseek(fd, offset, SEEK_CUR);

    if (bmc_nv_read(fd, alert_policy, sizeof(alert_policy_entry_t)) != sizeof(alert_policy_entry_t)) {
        pp_bmc_log_error("[PEF] could not read alert policy %d from persistent storage", idx);
        bmc_nv_close(fd);
        return PP_ERR;
    }

    bmc_nv_close(fd);
    return PP_SUC;
}

/**
 * set an entry of the alert policy table.
 * The set operation is directly written to nv_storage.
 * @returns PP_ERR or PP_SUC
 */
int pef_set_alert_policy(unsigned char idx, const alert_policy_entry_t* alert_policy) {
    int fd;
    if (PP_FAILED(open_check(&fd))) return PP_ERR;

    // seek
    int offset = 0;
    offset = offset + sizeof(pef_configuration_t);
    offset = offset + sizeof(event_filter_entry_t) * EFT_SIZE;
    offset = offset + sizeof(alert_policy_entry_t) * idx;
    bmc_nv_lseek(fd, offset, SEEK_CUR);

    if (bmc_nv_write(fd, alert_policy, sizeof(alert_policy_entry_t)) != sizeof(alert_policy_entry_t)) {
        pp_bmc_log_error("[PEF] could not write alert policy %d to persistent storage", idx);
        bmc_nv_close(fd);
        return PP_ERR;
    }

    bmc_nv_close(fd);
    return PP_SUC;
}

/**
 * get an entry of the alert string table. 
 * The operation may use a memory cache (but does not in the moment)
 * @returns PP_ERR or PP_SUC
 */
int pef_get_alert_string(unsigned char idx, alert_string_entry_t* alert_string) {
    int fd;
    if (PP_FAILED(open_check(&fd))) return PP_ERR;

    // seek
    int offset = 0;
    offset = offset + sizeof(pef_configuration_t);
    offset = offset + sizeof(event_filter_entry_t) * EFT_SIZE;
    offset = offset + sizeof(alert_policy_entry_t) * APT_SIZE;
    offset = offset + sizeof(alert_string_entry_t) * idx;
    bmc_nv_lseek(fd, offset, SEEK_CUR);

    if (bmc_nv_read(fd, alert_string, sizeof(alert_string_entry_t)) != sizeof(alert_string_entry_t)) {
        pp_bmc_log_error("[PEF] could not read alert string %d from persistent storage", idx);
        bmc_nv_close(fd);
        return PP_ERR;
    }

    bmc_nv_close(fd);
    return PP_SUC;
}

/**
 * set an entry of the alert string table.
 * The set operation is directly written to nv_storage.
 * @returns PP_ERR or PP_SUC
 */
int pef_set_alert_string(unsigned char idx, alert_string_entry_t* alert_string) {
    int fd;
    if (PP_FAILED(open_check(&fd))) return PP_ERR;

    // seek
    int offset = 0;
    offset = offset + sizeof(pef_configuration_t);
    offset = offset + sizeof(event_filter_entry_t) * EFT_SIZE;
    offset = offset + sizeof(alert_policy_entry_t) * APT_SIZE;
    offset = offset + sizeof(alert_string_entry_t) * idx;
    bmc_nv_lseek(fd, offset, SEEK_CUR);

    if (bmc_nv_write(fd, alert_string, sizeof(alert_string_entry_t)) != sizeof(alert_string_entry_t)) {
        pp_bmc_log_error("[PEF] could not write alert string %d to persistent storage", idx);
        bmc_nv_close(fd);
        return PP_ERR;
    }

    bmc_nv_close(fd);
    return PP_SUC;
}


/**
 * Initialize PEF. Create a file with the necessary header and
 * initialize each section with the passed data structure.
 * (data types are blueprints that are written to every entry)
 */
int pef_initialize(pef_configuration_t* config,
                   event_filter_entry_t* filter,
                   alert_policy_entry_t* alert_policy,
                   alert_string_entry_t* alert_string) {
    int fd;
    int i;

    if (PP_FAILED(bmc_nv_open(&fd, "pef"))) {
        pp_bmc_log_error("[PEF] could not open persistent storage");
        return PP_ERR;
    }

    pef_file_header_t header;
    header.ver = pefHeaderVersion;
    strncpy(header.magic,pefHeaderMagic, 16);
    if (bmc_nv_write(fd, &header, sizeof(pef_file_header_t)) != sizeof(pef_file_header_t)) {
        pp_bmc_log_error("[PEF] could not write file header to persistent storage");
        bmc_nv_close(fd);
        return PP_ERR;
    }
    
    if (bmc_nv_write(fd, config, sizeof(pef_configuration_t)) != sizeof(pef_configuration_t)) {
        pp_bmc_log_error("[PEF] could not initialize configuration");
        bmc_nv_close(fd);
        return PP_ERR;
    }
    
    for (i=0; i<EFT_SIZE; i++) {
        if (bmc_nv_write(fd, filter, sizeof(event_filter_entry_t)) != sizeof(event_filter_entry_t)) {
            pp_bmc_log_error("[PEF] could not initialize event filter %d", i);
            bmc_nv_close(fd);
            return PP_ERR;
        }
    }
    for (i=0; i<APT_SIZE; i++) {
        if (bmc_nv_write(fd, alert_policy, sizeof(alert_policy_entry_t)) != sizeof(alert_policy_entry_t)) {
            pp_bmc_log_error("[PEF] could not initialize alert policy %d", i);
            bmc_nv_close(fd);
            return PP_ERR;
        }
    }
    for (i=0; i<AST_SIZE; i++) {
        if (bmc_nv_write(fd, alert_string, sizeof(alert_string_entry_t)) != sizeof(alert_string_entry_t)) {
            pp_bmc_log_error("[PEF] could not initialize alert string %d", i);
            bmc_nv_close(fd);
            return PP_ERR;
        }
    }

    bmc_nv_close(fd);
    pp_bmc_log_debug("[PEF] new PEF storage initialized");
    return PP_SUC;
}

/**
 * Performs a complete check on the pef. (File exists, file magic
 * correct, file length correct). If this returns PP_SUC, all future
 * operations on the file should not fail. If PP_ERR is returned,
 * pef_initialized() must be called to create a correct pef.
 */
int pef_complete_check() {
    alert_string_entry_t as;
    // If we can read the last entry of the pef (alertstring[AST_SIZE-1])
    // then we can assume that the file is correct
    return pef_get_alert_string(AST_SIZE-1, &as);
}
