/**
 * file bmc_dev_sel.c
 *
 * Description: BMC System Event Log Device
 * 
 * Delete single SEL entry not supported by this device.
 * (bmc_dev_sel_nv.c supports it ?)
 *
 * (c) 2004 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 */

#include "pp/base.h"
#include "pp/vector.h"

#include "pp/bmc/ipmi_cmd.h"
#include "pp/bmc/ipmi_sess.h"
#include "pp/bmc/ipmi_err.h"
#include "pp/bmc/bmc_imsg.h"
#include "pp/bmc/bmc_core.h"
#include "pp/bmc/bmc_router.h"
#include "pp/bmc/debug.h"
#include "pp/bmc/utils.h"

#include "event_receiver.h"
#include "bmc_dev_sel.h"
#include "bmc_dev_sel_nv.h"

#include "pp/cfg.h"
#include <time.h>

/*
 * Globals
 */

#define IPMI_SEL_TYPE_SYSTEM 0x02
#define IPMI_IS_SEL_TYPE_SYSTEM(x)  ((x) == IPMI_SEL_TYPE_SYSTEM)
#define IPMI_IS_SEL_TYPE_OEM_TS(x)  ((x) >= 0xc0 && (x) <= 0xdf)
#define IPMI_IS_SEL_TYPE_OEM_NTS(x) ((x) >= 0xe0)

/* static data */
static struct {
    reserv_t *reserv;
    
    time_t last_del_time;

} sel = { NULL, 0 };

static int sel_enabled;


/********************************************************************
 * SEL command hndls
 */

/*
 * Get SEL Info
 */

#define IPMI_SEL_SUPP_DEL           0x08
#define IPMI_SEL_SUPP_PART_ADD      0x04
#define IPMI_SEL_SUPP_RESERVE       0x02
#define IPMI_SEL_SUPP_ALLOC_INFO    0x01

struct get_sel_info_rs_s {
    unsigned char ver;
    unsigned short entry_count_le16;
    unsigned short free_bytes_le16;
    unsigned int last_add_time_le32;
    unsigned int last_del_time_le32;
    BITFIELD2(unsigned char, support : 7, overflow : 1);
} __attribute__ ((packed));
typedef struct get_sel_info_rs_s get_sel_info_rs_t;

static int sel_cmd_get_sel_info(imsg_t *imsg)
{
    unsigned short size = sel_nv_get_size();
    get_sel_info_rs_t *rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(get_sel_info_rs_t));
    
    rs->ver                 = 0x51; /* still valid for IPMI v2.0 */
    rs->entry_count_le16     = cpu_to_le16(size);
    rs->free_bytes_le16      = cpu_to_le16((SEL_MAX_SIZE - size) * sizeof(sel_entry_t));
    rs->last_add_time_le32   = cpu_to_le32(pp_bmc_get_most_recent_addition_timestamp());
    rs->last_del_time_le32   = cpu_to_le32(sel.last_del_time);
    rs->support             = IPMI_SEL_SUPP_RESERVE | IPMI_SEL_SUPP_ALLOC_INFO;
    rs->overflow            = size == SEL_MAX_SIZE ? 1 : 0;
    
    return pp_bmc_router_send_msg(imsg);
}

/*
 * Get SEL Alloc Info
 */

struct get_sel_alloc_info_rs_s {
    unsigned short max_unit_count_le16;
    unsigned short unit_size_le16;
    unsigned short free_unit_count_le16;
    unsigned short largest_free_block_le16;
    unsigned char  max_record_size;
} __attribute__ ((packed));
typedef struct get_sel_alloc_info_rs_s get_sel_alloc_info_rs_t;

static int sel_cmd_get_sel_alloc_info(imsg_t *imsg)
{
    unsigned short size = sel_nv_get_size();
    get_sel_alloc_info_rs_t *rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(get_sel_alloc_info_rs_t));

    /* TODO: fill this with usefull values */
    rs->max_unit_count_le16 = cpu_to_le16(SEL_MAX_SIZE);
    rs->unit_size_le16 = cpu_to_le16(sizeof(sel_entry_t));
    rs->free_unit_count_le16 = cpu_to_le16(SEL_MAX_SIZE - size);
    rs->largest_free_block_le16 = cpu_to_le16(SEL_MAX_SIZE - size);
    rs->max_record_size = 255;

    return pp_bmc_router_send_msg(imsg);
}

/*
 * Reserve SEL
 */

static int sel_cmd_reserve_sel(imsg_t *imsg)
{
    reserv_cancel(sel.reserv);
    unsigned short id = cpu_to_le16(reserv_make(sel.reserv, imsg));
    return pp_bmc_router_resp_msg(imsg, IPMI_ERR_SUCCESS, &id, sizeof(id));
}

/*
 * Get SEL Entry
 */

struct get_sel_entry_rq_s {
    unsigned short res_id_le16;
    unsigned short rec_id_le16;
    unsigned char offs;
    unsigned char size;
} __attribute__ ((packed));
typedef struct get_sel_entry_rq_s get_sel_entry_rq_t;

struct get_sel_entry_rs_s {
    unsigned short next_rec_id_le16;
    sel_entry_t rec[0];  // [0..16] bytes of selected SEL record
} __attribute__ ((packed));
typedef struct get_sel_entry_rs_s get_sel_entry_rs_t;

static int sel_cmd_get_sel_entry(imsg_t *imsg)
{
    sel_entry_t evt;
    get_sel_entry_rq_t *rq = (void*)imsg->data;
    unsigned short id;
    unsigned short next;
    int need_res;
    unsigned char roffset = rq->offs;
    unsigned char rlength = rq->size;
    
    /* calculate sizes (important for partial read */
    if (roffset > sizeof(sel_entry_t)) {
        pp_bmc_log_warn("[SEL] Get SEL Entry called with invalid offset %d", roffset);
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
    }
    if (rlength > (sizeof(sel_entry_t) - roffset)) {
        rlength = sizeof(sel_entry_t) - roffset;
    }

    need_res = (rq->offs == 0) ?
        PP_BMC_RESERV_OPTIONAL : PP_BMC_RESERV_MANDATORY;

    /* check reservation - only required for partial reads */
    if (PP_ERR == reserv_check(sel.reserv, le16_to_cpu(rq->res_id_le16), need_res, imsg))
    {
        pp_bmc_log_warn("[SEL] Get SEL Entry called with invalid reservation ID");
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_RESERVATION);
    }

    // SEL empty?
    if (sel_nv_get_size() == 0) {
        pp_bmc_log_warn("[SEL] Get SEL Entry called, but SEL is empty");
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_ITEM_NOT_PRESENT);
    }

    //pp_bmc_log_debug("[SEL] get_sel_entry id=%x", rq->rec_id);
    id = le16_to_cpu(rq->rec_id_le16);

    // fix special IDs
    if (id == IPMI_SEL_FIRST_REC) id = sel_nv_get_first_event_id();
    else if (id == IPMI_SEL_LAST_REC) id = sel_nv_get_last_event_id();

    if (PP_FAILED(sel_nv_get_event(id, &evt, &next))) {
        pp_bmc_log_warn("[SEL] Get SEL Entry failed to get event %d", id);
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_ITEM_NOT_PRESENT);
    }
    
    get_sel_entry_rs_t *rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(get_sel_entry_rs_t) + rlength);

    /* prepare response */
    rs->next_rec_id_le16 = cpu_to_le16(next == 0xffff ? IPMI_SEL_LAST_REC : next);
    memcpy(rs->rec, ((unsigned char*)&evt)+roffset, rlength);
    
    return pp_bmc_router_send_msg(imsg);
}

/*
 * Add SEL Entry
 */

static int sel_cmd_add_sel_entry(imsg_t *imsg)
{
    sel_entry_t *e = (void*)imsg->data;
    unsigned short id_le16 = cpu_to_le16(sel_nv_get_next_entry_id());

    e->id_le16 = id_le16;
    
    time_t t = time(NULL);
    if (IPMI_IS_SEL_TYPE_SYSTEM(e->type)) {
	e->sys.timestamp_le32 = cpu_to_le32(t);
    } else if (IPMI_IS_SEL_TYPE_OEM_TS(e->type)) {
	e->oem_ts.timestamp_le32 = cpu_to_le32(t);
    } else if (IPMI_IS_SEL_TYPE_OEM_NTS(e->type)) {
	// nothing
    } else {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_BAD_SEL_TYPE);
    }

    if (sel_nv_add_event(e) == 0xffff)
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
	
    pp_bmc_set_most_recent_addition_timestamp(t);
    reserv_cancel(sel.reserv);

    return pp_bmc_router_resp_msg(imsg, IPMI_ERR_SUCCESS, &id_le16, sizeof(id_le16));
}

/*
 * Clear SEL
 */

struct clear_sel_rq_s {
    unsigned short res_id_le16;
    unsigned int magic_le24 : 24; /* see IPMI_ERASE_MAGIC */
    unsigned char cmd; /* see IPMI_ERASE_xxx */
} __attribute__ ((packed));
typedef struct clear_sel_rq_s clear_sel_rq_t;

static int sel_cmd_clear_sel(imsg_t *imsg)
{
    clear_sel_rq_t *rq = (void*)imsg->data;

    /* check reservation */
    if (PP_ERR == reserv_check(sel.reserv, le16_to_cpu(rq->res_id_le16),
			       PP_BMC_RESERV_MANDATORY, imsg))
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_RESERVATION);
    
    /* security double check for heavy operation */
    if (rq->magic_le24 != cpu_to_le24(IPMI_ERASE_MAGIC))
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);

    /* really want to erase...? */
    if (rq->cmd == IPMI_ERASE_INITIATE) {
        sel_nv_clear_sel();
        sel.last_del_time = time(NULL);
    }

    /* erease complete (always work synchron) */
    unsigned char state = 0x1;
    return pp_bmc_router_resp_msg(imsg, IPMI_ERR_SUCCESS, &state, sizeof(state));
}

/*
 * Get SEL Time
 * This is used for SEL and for SDRR
 */
int pp_sel_cmd_get_sel_sdrr_time(imsg_t *imsg)
{
    unsigned int t = cpu_to_le32(time(NULL));
    return pp_bmc_router_resp_msg(imsg, IPMI_ERR_SUCCESS, &t, sizeof(t));
}
/*
 * Set SEL Time
 * This is used for SEL and for SDRR. It must consequently be used
 * for the event_receiver and alerting functions as well. Set local
 * time to achieve that. Do not set local time if we run ntp.
 */
int pp_sel_cmd_set_sel_sdrr_time(imsg_t *imsg)
{
    char* time_type_str;
    time_t ipmi_time;
    struct timeval ipmi_reftime;
    
    pp_bmc_log_info("[SEL] received set time command");
    pp_cfg_get(&time_type_str, "time.proto._c_");
    if (strcmp(time_type_str, "local") == 0) {
        /* adjust (set) time on KIM */
        ipmi_time = le32_to_cpu(*(unsigned int*)imsg->data);

        ipmi_reftime.tv_sec = ipmi_time;
        ipmi_reftime.tv_usec = 0;

        settimeofday(&ipmi_reftime, NULL);
        pp_set_hardware_clock_exact(ipmi_time, ipmi_reftime);
    }
    free(time_type_str);

    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}


/*
 * internal helper, called to notify the device of new entries in the
 * sel_nv that invalidate existing reservations
 */
static void sel_add_hndl(unsigned short id UNUSED) {
    reserv_cancel(sel.reserv);
}


/********************************************************************
 * SEL device command table
 */

static const dev_cmd_entry_t sel_cmd_tab[] = {
    {
        .cmd_hndlr = sel_cmd_get_sel_info,
        .netfn = IPMI_NETFN_STORAGE, .cmd = IPMI_CMD_GET_SEL_INFO,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_USER
    },
    {
        .cmd_hndlr = sel_cmd_get_sel_alloc_info,
        .netfn = IPMI_NETFN_STORAGE, .cmd = IPMI_CMD_GET_SEL_ALLOC_INFO,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_USER
    },
    {
        .cmd_hndlr = sel_cmd_reserve_sel,
        .netfn = IPMI_NETFN_STORAGE, .cmd = IPMI_CMD_RESERVE_SEL,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_OPERATOR
    },
    {
        .cmd_hndlr = sel_cmd_get_sel_entry,
        .netfn = IPMI_NETFN_STORAGE, .cmd = IPMI_CMD_GET_SEL_ENTRY,
        .min_data_size = sizeof(get_sel_entry_rq_t), .min_priv_level = IPMI_PRIV_USER
    },
    {
        .cmd_hndlr = sel_cmd_add_sel_entry,
        .netfn = IPMI_NETFN_STORAGE, .cmd = IPMI_CMD_ADD_SEL_ENTRY,
        .min_data_size = sizeof(sel_entry_t), .min_priv_level = IPMI_PRIV_OPERATOR
    },
    {
        .cmd_hndlr = sel_cmd_clear_sel,
        .netfn = IPMI_NETFN_STORAGE, .cmd = IPMI_CMD_CLEAR_SEL,
        .min_data_size = sizeof(clear_sel_rq_t), .min_priv_level = IPMI_PRIV_OPERATOR
    },
    {
        .cmd_hndlr = pp_sel_cmd_get_sel_sdrr_time,
        .netfn = IPMI_NETFN_STORAGE, .cmd = IPMI_CMD_GET_SEL_TIME,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_USER
    },
    {
        .cmd_hndlr = pp_sel_cmd_set_sel_sdrr_time,
        .netfn = IPMI_NETFN_STORAGE, .cmd = IPMI_CMD_SET_SEL_TIME,
        .min_data_size = sizeof(int), .min_priv_level = IPMI_PRIV_OPERATOR
    },
    { .cmd_hndlr = NULL }
};

/********************************************************************
 * SEL device c'tor/d'tor
 */

int pp_bmc_dev_sel_init()
{
    sel.reserv = reserv_new();

    if (PP_FAILED(sel_nv_init())) {
        pp_bmc_log_error("[SEL] could not initialize SEL in non volatile memory");
        return PP_ERR;
    }

    if (sel_nv_complete_check() == PP_ERR) {
        if (sel_nv_create() == PP_ERR) {
            pp_bmc_log_error("[SEL] could not create SEL in non volatile memory");
            return PP_ERR;
        }
    }

    /* register all entries of cmd tab */
    if (PP_ERR == pp_bmc_core_reg_cmd_tab(sel_cmd_tab)) return PP_ERR;

    /*
     * IPMI spec demands that timestamps are power-on time related until
     * the time is set explicitly by SMS.
     * We don't do this, because we have a real-time clock.
     */
    /* timestamp based on SEL startup */
    /* time_offs = time(NULL); */

    sel_listener_hndl_t slh;
    slh.add = sel_add_hndl;
    slh.del = NULL;    // all del events are initiated by us, no need to listen
    slh.clear = NULL;  // all clear events are initiated by us, no need to listen
    sel_set_listener_sel_dev(slh);

    sel_enabled = 1;
    pp_bmc_log_info("[SEL] device started");

    return PP_SUC;
}

void pp_bmc_dev_sel_cleanup()
{
    /* unregister all entries of cmd tab */
    pp_bmc_core_unreg_cmd_tab(sel_cmd_tab);

    reserv_delete(sel.reserv); sel.reserv = NULL;

    sel_nv_cleanup();

    pp_bmc_log_info("[SEL] device shut down");
}

void bmc_dev_sel_set_enable(unsigned char value) {
    sel_enabled = value;
}

unsigned char bmc_dev_sel_get_enable(){
    return sel_enabled;
}
