/**
 * event_receiver.c
 *
 * The event receiver is the central point for adding new events. It can
 * be configured to disable events from the PCI management bus and ICMB.
 * Valid events are forwarded to the SEL. If the SEL is full, they are
 * stored in an internal, volatile queue to guarantee correct PEF
 * operation.
 * 
 * TODO:
 * - detect multiple IPMB events
 * - reenable SEL at power_up ?
 * 
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#include <stdlib.h>
#include <pp/bmc/ipmi_chan.h>

#include "event_receiver.h"
#include "bmc_dev_sel.h"
#include "bmc_dev_sel_nv.h"
#include "pp/bmc/debug.h"
#include "pp/bmc/utils.h"
#include "bmc_dev_event_msg_buff.h"



#define EVENT_BUFFER_SIZE  20

#define IPMI_SEL_TYPE_SYSTEM 0x02


/* An internal event ringbuffer for storing events if the sel is full */
sel_entry_t internal_buffer[EVENT_BUFFER_SIZE];
int root_index;    // predecessor of the oldest element
int head_index;    // the newest element

/* global vars */
static sel_listener_hndl_t evtr_listeners = { NULL, NULL, NULL };
static int last_add_time;
static int sel_full = 0;



/* internal prototypes */
static void sel_delete_hndl(unsigned short id, unsigned short next_id);
static void sel_clear_hndl(void);



/**
 * Initialize the event receiver
 */
int pp_bmc_event_receiver_init() {
    sel_listener_hndl_t slh;
    slh.add = NULL;
    slh.del = sel_delete_hndl;
    slh.clear = sel_clear_hndl;
    sel_set_listener_evt_r(slh);
    
    last_add_time = 0;
    
    // initialize the internal message queue
    root_index = 0;
    head_index = 0;
    
    return PP_SUC;
}

/**
 * Clean up the event receiver
 */
void pp_bmc_event_receiver_cleanup() {
    /* we should cleanup the internal message queue and discard remaining events */
}

/* get the next index of the ringbuffer, wrapped around */
static int next_index(int i) {
    if (i+1 == EVENT_BUFFER_SIZE)
        return 0;
    else
        return i+1;
}


/**
 * Receive a new event. The event will be dropped if it originates from
 * PCI Mgmt Bus or ICMB and the SEL is disabled. If the event is valid,
 * it will be stored in the SEL and forwarded to PEF.
 */
void pp_bmc_receive_event(
    unsigned char is_swid, unsigned char addr_swid, unsigned char lun, unsigned char chan,
    unsigned char sensor_type, unsigned char sensor_no,
    unsigned char event_dir, unsigned char event_type,
    unsigned char data1, unsigned char data2, unsigned char data3)
{
    int event_processed = 0;
    time_t t;
    sel_entry_t e;

    /* events coming from SMI must be logged unconditionally
     * (see SetBmcGlobalEnables), for others logging may have
     * been disabled                                           */
#if !defined (PP_FEAT_BMC_OEMCMDS_ONLY)     
    if (!bmc_dev_sel_get_enable() && chan != IPMI_CHAN_SI) {
	return;
    }
#endif
    
    t = time(NULL);
    memset(&e, 0, sizeof(sel_entry_t));
    e.id_le16 = cpu_to_le16(sel_nv_get_next_entry_id());
    e.type   = IPMI_SEL_TYPE_SYSTEM;
    e.sys.timestamp_le32   = cpu_to_le32(t);
    e.sys.is_swid          = is_swid & 0x01;
    e.sys.addr_swid        = addr_swid & 0x7f;
    e.sys.lun              = lun & 0x03;
    e.sys._res1            = 0;
    e.sys.chan             = chan & 0x0f;
    e.sys.rev              = 0x04; /* IPMI 1.5/2.0 */
    e.sys.sensor_type      = sensor_type;
    e.sys.sensor_no        = sensor_no;
    e.sys.event_type       = event_type & 0x7f;
    e.sys.event_dir        = event_dir & 0x01;
    e.sys.event_data[0]    = data1;
    e.sys.event_data[1]    = data2;
    e.sys.event_data[2]    = data3;

    /* add to event message buffer, can fail silently because *
     * event message buffer is full or disabled               */

#if defined (PP_FEAT_IPMI_SERVER_SMI_CHAN) /* no msg buf if no smi */
    // TODO: only add events that do not come from system software
    bmc_dev_event_msg_buf_add(&e);
#endif

    /* add to sel or internal buffer, notify pef (implicitly) */
    if (sel_nv_add_event(&e) == 0xffff) {
        // try to put in internal queue
        int ni = next_index(head_index);
        if (ni != root_index) {
            // enough space for new entry
            internal_buffer[ni] = e;
            head_index = ni;
            event_processed = 1;
        }
    } else {
        event_processed = 1;
        if (sel_full) sel_full = 0; // deassert event dropping warning
    }
    
    if (event_processed) {
        if (evtr_listeners.add) evtr_listeners.add(le16_to_cpu(e.id_le16));
    } else {
        if (!sel_full) {
            pp_bmc_log_warn("[EVT_RCV] SEL and internal buffer full"
                            " - start dropping events");
            sel_full = 1;
        }
    }
        
    last_add_time = t;
}

/**
 * Returns a reference to the specified SEL event.
 * Attention: memory for the event is allocated and must
 * be freed by the caller. Returns NULL if the specified
 * event does not exist.
 */
sel_entry_t* pp_bmc_get_event(unsigned short eventID) {
    sel_entry_t event;
    sel_entry_t* evtP;
    int i;
    
    //pp_bmc_log_debug("[EVT_RCV] get_event (id = %.4x)", eventID);
    
    if (sel_nv_get_event(eventID, &event, NULL) == PP_ERR) {
        // search internal list
        
        i = root_index;
        while (i != head_index) {
            i = next_index(i);
            if (le16_to_cpu(internal_buffer[i].id_le16) == eventID) {
                break;
            }
        }
        if (le16_to_cpu(internal_buffer[i].id_le16) == eventID) {
            event = internal_buffer[i];
        } else {
            return NULL;
        }
    }
    
    evtP = malloc(sizeof(sel_entry_t));
    *evtP = event;
    
    return evtP;
}
    
/**
 * Gets the event with the specified id and returns the
 * id of the next event. Returns 0xFFFF if the specified
 * event is the last event and 0x0000 if the specified
 * event does not exist.
 */
unsigned short pp_bmc_get_next_event_id(unsigned short event_id) {
    unsigned short next_id;
    sel_entry_t event;

    //pp_bmc_log_debug("[EVT_RCV] get_next_event (id = %.4x)", event_id);

    if (sel_nv_get_event(event_id, &event, &next_id) == PP_SUC) {
        if (next_id == 0xffff) {
            // event was last sel event
            if (root_index != head_index) {
                next_id = internal_buffer[next_index(root_index)].id_le16;
                //pp_bmc_log_debug("[EVT_RVC] last event in sel has first event in queue as next (%.4x)", next_id);
            } else {
                //pp_bmc_log_debug("[EVT_RVC] last event in sel has no next event (queue is empty)");
            }
        }
        // else: regular next_id
        //pp_bmc_log_debug("[EVT_RVC] next event in sel is %.4x", next_id);

    } else {
        // search internal list
        int idx;
        next_id = 0x0000; // pre-set to unknown
        idx = root_index;
        while (idx != head_index) {
            idx = next_index(idx);

            if (le16_to_cpu(internal_buffer[idx].id_le16) == event_id) {
                // this is the element we're looking for
                if (idx == head_index) {
                    next_id = 0xffff;
                    //pp_bmc_log_debug("[EVT_RVC] last event in queue has no next event");
                }
                else {
                    next_id = le16_to_cpu(internal_buffer[next_index(idx)].id_le16);
                    //pp_bmc_log_debug("[EVT_RVC] next event for event in queue is %.4x", next_id);
                }
                break;
            }
        }

        //if (next_id == 0x0000) pp_bmc_log_debug("[EVT_RVC] event %.4x is unknown", event_id);
    }

    return next_id;
}

/**
 * The PEF calls this function to notify the event receiver that the
 * specified id is not needed any more. This takes effect only if the
 * SEL is full and events are stored in an internal queue.
 */
int pp_bmc_release_event(unsigned short id) {
    // delete event from internal list if in list

    if (root_index != head_index) {
        // the released element will always be the first element (if still in list)
        if (le16_to_cpu(internal_buffer[next_index(root_index)].id_le16) == id) {
            // delete this element (the first element)
            root_index = next_index(root_index);
        }
    }
    
    return PP_SUC;
}

/**
 * Get the timestamp of the most recent addition to the SEL.
 */
time_t pp_bmc_get_most_recent_addition_timestamp() {
    return last_add_time;
}

/**
 * Set the timestamp of the most recent addition to the SEL.
 */
void pp_bmc_set_most_recent_addition_timestamp(time_t la_time) {
    last_add_time = la_time;
}

/**
 * Get the id of the last SEL record
 */
unsigned short pp_bmc_get_last_event_id() {
    return sel_nv_get_last_event_id();
}

/**
 * Sets one listener that is notified of event changes. This is usually
 * the pef. Multiple calls of this function overwrite the handlers.
 */
void pp_event_set_listener(sel_listener_hndl_t handlers) {
    evtr_listeners.add = handlers.add;
    evtr_listeners.del = handlers.del;
    evtr_listeners.clear = handlers.clear;
}

static void sel_delete_hndl(unsigned short id, unsigned short next_id) {
    // move internal queue element to sel if necessary
    if (root_index != head_index) {
        if (next_id == 0xffff) {
            next_id = le16_to_cpu(internal_buffer[next_index(root_index)].id_le16);
        }
        
        if (sel_nv_add_event(&(internal_buffer[next_index(root_index)])) != 0xffff) {
            pp_bmc_log_debug("[EVT_RCV] moved event %.4x from queue to SEL",
                             le16_to_cpu(internal_buffer[next_index(root_index)].id_le16));
            root_index = next_index(root_index);
        } else {
            pp_bmc_log_debug("[EVT_RCV] cannot move event to SEL");
        }
    }

    if (evtr_listeners.del != NULL)
        evtr_listeners.del(id, next_id);
}

static void sel_clear_hndl(void) {    
    int has_more = 0;
    
    while (root_index != head_index) {
        has_more = 1;
        if (sel_nv_add_event(&(internal_buffer[next_index(root_index)])) != 0xffff) {
            pp_bmc_log_debug("[EVT_RCV] moved event %.4x from queue to SEL",
                             le16_to_cpu(internal_buffer[next_index(root_index)].id_le16));
            root_index = next_index(root_index);
        } else {
            break;
        }
    }

    if ((has_more == 0) && (evtr_listeners.clear != NULL))
        evtr_listeners.clear();
}   

#if 0
// interal debug helper
static void dump_event_buf(void)
{
    pp_bmc_log_notice("[EVT_RCV] dumping internal event buffer...");
    int idx = root_index;
    while (idx != head_index) {
        idx = next_index(idx);
        pp_bmc_log_notice("[EVT_RCV] entry %d: id=%d", idx, le16_to_cpu(internal_buffer[idx].id_le16));
    }
}
#endif
