/**
 * bmc_imsg.h
 *
 * Description: BMC internal IPMI Message Structure and Methods
 *
 * (c) 2004 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 */
 
#ifndef __BMC_IMSG_H__
#define __BMC_IMSG_H__

#include <pp/base.h>
#include <pp/bmc/ipmi_lan.h>

/* include debug.h to get default for PP_BMC_DEBUG_LEVEL if not
 * specified in the Makefile */
#include <pp/bmc/debug.h>

#define IPMI20_STATE_INVALID       0
#define IPMI20_STATE_WAITING_RAKP1 1
#define IPMI20_STATE_WAITING_RAKP3 2
#define IPMI20_STATE_ACTIVATED     3
#define IPMI20_STATE_ACTIVE        4
/* Note: activated = activated but rakp4 message not confirmed   *
 *       active    = activated and used (confirmed, no way back) */

/**
 * This structure contains all information related to ipmi sessions.
 * All pointers are freed
 */
typedef struct imsg_session_s {
    unsigned int   session_id;        // the id assigned to this session
    unsigned char  session_handle;    // 1..63 possible sessions, absolute handle
    unsigned char* password;          // not a string, empty chars contain 0s, may be null for IPMIv20 sessions
    unsigned char  password_len;      // currently only 16 byte passwords are specified
    
    unsigned char  cur_priv_level;    // the priv. lvl assigned to this session
    unsigned char  max_priv_level;    // max priv. lvl allowed on this session
    
    unsigned int   inbound_sequence;  // seq No. from client to bmc
    unsigned int   outbound_sequence; // seq No. from bmc to client
    unsigned int   last_message_seq;  // inbound seq No. of most recent accepted request message.
    unsigned char  authtype;          // the authentication type for this session (none, md2, md5, straight, rmcp+)
                                      // all msgs within this session must be authenticated using this type
                                      // rmcp+ messages must use the negotiated ciphersuites
                                      
    unsigned char* challenge;         // 16 bytes, may be null, do not free

    // RAKP/IPMI 2.0 specific data, unused on other channels
    unsigned char  ipmi20_session_state;   // state of the session (inactive, rakp1, rakp3, activated)
    
    unsigned int   manager_session_id;     // the session id from the console

    unsigned char  authentication_cipher;  // authentication cipher used for this session
    unsigned char  integrity_cipher;       // integrity cipher used for this session
    unsigned char  confidentiality_cipher; // confidentiality cipher used for this session
    unsigned char  req_priv_level;         // requested privilege level, capped by chosen cipher suite
    unsigned char  name_only_lookup;       // set to 0 to indicate a role login (username+privlevel match)

    unsigned char* sik;                    // Session integrity key, length depends on used ciphers, may be NULL
    unsigned char  sik_len;                // length of authcode cipher hash, stored for convenience only
                                           // (same as authentication-hash length)
    unsigned char* k1;                     // same length as sik
    unsigned char* k2;                     // same length as sik
    
    
  //unsigned char* bmc_random;             // 16 bytes, bmc random number used in rakp (use challenge instead)
    unsigned char* manager_random;         // 16 bytes, manager random number used in rakp, may be NULL
    unsigned char* last_message_data;      // binary content of last sent message within this session,
    int            last_message_len;       // may be resent with pp_send_rmcp_msg() to handle incoming
    lan_addr_t     last_message_addr;      // requests with a processed sequence number. 
    
    unsigned char* username;               // the username (for RAKP messages)
    unsigned char  username_len;

    /* sol stuff */
    /* handles ipmi20 sol payload */
    int (*sol_handler)(unsigned char* data, size_t datelen,
                       struct imsg_session_s* session, lan_addr_t* addr);
    
    /* needs to be called for deactivating ipmi20 sol payload handler */
    void (*sol_deactivate)(struct imsg_session_s* session);

    /*
     *  internals
     */
    struct list_head link;            // for internal use of session_manager only
    unsigned short ref_count;         // reference counting. free the session if this value drops to 0
    
    int inactivity_timeout;           // session inactivity timeout handler, -1 = no active timeout
    
    unsigned char chan;               // channel id, the session is opened for
    unsigned char uid;                // user id, the session is opened for
    
    lan_addr_t* addr;                 // lan channel specific data
    unsigned int session_secret;      // IPMI-over-SCSI specific data
    
} imsg_session_t;

/**
 * BMC internal message structure
 */
typedef struct {
    unsigned char netfn;
    unsigned char cmd;

    unsigned char rs_addr;       // I2C slave addr or SWID
    unsigned char rs_lun;
    unsigned char rq_addr;       // I2C slave addr or SWID
    unsigned char rq_lun;
    unsigned char rq_seq;        // provided by client to match request/response
    
    unsigned char *data;         // cmd data (+completion code), may be NULL
    unsigned int data_size;
    
    unsigned char chan;          // the channel this msg originated
    
    unsigned char *buf;          // additional context data (address information)
    unsigned int buf_size;       // from channel adapters, may be NULL, release with free() if present

    imsg_session_t* session;     // session data, may be NULL for unencrypted messages outside a session
    unsigned int session_secret; // IPMI-over-SCSI specific data
                                 
    unsigned char priv_level;    // privilege level of this message
                                 // (copied from session if session != NULL)
                                 // always allowed maximum for sessionless channels (SI, IPMB)
                                 // always unspec for sessionless messages from sessionbased channels
} imsg_t;


/**
 * Create a new empty session struct.
 */
imsg_session_t* pp_bmc_session_new(void);

/**
 * Release the reference to the specified session. If the reference count
 * drops to 0 the session and all subdata (passwords, challenges, ...) are
 * deleted.
 */
void pp_bmc_session_delete(imsg_session_t* session);

/**
 * Clear the buffer with the last sent message. This will free any
 * allocated memory. Usually only used before storing a new 'last
 * message' or before deleting the session.
 */
void pp_bmc_session_clear_last_message(imsg_session_t* session);

/**
 * Creates a new IPMI message type including body.
 * The header and the databuffer of the message will
 * be initialized to zero.
 *
 * @param data_size Size in bytes of body to be allocated
 * @return Pointer to new message
 * @see impi_msg_delete()
 */
imsg_t* pp_bmc_imsg_new(unsigned int data_size);

/**
 * Deletes an IPMI message and frees memory.
 *
 * @param ipmi_msg_ptr addr of message pointer (point is NULL on exit)
 * @see impi_msg_new()
 */
void pp_bmc_imsg_delete(imsg_t *imsg_ptr);

/**
 * flags to be supplied to pp_bmc_imsg_copy
 * @see ipp_bmc_msg_copy
 */
#define IMSG_COPY_SHALLOW 0x0
#define IMSG_COPY_DATA    0x1

/**
 * copies an IMPI message
 * ATTENTION: buf is never ever copied, currently
 *            (not sure whether this will do, if not we may think
 *            again about reference counting)
 * @param flags determines whether deep copy or not
 * @return pointer to copied message
 */
imsg_t* pp_bmc_imsg_copy(imsg_t* imsg, int flags);

/**
 * Converts a command (request) into a response message.
 * The response databuffer of the message will be initialized to zero.
 *
 * @param ipmi_msg_ptr Command message
 * @param resp_data_size Size in bytes of body to be allocated
                         (1 is added automatically for completion code)
 * @param compl_code Error code copied to data[0]
 * @return returns pointer to actual response data (&data[1])
 */
void* pp_bmc_imsg_resp(imsg_t *imsg,
                unsigned char compl_code,
                unsigned int resp_data_size);

/**
 * Converts an already allocated response into a response error.
 * The message should be given to the msg-router after this and
 * not used any further.
 * 
 * @param ipmi_msg_ptr Command message
 * @param compl_code Error code copied to data[0]
 */ 
void pp_bmc_imsg_err(imsg_t* imsg, unsigned char compl_code);

/**
 * Converts an IPMI message into a readable string (e.g. for logging).
 *
 * @param ipmi_msg The message
 * @param buf String buffer
 * @param buf_size Allocated size of string buffer
 * @return Pointer to buf
 */
char* pp_bmc_imsg_to_str(const imsg_t *imsg, char* buf, int buf_size);

#if PP_BMC_DEBUG_LEVEL>0
/**
 * dumps an msg
 */
void pp_bmc_imsg_dump(int dbg_level, const char* originator, const imsg_t *imsg);
#else
#  define pp_bmc_imsg_dump(...)
#endif

#endif /* !__BMC_IMSG_H__ */
